/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2266 $
|                     $Date: 2009-04-16 09:36:11 +0200 (jeu, 16 avr 2009) $
\*============================================================================*/

/*******************************************************************************
*                               DPWS RUNTIME                                   *
*******************************************************************************/
#include "dcDPWS_Dpws.h"
#include "dc/dc_DpwsRequest.h"
#include "dc/dc_Runtime.h"
#include "dcDPWS_Memory.h"
#include "dcDPWS_Discovery.h"
#include "dcDPWS_Metadata.h"
#include "dcCOMN_Tools.h"
#include "dcDCPL_Os.h"
#include "dcDCPL_Net.h"
#include "dcDCPL_Socket.h"
#include "dcCOMN_Uuid.h"
#include "dcDPWS_Network.h"
#include "dcDPWS_Configuration.h"
#include "dcDPWS_Registry.h"
#include "dcDPWS_Cache.h"
#include "dcDPWS_SubscManager.h"
#include "dcDPWS_Event.h"
#include "dcGSOAP_Runtime.h"
#include "dc/dc_Plugin.h"
#include "dcDPWS_Utils.h"
#include "dcDPWS_Http.h"
#include "dcDPWS_Udp.h"

reactor_t reactor;
dcpl_mutex_t * reactor_lock = NULL;
handlePool_t reactor_handle_pool;

// common namespace table for headers
struct Namespace dpws10_default_namespaces[] =
{
  {SOAP_ENV_PREFIX, SOAP_ENV_URI, SOAP_ENV_WILDCARD, NULL},
  {SOAP_ENC_PREFIX, SOAP_ENC_URI, SOAP_ENC_WILDCARD, NULL},
  {XSI_PREFIX, XSI_URI, XSI_WILDCARD, NULL},
  {WSA_PREFIX, WSA_200408_URI, WSA_WILDCARD, NULL},
  {WDP_PREFIX, DPWS10_WDP_URI, WDP_WILDCARD, NULL},
  {WSD_PREFIX, DPWS10_WSD_URI, WSD_WILDCARD, NULL},
  {WSE_PREFIX, WSE_URI, WSE_WILDCARD, NULL},
  {WSM_PREFIX, WSM_URI, WSM_WILDCARD, NULL},
  {WST_PREFIX, WST_URI, WST_WILDCARD, NULL},
#ifdef WITH_WSMAN
  {WSMAN_PREFIX, WSMAN_URI, WSMAN_WILDCARD, NULL},
#endif
  {NULL, NULL, NULL, NULL}
};

struct Namespace dpws11_default_namespaces[] =
{
  {SOAP_ENV_PREFIX, SOAP_ENV_URI, SOAP_ENV_WILDCARD, NULL},
  {SOAP_ENC_PREFIX, SOAP_ENC_URI, SOAP_ENC_WILDCARD, NULL},
  {XSI_PREFIX, XSI_URI, XSI_WILDCARD, NULL},
  {WSA_PREFIX, WSA_200508_URI, WSA_WILDCARD, NULL},
  {WDP_PREFIX, DPWS11_WDP_URI, WDP_WILDCARD, NULL},
  {WSD_PREFIX, DPWS11_WSD_URI, WSD_WILDCARD, NULL},
  {WSE_PREFIX, WSE_URI, WSE_WILDCARD, NULL},
  {WSM_PREFIX, WSM_URI, WSM_WILDCARD, NULL},
  {WST_PREFIX, WST_URI, WST_WILDCARD, NULL},
#ifdef WITH_WSMAN
  {WSMAN_PREFIX, WSMAN_URI, WSMAN_WILDCARD, NULL},
#endif
  {NULL, NULL, NULL, NULL}
};

struct wsa_version wsa_200408_version = {
	WSA_200408_URI,
	WSA_200408_ANONYMOUS_ENDPOINT_REF_URI,
	WSA_200408_SOAP_FAULT_URI,
};

struct wsa_version wsa_200508_version = {
	WSA_200508_URI,
	WSA_200508_ANONYMOUS_ENDPOINT_REF_URI,
	WSA_200508_SOAP_FAULT_URI,
};

struct dpws_protocols dpws10_protocols = {
	&wsa_200408_version,
	DPWS10_WDP_URI,
	DPWS10_WDP_MODEL_URI,
	DPWS10_WDP_DEVICE_URI,
	DPWS10_WDP_HOSTING_URI,
	DPWS10_WDP_RELATIONSHIP_HOST_TYPE_URI,
	DPWS10_WSE_FILTERING_DIALECT,
	DPWS10_WSD_URN,
	DPWS10_WSD_URI,
	DPWS10_WSD_HELLO_ACTION,
	DPWS10_WSD_BYE_ACTION,
	DPWS10_WSD_PROBE_ACTION,
	DPWS10_WSD_RESOLVE_ACTION,
	DPWS10_WSD_PROBEMATCHES_ACTION,
	DPWS10_WSD_RESOLVEMATCHES_ACTION,
	DPWS10_WSD_MATCH_BY_RFC2396_URI,
	DPWS10_WSD_MATCH_BY_UUID_URI,
	DPWS10_WSD_MATCH_BY_LDAP_URI,
	DPWS10_WSD_MATCH_BY_STRCMP_URI,
	WSM_URI,
	WST_URI,
	WST_GET_ACTION,
	WST_GET_RESPONSE_ACTION,
	WSE_URI,
	WSE_SUBSCRIBE_ACTION,
	WSE_SUBSCRIBE_RESPONSE_ACTION,
	WSE_SUBSCRIPTION_END_ACTION,
	WSE_RENEW_ACTION,
	WSE_RENEW_RESPONSE_ACTION,
	WSE_UNSUBSCRIBE_ACTION,
	WSE_UNSUBSCRIBE_RESPONSE_ACTION,
	WSE_GET_STATUS_ACTION,
	WSE_GET_STATUS_RESPONSE_ACTION,
	dpws10_default_namespaces,
	dpws10_discovery_snd_namespaces,
	dpws10_transfer_snd_namespaces,
	dpws10_eventing_snd_namespaces
};

struct dpws_protocols dpws11_protocols = {
	&wsa_200508_version,
	DPWS11_WDP_URI,
	DPWS11_WDP_MODEL_URI,
	DPWS11_WDP_DEVICE_URI,
	DPWS11_WDP_HOSTING_URI,
	DPWS11_WDP_RELATIONSHIP_HOST_TYPE_URI,
	DPWS11_WSE_FILTERING_DIALECT,
	DPWS11_WSD_URN,
	DPWS11_WSD_URI,
	DPWS11_WSD_HELLO_ACTION,
	DPWS11_WSD_BYE_ACTION,
	DPWS11_WSD_PROBE_ACTION,
	DPWS11_WSD_RESOLVE_ACTION,
	DPWS11_WSD_PROBEMATCHES_ACTION,
	DPWS11_WSD_RESOLVEMATCHES_ACTION,
	DPWS11_WSD_MATCH_BY_RFC2396_URI,
	DPWS11_WSD_MATCH_BY_UUID_URI,
	DPWS11_WSD_MATCH_BY_LDAP_URI,
	DPWS11_WSD_MATCH_BY_STRCMP_URI,
	WSM_URI,
	WST_URI,
	WST_GET_ACTION,
	WST_GET_RESPONSE_ACTION,
	WSE_URI,
	WSE_SUBSCRIBE_ACTION,
	WSE_SUBSCRIBE_RESPONSE_ACTION,
	WSE_SUBSCRIPTION_END_ACTION,
	WSE_RENEW_ACTION,
	WSE_RENEW_RESPONSE_ACTION,
	WSE_UNSUBSCRIBE_ACTION,
	WSE_UNSUBSCRIBE_RESPONSE_ACTION,
	WSE_GET_STATUS_ACTION,
	WSE_GET_STATUS_RESPONSE_ACTION,
	dpws11_default_namespaces,
	dpws11_discovery_snd_namespaces,
	dpws11_transfer_snd_namespaces,
	dpws11_eventing_snd_namespaces
};

struct dpws_protocols * dpws_protocols[N_DPWS_VERSIONS];
struct dpws_protocols * default_protocols;
struct wsa_version * wsa_version[N_DPWS_VERSIONS];
int n_dpws_versions = 0;
int n_wsa_versions = 0;

extern const char ** SOAP_FMAC4 soap_faultsubcode(struct soap *);

#ifdef DPWS_DEBUG
	FILE *fdebug[3];
#endif

/*-------------------------------------- Types -------------------------------*/

#ifdef WITH_WSMAN
struct wse_sm_plugin default_sm_plugin = {
	wse_create_subsc_manager,
	wse_renew_subsc,
	wse_subsc_status,
	wse_delete_subscription
};
#endif


struct probe_send_info {
	reactor_t reactor;
	DCPL_SOCKET s4;
	DCPL_SOCKET s6;
	discovery_filter_t * filter;
	int f_size;
	struct send_probe_data * w_prb_data;
	struct dpws_protocols ** protocols;
	int p_size;
};

struct send_probe_data {
	DCPL_SOCKET socket;
	dcpl_ip_addr_t * netif_addr;
	char message_uuid[SCHEME_UUID_STRING_SIZE];
	discovery_filter_t * filter;
	struct dpws_protocols * protocols;
};

struct recv_probematch_data {
	int res_size;
	discovery_filter_t * filter;
	int f_size;
};

struct send_resolve_data {
	DCPL_SOCKET socket;
	dcpl_ip_addr_t * netif_addr;
	char message_uuid[SCHEME_UUID_STRING_SIZE];
	struct wsa_endpoint_ref endpoint;
	struct dpws_protocols * protocols;
};

struct recv_resolvematch_data {
	char * id;
	struct device_proxy * cache_entry;
};

struct resolve_send_info {
	reactor_t reactor;
	DCPL_SOCKET s4;
	DCPL_SOCKET s6;
	char * device_uuid;
	struct send_resolve_data * w_rslv_data;
	struct dpws_protocols ** protocols;
	int p_size;
};

DA_TYPED_DECL(wsmansel, struct wsman_selector);

/*------------------------- Static Functions prototypes ----------------------*/

static struct device_proxy* lookup_by_id(struct dpws* dpws, char * id, unsigned int time_out);
static struct dpws * dpws_in_wsd_AppSequence(struct dpws *dpws);
static int dpws_in_wsa_endpoint_ref(struct soap *soap, const char *tag, struct wsa_endpoint_ref *a);
static int generate_service_port(short * href, struct soap * soap);
static int dpws_http_get(struct soap* soap);
static int dpws_close_callback(struct soap * soap);
static int dpws_send_callback(struct soap*, const char*, size_t);
static size_t dpws_recv_callback(struct soap*, char*, size_t);
static void dpws_struct_init(struct dpws* dpws);
static int buffer_send(struct soap *soap, const char *s, size_t n);
static void buffer_marshalling_init(struct dpws *dpws);
static char * buffer_marshalling_end(struct dpws *dpws);
static int spawn_resolve(struct dpws *dpws, const char ** phys_addr, const char * log_addr);
static int dpws_handle_fault(struct dpws *dpws, int (* ffault)(struct dpws*));
static int convert_reactor_error(struct dpws * dpws, struct reactor * reactor, int error);
static int _init_ex(dc_ip_filter_t * sel, const char * hostname, int versions, DC_BOOL mono_netif);
#ifndef DCPL_HAVE_PKTINFO
static int create_multicast_listener_cbk(dcpl_ip_addr_t * netif_addr, void * cbk_data);
#endif
static int stop_server(DC_BOOL blocking, long delay_ms);
static struct dpws* dpws_copy_context(struct dpws* dest, struct dpws* src);
static int accept_msg(struct dpws * dpws, struct dpws * dpws_s);
static int init_dpws_filter(struct dpws * dpws, discovery_filter_t * user_filter, discovery_filter_t * dpws_filter);
static int send_probe_callback(struct dpws* dpws, struct reactor_item * item, struct send_probe_data * callback_data);
static int recv_probematch_callback(struct dpws* dpws, struct reactor_item * item, struct recv_probematch_data * callback_data);
static int schedule_probes_cbk(dcpl_ip_addr_t * netif_addr, struct probe_send_info * psinfo);
static short * lookup_ex(struct dpws* dpws, discovery_filter_t * filter, int f_size, int * res_size, unsigned int time_out, cache_mode_t mode);
static int send_resolve_callback(struct dpws* dpws, struct reactor_item * item, struct send_resolve_data * callback_data);
static int recv_resolvematch_callback(struct dpws* dpws, struct reactor_item * item, struct recv_resolvematch_data * callback_data);
static int schedule_resolves_cbk(dcpl_ip_addr_t * netif_addr, struct resolve_send_info * rsinfo);
static struct device_proxy* lookup_by_id(struct dpws* dpws, char * id, unsigned int time_out);
static short _probe_address_ex(struct dpws* dpws, char * addr, discovery_filter_t * filter);
static struct device_proxy * fill_device_services(struct dpws* dpws, short hrefDevice);
static int wsman_selector_set_in(struct dpws* dpws);
static int wsman_xpath_in(struct dpws* dpws, const char * tag);
static int dpws_is_oneway(struct dpws* dpws);
static int dpws_set_fault(struct dpws *dpws);
static struct media_type_param * set_content_type_param(struct media_type_param * param, const char * att, const char * value, struct media_type_param * next);
static void get_content_type(struct dpws * dpws, int status, struct media_type * mtype, struct media_type_param * params, int nparams);
static int dpws_use_anonymous_response(struct dpws* dpws);

/*******************************************************************************
*                         VERSION MANAGEMENT                                   *
*******************************************************************************/

int init_versions(int dpws_versions)
{
	if (dpws_versions & DPWS10_VERSION) {
		dpws_protocols[n_dpws_versions++] = &dpws10_protocols;
		wsa_version[n_wsa_versions++] = dpws10_protocols.wsa_version;
		if (DPWS_DEFAULT_VERSION == DPWS10_VERSION)
			default_protocols = &dpws10_protocols;
	}
	if (dpws_versions & DPWS11_VERSION) {
		dpws_protocols[n_dpws_versions++] = &dpws11_protocols;
		wsa_version[n_wsa_versions++] = dpws11_protocols.wsa_version;
		if (DPWS_DEFAULT_VERSION == DPWS11_VERSION)
			default_protocols = &dpws11_protocols;
	}
	if (n_dpws_versions == 0)
		return DPWS_ERR_UNSUPPORTED_VERSION;
	if (!default_protocols)
		default_protocols = dpws_protocols[0];
	return 0;
}

int set_wsa_version(struct dpws * dpws, const char* wsa_uri)
{
	int i;
	for (i = 0; i < n_wsa_versions; i++)
		if (!strcmp(wsa_version[i]->wsa_uri, wsa_uri)) {
			dpws->wsa_version = wsa_version[i];
			return 0;
		}
	return -1;
}

int get_protocols_versions(struct dpws * dpws, struct dpws_protocols ** protocols, int maxsize)
{
	int i, ret = 0;
	if (dpws && dpws->protocols) {
		protocols[0] = dpws->protocols;
		return 1;
	}
	for (i = 0; i < n_dpws_versions; i++) {
		if (!dpws || !dpws->wsa_version || dpws_protocols[i]->wsa_version == dpws->wsa_version) {
			protocols[ret++] = dpws_protocols[i];
			if (ret == maxsize) return ret;
		}
	}
	return ret;
}

int is_discovery_urn(struct dpws * dpws, const char * disco_urn)
{
	struct dpws_protocols * protocols[N_DPWS_VERSIONS];
	int i, size;
	size = get_protocols_versions(dpws, protocols, N_DPWS_VERSIONS);
	for (i = 0; i < size; i++) {
		if (!strcmp(protocols[i]->wsd_urn, disco_urn)) {
			dpws->protocols = protocols[i];
			return 1;
		}
	}
	return 0;
}

int dpws_is_subscribe_action(struct dpws * dpws, const char * subscribe_action)
{
	struct dpws_protocols * protocols[N_DPWS_VERSIONS];
	int i, size;
	size = get_protocols_versions(dpws, protocols, N_DPWS_VERSIONS);
	for (i = 0; i < size; i++) {
		if (!strcmp(protocols[i]->wse_subscribe_action, subscribe_action)) {
			dpws->protocols = protocols[i];
			return 1;
		}
	}
	return 0;
}

/*******************************************************************************
*                              PRIVATE STUFF                                   *
*******************************************************************************/
static struct dpws * dpws_in_wsd_AppSequence(struct dpws *dpws)
{
    struct soap *soap = dpws_dpws2soap(dpws);
    if (soap_element_begin_in(soap, "wsd:AppSequence", 0))
        return NULL;
    if (soap_s2unsignedLong(soap, soap_attr_value(soap, "InstanceId", 1 /* REQUIRED */), &dpws->instance_id))
        return NULL;
    if (soap_s2unsignedShort(soap, soap_attr_value(soap, "MessageNumber", 1), &dpws->msg_nb))
        return NULL;
    if (soap->body && soap_element_end_in(soap, "wsd:AppSequence"))
        return NULL;
    return dpws;
}

int dpws_out_wsa_endpoint_ref(struct soap *soap, const char *tag, struct wsa_endpoint_ref *a)
{
    return soap_out_endpoint_ref(soap, tag, -1, &a, NULL);
}

static int dpws_in_wsa_endpoint_ref(struct soap *soap, const char *tag, struct wsa_endpoint_ref *a)
{
    struct wsa_endpoint_ref ** ret = soap_in_endpoint_ref(soap, tag, &a, NULL);
    return ret ? DPWS_OK : -1;
}

void dpws_set_anonymous_wsa_endpoint_ref(struct dpws * dpws, struct wsa_endpoint_ref *a)
{
	a->address = dpws->wsa_version->wsa_anonymous_uri;
    a->subscription_id = NULL;
#ifdef WITH_WSMAN
	memset(&a->wsman_params, 0, sizeof(struct wsman_ref_params));
#endif
	a->dpws_version = NULL;
}

static int generate_service_port(short * href, struct soap * soap)
{
	char * XAddr;
	if (!(XAddr = (char*)get_transport_address(dpws_soap2dpws(soap), soap->host, *href))) {
		soap->error = SOAP_EOM;
		return DC_TRUE;
	}
    soap_send(soap, "<li>");
    soap_send(soap, XAddr);
    soap_send(soap, "</li>");
    return DC_FALSE;
}

static int dpws_http_get(struct soap* soap)
{
    char uuid[SCHEME_UUID_STRING_SIZE];
    char* context_path = strchr(soap->path, '/');
    struct service_endpoint* endpoint;
    struct service_port * port = NULL;
	struct dpws * dpws = dpws_soap2dpws(soap);
    strcpy(uuid, UUID_SCHEME);
    strncpy(uuid + UUID_SCHEME_SIZE, context_path + 1, UUID_SIZE);
    uuid[SCHEME_UUID_STRING_SIZE-1] = 0;
    dpws_response(dpws, SOAP_HTML); // HTTP response header with text/html
    soap_send(soap, "<html><head><title>");
    soap_send(soap, "Device presentation page");
    soap_send(soap, "</title></head><body>");
    port = find_endpoint_port(uuid);
    if (port)
    {
        short hrefDevice;
        int len = 10, i;
        short hrefs[10];
        DA_TYPED(pqn) types, *ptypes;
        DA_TYPED(str)* scopes;
        endpoint = port->service;
        hrefDevice = endpoint->device->href;
        soap_send(soap, "<h1>");
        soap_send(soap, ((struct localized_string *)dpws_get_ptr_att_item(hrefDevice, DPWS_PTR_FRIENDLY_NAME, 0))->s);
        soap_send(soap, "</h1>");
        soap_send(soap, "<h2>Model information</h2>");
        soap_send(soap, "<p><b>Manufacturer: </b><a href='");
        soap_send(soap, (char*)dpws_get_ptr_att(hrefDevice, DPWS_STR_MANUFACTURER_URL));
        soap_send(soap, "'>");
        soap_send(soap, ((struct localized_string *)dpws_get_ptr_att_item(hrefDevice, DPWS_PTR_MANUFACTURER, 0))->s);
        soap_send(soap, "</a></p>");
        soap_send(soap, "<p><b>Model name: </b><a href='");
        soap_send(soap, (char*)dpws_get_ptr_att(hrefDevice, DPWS_STR_MODEL_URL));
        soap_send(soap, "'>");
        soap_send(soap, ((struct localized_string *)dpws_get_ptr_att_item(hrefDevice, DPWS_PTR_MODEL_NAME, 0))->s);
        soap_send(soap, "</a></p>");
        soap_send(soap, "<p><b>Model number: </b>");
        soap_send(soap, (char*)dpws_get_ptr_att(hrefDevice, DPWS_STR_MODEL_NUMBER));
        soap_send(soap, "</p>");
        soap_send(soap, "<h2>Device discovery information</h2>");
        soap_send(soap, "<p><b>Logical address: </b>");
        soap_send(soap, uuid);
        soap_send(soap, "</p>");
        soap_send(soap, "<p><b>Supported types: </b></p><ul>");
        ptypes = get_device_types(dpws, endpoint->device, &types, DC_TRUE);
        for (i = 0; i < ptypes->nb; i++)
        {
            struct prefixed_qname * qname = DA_GET(ptypes, i);
            soap_send(soap, "<li>{");
            soap_send(soap, qname->qname.ns);
            soap_send(soap, "}");
            soap_send(soap, qname->qname.lname);
            soap_send(soap, "</li>");
        }
        soap_send(soap, "</ul>");
        scopes = &endpoint->device->scopes;
        if (scopes->nb > 0)
            soap_send(soap, "<p><b>Scopes: </b></p><ul>");
        for (i = 0; i < scopes->nb; i++)
        {
            char * scope = *DA_GET(scopes, i);
            soap_send(soap, "<li>");
            soap_send(soap, scope);
            soap_send(soap, "</li>");
        }
        soap_send(soap, "</ul>");
        soap_send(soap, "<h2>Troubleshooting information</h2>");
        soap_send(soap, "<p><b>Serial number: </b>");
        soap_send(soap, (char*)dpws_get_ptr_att(hrefDevice, DPWS_STR_SERIAL_NUMBER));
        soap_send(soap, "</p>");
        soap_send(soap, "<p><b>Firmware version: </b>");
        soap_send(soap, (char*)dpws_get_ptr_att(hrefDevice, DPWS_STR_FIRMWARE_VERSION));
        soap_send(soap, "</p>");
        if (!dpws_get_service_handles(hrefDevice, hrefs, &len))
        {
            soap_send(soap, "<h2>Hosted services</h2>");
            for (i = 0; i < len; i++)
            {
                int j, k, n;
                struct service_endpoint * service = get_endpoint(hrefs[i]);
                struct wsdl_info * info;
                ptypes = get_service_types(service);
                soap_send(soap, "<h3>Service ");
                soap_send(soap, (char*)dpws_get_ptr_att(hrefs[i], DPWS_STR_SERVICE_ID));
                soap_send(soap, "</h3>");
                soap_send(soap, "<p><b>Addresses: </b></p><ul>");
                if (browse_service_ports(endpoint->href, (service_port_cbk)generate_service_port, soap))
                	return soap->error;
                soap_send(soap, "</ul>");
                soap_send(soap, "<p><b>WSDL documents: </b><ul>");
                n = dpws_get_att_count(service->hrefServiceClass, DPWS_PTR_WSDL);
                for (j = 0; j < n; j++)
                {
                	info = (struct wsdl_info *)dpws_get_ptr_att_item(service->hrefServiceClass, DPWS_PTR_WSDL, j);
                    soap_send(soap, "<li><b>Target namespace:  </b>");
                    soap_send(soap, info->target_ns);
                    soap_send(soap, "</li><ul><li>Location: <a href='");
                    soap_send(soap, info->location);
                    soap_send(soap, "'>");
                    soap_send(soap, info->location);
                    soap_send(soap, "</a></li><li>Supported port types: ");
                    for (k = 0; k < ptypes->nb; k++)
                    {
                        struct prefixed_qname * qname = DA_GET(ptypes, k);
                        if (!strcmp(info->target_ns, qname->qname.ns))
                        {
                            soap_send(soap, qname->qname.lname);
                            soap_send(soap, " ");
                        }
                    }
                    soap_send(soap, "</ul></li>");
                }
                soap_send(soap, "</ul>");
            }
        }
        release_endpoint_port(port);
   }
    else
        soap_send(soap, "<h1>Unknown device<h1>");

    soap_send(soap, "</body></html>");
    soap_end_send(soap);
    return SOAP_OK;
}

static int dpws_close_callback(struct soap * soap)
{
	struct dpws * dpws = dpws_soap2dpws(soap);
	if (!dpws->transport_fns) { // Not using the external transport mode
		transport_data_t * tdata = dpws->transport_data;
		if (tdata)
			return tdata->tclass->fclose(dpws);
	}
	return SOAP_OK;
}

static int dpws_send_callback(struct soap* soap, const char* buf, size_t buflen)
{
	struct dpws * dpws = dpws_soap2dpws(soap);
	transport_data_t * tdata = NULL;
	int nwritten;

	if (dpws->transport_fns) { // External transport mode
		nwritten = dpws->transport_fns->fsend(dpws, dpws->transport_data, buf, buflen);
	} else {
		tdata = (transport_data_t *)dpws->transport_data;
		nwritten = tdata->tclass->fsend(dpws, tdata, buf, buflen);
	}
	if (nwritten < 0) {
		if (tdata)
			soap->errnum = tdata->error.syserr;
		return SOAP_EOF;
	}
	return SOAP_OK;
}

static size_t dpws_recv_callback(struct soap* soap, char* buf, size_t buflen)
{
	struct dpws * dpws = dpws_soap2dpws(soap);
	transport_data_t * tdata = NULL;
	int nread;

	if (dpws->transport_fns) { // External transport mode
		nread = dpws->transport_fns->frecv(dpws, dpws->transport_data, buf, buflen);
	} else {
		tdata = (transport_data_t *)dpws->transport_data;
		nread = tdata->tclass->frecv(dpws, tdata, buf, buflen);
	}
	if (nread < 0) {
		if (tdata)
			soap->errnum = tdata->error.syserr;
		return 0;
	}
	return nread;
}

static void dpws_struct_init(struct dpws* dpws)
{
    memset(dpws, 0, sizeof(struct dpws));
    soap_init(dpws_dpws2soap(dpws));
    dpws_init_headers_and_version(dpws);	// also reset status
    if (!dpws->soap.namespaces)
        dpws->soap.namespaces = default_protocols->default_namespaces;
    dpws->err_detail_type = -1;
    // Tell the SOAP runtime that we are in DPWS mode
    dpws->soap.imode |= DPWS_HEADERS;
    dpws->soap.omode |= DPWS_HEADERS;
    // Add support for device page display
    dpws->soap.fget = dpws_http_get;
	dpws->soap.frecv = dpws_recv_callback;
	dpws->soap.fsend = dpws_send_callback;
	dpws->soap.fclose = dpws_close_callback;
	dpws->soap.fclosesocket = NULL;
	dpws->soap.fshutdownsocket = NULL;
	dpws->soap.fopen = NULL;
	dpws->soap.fconnect = NULL;
	dpws->soap.fresolve = NULL;
	dpws->soap.faccept = NULL;
	dpws->soap.fparse = NULL;
	dpws->soap.fparsehdr = NULL;
	dpws->soap.fpost = NULL;
	dpws->soap.fposthdr = NULL;
	dpws->soap.fresponse = NULL;
}

static int buffer_send(struct soap *soap, const char *s, size_t n)
{
    n = MIN(n, (size_t)(1024 - dpws_soap2dpws(soap)->tmp_buf_count));
    strncat(soap->msgbuf, s, n);
    dpws_soap2dpws(soap)->tmp_buf_count += n;
    return SOAP_OK;
}

static void buffer_marshalling_init(struct dpws *dpws)
{
    dpws->tmp_buf_count = 0;
    dpws->backup_send_f = dpws_dpws2soap(dpws)->fsend;
    dpws_dpws2soap(dpws)->fsend = &buffer_send;	// QUESTION: is it correct ? Aren't flags modified ? A dedicated SOAP structure would be costly.
    strcpy(dpws_dpws2soap(dpws)->msgbuf, "<![CDATA[");
}

static char * buffer_marshalling_end(struct dpws *dpws)
{
    char *buf = dpws_dpws2soap(dpws)->msgbuf;
    buf += MIN(1018, dpws->tmp_buf_count);	// 1024 - 6
    strcpy(buf, "...]]>");
    dpws_dpws2soap(dpws)->fsend = dpws->backup_send_f;
    return dpws_dpws2soap(dpws)->msgbuf;
}

static int spawn_resolve(struct dpws *dpws, const char ** phys_addr, const char * log_addr)
{
    struct dpws *spawn_dpws = NULL;
    struct device_proxy* device;
    int ret = DPWS_OK;

	DC_RETURN_ASSERT_ALLOC(spawn_dpws = (struct dpws*)DC_MALLOC(DC_MEM_TRANSIENT, sizeof(struct dpws)));	// to avoid stack overflow
    dpws_struct_init(spawn_dpws);
    device = lookup_by_id(spawn_dpws, (char *)log_addr, WSD_MATCH_TIMEOUT);
	DC_ERROR_ASSERT(device, DPWS_ERR_COULD_NOT_RESOLVE_ADDRESS);
   	DC_ERROR_ASSERT_ALLOC(*phys_addr = DC_MSG_STRDUP(DC_MEM_API, dpws, *DA_GET(&device->transport_addresses, 0)));

DC_FUNC_ERROR
	dpws_end(spawn_dpws);
    DC_FREE(DC_MEM_TRANSIENT, spawn_dpws);
	return ret;
}

static int dpws_handle_fault(struct dpws *dpws, int (* ffault)(struct dpws*))
{
    struct soap *soap = dpws_dpws2soap(dpws);
    const char* action = dpws->action ? dpws->action : "";
	if (!strcmp(action, dpws->wsa_version->wsa_fault_uri))
    {
		dpws_set_namespaces(dpws, default_protocols->default_namespaces);
        soap_recv_fault(soap);
        return ffault(dpws);
    }
    else
        return soap->error = SOAP_NO_METHOD;
}

static int convert_reactor_error(struct dpws * dpws, struct reactor * reactor, int error)
{
	struct soap * soap = dpws_dpws2soap(dpws);
	struct dcpl_error error_info;

	error_info.error = reactor->error;
	error_info.syserr = reactor->syserr;
	error_info.detail = reactor->detail;
	soap->errnum = reactor->syserr;
	soap_set_receiver_error(soap, dcpl_error_string(&error_info, soap->msgbuf, sizeof(soap->msgbuf)), error_info.detail, SOAP_TCP_ERROR);

	return error;
}

/*******************************************************************************
*                                    API                                       *
*******************************************************************************/
void dpws_default_wsa_endpoint_ref(struct wsa_endpoint_ref *a)
{
	DC_CHECK_PARAM_RETURN(a,);
    memset(a, 0, sizeof(struct wsa_endpoint_ref));
}

static int _init_ex(dc_ip_filter_t * sel, const char * hostname, int versions, DC_BOOL mono_netif)
{
    int ret = DPWS_OK;

	DC_CHECK_PARAM(sel);

    if (!reactor_lock)
    {
#ifdef DPWS_DEBUG
		fdebug[DPWS_LOG_INDEX] = fopen("DPWS.log", "w");
		fdebug[DPWS_LOG_INDEX_SENT] = fopen("DPWS_SENT.log", "w");
		fdebug[DPWS_LOG_INDEX_RECV] = fopen("DPWS_RECV.log", "w");
#endif
        if ((ret = init_network(sel, hostname, mono_netif)) != DPWS_OK)
            return ret;

        if ((ret = init_registry()) != DPWS_OK)
            return ret;

        uuid_init();

        if ((reactor_lock = (dcpl_mutex_t*)dcpl_mutex_init()) == NULL)
            ret = DPWS_ERR_CREATING_MUTEX;
        else if (createHandlePool(DC_MEM_UNKNOWN, &reactor_handle_pool, 5, reactor_lock, NULL))
        	ret = DPWS_ERR_EOM;
        else if ((ret = init_discovery()) == DPWS_OK && (ret = init_cache()) == DPWS_OK)
        	ret = init_versions(versions);
    }
    return ret;
}

int dpws_init()
{
	dc_ip_filter_t sel = {NULL, DC_FALSE, DC_PROTO_INET, 0, NULL};
	return _init_ex(&sel, NULL, DPWS_DEFAULT_VERSION, DC_TRUE);
}

int dpws_init6()
{
	dc_ip_filter_t sel = {NULL, DC_FALSE, DC_PROTO_INET6, 0, NULL};
	return _init_ex(&sel, NULL, DPWS_DEFAULT_VERSION, DC_TRUE);
}

int dpws_init_ex(dc_ip_filter_t * sel, const char * hostname, int versions)
{
	return _init_ex(sel, hostname, versions, DC_FALSE);
}

int dpws_client_init(struct dpws* dpws, struct wsa_endpoint_ref* source)
{
	DC_CHECK_PARAM(dpws);

    dpws_struct_init(dpws);
    dpws->source = source;
    return DPWS_OK;
}

int dpws_server_init(struct dpws* dpws, struct wsa_endpoint_ref* source)
{
	return dpws_server_init_ex(
				dpws,
				source,
				registry_cfg.tcp_listen ? DC_LISTENER_DISCOVERY|DC_LISTENER_HTTP : DC_LISTENER_DISCOVERY,
				0,
				registry_cfg.tcp_backlog
			);
}

#ifndef DCPL_HAVE_PKTINFO
static int create_multicast_listener_cbk(dcpl_ip_addr_t * netif_addr, void * cbk_data)
{
	return DCPL_ADDR_IS_IPV4(netif_addr) ?
		dc_udp_add_multicast_listener(&reactor, DCPL_AF_INET, netif_addr, DISCOVERY_PORT, DISCOVERY_MULTICAST_ADDR) :
		dc_udp_add_multicast_listener(&reactor, DCPL_AF_INET6, netif_addr, DISCOVERY_PORT, DISCOVERY_MULTICAST_ADDRV6);
}
#endif

int dpws_server_init_ex(struct dpws* dpws, struct wsa_endpoint_ref* source, int listeners, int bind_flags, int backlog)
{
    int ret = DPWS_OK;

	DC_CHECK_PARAM(dpws && (!source || source->address) && listeners && backlog > 0);

	dpws_struct_init(dpws);
	dpws->soap.bind_flags = bind_flags;
    dpws->source = source;
    if ((ret = check_configuration()))
		return ret;
	if ((ret = dc_reactor_init(&reactor, DC_TRUE)))
		return ret;

	// Discovery setup
	if (listeners & DC_LISTENER_DISCOVERY) {
#ifdef DCPL_HAVE_PKTINFO
		// One socket & protocol
		if (dc_network_mode & DCPL_AF_INET) {
			if ((ret = dc_udp_add_multicast_listener(&reactor, DCPL_AF_INET, NULL, DISCOVERY_PORT, DISCOVERY_MULTICAST_ADDR)))
				return convert_reactor_error(dpws, &reactor, ret);
		}
		if (dc_network_mode & DCPL_AF_INET6) {
			if ((ret = dc_udp_add_multicast_listener(&reactor, DCPL_AF_INET6, NULL, DISCOVERY_PORT, DISCOVERY_MULTICAST_ADDRV6)))
				return convert_reactor_error(dpws, &reactor, ret);
		}
#else
		// One socket by interface & protocol
		if ((ret = network_browse_interfaces(DCPL_AF_INET|DCPL_AF_INET6, (network_addr_cbk)create_multicast_listener_cbk, NULL)))
			return convert_reactor_error(dpws, &reactor, ret);
#endif
	}
	// HTTP setup
	if (listeners & DC_LISTENER_HTTP) {
		if (dc_network_mode & DCPL_AF_INET) {
			if ((ret = dc_http_add_listener(&reactor, bind_flags, DCPL_AF_INET, NULL, http_endpoint.port, 0, backlog)))
				return convert_reactor_error(dpws, &reactor, ret);
		}
		if (dc_network_mode & DCPL_AF_INET6) {
			if ((ret = dc_http_add_listener(&reactor, bind_flags, DCPL_AF_INET6, NULL, http_endpoint.port, 0, backlog)))
				return convert_reactor_error(dpws, &reactor, ret);
		}
	}
	dpws->reactor = &reactor;
    advertize_enabled_devices();
    return ret;
}

int dpws_server_shutdown(struct dpws* dpws)
{
	// Note: reactor is automatically cleaned up on exit
	clean_registry();
	return DPWS_OK;
}

int dpws_shutdown()
{
    int ret = DPWS_OK;

	if (reactor_lock)	// watchguard
    {

		uninit_registry();
		clean_cache();
		uninit_cache();

		deleteHandlePool(&reactor_handle_pool);
        ret = dcpl_mutex_delete(reactor_lock);
        reactor_lock = NULL;

    	uninit_discovery();	// No need to clean pending discovery message because they all should have been sent
    	uninit_network();

#ifdef DPWS_DEBUG
		fclose(fdebug[DPWS_LOG_INDEX]);
		fclose(fdebug[DPWS_LOG_INDEX_SENT]);
		fclose(fdebug[DPWS_LOG_INDEX_RECV]);
#endif
	}
    return ret;
}

static int stop_server(DC_BOOL blocking, long delay_ms)
{
    int ret = DPWS_OK;

    /* send byes */
    if (registry_cfg.server_up)
    {
        if (get_toolkit_mode() == DPWS_DEVICE_USE)
        {
            /* schedule eventing ends and byes for enabled devices */
            shutdown_enabled_devices();
        }

        if (blocking) {
			dc_reactor_shutdown(&reactor, delay_ms);
		} else {
			dc_reactor_stop(&reactor, 0);
		}
		ret = dc_reactor_interrupt(&reactor);


        /* Set the shutdown flag to exit the loop */
        registry_cfg.server_up = DC_FALSE;
    }

    return ret;
}

int dpws_stop_server(long delay_ms)
{
	DC_CHECK_PARAM(delay_ms >= 0);

    return stop_server(DC_TRUE, delay_ms);
}

int dpws_stop_server_no_wait()
{
    return stop_server(DC_FALSE, 0);
}

int dpws_join_network(unsigned long boot_seq)
{
	return new_instance_id(boot_seq)
		|| advertize_enabled_devices()
		|| dc_reactor_interrupt(&reactor);
}

int dpws_get_error(struct dpws* dpws)
{
    DC_CHECK_PARAM(dpws);
    return dpws->err ? dpws->err : dpws->soap.error;
}

const char * dpws_get_error_msg(struct dpws* dpws)
{
    struct soap *soap = dpws_dpws2soap(dpws);

    DC_CHECK_PARAM_NO_RC(dpws->err, dpws, NULL);

    if (soap->error)
    {
        const char *c, *v = NULL, *s, **d;
        d = soap_faultcode(soap);
        if (!*d)
            soap_set_fault(soap);
        c = *d;
        if (soap->version == 2)
            v = *soap_faultsubcode(soap);
        s = *soap_faultstring(soap);
        d = soap_faultdetail(soap);
        sprintf(soap->tmpbuf, "%s%d fault: %s [%s]\n\"%s\"\nDetail: %s", soap->version ? "SOAP 1." : "Error ", soap->version ? (int)soap->version : soap->error, c, v ? v : "no subcode", s ? s : "[no reason]", d && *d ? *d : "[no detail]");
    }
    else if (dpws->err)
    	sprintf(soap->tmpbuf, "No message available for %d error code.", dpws->err);
	else
		sprintf(soap->tmpbuf, "Connection reset by peer or timed out.");
    return soap->tmpbuf;
}

static struct dpws* dpws_copy_context(struct dpws* dest, struct dpws* src)
{
    soap_copy_context(&dest->soap, &src->soap);
    memcpy((char *)dest + sizeof(dest->soap), (char *)src + sizeof(src->soap), sizeof(struct dpws) - sizeof(src->soap));
    return dest;
}

static int accept_msg(struct dpws * dpws, struct dpws * dpws_s)
{
	int ret = DPWS_OK;
	struct reactor * reactor = dpws->reactor;
	struct reactor_item * item = NULL;
	int32_t timeout = -1;

	dpws->active_item = NULL;
	if (!reactor)
		return DPWS_ERR_INIT_SERVER;

	if (dpws->soap.accept_timeout > 0)
		timeout = dpws->soap.accept_timeout * 1000;
	else if (dpws->soap.accept_timeout < 0)
		timeout = -dpws->soap.accept_timeout / 1000;

	if ((ret = dc_reactor_wait_single(reactor, &item, dpws, timeout)) < 0) {
		return convert_reactor_error(dpws, reactor, ret);
	} else if (ret > 0) {
		return DPWS_ERR_SERVER_STOPPED;
	} else if (!item) {
		struct soap * soap = dpws_dpws2soap(dpws);
		soap->errnum = 0;
		soap_set_receiver_error(soap, "Timeout", NULL, SOAP_TCP_ERROR);
		return DPWS_ERR_SERVER_TIMEOUT;
	}

	dpws->active_item = item;
	// The following call may replace the active item by a newly created one
	if ((ret = dc_reactor_execute_factory_item(reactor, item, dpws))) {
		return ret;
	}

	if (dpws_s)
		dpws_copy_context(dpws_s, dpws);
	else
		dpws_s = dpws;

	return DPWS_OK;
}

int dpws_accept(struct dpws* dpws)
{
	DC_CHECK_PARAM(dpws);
    return accept_msg(dpws, NULL);
}

int dpws_accept_thr(struct dpws* dpws, struct dpws* dpws_thr)
{
	DC_CHECK_PARAM(dpws && dpws_thr);
    if (!dpws_thr)
        return DPWS_ERR_INVALID_PARAMETER;
    return accept_msg(dpws, dpws_thr);
}

void * dpws_malloc(struct dpws* dpws, size_t size)
{
	DC_CHECK_PARAM_NO_RC(dpws->err, dpws && size > 0, NULL);

	return soap_malloc(dpws_dpws2soap(dpws), size);
}

int dpws_fault(struct dpws *dpws, fault_value_t value, const char *faultstring, const char *subcode, const char *detail)
{
	DC_CHECK_PARAM(dpws && faultstring);

	if (value == RECEIVER_FAULT_VALUE)
		return soap_receiver_fault_subcode(dpws_dpws2soap(dpws), subcode, faultstring, detail);
	else
		return soap_sender_fault_subcode(dpws_dpws2soap(dpws), subcode, faultstring, detail);
}

int dpws_end(struct dpws* dpws)
{
	DC_CHECK_PARAM(dpws);

    dpws_init_headers_and_version(dpws);
    release_proxies(dpws);
    dpws->err = DPWS_OK;
    soap_end(dpws_dpws2soap(dpws));
	dpws->soap.namespaces = NULL;
    return DPWS_OK;
}

int dpws_serve(struct dpws *dpws)
{
	DC_CHECK_PARAM(dpws);

	DPWS_SET_RCV(dpws->status);
	DPWS_RESET_MSG_TYPE(dpws->status);
	if (dpws->reactor && dpws->active_item)
		return dc_reactor_execute_item(dpws->reactor, dpws->active_item, dpws);
	return DPWS_OK;
}

int dpws_dispatch_request(struct dpws *dpws, void * transport_data, transport_fns_t * fns, const char * host, const char * path, media_type_t * mtype, const char * action, DC_BOOL needs_length)
{
	struct soap * soap = dpws_dpws2soap(dpws);
    int i, ret = DPWS_OK;
    struct service_port * port = NULL;
    struct service_endpoint * service = NULL;
    struct service_class *servClass;
    int (*fserv) (struct dpws *) = NULL;
	const char *address = NULL;
	char *p;
	soap_mode backup_imode;

	// Setup DPWS/gSOAP structure for HTTP request handling
	DPWS_SET_TCP(dpws->status);
    DPWS_SET_USER(dpws->status);
	dpws->transport_data = transport_data;
	dpws->transport_fns = fns;
	backup_imode = soap->imode;
	dpws->user_mode = soap->omode;
	soap->imode &= ~SOAP_IO;
	soap->omode &= ~SOAP_IO;
	soap->imode &= ~SOAP_IO_KEEPALIVE;
	soap->omode &= ~SOAP_IO_KEEPALIVE;
	// Store the host only in the runtime structure
	p = strchr(host, '[');
	if (p)	// IPv6
	{
		char * p2 = strchr(host, ']');
		if (!p2) {
			ret = soap_set_sender_error(soap, "Incorrect 'Host' header.", NULL, SOAP_HTTP_ERROR);
			goto exit;
		}
		if (host == soap->host)	{	// gSOAP HTTP server
			for (; (p + 1) != p2; p++)
				*p = *(p + 1);
			*p = 0;
		}
		else {	// other server
			size_t len = p2 - p - 1;
			strncpy(soap->host, p + 1, len);
			soap->host[len] = 0;
		}
	}
	else	// IPv4
	{
		p = strchr(host, ':');
		if (host == soap->host)	{	// gSOAP HTTP server
			if (p)
				*p = 0;
		}
		else {	// other server
			if (p) {
				size_t len = p - host;
				strncpy(soap->host, host, p - host);
				soap->host[len] = 0;
			}
			else
				strcpy(soap->host, host);
		}
	}
	if (!needs_length)
		soap->omode |= SOAP_ENC_XML;
	if (mtype) {
		if (!strcmp(mtype->type, "application") && !strcmp(mtype->subtype, "dime")) {
			soap->imode |= SOAP_ENC_DIME;
		} else if (!strcmp(mtype->type, "multipart") && (!strcmp(mtype->subtype, "related") || !strcmp(mtype->subtype, "form-data"))) {
			media_type_param_t * p = mtype->params;
			while (p) {
				if (!strcmp(p->attribute, "boundary"))
					soap->mime.boundary = (char *)p->value;
				else if (!strcmp(p->attribute, "start"))
					soap->mime.start = p->value;
				p = p->next;
			}
			soap->imode |= SOAP_ENC_MIME;
		}
	}

	dpws_set_namespaces(dpws, default_protocols->default_namespaces);
    soap_begin(soap);
    if ((ret = soap_begin_recv(soap))
		|| (ret = soap_envelope_begin_in(soap))
    	|| (ret = soap_recv_header(soap))
    	|| (ret = dpws_check_headers(dpws)))
    	goto exit;

	if (dpws->to && is_discovery_urn(dpws, dpws->to)) {
		// Case of the directed probe
		// retrieve the HTTP request URL.
		address = path;
		if (!dpws->reply_to.address)
			dpws->reply_to.address = dpws->wsa_version->wsa_anonymous_uri;
	}
	else if (dpws->to)
		address = dpws->to;
	else // BP 1.1
		address = path;

    // Added to complete the HTTP request/response in oneway cases
    if (dpws_is_asynchronous(dpws) && (ret = dpws_send_empty_response(dpws)))
    	goto exit;

    if ((ret = soap_body_begin_in(soap)))
        goto exit;

	port = find_endpoint_port(address);

    if (!port)
    {
        ret = soap->error = WSA_ERR_DESTINATION_UNREACHABLE;
        goto exit;
    }
    service = port->service;

    dpws->href_endpoint = service->href;
#ifdef WITH_WSMAN
	dpws->eventing_sm = &default_sm_plugin;
#endif

    servClass = (struct service_class *)getObject(&registryPool, service->hrefServiceClass);

    // function election processing
    ret = SOAP_NO_METHOD;
    for (i = 0; i < servClass->funcs.nb; i++)
    {
        fserv = *DA_GET(&servClass->funcs, i);
        ret = fserv(dpws);
        if (ret)
        {
            if (ret != SOAP_NO_METHOD)							// NOTE: this is optimized to return ASAP
                goto exit;
        }
        else
            break;
    }

	// Block invalid messages for built-in protocols
	soap->mode |= SOAP_XML_STRICT;	// mode is set by soap_begin_recv above

	if (ret == SOAP_NO_METHOD)
        ret = transfer_serve_request(dpws);	// NOTE: mex although attached to device can address a hosted service

#ifdef WITH_WSMAN
    if (ret == SOAP_NO_METHOD && (service->subsc_manager || dpws->eventing_sm != &default_sm_plugin))
#else
    if (ret == SOAP_NO_METHOD && service->subsc_manager)
#endif
        ret = subscmanager_serve_request(dpws);

	if (ret == SOAP_NO_METHOD)
	{
		soap->mode &= ~SOAP_XML_STRICT;	// Because extensibility has been removed from generation since it adds code but brings nothing since gSOAP cannot control order (equivalent to ignore unknown)
        ret = wsd_serve_directed_probe(dpws);
		if (ret == DPWS_ERR_COULD_NOT_MATCH_DIRECTED_PROBE)
			ret = soap->error = SOAP_OK;
	}

	// For asynchronous fault handling
    if (ret == SOAP_NO_METHOD)
    {
        if (servClass->fault_func)
            ret = dpws_handle_fault(dpws, servClass->fault_func);
    }

    if (ret == SOAP_NO_METHOD)
    {
        ret = soap->error = WSA_ERR_ACTION_NOT_SUPPORTED;
        goto exit;
    }

	// Make sure the empty response is sent in case of one-way with no headers (BP1.1)
    if (!ret && !DPWS_IS_RESPONSE(dpws->status) && (ret = dpws_send_empty_response(dpws)))
    	goto exit;

exit:
    if (ret)
        ret = dpws_send_fault(dpws);
    if (port)
        release_endpoint_port(port);
	soap->imode = backup_imode;
	soap->omode = dpws->user_mode;
	dpws->transport_data = NULL;
	dpws->transport_fns = NULL;
    return ret;
}

static int init_dpws_filter(struct dpws * dpws, discovery_filter_t * user_filter, discovery_filter_t * dpws_filter)
{
	qname_t * types = DC_MSG_MALLOC(DC_MEM_DISCOVERY, dpws, (user_filter->nb_types+1)*sizeof(qname_t));

	if (!types)
		return DPWS_ERR_EOM;

	memcpy(types, user_filter->types, user_filter->nb_types*sizeof(qname_t));
	types[user_filter->nb_types].ns = dpws->protocols->wdp_uri;
	types[user_filter->nb_types].lname = WDP_DEVICE_TYPE;
	dpws_filter->types = types;
	dpws_filter->nb_types = user_filter->nb_types+1;
	dpws_filter->scopes = user_filter->scopes;
	dpws_filter->nb_scopes = user_filter->nb_scopes;
	dpws_filter->match_rule = user_filter->match_rule;
	return DPWS_OK;
}

static int send_probe_callback(struct dpws* dpws, struct reactor_item * item, struct send_probe_data * data)
{
	int ret = DPWS_OK;
	struct udp_transport_data tdata;
	discovery_filter_t dpws_filter;

	dpws->protocols = data->protocols;
	if ((ret = init_dpws_filter(dpws, data->filter, &dpws_filter)))
		return ret;

	if (!(ret = dc_udp_setup_multicast_output_channel(
			dpws,
			&tdata,
			data->socket,
			DCPL_ADDR_IS_IPV4(data->netif_addr) ? DISCOVERY_MULTICAST_ADDR:DISCOVERY_MULTICAST_ADDRV6,
			DISCOVERY_PORT,
			data->netif_addr
			)))
		ret = send_probe(dpws, &dpws_filter, data->message_uuid);
	dc_transport_teardown_channel(dpws, &tdata.base);
	dc_transport_data_clear(&tdata.base);
	dpws_end(dpws);
	return ret;
}

static int recv_probematch_callback(struct dpws* dpws, struct reactor_item * item, struct recv_probematch_data * data)
{
	int ret = DPWS_OK;
	DPWS_SET_UNICAST(dpws->status);	// fork using another way ?
	if (!(ret = dc_udp_serve_request(dpws, item, wsd_ucast_recv_probematch, NULL))) {
	    int nb = count_cache_entry(data->filter, data->f_size, data->res_size);
       	DPWSLOG1(DC_CACHE, "<- %d endpoints found in cache\n", nb);
		if (data->res_size > 0 && nb >= data->res_size)
			dc_reactor_stop(item->reactor, 0);
	}
	dpws_end(dpws);
	// Return should be OK even in case of bad messages, otherwise the reactor will stop
	return DPWS_OK;
}

static int schedule_probes_cbk(dcpl_ip_addr_t * netif_addr, struct probe_send_info * psinfo)
{
    int i, j;

	for (j = 0; j < psinfo->p_size; j++)
		for (i = 0; i < psinfo->f_size; i++)
		{
			short send_nb = 0;
			long delay = 0;

			psinfo->w_prb_data->netif_addr = netif_addr;
    		psinfo->w_prb_data->socket = DCPL_ADDR_IS_IPV4(netif_addr) ? psinfo->s4 : psinfo->s6;
			dpws_schemed_uuid_create(psinfo->w_prb_data->message_uuid);
			psinfo->w_prb_data->filter = psinfo->filter + i;
			psinfo->w_prb_data->protocols = psinfo->protocols[j];

			while ((delay = get_soap_over_udp_schedule(&send_nb,
					(DPWS_MULTICAST_MSG(DPWS_MSG_PROBE) ? MULTICAST_UDP_REPEAT : UNICAST_UDP_REPEAT),
					DPWS_APP_DELAY_MSG(DPWS_MSG_PROBE), delay)) != 0xFFFFFFFF)
			{
				dc_reactor_register_timer(&psinfo->reactor, DC_FALSE, delay, 0, (reactor_cbk)send_probe_callback, psinfo->w_prb_data, NULL, NULL);
				DPWSLOG1(DC_CACHE, "-> schedule Probe %d sending\n", delay);
			}
			psinfo->w_prb_data++;
		}
	return DPWS_OK;
}

static short * lookup_ex(struct dpws* dpws, discovery_filter_t * filter, int f_size, int * res_size, unsigned int time_out, cache_mode_t mode)
{
    int nb = 0;
    short * results = NULL;
	discovery_filter_t null_filter = {NULL, 0, NULL, 0, WSD_MATCH_BY_UNSPECIFIED};
	struct wsa_endpoint_ref * dp;

	DC_CHECK_PARAM_RETURN(dpws, NULL);
	DC_CHECK_PARAM_NO_RC(dpws->err, (filter == NULL && f_size == 0) || (filter != NULL && f_size > 0), NULL);
#ifdef DC_API_PARAM_CONTROL
	if (filter)
	{
		int i;
		for (i = 0; i < f_size; i++) {
			DC_CHECK_PARAM_NO_RC(dpws->err, check_discovery_filter(&filter[i]), NULL);
		}
	}
#endif
	DC_CHECK_PARAM_NO_RC(dpws->err, res_size && time_out > 0, NULL);

	dcpl_mutex_lock(cache_lock);

	dp = dp_EPR;

	/* Check the number of asked devices does not exceed the cache size */
	if (*res_size > 0 && (unsigned long)(*res_size) > get_cache_config_int_att(DPWS_INT_MAX_DEVICE_PROXIES, 0)) {
		dpws->err = DPWS_ERR_NB_DEVICES_EXCEEDS_CACHE_SIZE;
		*res_size = 0;
		return NULL;
	}
	dcpl_mutex_unlock(cache_lock);

	if (!filter) {
		filter = &null_filter;
		f_size = 1;
	}

	/* retrieve endpoints from cache */
    if (mode == WSD_LOOKUP_MODE_CACHE_FIRST)
        nb = count_cache_entry(filter, f_size, *res_size);

	DPWSLOG1(DC_CACHE, "<- %d endpoints initially found in cache created\n", nb);

	/* Use DP if present */
	if (dp && time_out > 0 && mode != WSD_LOOKUP_MODE_CACHE_ONLY && (*res_size <= 0 || nb < *res_size)) {
		int i;
		if ((dpws->err = dpws_check_endpoint(dpws, dp))) {
				*res_size = 0;
				return NULL;
		}
		for (i = 0; i < f_size; i++) {
			discovery_filter_t dpws_filter;
			if (init_dpws_filter(dpws, filter + i, &dpws_filter)) {
				dpws->err = DPWS_ERR_EOM;
				*res_size = 0;
				return NULL;
			}
			call_discovery_proxy_probe(dpws, dp, &dpws_filter, time_out/f_size);
		}
		dpws_end(dpws);
	}

	/* send probe */
	if (!dp && time_out > 0 && mode != WSD_LOOKUP_MODE_CACHE_ONLY && (*res_size <= 0 || nb < *res_size))
	{
		struct recv_probematch_data prbm_data;
		struct probe_send_info psinfo;
		struct send_probe_data * prb_data;
		int p_size;
		struct dpws_protocols * protocols[N_DPWS_VERSIONS];

		p_size = get_protocols_versions(dpws, protocols, N_DPWS_VERSIONS);

		dc_reactor_init(&psinfo.reactor, DC_FALSE);

		/* Allocate messages structures (not on soap heap because of dpws_end) */
		prb_data = DC_MALLOC(
				DC_MEM_TRANSIENT,
				network_count_interfaces(DCPL_AF_INET|DCPL_AF_INET6) * f_size * p_size * sizeof(struct send_probe_data)
				);
		if (!prb_data) {
        	dpws->err = DPWS_ERR_EOM;
			*res_size = 0;
        	return NULL;
		}

		prbm_data.filter = filter;
		prbm_data.f_size = f_size;
		prbm_data.res_size = *res_size;

		if (dc_network_mode & DCPL_AF_INET) {
			// Note: the socket is closed when exiting the reactor
			psinfo.s4 = dcpl_udp_open(DCPL_AF_INET, NULL);
			if (psinfo.s4 == DCPL_INVALID_SOCKET) {
				dpws->err = DPWS_ERR_COULD_NOT_CREATE_SOCKET;
				DC_FREE(DC_MEM_TRANSIENT, prb_data);
				*res_size = 0;
				return NULL;
			}
			/* register UDP socket listener */
			dc_reactor_register_socket(&psinfo.reactor, psinfo.s4, DC_RI_READ, (reactor_cbk)recv_probematch_callback, &prbm_data, NULL, NULL);
		}
		if (dc_network_mode & DCPL_AF_INET6) {
			// Note: the socket is closed when exiting the reactor
			psinfo.s6 = dcpl_udp_open(DCPL_AF_INET6, NULL);
			if (psinfo.s6 == DCPL_INVALID_SOCKET) {
				dpws->err = DPWS_ERR_COULD_NOT_CREATE_SOCKET;
				DC_FREE(DC_MEM_TRANSIENT, prb_data);
				*res_size = 0;
				return NULL;
			}
			/* register UDP socket listener */
			dc_reactor_register_socket(&psinfo.reactor, psinfo.s6, DC_RI_READ, (reactor_cbk)recv_probematch_callback, &prbm_data, NULL, NULL);
		}

        /* schedule probes */
		psinfo.filter = filter;
		psinfo.f_size = f_size;
		psinfo.w_prb_data = prb_data;
		psinfo.protocols = protocols;
		psinfo.p_size = p_size;
		network_browse_interfaces(DCPL_AF_INET|DCPL_AF_INET6, (network_addr_cbk)schedule_probes_cbk, &psinfo);


		DPWSLOG1(DC_CACHE, "-> schedule timeout %d\n", MIN(time_out, MATCH_TIMEOUT));
		dpws->err = dc_reactor_start(&psinfo.reactor, dpws, MIN(time_out, MATCH_TIMEOUT));
		// Reactor will exit and cleanup on callback error, so nothing more to do

		DC_FREE(DC_MEM_TRANSIENT, prb_data);
    }

	results = cache_lookup(dpws, filter, f_size, res_size);	// NOTE: this function copy cache contents using dpws heap since dpws_end has been called to allow multiple receive
	DPWSLOG1(DC_CACHE, "<- %d endpoints retrieved in cache\n", *res_size);

	if (results) {
		dcpl_mutex_lock(cache_lock);
		checkout_proxies(dpws, results, *res_size);
		dcpl_mutex_unlock(cache_lock);
	}

    return results;
}

short * dpws_lookup(struct dpws* dpws, char * ns, char * type, char * scope, int *res_size)
{
	discovery_filter_t filter = {NULL, 0, NULL, 0, WSD_MATCH_BY_UNSPECIFIED};
	qname_t qn;

	DC_CHECK_PARAM_RETURN(dpws, NULL);
	DC_CHECK_PARAM_NO_RC(dpws->err, (ns == NULL && type == NULL) || (ns != NULL && type != NULL), NULL);

	if (type) {
		qn.ns = ns;
		qn.lname = type;
		filter.types = &qn;
		filter.nb_types = 1;
	}
	if (scope) {
		filter.scopes = &scope;
		filter.nb_scopes = 1;
	}

	return lookup_ex(dpws, &filter, 1, res_size, WSD_MATCH_TIMEOUT, WSD_LOOKUP_MODE_CACHE_FIRST);
}

#ifdef WITH_LEGACY_DC_API
short * dpws_lookup_ex(struct dpws* dpws, struct qname types[],
                       char ** scopes, scope_match_rule_t match_rule,
                       int * res_size, long time_out, int mode)
{
	int i;
	discovery_filter_t filter = {NULL, 0, NULL, 0, WSD_MATCH_BY_UNSPECIFIED};

    /* Build dynamic array */
    if (types) {
    	filter.types = types;
	    for (i = 0; types[i].lname; i++);
	    filter.nb_types = i;
    }

    if (scopes) {
	    filter.scopes = scopes;
	    for (i = 0; scopes[i]; i++);
	    filter.nb_scopes = i;
    }
    filter.match_rule = match_rule;

    return lookup_ex(dpws, &filter, 1, res_size, (unsigned int)time_out, mode);
}

#else
short * dpws_lookup_ex(struct dpws* dpws, discovery_filter_t * filter, int f_size, int * res_size, unsigned int time_out, cache_mode_t mode)
{
    return lookup_ex(dpws, filter, f_size, res_size, time_out, mode);
}
#endif

static int send_resolve_callback(struct dpws* dpws, struct reactor_item * item, struct send_resolve_data * data)
{
	int ret = DPWS_OK;

	struct udp_transport_data tdata;

	dpws->protocols = data->protocols;

	if (!(ret = dc_udp_setup_multicast_output_channel(
			dpws,
			&tdata,
			data->socket,
			DCPL_ADDR_IS_IPV4(data->netif_addr) ? DISCOVERY_MULTICAST_ADDR:DISCOVERY_MULTICAST_ADDRV6,
			DISCOVERY_PORT,
			data->netif_addr
			)))
		ret = send_resolve(dpws, &data->endpoint, data->message_uuid);
	dc_transport_teardown_channel(dpws, &tdata.base);
	dc_transport_data_clear(&tdata.base);
	dpws_end(dpws);
	return ret;
}

static int recv_resolvematch_callback(struct dpws* dpws, struct reactor_item * item, struct recv_resolvematch_data * data)
{
	int ret = DPWS_OK;
	DPWS_SET_UNICAST(dpws->status);	// fork using another way ?
	if (!(ret = dc_udp_serve_request(dpws, item, wsd_ucast_recv_resolvematch, NULL))) {
		data->cache_entry = find_device_proxy(data->id);
		if (data->cache_entry && data->cache_entry->transport_addresses.nb > 0)
			dc_reactor_stop(item->reactor, 0);
	}
	dpws_end(dpws);
	// Return should be OK even in case of bad messages, otherwise the reactor will stop
	return DPWS_OK;
}

static int schedule_resolves_cbk(dcpl_ip_addr_t * netif_addr, struct resolve_send_info * rsinfo)
{
	int i;

	for (i = 0; i < rsinfo->p_size; i++) {
		short send_nb = 0;
		long delay = 0;
		rsinfo->w_rslv_data->netif_addr = netif_addr;
		rsinfo->w_rslv_data->socket = DCPL_ADDR_IS_IPV4(netif_addr) ? rsinfo->s4 : rsinfo->s6;
		dpws_schemed_uuid_create(rsinfo->w_rslv_data->message_uuid);
		dpws_default_wsa_endpoint_ref(&rsinfo->w_rslv_data->endpoint);
		rsinfo->w_rslv_data->endpoint.address = rsinfo->device_uuid;
		rsinfo->w_rslv_data->protocols = rsinfo->protocols[i];

		while ((delay = get_soap_over_udp_schedule(&send_nb,
				(DPWS_MULTICAST_MSG(DPWS_MSG_RESOLVE) ? MULTICAST_UDP_REPEAT : UNICAST_UDP_REPEAT),
				DPWS_APP_DELAY_MSG(DPWS_MSG_RESOLVE), delay)) != 0xFFFFFFFF) {
			dc_reactor_register_timer(&rsinfo->reactor, DC_FALSE, delay, 0, (reactor_cbk)send_resolve_callback, rsinfo->w_rslv_data, NULL, NULL);
			DPWSLOG1(DC_CACHE, "-> schedule Probe %d sending\n", delay);
		}
		rsinfo->w_rslv_data++;
	}
	return DPWS_OK;
}

static struct device_proxy* lookup_by_id(struct dpws* dpws, char * id, unsigned int time_out)
{
	struct wsa_endpoint_ref * dp;
	struct recv_resolvematch_data rslvm_data;

	rslvm_data.id = id;
	rslvm_data.cache_entry = NULL;

	rslvm_data.cache_entry = find_device_proxy(id);

	dcpl_mutex_lock(cache_lock);
	dp = dp_EPR;
	dcpl_mutex_unlock(cache_lock);

	if (!rslvm_data.cache_entry ||
		(!is_physical_address(rslvm_data.cache_entry->superEPR.super.address)
			&& rslvm_data.cache_entry->transport_addresses.nb == 0)) {
		if (dp) {
			if ((dpws->err = dpws_check_endpoint(dpws, dp))) {
					return NULL;
			}
			rslvm_data.cache_entry = call_discovery_proxy_resolve(dpws, dp, id, time_out);
			dpws_end(dpws);
		}
		if (!dp) {
			struct send_resolve_data * rslv_data;
			struct resolve_send_info rsinfo;
			int p_size;
			struct dpws_protocols * protocols[N_DPWS_VERSIONS];

			p_size = get_protocols_versions(dpws, protocols, N_DPWS_VERSIONS);

			dc_reactor_init(&rsinfo.reactor, DC_FALSE);

			/* Allocate messages structures (not on soap heap because of dpws_end) */
			rslv_data = DC_MALLOC(
					DC_MEM_TRANSIENT,
					network_count_interfaces(DCPL_AF_INET|DCPL_AF_INET6) * p_size * sizeof(struct send_resolve_data)
					);
			if (!rslv_data) {
	        	dpws->err = DPWS_ERR_EOM;
	        	return NULL;
			}

			rslvm_data.id = id;
			rslvm_data.cache_entry = NULL;

			if (dc_network_mode & DCPL_AF_INET) {
				// Note: the socket is closed when exiting the reactor
				rsinfo.s4 = dcpl_udp_open(DCPL_AF_INET, NULL);
				if (rsinfo.s4 == DCPL_INVALID_SOCKET) {
					dpws->err = DPWS_ERR_COULD_NOT_CREATE_SOCKET;
					DC_FREE(DC_MEM_TRANSIENT, rslv_data);
					return NULL;
				}
				/* register UDP socket listener */
				dc_reactor_register_socket(&rsinfo.reactor, rsinfo.s4, DC_RI_READ, (reactor_cbk)recv_resolvematch_callback, &rslvm_data, NULL, NULL);
			}
			if (dc_network_mode & DCPL_AF_INET6) {
				// Note: the socket is closed when exiting the reactor
				rsinfo.s6 = dcpl_udp_open(DCPL_AF_INET6, NULL);
				if (rsinfo.s6 == DCPL_INVALID_SOCKET) {
					dpws->err = DPWS_ERR_COULD_NOT_CREATE_SOCKET;
					DC_FREE(DC_MEM_TRANSIENT, rslv_data);
					return NULL;
				}
				/* register UDP socket listener */
				dc_reactor_register_socket(&rsinfo.reactor, rsinfo.s6, DC_RI_READ, (reactor_cbk)recv_resolvematch_callback, &rslvm_data, NULL, NULL);
			}

	        /* schedule resolves */
			rsinfo.device_uuid = id;
			rsinfo.w_rslv_data = rslv_data;
			rsinfo.protocols = protocols;
			rsinfo.p_size = p_size;
			network_browse_interfaces(DCPL_AF_INET|DCPL_AF_INET6, (network_addr_cbk)schedule_resolves_cbk, &rsinfo);

			DPWSLOG1(DC_CACHE, "-> schedule timeout %d\n", MIN(time_out, MATCH_TIMEOUT));
			dpws->err = dc_reactor_start(&rsinfo.reactor, dpws, MIN(time_out, MATCH_TIMEOUT));
			// Reactor will exit and cleanup on callback error, so nothing more to do

			DC_FREE(DC_MEM_TRANSIENT, rslv_data);
		}
	}
	return rslvm_data.cache_entry;
}

short dpws_lookup_by_id(struct dpws* dpws, const char * id)
{
    struct device_proxy* proxy;
    char uuid[SCHEME_UUID_STRING_SIZE];
	char* valid_id;

	DC_CHECK_PARAM(dpws && id);

	if (is_physical_address(id)) {
		valid_id = (char *)id;
	} else if (!(valid_id = format_scheme_uuid(uuid, (char *)id))) {
    	dpws->err = DPWS_ERR_INVALID_PARAMETER;
    	return -1;
    }
    proxy = lookup_by_id(dpws, valid_id, WSD_MATCH_TIMEOUT);
    if (proxy)
    {
        dcpl_mutex_lock(cache_lock);
        dpws->err = checkout_proxies(dpws, &proxy->super.href, 1);
        dcpl_mutex_unlock(cache_lock);
		return dpws->err ? dpws->err : proxy->super.href;
    }
    else
        return DPWS_ERR_NO_HANDLE_FOUND;
}

static short _probe_address_ex(struct dpws* dpws, char * addr, discovery_filter_t * filter)
{
    char * device_id;
	int i, p_size;
	struct dpws_protocols * protocols[N_DPWS_VERSIONS];

	DC_CHECK_PARAM(dpws && addr && filter && check_discovery_filter(filter));	// a filter must be provided

	p_size = get_protocols_versions(dpws, protocols, N_DPWS_VERSIONS);

	for (i = 0; i < p_size; i++) {
		discovery_filter_t dpws_filter;
		dpws->protocols = protocols[i];

		if ((dpws->err = init_dpws_filter(dpws, filter, &dpws_filter))) {
			return dpws->err;
		}

		if ((device_id = call_directed_probe(dpws, addr, &dpws_filter)))
		{
			struct device_proxy * proxy;
			dcpl_mutex_lock(cache_lock);
			proxy = find_device_proxy(device_id);
			dpws->err = checkout_proxies(dpws, &proxy->super.href, 1);
			dcpl_mutex_unlock(cache_lock);
			return dpws->err ? dpws->err : proxy->super.href;
		}
	}
    return -1;
}

short dpws_probe_address(struct dpws* dpws, char * addr, char* ns, char* type, char* scope)
{
	discovery_filter_t filter = {NULL, 0, NULL, 0, WSD_MATCH_BY_UNSPECIFIED};
	qname_t qn;

	DC_CHECK_PARAM((ns == NULL && type == NULL) || (ns != NULL && type != NULL));

	if (type) {
		qn.ns = ns;
		qn.lname = type;
		filter.types = &qn;
		filter.nb_types = 1;
	}
	if (scope) {
		filter.scopes = &scope;
		filter.nb_scopes = 1;
	}

    return _probe_address_ex(dpws, addr, &filter);
}

#ifdef WITH_LEGACY_DC_API
short dpws_probe_address_ex(struct dpws* dpws, char* address, struct qname *types, char ** scopes, scope_match_rule_t match_rule)
{
	int i;
	discovery_filter_t filter = {NULL, 0, NULL, 0, WSD_MATCH_BY_UNSPECIFIED};

    /* Build dynamic array */
    if (types) {
    	filter.types = types;
	    for (i = 0; types[i].lname; i++);
	    filter.nb_types = i;
    }

    if (scopes) {
	    filter.scopes = scopes;
	    for (i = 0; scopes[i]; i++);
	    filter.nb_scopes = i;
    }
    filter.match_rule = match_rule;

    return _probe_address_ex(dpws, addr, &filter);
}

#else
short dpws_probe_address_ex(struct dpws* dpws, char* address, discovery_filter_t * filter)
{
    return _probe_address_ex(dpws, address, filter);
}
#endif

static struct device_proxy * fill_device_services(struct dpws* dpws, short hrefDevice)
{
    struct device_proxy * device = NULL;
    DC_BOOL isDevice;

    // find the cache entry
    dcpl_mutex_lock(cache_lock);

    // perform a getMetadata if hosted services were not retrieved yet
    device = get_device_proxy(hrefDevice, &isDevice);

    if (!device)
    {
        dpws->err = DPWS_ERR_NO_HANDLE_FOUND;
        return NULL;
    }
    if (!device->hosted_retrieved)
    {
		int ret;
        dcpl_mutex_unlock(cache_lock);
        ret = get_endpoint_metadata(dpws, hrefDevice, device, NULL, NULL, NULL, NULL, NULL);
        dcpl_mutex_lock(cache_lock);
	    if (ret) return NULL;
    }
    // do not UNLOCK (performed by the caller)
    return device;
}

short * dpws_get_services(struct dpws* dpws,
                          short href_device, char * ns, char * port_type, int * res_size)
{
    short * ret = NULL;
    struct device_proxy * device_proxy;

	DC_CHECK_PARAM_RETURN(dpws, NULL);
	DC_CHECK_PARAM_NO_RC(dpws->err, href_device >= 0 && res_size && ((ns && port_type) || (!ns && !port_type)), NULL);

    device_proxy = fill_device_services(dpws, href_device);	// takes lock
    if (!device_proxy)
    {
        *res_size = dpws->err = DPWS_ERR_COULD_NOT_FIND_CACHE_ENTRY;
        goto exit;
    }
    // filter cache and copy to results for asked
    ret = filter_services_by_type(dpws, device_proxy, ns, port_type, res_size);
    if (ret)
        checkout_proxies(dpws, ret, *res_size);

exit:
    dcpl_mutex_unlock(cache_lock);
    dpws_init_headers_and_version(dpws);
    return ret;
}

short dpws_get_service_by_id(struct dpws* dpws, short href_device, char * id)
{
    short ret = -1;
    struct device_proxy * cache_entry;

    DC_CHECK_PARAM(dpws && href_device >= 0 && id);

    cache_entry = fill_device_services(dpws, href_device);	// takes lock
    if (!cache_entry)
        goto exit;

    // filter cache and copy to results for asked
    ret = filter_services_by_id(cache_entry, id);
    if (ret >= 0)
    	dpws->err = checkout_proxies(dpws, &ret, 1);

exit:
    dcpl_mutex_unlock(cache_lock);
    dpws_init_headers_and_version(dpws);
    return ret;
}

struct device_metadata* dpws_get_device_metadata(struct dpws* dpws, short href_device)
{
    struct device_metadata* metadata;
    da_allocator_t daa;
    DA_TYPED(pepr) host_endpoints;
    DA_TYPED(pwi) wsdlList;
    DA_TYPED(psi) services;
    struct device_proxy * device;
    DC_BOOL isDevice;

    DC_CHECK_PARAM_RETURN(dpws, NULL);
    DC_CHECK_PARAM_NO_RC(dpws->err, href_device >= 0, NULL);

    soap_get_da_allocator(dpws_dpws2soap(dpws), &daa);
    DA_INIT(struct wsa_endpoint_ref *, &host_endpoints, DC_MEM_API, &daa, 1);
    DA_INIT(struct wsdl_info *, &wsdlList, DC_MEM_API, &daa, 4);
    DA_INIT(struct service_info *, &services, DC_MEM_API, &daa, 4);

    if (!(metadata = (struct device_metadata*)DC_MSG_MALLOC(DC_MEM_API, dpws, sizeof(struct device_metadata)))) {
    	dpws->err = DPWS_ERR_EOM;
    	return NULL;
    }

    dcpl_mutex_lock(cache_lock);
    device = get_device_proxy(href_device, &isDevice);
    dcpl_mutex_unlock(cache_lock);

    if (!device)
    {
        dpws->err = DPWS_ERR_NO_HANDLE_FOUND;
        return NULL;
    }
    if (!isDevice)
    {
        dpws->err = DPWS_ERR_IS_A_HOSTED_SERVICE;
        return NULL;
    }
    if (get_endpoint_metadata(dpws, href_device, device->hosted_retrieved ? NULL : device, &metadata->model_info, &metadata->device_info, &host_endpoints, &services, &wsdlList))
        return NULL;

    metadata->host_endpoints = (struct wsa_endpoint_ref **)host_endpoints.tab;
//    metadata->host_endpoints = NULL;
    metadata->services = (struct service_info **)services.tab;
    metadata->wsdls = (struct wsdl_info **)wsdlList.tab;
    return metadata;
}

struct service_metadata* dpws_get_service_metadata(struct dpws* dpws, short href_service)
{
    struct service_metadata* metadata;
    da_allocator_t daa;
    DA_TYPED(pepr) host_endpoints;
    DA_TYPED(pwi) wsdlList;
    DA_TYPED(psi) services;
    struct device_proxy * device;
    DC_BOOL isDevice;

    DC_CHECK_PARAM_RETURN(dpws, NULL);
    DC_CHECK_PARAM_NO_RC(dpws->err, href_service >= 0, NULL);

    soap_get_da_allocator(dpws_dpws2soap(dpws), &daa);
    DA_INIT(struct wsa_endpoint_ref *, &host_endpoints, DC_MEM_API, &daa, 1);
    DA_INIT(struct wsdl_info *, &wsdlList, DC_MEM_API, &daa, 1);
    DA_INIT(struct service_info *, &services, DC_MEM_API, &daa, 1);

    if (!(metadata = (struct service_metadata*)DC_MSG_MALLOC(DC_MEM_API, dpws, sizeof(struct service_metadata)))) {
    	dpws->err = DPWS_ERR_EOM;
    	return NULL;
    }

    dcpl_mutex_lock(cache_lock);
    device = get_device_proxy(href_service, &isDevice);
    dcpl_mutex_unlock(cache_lock);

    if (!device)
    {
        dpws->err = DPWS_ERR_NO_HANDLE_FOUND;
        return NULL;
    }
    if (isDevice)
    {
        dpws->err = DPWS_ERR_IS_A_DEVICE;
        return NULL;
    }
    if (get_endpoint_metadata(dpws, href_service, device->hosted_retrieved ? NULL : device, NULL, NULL, &host_endpoints, &services, &wsdlList))
        return NULL;

    metadata->host_endpoints = (struct wsa_endpoint_ref **)host_endpoints.tab;
    metadata->service_info = *DA_GET(&services, 0);
    metadata->wsdls = (struct wsdl_info **)wsdlList.tab;
    return metadata;
}

char * dpws_event_renew(struct dpws* dpws, struct wsa_endpoint_ref* subscription_manager, char* expiration)
{
    struct _wse__Renew renew;
    struct _wse__RenewResponse response;
    char * ret = NULL;

	DC_CHECK_PARAM_RETURN(dpws, NULL);
	DC_CHECK_PARAM_NO_RC(dpws->err, subscription_manager, NULL);

    renew.wse__Expires = expiration;
    if (dpws_call___wse__RenewOp(dpws, subscription_manager, NULL, &renew, &response))
        goto exit;

    ret = response.wse__Expires;

exit:
    dpws_init_headers_and_version(dpws);
    return ret;
}

char * dpws_event_status(struct dpws* dpws, struct wsa_endpoint_ref* subscription_manager)
{
    struct _wse__GetStatus get_status;
    struct _wse__GetStatusResponse status;
    char * ret = NULL;

	DC_CHECK_PARAM_RETURN(dpws, NULL);
	DC_CHECK_PARAM_NO_RC(dpws->err, subscription_manager, NULL);

    if (dpws_call___wse__GetStatusOp(dpws, subscription_manager, NULL, &get_status, &status))
        goto exit;

    ret = status.wse__Expires;

exit:
    dpws_init_headers_and_version(dpws);
    return ret;
}

int dpws_event_unsubscribe(struct dpws* dpws, struct wsa_endpoint_ref* subscription_manager)
{
    int ret;
    struct _wse__Unsubscribe unsubscribe;
    struct __wse__UnsubscribeOpResponse response;

    DC_CHECK_PARAM(dpws && subscription_manager);

    ret = dpws_call___wse__UnsubscribeOp(dpws, subscription_manager, NULL, &unsubscribe, &response);
    dpws_init_headers_and_version(dpws);
    return ret;
}

/*******************************************************************************
*                                     RUNTIME                                  *
*******************************************************************************/

/*----------------------------------------------------------------------------*\
 *                                  ADDRESSING                                *
\*----------------------------------------------------------------------------*/
int dpws_headers_out(struct dpws* dpws)
{
	int status = SOAP_OK;
    struct soap * soap = dpws_dpws2soap(dpws);
    // check that we are in DPWS context
    if (!(soap->omode & DPWS_HEADERS))
        goto exit;
    soap->part = SOAP_IN_ENVELOPE;	// NOTE: hack because gSoap 2.6.2 defines default namespaces locally for every element
    if (dpws->to)
        status = soap_simple_element_out(soap, "wsa:To", dpws->to);
    if (!status && dpws->action)
    	status = soap_simple_element_out(soap, "wsa:Action", dpws->action);
    if (!status && dpws->message_id)
    	status = soap_simple_element_out(soap, "wsa:MessageID", dpws->message_id);
    if (!status && dpws->relates_to)
    	status = soap_simple_element_out(soap, "wsa:RelatesTo", dpws->relates_to);

    if (!DPWS_IS_RESPONSE(dpws->status))
    {
        if (!status && !dpws_endpoint_ref_is_null(dpws->from))
        	status = dpws_out_wsa_endpoint_ref(soap, "wsa:From", &dpws->from);
        if (!status && !dpws_endpoint_ref_is_null(dpws->reply_to))
        	status = dpws_out_wsa_endpoint_ref(soap, "wsa:ReplyTo", &dpws->reply_to);
        if (!status && !dpws_endpoint_ref_is_null(dpws->fault_to))
        	status = dpws_out_wsa_endpoint_ref(soap, "wsa:FaultTo", &dpws->fault_to);
    }
    if (!status && DPWS_APP_SEQ_MSG(dpws->status))
    {
    	if ((status = soap_set_attr(soap, "InstanceId", soap_unsignedLong2s(soap, dpws->instance_id)))
    		|| (status = soap_set_attr(soap, "MessageNumber", soap_unsignedShort2s(soap, dpws->msg_nb)))
    		|| (status = soap_empty_element_out(soap, "wsd:AppSequence")))
    		goto exit;
    }
    if (!status && dpws->subscription_id)
    	status = soap_simple_element_out(soap, "wse:Identifier", dpws->subscription_id);

#ifdef WITH_WSMAN
	if (!status && dpws->wsman_headers.ref_params.resource_uri)
		status = soap_simple_element_out(soap, "wsman:ResourceURI", dpws->wsman_headers.ref_params.resource_uri);
	if (!status && dpws->wsman_headers.ref_params.nb_selectors > 0)
    {
	    int i;
	    if ((status = soap_element_begin_out(soap, "wsman:SelectorSet", -1, "")))
	    	goto exit;
        for (i = 0; i < dpws->wsman_headers.ref_params.nb_selectors; i++)
        {
        	if ((status = soap_set_attr(soap, "Name", dpws->wsman_headers.ref_params.selectors[i].name))
        		|| (status = soap_simple_element_out(soap, "wsman:Selector", dpws->wsman_headers.ref_params.selectors[i].value)))
        		goto exit;
        }
        status = soap_element_end_out(soap, "wsman:SelectorSet");
    }
    if (!status && dpws->wsman_headers.fragment_transfer) {
    	int i;
    	for (i = 0; i < dpws->wsman_headers.fragment_transfer->nb_pfx_def; i++) {
    		struct prefix_def pfx_def = dpws->wsman_headers.fragment_transfer->ns_ctx[i];
			if (!soap_check_prefix_definition(soap, pfx_def.ns_uri, pfx_def.ns_prefix)) {
				status = soap->error;
    			goto exit;
			}
    	}
    	if ((status = soap_simple_element_out(soap, "wsman:FragmentTransfer", dpws->wsman_headers.fragment_transfer->exp)))
    		goto exit;
		soap_pop_namespace(soap);
    }
	if (!status && dpws->wsman_headers.bookmark)
		status = soap_simple_element_out(soap, "wsman:Bookmark", dpws->wsman_headers.bookmark);
#endif

    dpws->soap.part = SOAP_IN_HEADER;

exit:
    return status;
}

#ifdef WITH_WSMAN

static int wsman_selector_set_in(struct dpws* dpws)
{
    struct soap * soap = dpws_dpws2soap(dpws);
    da_allocator_t daa;
	DA_TYPED(wsmansel) selectors;
	int ret = DPWS_OK;

	DA_INIT(struct wsman_selector, &selectors, DC_MEM_WSMAN, soap_get_da_allocator(soap, &daa), 2);

	if (soap_element_begin_in(soap, "wsman:SelectorSet", 0))
        return soap->error;
    while (soap_element_begin_in(soap, "wsman:Selector", 0) != SOAP_NO_TAG) // ?: SOAP_TAG_MISMATCH ?
    {
    	struct wsman_selector * p_sel;
        if (soap->error)
            return soap->error;
        DC_ERROR_ASSERT_ALLOC(p_sel = DA_ADD(&selectors));
		if (soap_s2string(soap, soap_attr_value(soap, "Name", 1), &p_sel->name))
			return soap->error;
        if (!(p_sel->value = soap_string_in(soap, 0, -1, -1)))
			return soap->error;
	    if (soap->body && soap_element_end_in(soap, "wsman:Selector"))
	        return soap->error;
    }
    if (soap->body && soap_element_end_in(soap, "wsman:SelectorSet"))
        return soap->error;
	dpws->wsman_headers.ref_params.nb_selectors = selectors.nb;
	dpws->wsman_headers.ref_params.selectors = selectors.tab;

DC_FUNC_ERROR
	return ret;
}

static int wsman_xpath_in(struct dpws* dpws, const char * tag)
{
	int status;
    if (!(status = soap_element_begin_in(dpws_dpws2soap(dpws), tag, 0)) && dpws_dpws2soap(dpws)->body) {
		if (!(dpws->wsman_headers.fragment_transfer =
			(struct xpath_expression*)DC_MSG_MALLOC(DC_MEM_WSMAN, dpws, sizeof(struct xpath_expression)))) {
			status = SOAP_EOM;
			goto error;
		}
		dpws->wsman_headers.fragment_transfer->exp = soap_string_in(dpws_dpws2soap(dpws), 0, -1, -1);
		dpws->wsman_headers.fragment_transfer->ns_ctx = soap_build_ns_context(dpws_dpws2soap(dpws), &dpws->wsman_headers.fragment_transfer->nb_pfx_def);
	    status = soap_element_end_in(dpws_dpws2soap(dpws), tag);
    }
error:
    return status;
}

#endif

int dpws_header_in(struct dpws* dpws)
{
    struct soap * soap = dpws_dpws2soap(dpws);
    // check that we are in DPWS context
    if (!(soap->imode & DPWS_HEADERS))
        return -1;
    if ((!dpws->to && !wsa_simple_element_in(dpws, "wsa:To", &dpws->to))
            || (!dpws->action && !wsa_simple_element_in(dpws, "wsa:Action", &dpws->action))
            || (!dpws->message_id && !wsa_simple_element_in(dpws, "wsa:MessageID", &dpws->message_id))
            || (!dpws->relates_to && !wsa_simple_element_in(dpws, "wsa:RelatesTo", &dpws->relates_to))
            || (!dpws->subscription_id && !soap_simple_element_in(dpws_dpws2soap(dpws), "wse:Identifier", &dpws->subscription_id))
            || (DPWS_IS_UDP_RCV_MULTICAST(dpws->status) && dpws->msg_nb == 0 && dpws_in_wsd_AppSequence(dpws))
            || (dpws_endpoint_ref_is_null(dpws->from) && !dpws_in_wsa_endpoint_ref(dpws_dpws2soap(dpws), "wsa:From", &dpws->from))
            || (dpws_endpoint_ref_is_null(dpws->reply_to) && !dpws_in_wsa_endpoint_ref(dpws_dpws2soap(dpws), "wsa:ReplyTo", &dpws->reply_to))
            || (dpws_endpoint_ref_is_null(dpws->fault_to) && !dpws_in_wsa_endpoint_ref(dpws_dpws2soap(dpws), "wsa:FaultTo", &dpws->fault_to))
#ifdef WITH_WSMAN
			|| (!dpws->wsman_headers.ref_params.resource_uri && !soap_simple_element_in(dpws_dpws2soap(dpws), "wsman:ResourceURI", &dpws->wsman_headers.ref_params.resource_uri))
			|| (dpws->wsman_headers.ref_params.nb_selectors <= 0 && !wsman_selector_set_in(dpws))
            || (!dpws->wsman_headers.fragment_transfer && !wsman_xpath_in(dpws, "wsman:FragmentTransfer"))
            || (!dpws->wsman_headers.bookmark && !soap_simple_element_in(dpws_dpws2soap(dpws), "wsman:Bookmark", &dpws->wsman_headers.bookmark))
#endif
	)
		return 0;
    return soap->error;
}

void dpws_init_headers(struct dpws* dpws)
{
    dpws->to = NULL;
    dpws->action = NULL;
    dpws->message_id = NULL;
    dpws->relates_to = NULL;
    dpws->instance_id = 0;
    dpws->msg_nb = 0;
    dpws_default_wsa_endpoint_ref(&dpws->from);
    dpws_default_wsa_endpoint_ref(&dpws->reply_to);
    dpws_default_wsa_endpoint_ref(&dpws->fault_to);
    DPWS_INIT_STATUS(dpws->status);
    dpws->subscription_id = NULL;
#ifdef WITH_WSMAN
    memset(&dpws->wsman_headers, 0, sizeof(struct wsman_headers));
#endif
}

void dpws_init_headers_and_version(struct dpws* dpws)
{
	dpws_init_headers(dpws);
	dpws->protocols = NULL;
	dpws->wsa_version = NULL;
}

int dpws_check_headers(struct dpws *dpws)
{
    if (!registry_cfg.bp1_1_compatibility && (
    		!dpws->to
            || (!dpws_endpoint_ref_is_null(dpws->reply_to) && !dpws->message_id)
            || !dpws->action
            )
    )
        return dpws_dpws2soap(dpws)->error = WSA_ERR_REQUIRED_HEADER_INFO;
	if (!dpws->wsa_version)
		dpws->wsa_version = default_protocols->wsa_version;
    return SOAP_OK;
}

int dpws_set_call_headers(struct dpws* dpws,
                           struct wsa_endpoint_ref *to, struct wsa_endpoint_ref *reply_to, char *wsa_action)
{
	char * mid;
    DPWS_SET_REQUEST(dpws->status);
	dpws->to = DPWS_IS_DIRECTED_PROBE(dpws->status) ? dpws->protocols->wsd_urn : to->address;	// Handles the case of the directed probe
    if (dpws->source)
        memcpy(&dpws->from, dpws->source, sizeof(struct wsa_endpoint_ref));
    dpws->action = wsa_action;
    if (reply_to)
        memcpy(&dpws->reply_to, reply_to, sizeof(struct wsa_endpoint_ref));
    else
        dpws_set_anonymous_wsa_endpoint_ref(dpws, &dpws->reply_to);
    // fault is optional
    if (!(mid = (char *)DC_MSG_MALLOC(DC_MEM_MSG_PROCESSING, dpws, SCHEME_UUID_STRING_SIZE)))
    	return DPWS_ERR_EOM;
    dpws->message_id = dpws_schemed_uuid_create(mid);
    dpws->relates_to = NULL;
    dpws->subscription_id = to->subscription_id;
#ifdef WITH_WSMAN
	dpws->wsman_headers.ref_params = to->wsman_params;	// struct copy
#endif
	return DPWS_OK;
}

void dpws_set_oneway_headers(struct dpws* dpws,	struct wsa_endpoint_ref *to, char *wsa_action)
{
    if (!DPWS_IS_DISCOVERY(dpws->status))	// NOTE: no need normally if the user call dpws_end between every call. Discovery APIs resets headers (no dpws_end after possibly them)
        dpws->message_id = NULL;
    DPWS_SET_REQUEST(dpws->status);
    dpws->to = to->address;
    if (dpws->source)
        memcpy(&dpws->from, dpws->source, sizeof(struct wsa_endpoint_ref));
    dpws->action = wsa_action;
    dpws_default_wsa_endpoint_ref(&dpws->reply_to);
    dpws_default_wsa_endpoint_ref(&dpws->fault_to);
    if (!(dpws->status & (DPWS_MSG_PROBE_MATCH|DPWS_MSG_RESOLVE_MATCH|DPWS_MSG_DP_HELLO)))	// are responses even if formally not !
        dpws->relates_to = NULL;
    dpws->subscription_id = to->subscription_id;
#ifdef WITH_WSMAN
	dpws->wsman_headers.ref_params = to->wsman_params;
#endif
}

void dpws_set_reply_headers(struct dpws* dpws, char *wsa_action)
{
    DPWS_SET_RESPONSE(dpws->status);
	if (dpws->to)	// For BP1.1 compatibility
	{
		dpws->to = dpws->reply_to.address;
		if (dpws->source)
			memcpy(&dpws->from, dpws->source, sizeof(struct wsa_endpoint_ref));
		else
			dpws_default_wsa_endpoint_ref(&dpws->from);
		dpws_default_wsa_endpoint_ref(&dpws->reply_to);
		dpws_default_wsa_endpoint_ref(&dpws->fault_to);
		dpws->action = wsa_action;
		dpws->relates_to = dpws->message_id;
		dpws->message_id = NULL;
	    dpws->subscription_id = dpws->reply_to.subscription_id;
	}
#ifdef WITH_WSMAN
    // set management headers for response
    // 1. ResourceId is ommitted R2.1.1-4
    // 2. Selector
	dpws->wsman_headers.ref_params.resource_uri = NULL;
	dpws->wsman_headers.ref_params.nb_selectors = -1;
#endif
}

static int dpws_is_oneway(struct dpws* dpws)
{
    return dpws->to && // BP1.1 compatibility
		dpws_endpoint_ref_is_null(dpws->reply_to);
}

int dpws_is_asynchronous(struct dpws* dpws)
{
    return dpws->to	&& // BP1.1 compatibility
		(dpws_endpoint_ref_is_null(dpws->reply_to) || strcmp(dpws->reply_to.address, dpws->wsa_version->wsa_anonymous_uri));
}

int dpws_receive_empty_response(struct dpws* dpws)
{
	transport_data_t * tdata = (transport_data_t *)dpws->transport_data;
	struct soap* soap = dpws_dpws2soap(dpws);
	if (tdata && tdata->tclass->frecvemptyresponse)
		if (tdata->tclass->frecvemptyresponse(dpws)) {
			soap->error = SOAP_EOF;
			soap->errnum = tdata->error.syserr;
		}
    return soap_closesock(soap);
}

int dpws_send_empty_response(struct dpws* dpws)
{
	DPWS_SET_RESPONSE(dpws->status);
	return dpws->transport_fns->fresponse(dpws, dpws->transport_data, DC_SOAP_EMPTY_RESPONSE, NULL, 0);
}

static int dpws_set_fault(struct dpws *dpws)
{
	int ret = SOAP_OK;
    struct soap *soap = dpws_dpws2soap(dpws);
    const char **c = soap_faultcode(soap);
    const char **sc = soap_faultsubcode(soap);
    const char **s = soap_faultstring(soap);
    const char **d = soap_faultdetail(soap);

    /* Set the fault body */
    switch (soap->error)
    {
    case WSA_ERR_INVALID_HEADER_INFO_MESSAGE_ID:
    case WSA_ERR_INVALID_HEADER_INFO_RELATES_TO:
    case WSA_ERR_INVALID_HEADER_INFO_TO:
    case WSA_ERR_INVALID_HEADER_INFO_ACTION:
    case WSA_ERR_INVALID_HEADER_INFO_FROM:
    case WSA_ERR_INVALID_HEADER_INFO_REPLY_TO:
    case WSA_ERR_INVALID_HEADER_INFO_FAULT_TO:
        *c = "SOAP-ENV:Sender";
        *sc = "wsa:InvalidMessageInformationHeader";
        *s = "A message information header is not valid and the message cannot be processed"; // spec message truncated
        buffer_marshalling_init(dpws);
        dpws_headers_out(dpws);
        *d = buffer_marshalling_end(dpws);
        break;
    case WSA_ERR_REQUIRED_HEADER_INFO:
        *c = "SOAP-ENV:Sender";
        *sc = "wsa:MessageInformationHeaderRequired";
        *s = "A required message information header, To, MessageID, or Action, is not present.";
        *d = soap->tag; // put it in when raising the error if not already done
        break;
    case WSA_ERR_DESTINATION_UNREACHABLE:
        *c = "SOAP-ENV:Sender";
        *sc = "wsa:DestinationUnreachable";
        *s = "No route can be determined to reach the destination role defined by the WS-Addressing To.";
        break;
    case WSA_ERR_ACTION_NOT_SUPPORTED:
        *c = "SOAP-ENV:Sender";
        *sc = "wsa:ActionNotSupported";
        *s = "The [action] cannot be processed at the receiver.";
        *d = dpws->action;
        break;
        //case WSA_ERR_ENDPOINT_UNAVAILABLE:
        //	*c = "SOAP-ENV:Receiver";
        //break;
    case WSE_ERR_DELIVERY_MODE_REQUESTED_UNAVAILABLE:
        *c = "SOAP-ENV:Sender";
        *sc = "wse:DeliveryModeRequestedUnavailable";
        *s = "The requested delivery mode is not supported.";
        strcpy(soap->msgbuf, "<wse:SupportedDeliveryMode>");
        strcat(soap->msgbuf, WSE_MODE_PUSH);
        strcat(soap->msgbuf, "</wse:SupportedDeliveryMode>");
        *d = soap->msgbuf;
        break;
    case WSE_ERR_INVALID_EXPIRATION_TIME:
        *c = "SOAP-ENV:Sender";
        *sc = "wse:InvalidExpirationTime";
        *s = "The expiration time requested is invalid.";
        break;
    case WSE_ERR_UNSUPPORTED_EXPIRATION_TYPE:
        *c = "SOAP-ENV:Sender";
        *sc = "wse:UnsupportedExpirationType";
        *s = "Only expiration durations are supported.";
        break;
    case WSE_ERR_FILTERING_NOT_SUPPORTED:
        *c = "SOAP-ENV:Sender";
        *sc = "wse:FilteringNotSupported";
        *s = "Filtering is not supported.";
        break;
    case WSE_ERR_FILTERING_REQUESTED_UNAVAILABLE:
        *c = "SOAP-ENV:Sender";
        *sc = "wse:FilteringRequestedUnavailable";
        *s = "The requested filter dialect is not supported.";
        strcpy(soap->msgbuf, "<wse:SupportedDialect>");
        strcat(soap->msgbuf, dpws->protocols->wdp_wse_filtering_dialect);
        strcat(soap->msgbuf, "</wse:SupportedDialect>");
        *d = soap->msgbuf;
        break;
        //case WSE_ERR_EVENT_SOURCE_UNABLE_TO_PROCESS:
        //	*c = "SOAP-ENV:Receiver";
        //break;
    case WSE_ERR_UNABLE_TO_RENEW:
        *c = "SOAP-ENV:Receiver";
        *sc = "wse:UnableToRenew";
        *s = "The referenced subscription does not exist";
        break;
    case WSE_ERR_INVALID_SUBSCRIPTION_ID:	// Not in the spec : invented
        *c = "SOAP-ENV:Sender";
        *sc = "wse:InvalidMessage";
        *s = "The message is not valid and cannot be processed.";
        buffer_marshalling_init(dpws);
        subscManager_putelement(dpws, dpws->err_detail_data, dpws->err_detail_data_elt, dpws->err_detail_type);
        *d = buffer_marshalling_end(dpws);
        break;
	case WSE_ERR_INVALID_MESSAGE:
	case WSE_ERR_INVALID_SUBSCRIPTION:	// Not in the spec : invented
        *c = "SOAP-ENV:Sender";
        *sc = "wse:InvalidMessage";
        *s = "The message is not valid and cannot be processed.";
        buffer_marshalling_init(dpws);
        event_putelement(dpws, dpws->err_detail_data, dpws->err_detail_data_elt, dpws->err_detail_type);
        *d = buffer_marshalling_end(dpws);
        break;
    }

    /* Set the fault WSA Headers */
    DPWS_SET_RESPONSE(dpws->status);

    dpws->to = NULL;
    if (!dpws_endpoint_ref_is_null(dpws->fault_to))
    {
        dpws->to = dpws->fault_to.address;
        dpws->subscription_id = dpws->fault_to.subscription_id;
        dpws_default_wsa_endpoint_ref(&dpws->fault_to);
    }
    if (!dpws_endpoint_ref_is_null(dpws->reply_to))
    {
        if (!dpws->to)
        {
            dpws->to = dpws->reply_to.address;
            dpws->subscription_id = dpws->reply_to.subscription_id;
        }
        dpws_default_wsa_endpoint_ref(&dpws->reply_to);
    }
    if (!dpws_endpoint_ref_is_null(dpws->from))
    {
        if (!dpws->to)
        {
            dpws->to = dpws->from.address;
            dpws->subscription_id = dpws->from.subscription_id;
        }
        dpws_default_wsa_endpoint_ref(&dpws->from);
    }
    if (!dpws->to)
    {
        dpws->to = dpws->wsa_version->wsa_anonymous_uri;
        dpws->subscription_id = NULL;
    }

	if (soap->error != WSA_ERR_SOAP_FAULT || !dpws->action)
    	dpws->action = dpws->wsa_version->wsa_fault_uri;

    dpws->relates_to = dpws->message_id;
    if (!(dpws->message_id = dpws_schemed_uuid_create((char *)DC_MSG_MALLOC(DC_MEM_MSG_PROCESSING, dpws, SCHEME_UUID_STRING_SIZE)))) {
    	ret = DPWS_ERR_EOM;
    	goto error;
    }
    soap_set_fault(soap);

error:
    return ret;
}

int dpws_send_fault(struct dpws *dpws)
{
    struct soap *soap = dpws_dpws2soap(dpws);
    register int status = soap->error;

	if ((status != WSA_ERR_REQUIRED_HEADER_INFO) && DPWS_IS_RESPONSE(dpws->status))	// no need to send fault in one way (waiting for "robust oneway") or if response already sent
        return (soap->error = SOAP_OK);

	if (!dpws->protocols) {
		dpws->protocols = default_protocols;
		dpws->wsa_version = dpws->protocols->wsa_version;
	}

	if (dpws_set_fault(dpws))
		return DPWS_ERR_EOM;

	if (status != SOAP_EOF || (!soap->recv_timeout && !soap->send_timeout))
    {
 		soap->encodingStyle = NULL;
        soap->error = SOAP_OK;
        soap_serializeheader(soap);
        soap_serializefault(soap);
        soap_begin_count(soap);
        if (soap->mode & SOAP_IO_LENGTH)
        {
            soap_envelope_begin_out(soap);
            soap_putheader(soap);
            soap_body_begin_out(soap);
            soap_putfault(soap);
            soap_body_end_out(soap);
            soap_envelope_end_out(soap);
        }
		soap_end_count(soap);
        if (dpws_response(dpws, status)
            || soap_envelope_begin_out(soap)
            || soap_putheader(soap)
            || soap_body_begin_out(soap)
            || soap_putfault(soap)
            || soap_body_end_out(soap)
            || soap_envelope_end_out(soap))
	      return soap_closesock(soap);
		soap_end_send(soap);
	}
	soap->error = status;
	return soap_closesock(soap);
}

static struct media_type_param * set_content_type_param(struct media_type_param * param, const char * att, const char * value, struct media_type_param * next)
{
	param->attribute = att;
	param->value = value;
	param->next = next;
	return param;
}

static void get_content_type(struct dpws * dpws, int status, struct media_type * mtype, struct media_type_param * params, int nparams)
{
	struct soap * soap = dpws_dpws2soap(dpws);
	char * type = NULL, * startinfo = NULL;

	mtype->subtype = NULL;
	mtype->params = NULL;
	if (status == SOAP_FILE && soap->http_content) {
		// Hack: use complete serialized media type as main type
		mtype->type = soap->http_content;
	} else if (status == SOAP_HTML) {
		mtype->type = "text";
		mtype->subtype = "html";
		if (nparams > 0)
			mtype->params = set_content_type_param(params, "charset", "utf-8", NULL);
	} else if (soap->version == 2) {
		mtype->type = "application";
		mtype->subtype = "soap+xml";
		if (nparams > 0)
			mtype->params = set_content_type_param(params, "charset", "utf-8", NULL);
		type = "application/soap+xml";
	} else {
		mtype->type = "text";
		mtype->subtype = "xml";
		if (nparams > 0)
			mtype->params = set_content_type_param(params, "charset", "utf-8", NULL);
		type = "text/xml";
	}
	if (soap->mode & (SOAP_ENC_DIME | SOAP_ENC_MTOM)) {
		if (soap->mode & SOAP_ENC_MTOM)
		{
			startinfo = "application/soap+xml";
			type = "application/xop+xml";
		}
		else
			type = "application/dime";
	}
	if ((soap->mode & SOAP_ENC_MIME) && soap->mime.boundary) {
		int index = 0;
		mtype->type = "multipart";
		mtype->subtype = "related";
		if (index < nparams) {
			mtype->params = set_content_type_param(params+index, "boundary", soap->mime.boundary, NULL);
			index++;
		}
		if (index < nparams) {
			mtype->params = set_content_type_param(params+index, "type", type, params+index-1);
			index++;
		}
		if (index < nparams && soap->mime.start) {
			mtype->params = set_content_type_param(params+index, "start", soap->mime.start, params+index-1);
			index++;
		}
		if (index < nparams && startinfo) {
			mtype->params = set_content_type_param(params+index, "start-info", startinfo, params+index-1);
			index++;
		}
	}
}

static int dpws_use_anonymous_response(struct dpws* dpws)
{
    return !dpws->to	// For BP 1.1 compatibility (cannot happen if turned off)
		|| !strcmp(dpws->to, dpws->wsa_version->wsa_anonymous_uri);
}

int dpws_connect(struct dpws* dpws, const char* address, const char* action)
{
	int ret = DPWS_OK;
	const char * endpoint = NULL;
	struct soap * soap = dpws_dpws2soap(dpws);
	transport_data_t * tdata = (transport_data_t *)dpws->transport_data;
	size_t count = dc_soap_count_attachments(soap);
	media_type_t mtype;
	media_type_param_t params[4];

	get_content_type(dpws, 0, &mtype, params, 4);
	if (is_discovery_urn(dpws, address)) {
		if (!tdata)
			ret = WSA_ERR_DESTINATION_UNREACHABLE;
    } else if (!strcmp(address, dpws->wsa_version->wsa_anonymous_uri)) {
		endpoint = NULL;
	} else if (!is_physical_address(address)) {
		ret = spawn_resolve(dpws, &endpoint, address);
	} else {
		endpoint = address;
	}
	if (ret)
		return ret;

	if (!endpoint) {
		if (tdata)
			ret = tdata->tclass->fconnect(dpws, NULL, action, &mtype, count);
		else
			ret = WSA_ERR_DESTINATION_UNREACHABLE;
	} else {
		if (tdata)
			ret = tdata->tclass->fconnect(dpws, endpoint, action, &mtype, count);
		else
			ret = dc_http_connect(dpws, endpoint, action, &mtype, count);
	}
	if (ret)
		return ret;

	return soap_begin_send(soap);
}

int dpws_response(struct dpws *dpws, int status)
{
	int ret = DPWS_OK;
    const char * reply_host = dpws->to;
	struct soap * soap = dpws_dpws2soap(dpws);
	size_t count = dc_soap_count_attachments(soap);
	media_type_t mtype;
	media_type_param_t params[4];

	get_content_type(dpws, status, &mtype, params, 4);

	if (status == 0 || status == 200 || status == SOAP_HTML || status == SOAP_FILE) {
		status = DC_SOAP_RESPONSE;
	} else if (status < 200 || status > 600) { // Not an HTTP status code: SOAP processing error
		const char *s = *soap_faultcode(soap);
		if (s && soap->version == 2 && !strcmp(s, "SOAP-ENV:Sender"))
			status = DC_SOAP_SENDER_ERROR;
		else
			status = DC_SOAP_RECEIVER_ERROR;
	}

	if (dpws_use_anonymous_response(dpws)) {
		ret = dpws->transport_fns->fresponse(dpws, dpws->transport_data, status, &mtype, count);
		if (!ret) {
			ret = soap_begin_send(soap);
		}
	} else {
		// This will replace the input channel by a new output channel
		dpws->transport_data = NULL;
		dpws->transport_fns = NULL;
		soap->omode = dpws->user_mode;
		ret = dpws_connect(dpws, reply_host, dpws->action);
    }
    return ret;
}

void dpws_set_namespaces(struct dpws *dpws, struct Namespace *p)
{
	struct soap * soap = dpws_dpws2soap(dpws);
	if (p != soap->namespaces || !soap->local_namespaces) {
		char* soap_uri = soap->local_namespaces ? soap->local_namespaces[0].out : NULL;
		if (soap_uri)
			soap->local_namespaces[0].out = NULL;
        soap_set_namespaces(soap, p);
		if (soap_uri) {
			if (soap->local_namespaces[0].out)
				SOAP_FREE(soap, soap->local_namespaces[0].out); // Must use SOAP_FREE to be consistent with allocation
			soap->local_namespaces[0].out = soap_uri;
			if (!strcmp(soap_uri, SOAP_ENV_URI))
				soap->version = 2;
			else
				soap->version = 1;
		}	
	}
	if (dpws->wsa_version)
		soap_update_namespace_uri(soap, "wsa", dpws->wsa_version->wsa_uri);
}

/* provide a SCHEME_UUID_STRING_SIZE octet char buffer */
char * dpws_schemed_uuid_create(char *uuid)	// QUESTION: should protect ?
{
    strcpy(uuid, UUID_SCHEME);
    uuid_create(uuid + UUID_SCHEME_SIZE, dcpl_mac_address);
    return uuid;
}

int dpws_check_endpoint(struct dpws * dpws, struct wsa_endpoint_ref * to)
{
	if (to && to->dpws_version) {
		dpws->protocols = to->dpws_version;
	} else if (!dpws->protocols) {
		dpws->protocols = default_protocols;
	}
	dpws->wsa_version = dpws->protocols->wsa_version;
	return DPWS_OK;
}
