/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|                      ->>  Copyright 2008 Odonata <<-                         |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1772 $
|                     $Date: 2008-10-09 11:57:03 +0200 (jeu., 09 oct. 2008) $
\*============================================================================*/

#include "dcDPWS_Http.h"
#include "dcDPWS_Reactor.h"
#include "dcDPWS_ConnPool.h"
#include "dcDPWS_Network.h"
#include "dcDCPL_Socket.h"
#include "dcDPWS_Dpws.h"
#include "dcCOMN_Tools.h"
#include "dc/dc_Error.h"
#include "dc/dc_Dpws.h"
#include "dc/dc_DpwsRequest.h"


/*-------------------------------------- Types -------------------------------*/

#define DC_HTTP_HEADERS_PEEK_SIZE 1024
#define DC_HTTP_UNKNOWN_LENGTH (-1)
#define DC_HTTP_UNSUPPORTED_CONTENT_ENCODING_ERROR		-100

#define http_blank(c)				((c) >= 0 && (c) <= 32)
#define http_notblank(c)			((c) > 32)
#define http_xdigit(c)				(((c) >= '0' && (c) <= '9') ? (c) - '0' : ((c) >= 'a' && (c) <= 'f') ? 10 + (c) - 'a' : ((c) >= 'A' && (c) <= 'F') ? 10 + (c) - 'A' : -1)
#define http_error(soap, status)	""

#define http_buf_unget(tdata, c) ((tdata)->ahead = c)
#define http_buf_get0(tdata) (((tdata)->index >= (tdata)->buflen && http_buf_fill(tdata)) ? EOF : (unsigned char)(tdata)->buf[(tdata)->index])
#define http_buf_get1(tdata) (((tdata)->index >= (tdata)->buflen && http_buf_fill(tdata)) ? EOF : (unsigned char)(tdata)->buf[(tdata)->index++])

/*------------------------- Static Functions prototypes ----------------------*/

static int convert_dcpl_tcp_error(struct dcpl_error * dcpl_error);
static int convert_dcpl_server_error(struct dpws * dpws, struct dcpl_error * dcpl_error);
static int convert_dcpl_client_error(struct dpws * dpws, struct dcpl_error * dcpl_error);
static http_transport_data_t * init_http_transport_data(http_transport_data_t * tdata);
static int http_send(struct dpws * dpws, void * transport_data, const char * buf, size_t buflen);
static int http_response(struct dpws * dpws, void * transport_data, int status, media_type_t * mtype, size_t len);
static int http_recv(struct dpws * dpws, void * transport_data, char * buf, size_t buflen);
static int http_recvemptyresponse(struct dpws * dpws);
static int http_close(struct dpws * dpws);
static int http_setup_channel(struct dpws * dpws, struct transport_data * tdata);
static int http_teardown_channel(struct dpws * dpws, struct transport_data * tdata);
static int http_close_channel(struct transport_data * tdata);
static int http_serve_request(struct dpws * dpws, struct reactor_item * item, void* callback_data);
static int http_free_transport_data(struct reactor_item * item, DCPL_SOCKET socket, void* callback_data);
static int http_recv_headers(struct dpws * dpws);
static int http_send_request_headers(struct dpws * dpws, const char * endpoint, const char * host, unsigned short port, const char * path, const char * action, struct media_type * mtype, size_t count);
static int http_send_response_headers(struct dpws * dpws, int status, struct media_type * mtype, size_t len);
static int http_flush_output(struct http_transport_data * tdata);
static int http_cleanup_input(struct http_transport_data * tdata);
static int http_buf_flush(struct http_transport_data * tdata);
static int http_buf_write(struct http_transport_data * tdata, const char *s, size_t len);
static int http_buf_fill(struct http_transport_data * tdata);
static int http_buf_skip(struct http_transport_data * tdata, int len);
static int http_buf_getchar(struct http_transport_data * tdata);
static int http_buf_getline(struct http_transport_data * tdata, char *s, int len);
static int http_post_header(struct http_transport_data * tdata, const char *key, const char *val);
static int http_send_header(struct http_transport_data * tdata, const char *s);

/*----------------------------------- Data -----------------------------------*/

static struct transport_class http_transport_class = {
	http_send,
	http_recv,
	http_response,
	dc_http_connect,
	http_recvemptyresponse,
	http_close,
	NULL,
	http_setup_channel,
	http_teardown_channel,
	http_close_channel
};

/*----------------------------------------------------------------------------*/

static int convert_dcpl_tcp_error(struct dcpl_error * dcpl_error)
{
	int dpwserror;

	switch (dcpl_error->error) {
		case DCPL_SOCKET_CREATE_ERROR: dpwserror = DPWS_ERR_COULD_NOT_CREATE_SOCKET; break;
		case DCPL_SOCKET_BIND_ERROR: dpwserror = DPWS_ERR_COULD_NOT_BIND_SOCKET; break;
		case DCPL_SOCKET_REUSEADDR_ERROR: dpwserror = DPWS_ERR_COULD_NOT_SET_REUSE_ADDR_OPTION; break;
		case DCPL_SOCKET_GETSOCKNAME_ERROR: dpwserror = DPWS_ERR_SOCKET_ERROR; break;
		default: dpwserror = DPWS_ERR_SOCKET_ERROR; break;
	}
	return dpwserror;
}

static int convert_dcpl_server_error(struct dpws * dpws, struct dcpl_error * dcpl_error)
{
	int dpwserror = convert_dcpl_tcp_error(dcpl_error);
	if (dpws) {
		struct soap * soap = dpws_dpws2soap(dpws);
		soap->errnum = dcpl_error->syserr;
		soap_set_receiver_error(soap, dcpl_error_string(dcpl_error, soap->msgbuf, sizeof(soap->msgbuf)), dcpl_error->detail, SOAP_TCP_ERROR);
	}
	return dpwserror;
}

static int convert_dcpl_client_error(struct dpws * dpws, struct dcpl_error * dcpl_error)
{
	int dpwserror = convert_dcpl_tcp_error(dcpl_error);
	if (dpws) {
		struct soap * soap = dpws_dpws2soap(dpws);
		soap->errnum = dcpl_error->syserr;
		soap_set_sender_error(soap, dcpl_error_string(dcpl_error, soap->msgbuf, sizeof(soap->msgbuf)), dcpl_error->detail, SOAP_TCP_ERROR);
	}
	return dpwserror;
}

static http_transport_data_t * init_http_transport_data(http_transport_data_t * tdata)
{
	http_transport_data_t * result =
		(http_transport_data_t *)dc_transport_data_init((transport_data_t *)tdata, sizeof(http_transport_data_t), &http_transport_class);

	if (!result) {
		return NULL;
	}
	return result;
}

int dc_http_add_listener(struct reactor * reactor, int bind_flags, int family, const char* addr, unsigned short port, uint32_t itf_selector, int backlog)
{
	struct socket_data bind_data;
	struct dcpl_error error;
	DCPL_SOCKET socket;
	int ret = DPWS_OK;

	bind_data.flags = bind_flags;
	bind_data.buflen = SOAP_BUFLEN;
	bind_data.keep_alive = 0;
	bind_data.tcp_nodelay = 1;
	bind_data.linger_time = 0;
	if ((socket = dcpl_tcp_bind_listener(family, addr, itf_selector, port, backlog, &bind_data, &error)) == DCPL_INVALID_SOCKET) {
		reactor->error = error.error;
		reactor->detail = error.detail;
		reactor->syserr = error.syserr;
		return convert_dcpl_tcp_error(&error);
	}
	if ((ret = dc_reactor_register_socket(reactor, socket, DC_RI_READ | DC_RI_FACTORY, dc_http_accept_request, NULL, NULL, NULL))) {
		dcpl_closesocket(socket, NULL);
		return ret;
	}
	return ret;
}

int dc_http_accept_request(struct dpws * dpws, struct reactor_item * item, void* callback_data)
{
	DCPL_SOCKET socket = DCPL_INVALID_SOCKET;
	dcpl_error_t error;
	socket_data_t accept_data;
	struct soap * soap = dpws_dpws2soap(dpws);
	http_transport_data_t * tdata = init_http_transport_data(NULL);
	struct reactor_item * new_item;

	if (!tdata)
		return DPWS_ERR_EOM;
	accept_data.flags = soap->accept_flags;
	accept_data.buflen = SOAP_BUFLEN;
	accept_data.keep_alive = ((soap->imode & SOAP_IO_KEEPALIVE) != 0);
	accept_data.tcp_nodelay = 1;
	accept_data.linger_time = 0;
	if ((socket = dcpl_tcp_accept(item->socket, &accept_data, tdata->base.peer_addr, sizeof(tdata->base.peer_addr), &tdata->base.peer_port, &tdata->base.itf, &error)) == DCPL_INVALID_SOCKET) {
		dc_transport_data_clear((transport_data_t *)tdata);
		convert_dcpl_server_error(dpws, &error);
		return DPWS_ERR_COULD_NOT_CREATE_SOCKET;
	}
	tdata->base.socket = socket;
	new_item = dc_reactor_create_socket_item(item->reactor, socket, DC_RI_READ, http_serve_request, tdata, http_free_transport_data, NULL);
	if (!new_item) {
		dcpl_closesocket(socket, &error);
		dc_transport_data_clear((transport_data_t *)tdata);
		return DPWS_ERR_EOM;
	}
	dpws->active_item = new_item;
	DPWSLOG1(DC_TRANSPORT, "Opening HTTP socket %d for reception", socket);
	return DPWS_OK;
}

static int http_serve_request(struct dpws * dpws, struct reactor_item * item, void* callback_data)
{
	http_transport_data_t * tdata = (http_transport_data_t *)callback_data;
	struct soap * soap = dpws_dpws2soap(dpws);
	int n = soap->max_keep_alive, ret = DPWS_OK;

	do {
		dc_transport_setup_channel(dpws, (transport_data_t *)tdata, tdata->base.socket, DC_TRANSPORT_INPUT);
		dcpl_init_error(&tdata->base.error);
		if (--n <= 0) {
			tdata->base.status &= ~DC_HTTP_KEEPALIVE;
			tdata->base.status |= DC_HTTP_NOKEEPALIVE;
		}
		DPWSLOG1(DC_TRANSPORT, "Waiting for/receiving HTTP request headers on socket %d", tdata->base.socket);
		if (!(ret = http_recv_headers(dpws))) {
			if (tdata->code == DC_HTTP_POST) {
				media_type_t * ctype = NULL; // Already processed in headers
				DC_BOOL needs_length = ((tdata->base.status & DC_HTTP_CHUNKED_OUTPUT) || !(tdata->base.status & DC_HTTP_KEEPALIVE)) ? DC_FALSE : DC_TRUE;
				DPWSLOG1(DC_TRANSPORT, "Processing POST request for URL %s", soap->path);
				if ((ret = dpws_dispatch_request(dpws, tdata, (transport_fns_t *)tdata->base.tclass, soap->host, soap->path, ctype, soap->action, needs_length))) {
					n = 0;
				}
			} else if (tdata->code == DC_HTTP_GET && soap->fget) {
				DPWSLOG1(DC_TRANSPORT, "Processing GET request for URL %s", soap->path);
				dpws->transport_data = tdata;
				dpws->transport_fns = (transport_fns_t *)tdata->base.tclass;
				if ((ret = soap->fget(soap))) {
					n = 0;
				}
				dpws->transport_data = NULL;
				dpws->transport_fns = NULL;
			}
			if ((ret = http_flush_output(tdata))) {
				// Transport error
				n = 0;
			} else {
				DPWSLOG(DC_TRANSPORT, "HTTP response sent");
			}
			if ((ret = http_cleanup_input(tdata))) {
				// Transport error
				n = 0;
			} else {
				DPWSLOG(DC_TRANSPORT, "HTTP request fully read");
			}
		} else if (ret > 0) {
			// Invalid header (ignored in current impl)
			DPWSLOG1(DC_TRANSPORT, "Invalid HTTP header (status: %d)", ret);
			if (http_send_response_headers(dpws, ret, NULL, 0)) {
				n = 0;
			}
		} else {
			// EOF or transport error
			n = 0;
		}
		if (!(tdata->base.status & DC_HTTP_KEEPALIVE))
			n = 0;
		dc_transport_teardown_channel(dpws, (transport_data_t *)tdata);
		if (n > 0) {
			dpws_end(dpws);
			DPWSLOG1(DC_TRANSPORT, "Keep-alive connection: %d remaining\n", n);
		}
	} while (n > 0);
	dc_reactor_item_invalidate(item);
	return ret;
}

static int http_free_transport_data(struct reactor_item * item, DCPL_SOCKET socket, void* callback_data)
{
	if (callback_data)
		dc_transport_data_clear((transport_data_t *)callback_data);
	if (socket != DCPL_INVALID_SOCKET) {
		DPWSLOG1(DC_TRANSPORT, "Closing HTTP socket %d for reception", socket);
		dcpl_tcp_shutdown(socket, 2, NULL);
		dcpl_closesocket(socket, NULL);
	}
	return DPWS_OK;
}

int dc_http_connect(struct dpws * dpws, const char * endpoint, const char * action, struct media_type * mtype, size_t len)
{
	http_transport_data_t * tdata = (http_transport_data_t *)dpws->transport_data;
	struct soap * soap = dpws_dpws2soap(dpws);
	int ret = DPWS_OK;

	soap_set_endpoint(soap, endpoint);
	if (!tdata) {
		if ((ret = dc_http_open_output_channel(dpws, NULL, soap->host, soap->port)))
			return ret;
		tdata = (http_transport_data_t *)dpws->transport_data;
		tdata->base.status |= DC_TRANSPORT_INTERNAL;
	}
	DPWSLOG1(DC_TRANSPORT, "Sending HTTP request headers to %s", endpoint);
	ret = http_send_request_headers(dpws, endpoint, soap->host, soap->port, soap->path, action, mtype, len);
	return ret;
}

int dc_http_open_output_channel(struct dpws * dpws, http_transport_data_t * tdata, const char * host, unsigned short port)
{
	struct dcpl_error error;
	struct soap * soap = dpws_dpws2soap(dpws);
	struct socket_data connect_data;
	DCPL_SOCKET socket;
	int ret = DPWS_OK;
	int timeout = soap->connect_timeout < 0 ? -soap->connect_timeout/1000 : soap->connect_timeout*1000;

	connect_data.flags = soap->connect_flags;
	connect_data.buflen = SOAP_BUFLEN;
	connect_data.keep_alive = ((soap->omode & SOAP_IO_KEEPALIVE) != 0);
	connect_data.tcp_nodelay = 1;
	connect_data.linger_time = 0;

	if (soap->omode & SOAP_IO_KEEPALIVE) {
		socket = dc_connpool_connect(host, port, timeout, &connect_data, &error);
	} else {
		DPWSLOG2(DC_TRANSPORT, "Opening new client connection for %s:%d", host, port);
		socket = network_tcp_connect(host, port, timeout, &connect_data, &error);
	}
	if (socket == DCPL_INVALID_SOCKET) {
		DPWSLOG2(DC_TRANSPORT, "Could not open client connection for %s:%d", host, port);
		return convert_dcpl_client_error(dpws, &error);
	}
	ret = dc_http_setup_output_channel(dpws, tdata, socket);
	if (ret) {
		DPWSLOG2(DC_TRANSPORT, "Could not setup HTTP channel for %s:%d", host, port);
		if (soap->omode & SOAP_IO_KEEPALIVE) {
			dc_connpool_closesocket(socket, 1, NULL);
		} else {
			DPWSLOG3(DC_TRANSPORT, "Closing client HTTP socket %d for %s:%d", socket, host, port);
			dcpl_closesocket(socket, NULL);
		}
	}
	return ret;
}

int dc_http_get_local_address(struct dpws * dpws, DCPL_SOCKET s, char * local_addr_buf, int buf_size)
{
	struct dcpl_error error;
	if (dcpl_tcp_get_local_address(s, local_addr_buf, buf_size, &error))
		return convert_dcpl_server_error(dpws, &error);
	return DPWS_OK;
}

int dc_http_setup_output_channel(struct dpws * dpws, http_transport_data_t * tdata, DCPL_SOCKET socket)
{
	struct soap * soap = dpws_dpws2soap(dpws);
	int ret;

	tdata = init_http_transport_data(tdata);
	if (!tdata)
		return DPWS_ERR_EOM;
	ret = dc_transport_setup_channel(dpws, (transport_data_t *)tdata, socket, DC_TRANSPORT_OUTPUT);
	tdata->code = DC_HTTP_POST;
	soap->omode &= ~SOAP_IO;
	soap->omode &= ~SOAP_IO_KEEPALIVE;
	return ret;
}

static int http_setup_channel(struct dpws * dpws, struct transport_data * tdata)
{
	struct soap * soap = dpws_dpws2soap(dpws);
	http_transport_data_t * htdata = (http_transport_data_t *)tdata;

	htdata->length = DC_HTTP_UNKNOWN_LENGTH;
	htdata->buf = soap->buf;
	htdata->bufsize = sizeof(soap->buf);
	htdata->buflen = 0;
	htdata->index = 0;
	htdata->ahead = 0;
	htdata->base.status &= ~DC_HTTP_MASK;
	if (soap->imode & SOAP_IO_KEEPALIVE)
		htdata->base.status |= (DC_HTTP_NOKEEPALIVE | DC_HTTP_KEEPALIVE); // Set both to see which one is changed in headers (input mode only)
	if (soap->omode & SOAP_IO_KEEPALIVE)
		htdata->base.status |= (DC_TRANSPORT_POOLED | DC_HTTP_KEEPALIVE);
	if ((soap->omode & SOAP_IO) == SOAP_IO_CHUNK)
		htdata->base.status |= DC_HTTP_CHUNKED_OUTPUT;
	htdata->socket_flags = soap->socket_flags;
	htdata->recv_timeout = soap->recv_timeout < 0 ? -soap->recv_timeout/1000 : soap->recv_timeout*1000;
	htdata->send_timeout = soap->send_timeout < 0 ? -soap->send_timeout/1000 : soap->send_timeout*1000;
	return DPWS_OK;
}

static int http_teardown_channel(struct dpws * dpws, struct transport_data * tdata)
{
	return DPWS_OK;
}

static int http_close_channel(struct transport_data * tdata)
{
	DCPL_SOCKET socket = tdata->socket;

	if (socket != DCPL_INVALID_SOCKET) {
		if (tdata->status & DC_TRANSPORT_POOLED) {
			int reuse = (!tdata->error.syserr && (tdata->status & DC_HTTP_KEEPALIVE));
			if (reuse)
				http_cleanup_input((http_transport_data_t *)tdata);
			dc_connpool_closesocket(socket, reuse, NULL);
			tdata->status &= ~DC_TRANSPORT_POOLED;
		} else {
			DPWSLOG1(DC_TRANSPORT, "Closing client HTTP socket %d", socket);
			dcpl_tcp_shutdown(socket, 2, NULL);
			dcpl_closesocket(socket, NULL);
		}
		tdata->socket = DCPL_INVALID_SOCKET;
	}
	return DPWS_OK;
}

static int http_flush_output(struct http_transport_data * tdata)
{
	int ret;
	DPWSLOG1(DC_TRANSPORT, "Flushing output on socket %d", tdata->base.socket);
	if ((ret = http_buf_flush(tdata)))
		return ret;
	if (tdata->base.status & DC_HTTP_CHUNKED_OUTPUT) {
		DPWSLOG1(DC_TRANSPORT, "Sending final chunk on socket %d", tdata->base.socket);
		DPWSMSG(SENT, "\r\n0\r\n\r\n", 7);
		if (dcpl_tcp_send(tdata->base.socket, "\r\n0\r\n\r\n", 7, tdata->socket_flags, tdata->send_timeout, &tdata->base.error) < 0) {
			DPWSLOG3(DC_TRANSPORT, "TCP send error on socket %d (DCPL error: %d, system error: %d)", tdata->base.socket, tdata->base.error.error, tdata->base.error.syserr);
			return EOF;
		}
	}
	return DPWS_OK;
}

static int http_direct_send(struct http_transport_data * tdata, const char * buf, size_t buflen)
{
	int ret;
	if ((tdata->base.status & (DC_HTTP_HEADERS_SENT | DC_HTTP_CHUNKED_OUTPUT)) == (DC_HTTP_HEADERS_SENT | DC_HTTP_CHUNKED_OUTPUT)) {
		char chunkspec[16];
		int speclen;
		sprintf(chunkspec, "\r\n%lX\r\n", (unsigned long)buflen);
		speclen = strlen(chunkspec);
		DPWSLOG2(DC_TRANSPORT, "Sending chunk size = %d on socket %d", buflen, tdata->base.socket);
		DPWSMSG(SENT, chunkspec, speclen);
		if ((ret = dcpl_tcp_send(tdata->base.socket, chunkspec, speclen, tdata->socket_flags, tdata->send_timeout, &tdata->base.error)) < 0) {
			DPWSLOG3(DC_TRANSPORT, "TCP send error on socket %d (DCPL error: %d, system error: %d)", tdata->base.socket, tdata->base.error.error, tdata->base.error.syserr);
			return ret;
		}
	}
	DPWSLOG2(DC_TRANSPORT, "Sending %d bytes of data on socket %d", buflen, tdata->base.socket);
	DPWSMSG(SENT, buf, buflen);
	if ((ret = dcpl_tcp_send(tdata->base.socket, buf, buflen, tdata->socket_flags, tdata->send_timeout, &tdata->base.error)) < 0) {
		DPWSLOG3(DC_TRANSPORT, "TCP send error on socket %d (DCPL error: %d, system error: %d)", tdata->base.socket, tdata->base.error.error, tdata->base.error.syserr);
	}
	return ret;
}

static int http_send(struct dpws * dpws, void * transport_data, const char * buf, size_t buflen)
{
	http_transport_data_t * tdata = (http_transport_data_t *)transport_data;

	return http_buf_write(tdata, buf, buflen);
}

static int http_get_chunksize(struct http_transport_data * tdata)
{
	int c;
	int xdig, chunksize = 0, bufsize = tdata->bufsize;
	// Reduce bufsize for peeking
	tdata->bufsize = bufsize < 80 ? bufsize : 80;
	c = http_buf_getchar(tdata);
	if (c == '\r') { // Skip the preceding CRLF
		if (http_buf_getchar(tdata) == '\n') {
			c = http_buf_getchar(tdata);
		} else { // Invalid chunk format
			chunksize = 0;
			goto exit;
		}
	}
	while (http_blank(c))
		c = http_buf_getchar(tdata);
	while ((xdig = http_xdigit(c)) >= 0) {
		chunksize <<= 4;	// Multiply existing value by 16
		chunksize += xdig;  // Add new digit
		c = http_buf_getchar(tdata);
	}
	DPWSLOG2(DC_TRANSPORT, "New chunk of size %d on socket %d", chunksize, tdata->base.socket);
	while (c != '\n' && c != EOF)
		c = http_buf_getchar(tdata);
	if (c == EOF) {
		chunksize = 0;
		goto exit;
	}
	if (!chunksize) { // Also read the empty line after last chunk
		c = http_buf_getchar(tdata);
		c = http_buf_getchar(tdata);
	}
	if (tdata->index > 0 && http_buf_skip(tdata, tdata->index))
		chunksize = 0;
exit:
	tdata->bufsize = bufsize;
	return chunksize;
}

static int http_direct_recv(struct http_transport_data * tdata, char * buf, size_t buflen)
{
	int nread;
	if ((tdata->base.status & DC_HTTP_CHUNKED_INPUT) && tdata->length <= 0) {
		tdata->length = http_get_chunksize(tdata);
	}
	if (!tdata->length)
		return 0;
	if (tdata->length > 0 && tdata->length < (int)buflen) {
		buflen = tdata->length;
	}
	nread = dcpl_tcp_recv(tdata->base.socket, tdata->buf, buflen, tdata->socket_flags, tdata->recv_timeout, &tdata->base.error);
	DPWSMSG(RECV, tdata->buf, nread);
	if (nread >= 0) {
		DPWSLOG3(DC_TRANSPORT, "Receiving %d bytes of data on socket %d (%d requested)", nread, tdata->base.socket, buflen);
	} else {
		DPWSLOG4(DC_TRANSPORT, "TCP recv error on socket %d while receiving %d bytes (DCPL error: %d, system error: %d)", tdata->base.socket, buflen, tdata->base.error.error, tdata->base.error.syserr);
	}
	if (tdata->length > 0) {
		tdata->length -= nread;
	}
	return nread;
}

static int http_cleanup_input(struct http_transport_data * tdata)
{
	int nread = 0;
	if ((tdata->base.status & DC_HTTP_CHUNKED_INPUT) || tdata->length > 0) {
		do {
			nread = http_direct_recv(tdata, tdata->buf, tdata->bufsize);
		} while (nread > 0);
	}
	return nread;
}

static int http_recv(struct dpws * dpws, void * transport_data, char * buf, size_t buflen)
{
	http_transport_data_t * tdata = (http_transport_data_t *)transport_data;

	if (!(tdata->base.status & DC_HTTP_HEADERS_RECEIVED)) { // start reading response
		// send any remaining data in output buffer
		if (http_flush_output(tdata))
			return 0; // Will generate a EOF in gSOAP
		if (!(tdata->base.status & DC_HTTP_KEEPALIVE)) {
			DPWSLOG1(DC_TRANSPORT, "Shutting down sending half of client HTTP socket %d", tdata->base.socket);
			dcpl_tcp_shutdown(tdata->base.socket, 1, NULL);
		}
		DPWSLOG1(DC_TRANSPORT, "Waiting for/receiving HTTP response headers on socket %d", tdata->base.socket);
		if (http_recv_headers(dpws))
			return 0; // Will generate a EOF in gSOAP
	}
	return http_direct_recv(tdata, buf, buflen);
}

static int http_close(struct dpws * dpws)
{
	http_transport_data_t * tdata = (http_transport_data_t *)dpws->transport_data;
	if (tdata->base.status & DC_TRANSPORT_INTERNAL)
		dc_transport_close_channel(dpws);
	return DPWS_OK;
}

static const char* empty_response_close = "HTTP/1.1 202 Accepted\r\nServer: gSOAP/2.7\r\nConnection: close\r\n\r\n";
static const char* empty_response = "HTTP/1.1 202 Accepted\r\nServer: gSOAP/2.7\r\nConnection: keep-alive\r\nContent-Length: 0\r\n\r\n";

static int empty_http_response(struct dpws *dpws) {
	http_transport_data_t * tdata = (http_transport_data_t *)dpws->transport_data;
	const char * resp = (tdata->base.status & DC_HTTP_KEEPALIVE) ? empty_response : empty_response_close;
	int len = strlen(resp);
	// Bypass the buffering mechanism to allow output while input is being processed
	int nwritten = http_direct_send(tdata, resp, len);
	// Deactivates chunked mode to avoid outputting the final empty chunk
	tdata->base.status &= ~DC_HTTP_CHUNKED_OUTPUT;
	return nwritten < len ? SOAP_EOF : DPWS_OK;
}

static int http_response(struct dpws * dpws, void * transport_data, int status, struct media_type * mtype, size_t len)
{
	http_transport_data_t * tdata = (http_transport_data_t *)transport_data;
	tdata->index = 0;
	tdata->buflen = 0;
	if (status == DC_SOAP_EMPTY_RESPONSE) {
		DPWSLOG(DC_TRANSPORT, "Sending empty HTTP response headers");
		return empty_http_response(dpws);
	} else if (!len && !(tdata->base.status & DC_HTTP_CHUNKED_OUTPUT)) {
		// Unknown size: must use chunked or close connection
		tdata->base.status &= ~DC_HTTP_KEEPALIVE;
	}
	DPWSLOG(DC_TRANSPORT, "Sending HTTP response headers");
	return http_send_response_headers(dpws, status, mtype, len);
}

static int http_recvemptyresponse(struct dpws * dpws)
{
	http_transport_data_t * tdata = (http_transport_data_t *)dpws->transport_data;
	int ret;
	if ((ret = http_flush_output(tdata)))
		return ret;
	DPWSLOG1(DC_TRANSPORT, "Waiting for/receiving HTTP response headers on socket %d", tdata->base.socket);
	return http_recv_headers(dpws);
}

/******************************* HTTP headers *****************************************/

/*--------------------------------------------------------------------------------*\
| gSOAP public license.
| 
| The contents of this file are subject to the gSOAP Public License Version 1.3
| (the "License"); you may not use this file except in compliance with the
| License. You may obtain a copy of the License at
| http://www.cs.fsu.edu/~engelen/soaplicense.html
| Software distributed under the License is distributed on an "AS IS" basis,
| WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
| for the specific language governing rights and limitations under the License.
| 
| The Initial Developer of the Original Code is Robert A. van Engelen.
| Copyright (C) 2000-2005, Robert van Engelen, Genivia Inc., All Rights Reserved.
\*--------------------------------------------------------------------------------*/

static int http_buf_flush(struct http_transport_data * tdata)
{
	int len = tdata->index;
	if (len) {
		tdata->index = 0;
		if (http_direct_send(tdata, tdata->buf, len) <= 0)
			return EOF;
	}
	return DPWS_OK;
}

static int http_buf_write(struct http_transport_data * tdata, const char *s, size_t len)
{
	register size_t i = tdata->bufsize - tdata->index;
	while (len >= i)
	{
		memcpy(tdata->buf + tdata->index, s, i);
		tdata->index = tdata->bufsize;
		if (http_buf_flush(tdata))
			return EOF;
		s += i;
		len -= i;
		i = tdata->bufsize;
	}
	memcpy(tdata->buf + tdata->index, s, len);
	tdata->index += len;
	return DPWS_OK;
}

static int http_buf_skip(struct http_transport_data * tdata, int len)
{
	int nread;
	tdata->index = 0;
	tdata->buflen = 0;
	tdata->ahead = 0;
	nread = dcpl_tcp_recv(tdata->base.socket, tdata->buf, len, tdata->socket_flags, 0, &tdata->base.error);
	DPWSMSG(RECV, tdata->buf, nread);
	if (nread < 0) {
		DPWSLOG4(DC_TRANSPORT, "TCP recv error on socket %d while skipping %d bytes (DCPL error: %d, system error: %d)", tdata->base.socket, len, tdata->base.error.error, tdata->base.error.syserr);
		return EOF;
	} else {
		DPWSLOG2(DC_TRANSPORT, "Skipping %d bytes of data on socket %d", nread, tdata->base.socket);
		return DPWS_OK;
	}
}

static int http_buf_fill(struct http_transport_data * tdata)
{
	int size = tdata->bufsize;
	if (tdata->index > 0 && http_buf_skip(tdata, tdata->index))
		return EOF;
	tdata->buflen = dcpl_tcp_peek(tdata->base.socket, tdata->buf, size, tdata->socket_flags, tdata->recv_timeout, &tdata->base.error);
	if (tdata->buflen > 0) {
		DPWSLOG2(DC_TRANSPORT, "Peeking %d bytes of data on socket %d", tdata->buflen, tdata->base.socket);
		return DPWS_OK;
	} else {
		DPWSLOG3(DC_TRANSPORT, "TCP peek error on socket %d (DCPL error: %d, system error: %d)", tdata->base.socket, tdata->base.error.error, tdata->base.error.syserr);
		return EOF;
	}
}

static int http_buf_getchar(struct http_transport_data * tdata)
{
	register int c; // use int to support both unsigned chars and EOF (-1)
	c = tdata->ahead;
	if (c)
	{
		if (c != EOF)
			tdata->ahead = 0;
		return c;
	}
	return http_buf_get1(tdata);
}

static int http_buf_getline(struct http_transport_data * tdata, char *s, int len)
{
	int i = len;
	int c = 0;
	for (;;)
	{
		while (--i > 0)
		{
			c = http_buf_getchar(tdata);
			if (c == '\r' || c == '\n')
				break;
			if (c == EOF)
				return EOF;
			*s++ = (char)c;
		}
		if (c != '\n')
			c = http_buf_getchar(tdata); /* got \r, now get \n */
		if (c == '\n')
		{
			*s = '\0';
			if (i+1 == len) /* empty line: end of HTTP header */
				break;
			c = http_buf_unget(tdata, http_buf_getchar(tdata));
			if (c != ' ' && c != '\t') /* HTTP line continuation? */
				break;
		}
		else if (c == EOF)
			return EOF;
	}
	return DPWS_OK;
}

static int http_parse_header(struct soap *soap, struct http_transport_data * tdata, char *key, const char *val)
{
	if (!soap_tag_cmp(key, "Host"))
	{
		strncpy(soap->host, val, sizeof(soap->host) - 1);
		soap->host[sizeof(soap->host) - 1] = '\0';
		strcpy(soap->endpoint, "http://");
		strncat(soap->endpoint, val, sizeof(soap->endpoint) - 8);
		soap->endpoint[sizeof(soap->endpoint) - 1] = '\0';
	}
	else if (!soap_tag_cmp(key, "Content-Type"))
	{
		if (soap_get_header_attribute(soap, val, "application/dime"))
			soap->mode |= SOAP_ENC_DIME;
		else if (soap_get_header_attribute(soap, val, "multipart/related")
				|| soap_get_header_attribute(soap, val, "multipart/form-data"))
		{
			soap->mime.boundary = soap_strdup(soap, soap_get_header_attribute(
					soap, val, "boundary"));
			soap->mime.start = soap_strdup(soap, soap_get_header_attribute(
					soap, val, "start"));
			soap->mode |= SOAP_ENC_MIME;
		}
	}
	else if (!soap_tag_cmp(key, "Content-Length"))
	{
		tdata->length = soap->length = soap_strtoul(val, NULL, 10);
	}
	else if (!soap_tag_cmp(key, "Content-Encoding"))
	{
		if (!soap_tag_cmp(val, "deflate"))
			return DC_HTTP_UNSUPPORTED_CONTENT_ENCODING_ERROR;
		else if (!soap_tag_cmp(val, "gzip"))
			return DC_HTTP_UNSUPPORTED_CONTENT_ENCODING_ERROR;
	}
	else if (!soap_tag_cmp(key, "Transfer-Encoding"))
	{
		if (!soap_tag_cmp(val, "chunked"))
			tdata->base.status |= DC_HTTP_CHUNKED_INPUT;
	}
	else if (!soap_tag_cmp(key, "Connection"))
	{
		if (!soap_tag_cmp(val, "keep-alive"))
			tdata->base.status &= ~DC_HTTP_NOKEEPALIVE;
		else if (!soap_tag_cmp(val, "close"))
			tdata->base.status &= ~DC_HTTP_KEEPALIVE;
	}
	else if (!soap_tag_cmp(key, "Authorization"))
	{
		if (!soap_tag_cmp(val, "Basic *"))
		{
			int n;
			char *s;
			soap_base642s(soap, val + 6, soap->tmpbuf,
					sizeof(soap->tmpbuf) - 1, &n);
			soap->tmpbuf[n] = '\0';
			if ((s = strchr(soap->tmpbuf, ':')))
			{
				*s = '\0';
				soap->userid = soap_strdup(soap, soap->tmpbuf);
				soap->passwd = soap_strdup(soap, s + 1);
			}
		}
	}
	else if (!soap_tag_cmp(key, "WWW-Authenticate"))
		soap->authrealm = soap_strdup(soap, soap_get_header_attribute(soap, val
				+ 6, "realm"));
	else if (!soap_tag_cmp(key, "Expect"))
	{
		if (!soap_tag_cmp(val, "100-continue"))
		{
			const char * resp = "HTTP/1.1 100 Continue\r\n\r\n";
			int len = strlen(resp);
			// Bypass the buffering mechanism to allow output while input is being processed
			int nwritten = http_direct_send(tdata, resp, len);
			return nwritten < len ? SOAP_EOF : DPWS_OK;
		}
	}
	else if (!soap_tag_cmp(key, "SOAPAction"))
	{
		if (*val == '"')
		{
			soap->action = soap_strdup(soap, val + 1);
			soap->action[strlen(soap->action) - 1] = '\0';
		}
	}
	else if (!soap_tag_cmp(key, "Location"))
	{
		strncpy(soap->endpoint, val, sizeof(soap->endpoint));
		soap->endpoint[sizeof(soap->endpoint) - 1] = '\0';
	}
	return DPWS_OK;
}

static int http_recv_headers(struct dpws *dpws)
{
	http_transport_data_t * tdata = (http_transport_data_t *)dpws->transport_data;
	struct soap * soap = dpws_dpws2soap(dpws);
	int ret = DPWS_OK;
	char header[SOAP_HDRLEN], *s;
	unsigned short get = 0, k = 0;
	int bufsize = tdata->bufsize;

	tdata->base.status |= DC_HTTP_HEADERS_RECEIVED;
	// limit bufsize for more efficient peeking
	tdata->bufsize = bufsize < DC_HTTP_HEADERS_PEEK_SIZE ? bufsize : DC_HTTP_HEADERS_PEEK_SIZE;

	*soap->endpoint = '\0';
	soap->length = 0;
	soap->userid = NULL;
	soap->passwd = NULL;
	soap->action = NULL;
	soap->authrealm = NULL;
	do
	{
		if ((ret = http_buf_getline(tdata, soap->msgbuf, sizeof(soap->msgbuf))))
			return ret;
		DPWSLOG1(DC_TRANSPORT, "HTTP status: %s", soap->msgbuf);
		for (;;)
		{
			if ((ret = http_buf_getline(tdata, header, SOAP_HDRLEN)))
			{
				if (ret == EOF) {
					DPWSLOG(DC_TRANSPORT, "EOF in HTTP header, continue anyway");
					break;
				}
				return ret;
			}
			if (!*header)
				break;
			DPWSLOG1(DC_TRANSPORT, "HTTP header: %s", header);
			s = strchr(header, ':');
			if (s)
			{
				*s = '\0';
				do
					s++;
				while (*s && *s <= 32);
				if ((ret = http_parse_header(soap, tdata, header, s)))
				{
					return ret;
				}
			}
		}
		if ((s = strchr(soap->msgbuf, ' ')))
		{
			k = (unsigned short)soap_strtoul(s, &s, 10);
			if (!http_blank((int)*s))
				k = 0;
		}
		else
			k = 0;
	} while (k == 100);

	// Restore normal bufsize for later user
	tdata->bufsize = bufsize;
	// Actually read the headers to move the stream to the beginning of the payload
	if (tdata->index > 0 && (ret = http_buf_skip(tdata, tdata->index)))
		return ret;
	DPWSLOG(DC_TRANSPORT, "Finished HTTP header parsing");
	s = strstr(soap->msgbuf, "HTTP/");
	if (s && s[7] != '1')
	{
		if ((tdata->base.status & DC_HTTP_NOKEEPALIVE))  // HTTP/1.0 needs explicit keep-alive
			tdata->base.status &= ~DC_HTTP_KEEPALIVE;
		if (k == 0 && (tdata->base.status & DC_HTTP_CHUNKED_OUTPUT)) /* k == 0 for HTTP request */
		{
			tdata->base.status &= ~DC_HTTP_CHUNKED_OUTPUT;
			tdata->base.status &= ~DC_HTTP_KEEPALIVE; // Deactivates keep-alive when chunked mode is not possible (HTTP1.0)
		}
	}
	if ((tdata->base.status & DC_HTTP_KEEPALIVE))
		tdata->base.status &= ~DC_HTTP_NOKEEPALIVE;
	else
		tdata->base.status |= DC_HTTP_NOKEEPALIVE;
	DPWSLOG1(DC_TRANSPORT, "Keep alive connection = %d", (tdata->base.status & DC_HTTP_KEEPALIVE) != 0);
	if ((tdata->base.status & DC_HTTP_CHUNKED_INPUT))
		tdata->length = DC_HTTP_UNKNOWN_LENGTH;

	if (k == 0 && (((get = !strncmp(soap->msgbuf, "GET ", 4))) || !strncmp(soap->msgbuf, "POST ", 5)))
	{
		size_t m = strlen(soap->endpoint);
		size_t n = m + (s - soap->msgbuf) - 5 - (!get);
		if (n >= sizeof(soap->endpoint))
			n = sizeof(soap->endpoint) - 1;
		strncpy(soap->path, soap->msgbuf + 4 + (!get), n - m);
		soap->path[n - m] = '\0';
		strcat(soap->endpoint, soap->path);
		if (get)
			tdata->code = DC_HTTP_GET;
		else
			tdata->code = DC_HTTP_POST;
		DPWSLOG1(DC_TRANSPORT, "Target endpoint='%s'", soap->endpoint);
		return DPWS_OK;
	}
	tdata->code = k;
	if ((k >= 200 && k <= 299) || k == 400 || k == 500)
		return DPWS_OK;
	DPWSLOG1(DC_TRANSPORT, "HTTP error %d", k);
	return soap_set_receiver_error(soap, "HTTP error", soap->msgbuf, k);
}

static int http_puthttphdr(struct soap *soap, struct http_transport_data * tdata, struct media_type * mtype, size_t count)
{
	register int err;

	if (mtype && mtype->type) {
		struct media_type_param * param = mtype->params;
		if ((err = http_send_header(tdata, "Content-Type: "))
			|| (err = http_send_header(tdata, mtype->type)))
			return err;
		if (mtype->subtype) {
			if ((err = http_buf_write(tdata, "/", 1))
				|| (err = http_send_header(tdata, mtype->subtype)))
				return err;
		}
		while (param) {
			if ((err = http_buf_write(tdata, "; ", 2))
				|| (err = http_send_header(tdata, param->attribute))
				|| (err = http_buf_write(tdata, "=\"", 2))
				|| (err = http_send_header(tdata, param->value))
				|| (err = http_buf_write(tdata, "\"", 1)))
				return err;
			param = param->next;
		}
		if ((err = http_buf_write(tdata, "\r\n", 2)))
			return err;
	}
	if ((tdata->base.status & DC_HTTP_CHUNKED_OUTPUT))
		err = http_post_header(tdata, "Transfer-Encoding", "chunked");
	else if (count > 0)
	{
		sprintf(soap->tmpbuf, "%lu", (unsigned long)count);
		err = http_post_header(tdata, "Content-Length", soap->tmpbuf);
	}
	if (err)
		return err;
	return http_post_header(tdata, "Connection", (tdata->base.status & DC_HTTP_KEEPALIVE) ? "keep-alive" : "close");
}

static int http_send_request_headers(struct dpws * dpws, const char *endpoint, const char * host, unsigned short port, const char * path, const char * action, struct media_type * mtype, size_t count)
{
	http_transport_data_t * tdata = (http_transport_data_t *)dpws->transport_data;
	struct soap * soap = dpws_dpws2soap(dpws);
	const char *s;
	register char *p;
	register int err = DPWS_OK;
	int ipv6Addr, i;
	if (tdata->code == DC_HTTP_GET)
	{
		s = "GET";
		count = 0;
	}
	else
		s = "POST";
	if (!endpoint || (strncmp(endpoint, "http:", 5) && strncmp(endpoint,
			"https:", 6) && strncmp(endpoint, "httpg:", 6)))
		return DPWS_OK;
	if (soap->proxy_host && strncmp(endpoint, "https:", 6))
		sprintf(soap->tmpbuf, "%s %s HTTP/%s", s, endpoint, soap->http_version);
	else
		sprintf(soap->tmpbuf, "%s /%s HTTP/%s", s, (*path == '/' ? path + 1
				: path), soap->http_version);
	if ((err = http_post_header(tdata, soap->tmpbuf, NULL)))
		return err;
	ipv6Addr = (int)is_ipv6_addr(host);
	p = soap->tmpbuf;
	if (ipv6Addr) {
		*p = '[';
		p++;
	}
	for (i = 0; host[i]; i++) {
		*p = host[i];
		p++;
	}
	if (ipv6Addr) {
		*p = ']';
		p++;
	}
	if (port != 80)
		sprintf(p, ":%d", port);
	else
		*p = 0;
	if ((err = http_post_header(tdata, "Host", soap->tmpbuf)) || (err
			= http_post_header(tdata, "User-Agent", "gSOAP/2.7")) || (err
			= http_puthttphdr(soap, tdata, mtype, count)))
		return err;
	if (soap->userid && soap->passwd && strlen(soap->userid)
			+ strlen(soap->passwd) < 761)
	{
		sprintf(soap->tmpbuf + 262, "%s:%s", soap->userid, soap->passwd);
		strcpy(soap->tmpbuf, "Basic ");
		soap_s2base64(soap, (const unsigned char*)(soap->tmpbuf + 262),
				soap->tmpbuf + 6, strlen(soap->tmpbuf + 262));
		if ((err = http_post_header(tdata, "Authorization", soap->tmpbuf)))
			return err;
	}
	if (soap->proxy_userid && soap->proxy_passwd && strlen(soap->proxy_userid)
			+ strlen(soap->proxy_passwd) < 761)
	{
		sprintf(soap->tmpbuf + 262, "%s:%s", soap->proxy_userid,
				soap->proxy_passwd);
		strcpy(soap->tmpbuf, "Basic ");
		soap_s2base64(soap, (const unsigned char*)(soap->tmpbuf + 262),
				soap->tmpbuf + 6, strlen(soap->tmpbuf + 262));
		if ((err = http_post_header(tdata, "Proxy-Authorization", soap->tmpbuf)))
			return err;
	}
	if (action && soap->version == 1)
	{
		sprintf(soap->tmpbuf, "\"%s\"", action);
		if ((err = http_post_header(tdata, "SOAPAction", soap->tmpbuf)))
			return err;
	}
	if (tdata->base.status & DC_HTTP_CHUNKED_OUTPUT) {
		err = http_buf_flush(tdata);
	} else {
		err = http_post_header(tdata, NULL, NULL);
	}
	tdata->base.status |= DC_HTTP_HEADERS_SENT;
	return err;
}

static int http_send_header(struct http_transport_data * tdata, const char *s)
{
	register const char *t;
	do
	{
		t = strchr(s, '\n'); /* disallow \n in HTTP headers */
		if (!t)
			t = s + strlen(s);
		if (http_buf_write(tdata, s, t - s))
			return EOF;
		s = t + 1;
	} while (*t);
	return DPWS_OK;
}

static int http_post_header(struct http_transport_data * tdata, const char *key, const char *val)
{
	if (key)
	{
		if (http_send_header(tdata, key))
			return EOF;
		if (val	&& (http_buf_write(tdata, ": ", 2) || http_send_header(tdata, val)))
			return EOF;
	}
	return http_buf_write(tdata, "\r\n", 2);
}

static int http_send_response_headers(struct dpws * dpws, int status, struct media_type * mtype, size_t count)
{
	http_transport_data_t * tdata = (http_transport_data_t *)dpws->transport_data;
	struct soap * soap = dpws_dpws2soap(dpws);
	register int err = DPWS_OK;
	if (status == DC_SOAP_RESPONSE)	{
		DPWSLOG(DC_TRANSPORT, "200 OK");
		sprintf(soap->tmpbuf, "HTTP/%s 200 OK", soap->http_version);
		if ((err = http_post_header(tdata, soap->tmpbuf, NULL)))
			return err;
	} else if (status == DC_SOAP_SENDER_ERROR) {
		DPWSLOG(DC_TRANSPORT, "400 Bad Request");
		sprintf(soap->tmpbuf, "HTTP/%s 400 Bad Request", soap->http_version);
		if ((err = http_post_header(tdata, soap->tmpbuf, NULL)))
			return err;
	} else if (status == DC_SOAP_RECEIVER_ERROR) {
		DPWSLOG(DC_TRANSPORT, "500 Internal Server Error");
		sprintf(soap->tmpbuf, "HTTP/%s 500 Internal Server Error", soap->http_version);
		if ((err = http_post_header(tdata, soap->tmpbuf, NULL)))
			return err;
	} else if (status > 200 && status < 600) {
		sprintf(soap->tmpbuf, "HTTP/%s %d %s", soap->http_version, status,
				http_error(soap, status));
		if ((err = http_post_header(tdata, soap->tmpbuf, NULL)))
			return err;
		if (status == 401) {
			sprintf(soap->tmpbuf, "Basic realm=\"%s\"",
					soap->authrealm ? soap->authrealm : "gSOAP Web Service");
			if ((err = http_post_header(tdata, "WWW-Authenticate", soap->tmpbuf)))
				return err;
		} else if ((status >= 301 && status <= 303) || status == 307) {
			if ((err = http_post_header(tdata, "Location", soap->endpoint)))
				return err;
		}
	}
	if ((err = http_post_header(tdata, "Server", "gSOAP/2.7"))
		|| (err = http_puthttphdr(soap, tdata, mtype, count)))
		return err;
	if (tdata->base.status & DC_HTTP_CHUNKED_OUTPUT) {
		err = http_buf_flush(tdata);
	} else {
		err = http_post_header(tdata, NULL, NULL);
	}
	tdata->base.status |= DC_HTTP_HEADERS_SENT;
	return err;
}

