/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1863 $
|                     $Date: 2008-11-24 15:00:53 +0100 (lun., 24 nov. 2008) $
\*============================================================================*/

/*******************************************************************************
*                         Network Interface Manager                            *
*******************************************************************************/

#include "dcDPWS_Network.h"
#include "dc/dc_Error.h"
#include "dcCOMN_Tools.h"

/*----------------------------------- Types ----------------------------------*/

#define LL_TIMEOUT 5	// ms

struct connect_data {
	const char *host;
	uint16_t port;
	int timeout;
	struct socket_data * data;
	struct dcpl_error * error;
	DCPL_SOCKET s;
};

enum network_addr_type{NOT_LL, LL, SCOPED_LL};

/*----------------------------------- Data -----------------------------------*/

// Network interface configuration
dcpl_ip_addr_t *	dc_netif_addrs = NULL;
int					dc_netif_nb_addrs = 0, dc_real_netif_nb_addrs;
char *				dc_hostname = NULL;
int					dc_network_mode;

/*------------------------- Static Functions prototypes ----------------------*/

static int count_addr_cbk(dcpl_ip_addr_t * netif_addr, int * count);
static enum network_addr_type is_ipv6_link_local(char *p);
static int ll_connect_cbk(dcpl_ip_addr_t * netif_addr, struct connect_data * conn_data);

/*----------------------------------------------------------------------------*/
int init_network(dc_ip_filter_t * filter, const char * hostname, DC_BOOL mono_netif)
{
	int i;

	if (dcpl_retrieve_ip_info(filter, &dc_netif_addrs, &dc_real_netif_nb_addrs))
		return DPWS_ERR_RETRIEVING_NETWORK_INTERFACE_INFO;

	if (mono_netif)
		dc_netif_nb_addrs = 1;
	else
		dc_netif_nb_addrs = dc_real_netif_nb_addrs;

	if (dc_netif_nb_addrs == 0)
		return DPWS_ERR_NO_NETWORK_INTERFACE;

	if (dc_hostname && !(dc_hostname = DC_STRDUP(DC_MEM_NETWORK, hostname)))
		return DPWS_ERR_EOM;

	for (i = 0; i < dc_netif_nb_addrs; i++)
	{
		dc_network_mode |= DCPL_GET_ADDR_FAMILY(&dc_netif_addrs[i]);
	}

	return DPWS_OK;
}

void uninit_network()
{
	dcpl_free_ip_info(dc_netif_addrs, dc_real_netif_nb_addrs);
}


int network_browse_interfaces(int protocols, network_addr_cbk cbk, void * cbk_data)
{
	int i = 0, ret = DPWS_OK, protos;
	uint32_t prev_itf_nb = 0;

	for (; i < dc_netif_nb_addrs; i++)
	{
		if (dc_netif_addrs[i].itf_nb != prev_itf_nb)
			protos = 0;
		if ((protocols & DCPL_AF_INET) && !(protos & DCPL_AF_INET) && DCPL_GET_ADDR_FAMILY(&dc_netif_addrs[i]) == DCPL_AF_INET) {
			protos |= DCPL_AF_INET;
			if ((ret = cbk(dc_netif_addrs + i, cbk_data)))
				break;
		}
		if ((protocols & DCPL_AF_INET6) && !(protos & DCPL_AF_INET6) && DCPL_GET_ADDR_FAMILY(&dc_netif_addrs[i]) == DCPL_AF_INET6) {
			protos |= DCPL_AF_INET6;
			if ((ret = cbk(dc_netif_addrs + i, cbk_data)))
				break;
		}
		prev_itf_nb = dc_netif_addrs[i].itf_nb;
	}
	return ret;
}

static int count_addr_cbk(dcpl_ip_addr_t * netif_addr, int * count)
{
	(*count)++;
	return 0;
}

int network_count_interfaces(int protocols)
{
	int count = 0;
	network_browse_interfaces(protocols, (network_addr_cbk)count_addr_cbk, &count);
	return count;
}

int network_browse_interface(uint32_t itf_nb, int protocols, network_addr_cbk cbk, void * cbk_data)
{
	int i = 0, ret = DPWS_OK;

	for (; i < dc_netif_nb_addrs; i++)
	{
		if (dc_netif_addrs[i].itf_nb == itf_nb
				&& ((((protocols & DCPL_AF_INET) && DCPL_GET_ADDR_FAMILY(&dc_netif_addrs[i]) == DCPL_AF_INET)) ||
					((protocols & DCPL_AF_INET6) && DCPL_GET_ADDR_FAMILY(&dc_netif_addrs[i]) == DCPL_AF_INET6)))
		{
			if ((ret = cbk(dc_netif_addrs + i, cbk_data)))
				break;
		}
	}
	return ret;
}

// Does not ensure the address is valid but "label" it (if LL, cannot be DNS host, non-LL IPv6 nor IPv4)
static enum network_addr_type is_ipv6_link_local(char *p)
{
	/* Check the host is an IPv6 Link Local */
	for (; IS_WS(*p); p++);	// skip whitespace
	if (*p == 'f' || *p == 'F') {
		p++;
		if (*p == 'e' || *p == 'E') {
			p++;
			if (*p == '8' || *p == '9' || *p == 'a' || *p == 'b' || *p == 'A' || *p == 'B') {
				p++;
				if (*p != ':') {	// else first hexa digit was 0
					p++;
					if (*p == ':') {	// not a DNS name (LDH) nor an IPv4
						for (; *p; p++) {
							if (*p == '%')
								return SCOPED_LL;
						}
						return LL;
					}
				}
			}
		}
	}
	return NOT_LL;
}

static int ll_connect_cbk(dcpl_ip_addr_t * netif_addr, struct connect_data * conn_data)
{
	if (conn_data->s == DCPL_INVALID_SOCKET)
	{
		conn_data->s = dcpl_tcp_connect(
								conn_data->host,
								netif_addr->itf_nb,
								conn_data->port,
								conn_data->timeout,
								conn_data->data,
								conn_data->error
								);
	}
	return DPWS_OK;
}

DCPL_SOCKET network_tcp_connect(const char *host, uint16_t port, int timeout, struct socket_data * data, struct dcpl_error * error)
{
	enum network_addr_type addr_type;

	addr_type = is_ipv6_link_local((char *)host);

	if (addr_type == LL)
	{
		struct connect_data cbk_data;
		
		cbk_data.host = host;
		cbk_data.port = port;
		cbk_data.data = data;
		cbk_data.error = error;
		cbk_data.s = DCPL_INVALID_SOCKET;
		
		cbk_data.timeout = timeout / network_count_interfaces(DCPL_AF_INET6);	// Not strict but since the "iterative" solution may be improved...
		network_browse_interfaces(DCPL_AF_INET6, (network_addr_cbk)ll_connect_cbk, &cbk_data);
		return cbk_data.s;
	}
	else
		return dcpl_tcp_connect(host, 0, port, timeout, data, error);
}
