/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|                      ->>  Copyright 2008 Odonata <<-                         |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1772 $
|                     $Date: 2008-10-09 11:57:03 +0200 (jeu., 09 oct. 2008) $
\*============================================================================*/

/**
 * Design issues:
 * 1. Only one queue used to store incoming reactor items: dispatch in internal queues must be performed
 * by the reactor. The advantage is that only one queue needs to be lock-protected. This should bring
 * benefits as long as incoming items are not too numerous.
 * 2. Only one queue is used to store active items: this requires an additional iteration in the result list
 * to execute internal items. Two queues could be used instead.
 * 3. Items are stored in lists using push_item: therefore, the order of items depends on the number of moves
 * from one list to another.
 */

#include "dcDPWS_Reactor.h"
#include "dcDCPL_Socket.h"
#include "dcDCPL_Reactor.h"
#include "dcDPWS_Registry.h"
#include "dcDPWS_Network.h"
#include "dcDPWS_Dpws.h"
#include "dcCOMN_Handle.h"


/*----------------------------------- Data -----------------------------------*/

/*------------------------- Static Functions prototypes ----------------------*/
// List management

static struct reactor_item * pop_item(struct reactor_item ** list);
static void push_item(struct reactor_item * item, struct reactor_item ** list);
static void insert_in_timer_list(struct reactor_item * timer_item, struct reactor_item ** timer_list);

// Construction/allocation of reactor and reactor items

static int delete_reactor(struct reactor * reactor);
static struct reactor_item * create_reactor_item(struct reactor * reactor, uint8_t status, reactor_cbk callback, void * callback_data, reactor_del_cbk del_callback);
static int delete_reactor_item(struct reactor * reactor, struct reactor_item * item);
static int delete_item_list(struct reactor * reactor, struct reactor_item ** item_list);

// Utilities

static void dispatch_reactor_item(struct reactor * reactor, struct reactor_item * item, struct reactor_item ** active_list);
static int filter_results(struct reactor * reactor, struct reactor_item ** active_list, int * nresults, struct dpws * dpws);
static int reactor_is_done(struct reactor * reactor);
static int return_reactor_item(struct reactor * reactor, struct reactor_item * item);
static int register_reactor_item(struct reactor * reactor, struct reactor_item * item, DC_BOOL notify);

/**************************** Implementation ***************************/
static struct reactor_item * pop_item(struct reactor_item ** list)
{
	struct reactor_item * item = *list;
	if (item) {
		*list = item->next;
		item->next = NULL;
	}
	return item;
}

static void push_item(struct reactor_item * item, struct reactor_item ** list)
{
	if (item) {
		item->next = *list;
		*list = item;
	}
}

static void insert_in_timer_list(struct reactor_item * timer_item, struct reactor_item ** timer_list)
{
	struct reactor_item * item = *timer_list;
	struct reactor_item * previous = NULL;

	while (item && item->exp_date <= timer_item->exp_date) {
		previous = item;
		item = item->next;
	}
	timer_item->next = item;
	if (previous) {
		previous->next = timer_item;
	} else {
		*timer_list = timer_item;
	}
}

static int set_reactor_error(struct reactor * reactor, struct dcpl_error * error_info, int error)
{
	if (error_info) {
		reactor->error = error_info->error;
		reactor->syserr = error_info->syserr;
		reactor->detail = error_info->detail;
	}
	if (error_info->error == DCPL_EOM_ERROR)
		return DPWS_ERR_EOM;
	else
		return error;
}

int dc_reactor_init(struct reactor * reactor, DC_BOOL shared)
{
	struct dcpl_error error_info;

	// Field default is mostly NULL or 0
	memset(reactor, 0, sizeof(struct reactor));
	reactor->status = DC_REACTOR_DONE_STATUS;
	reactor->state = DC_REACTOR_DONE_STATE;
	if (shared)
		reactor->lock = reactor_lock;
	// Choose any of the selected addresses to choose a protocol for wakeup
	if (dcpl_reactor_init(
			reactor,
			DCPL_GET_ADDR_FAMILY(&dc_netif_addrs[0]),
			http_endpoint.port,
			&error_info)) {
		return set_reactor_error(reactor, &error_info, DPWS_ERR_INTERNAL_ERROR);
	}
	reactor->status = DC_REACTOR_INIT_STATUS;
	reactor->state = DC_REACTOR_RUNNING_STATE;
	return DPWS_OK;
}

static int delete_reactor(struct reactor * reactor)
{
	struct reactor_item * wait_list = NULL;
	struct reactor_item * item = NULL;
	struct dcpl_error error_info;
	int ret = DPWS_OK;

	reactor->status = DC_REACTOR_DONE_STATUS;
	if (dcpl_reactor_free(reactor, &error_info)) {
		ret = set_reactor_error(reactor, &error_info, DPWS_ERR_INTERNAL_ERROR);
	}
	delete_item_list(reactor, &reactor->read_list);
	delete_item_list(reactor, &reactor->write_list);
	delete_item_list(reactor, &reactor->exception_list);
	delete_item_list(reactor, &reactor->timer_list);
	delete_item_list(reactor, &reactor->pending_list);
	delete_item_list(reactor, &reactor->results);
	if (reactor->lock) dcpl_mutex_lock(reactor->lock);
	reactor->state = DC_REACTOR_DONE_STATE;
	wait_list = reactor->wait_list;
	reactor->wait_list = NULL;
	if (reactor->lock) dcpl_mutex_unlock(reactor->lock);
	delete_item_list(reactor, &wait_list);
	if (reactor->lock) dcpl_mutex_lock(reactor->lock);
	item = reactor->free_list;
	reactor->free_list = NULL;
	if (reactor->lock) dcpl_mutex_unlock(reactor->lock);
	while (item) {
		struct reactor_item * next = item->next;
		DC_FREE(DC_MEM_TRANSIENT, item);
		item = next;
	}
	reactor->lock = NULL;
	return ret;
}

static struct reactor_item * create_reactor_item(struct reactor * reactor, uint8_t status, reactor_cbk callback, void * callback_data, reactor_del_cbk del_callback)
{
	struct reactor_item * new_item;
	if (reactor->lock) dcpl_mutex_lock(reactor->lock);
	new_item = pop_item(&reactor->free_list);
	if (reactor->lock) dcpl_mutex_unlock(reactor->lock);
	if (!new_item) {
		new_item = (struct reactor_item *)DC_MALLOC(DC_MEM_TRANSIENT, sizeof(struct reactor_item));
	}
	if (new_item) {
		new_item->reactor = reactor;
		new_item->status = status;
		new_item->priority = 0;
		new_item->href = -1;
		new_item->callback = callback;
		new_item->callback_data = callback_data;
		new_item->del_callback = del_callback;
		new_item->socket = SOAP_INVALID_SOCKET;
		new_item->exp_date = 0;
		new_item->period = 0;
	}
	return new_item;
}

static int delete_reactor_item(struct reactor * reactor, struct reactor_item * item)
{
	struct reactor_item * list = item;
	item->next = NULL; // to make sure we don't delete a whole list by mistake
	return delete_item_list(reactor, &list);
}

static int delete_item_list(struct reactor * reactor, struct reactor_item ** item_list)
{
	int ret = DPWS_OK;
	struct reactor_item * item = *item_list;
	struct reactor_item * prev = NULL;
	while (item) {
		int ret1 = dc_reactor_item_invalidate(item);
		if (!ret) ret = ret1;
		prev = item;
		item = item->next;
	}
	if (prev) {
		if (reactor->lock) dcpl_mutex_lock(reactor->lock);
		prev->next = reactor->free_list;
		reactor->free_list = *item_list;
		if (reactor->lock) dcpl_mutex_unlock(reactor->lock);
	}
	*item_list = NULL;
	return ret;
}

/**
 * Adds the item to the proper list, depending on its type and timeout.
 */
static void dispatch_reactor_item(struct reactor * reactor, struct reactor_item * item, struct reactor_item ** active_list)
{
	switch (item->status & DC_RI_TYPE_MASK) {
	case DC_RI_TIMER:
		if (active_list && item->exp_date == 0) {
			push_item(item, active_list);
		} else {
			insert_in_timer_list(item, &reactor->timer_list);
		}
		break;
	case DC_RI_READ:
		push_item(item, &reactor->read_list);
		break;
	case DC_RI_WRITE:
		push_item(item, &reactor->write_list);
		break;
	case DC_RI_EXCEPTION:
		push_item(item, &reactor->exception_list);
		break;
	}
}

/**
 * Handles internal active items and periodic timer items.
 */
static int filter_results(struct reactor * reactor, struct reactor_item ** active_list, int * nresults, struct dpws * dpws)
{
	int ret = DPWS_OK;
	struct reactor_item * item = *active_list;
	struct reactor_item * previous = NULL;

	while (item != NULL) {
		if (item->href >= 0 && 	getHandleUseCount(&reactor_handle_pool, item->href) == 1) {	// unregistered item
			struct reactor_item * next = item->next;
			if (previous) previous->next = next;
			else *active_list = next;
			delete_reactor_item(reactor, item);
			item = next;
		}
		else {
			if ((item->status & DC_RI_TYPE_MASK) == DC_RI_TIMER && item->period > 0 && !reactor->pending_list) {
				// replace item in result list by a copy and reschedule existing item
				// Not performed when reactor is shutting down (pending_list not null)
				struct reactor_item * new_item =
					create_reactor_item(reactor, item->status, item->callback, item->callback_data, item->del_callback);
				if (!new_item) {
					ret = DPWS_ERR_EOM;
					break;
				}
				if (previous) previous->next = new_item;
				else *active_list = new_item;
				new_item->next = item->next;
				item->next = NULL;
				item->exp_date += item->period;
				insert_in_timer_list(item, &reactor->timer_list);
				item = new_item;
			}
			if (item->status & DC_RI_INTERNAL) {
				struct reactor_item * next = item->next;
				if (previous) previous->next = next;
				else *active_list = next;
				item->next = NULL;
				if (item->callback)
					ret = item->callback(dpws, item, item->callback_data);
				if (ret || (item->status & DC_RI_ONCE)) {
					int ret1 = delete_reactor_item(reactor, item);
					if (!ret) ret = ret1;
				} else {
					dispatch_reactor_item(reactor, item, NULL);
				}
				if (ret) break;
				item = next;
			} else {
				*nresults = *nresults + 1;
				previous = item;
				item = item->next;
			}
		}
	}
	return ret;
}

static int reactor_is_done(struct reactor * reactor)
{
	uint16_t state;
	if (reactor->lock) dcpl_mutex_lock(reactor->lock);
	state = reactor->state;
	if (reactor->lock) dcpl_mutex_unlock(reactor->lock);
	return (state == DC_REACTOR_DONE_STATE);
}

static int register_reactor_item(struct reactor * reactor, struct reactor_item * item, DC_BOOL notify)
{
	DC_BOOL needs_wakeup = DC_FALSE;

	if (reactor->lock) dcpl_mutex_lock(reactor->lock);
	push_item(item, &reactor->wait_list);
	if (notify && (reactor->state & DC_REACTOR_WAITING_STATE)
		&& (((item->status & DC_RI_TYPE_MASK) != DC_RI_TIMER)
		     || reactor->next_wakeup_date == 0
		     || item->exp_date < reactor->next_wakeup_date))
	{
		reactor->state = DC_REACTOR_INTERRUPTED_STATE;
		needs_wakeup = DC_TRUE;
	}
	if (reactor->lock) dcpl_mutex_unlock(reactor->lock);
	if (needs_wakeup && dcpl_reactor_interrupt(reactor, NULL))
		return DPWS_ERR_INTERNAL_ERROR;
	return DPWS_OK;
}

static int return_reactor_item(struct reactor * reactor, struct reactor_item * item)
{
	int ret = DPWS_OK;
	uint16_t state;
	int outstanding_items = 0;

	if (reactor->lock) dcpl_mutex_lock(reactor->lock);
	state = reactor->state;
	outstanding_items = --reactor->outstanding_items;
	if (reactor->lock) dcpl_mutex_unlock(reactor->lock);

	if (state == DC_REACTOR_DONE_STATE) {
		ret = dc_reactor_item_invalidate(item);
		DC_FREE(DC_MEM_TRANSIENT, item);
	} else if (item->status == DC_RI_DONE || item->status & DC_RI_ONCE) {
		ret = delete_reactor_item(reactor, item);
		if ((state & DC_REACTOR_SHUTDOWN_STATE) && !outstanding_items && dcpl_reactor_interrupt(reactor, NULL))
			return DPWS_ERR_INTERNAL_ERROR;
	} else {
		ret = register_reactor_item(reactor, item, DC_TRUE);
	}
	return ret;
}

static int stop_callback(struct dpws * dpws, struct reactor_item * item, void* callback_data)
{
	item->reactor->status = DC_REACTOR_STOPPED_STATUS;
	return DPWS_OK;
}

static int shutdown_callback(struct dpws * dpws, struct reactor_item * item, void* callback_data)
{
	struct reactor * reactor = item->reactor;
	uint32_t timeout = (uint32_t)callback_data; // Should be OK on all platforms
	struct reactor_item * stop_item = NULL;
	uint8_t status = DC_RI_TIMER | DC_RI_ONCE | DC_RI_INTERNAL;

	reactor->pending_list = reactor->read_list;
	reactor->read_list = NULL;
	if (!reactor->pending_list) // Add dummy item to ensure list is not empty
		push_item(create_reactor_item(reactor, DC_RI_DONE, NULL, NULL, NULL), &reactor->pending_list);

	// Create a timer item for stopping the shutdown if it takes too long
	stop_item = create_reactor_item(reactor, status, stop_callback, NULL, NULL);
	if (!stop_item)
		return DPWS_ERR_EOM;
	stop_item->exp_date = dcpl_get_clock() + timeout;
	insert_in_timer_list(stop_item, &reactor->timer_list);
	if (stop_item->next)
		delete_item_list(reactor, &stop_item->next);  // Remove all timer items after the timeout
	return DPWS_OK;
}

static int timeout_callback(struct dpws * dpws, struct reactor_item * item, void* callback_data)
{
	item->reactor->status = DC_REACTOR_TIMED_OUT_STATUS;
	return DPWS_OK;
}

int dc_reactor_wait(struct reactor * reactor, struct reactor_item ** active_list, struct dpws * dpws, int32_t timeout)
{
	int ret = DPWS_OK;
	uint64_t clock = 0, next_date = 0;
	int outstanding_items = 0;
	DC_BOOL timer_checked = DC_FALSE, is_polling = DC_FALSE;
	struct reactor_item * results = NULL;
	struct reactor_item * new_items = NULL;
	struct dcpl_error error_info;

	if (reactor_is_done(reactor))
		return DPWS_ERR_ILLEGAL_STATE;

	reactor->status = DC_REACTOR_ACTIVE_STATUS;

	if (timeout >= 0)
		dc_reactor_register_timer(reactor, DC_TRUE, timeout, 0, timeout_callback, NULL, NULL, NULL);

	// Loop until external active items are available or an error occurs
	do {
		// Add new items in wait list to the reactor and compute next timeout.
		// Because we don't lock the wait list for the complete processing,
		// we need to loop to get possible new items added during the processing
		while (1) {
			if (reactor->lock) dcpl_mutex_lock(reactor->lock);
			new_items = reactor->wait_list;
			reactor->wait_list = NULL;
			if (!new_items && timer_checked) { // Loop exit condition
				// the following must be done inside the lock to guarantee atomicity
				reactor->next_wakeup_date = next_date;
				 // used to know if reactor must be woken up when new items are added to the wait list
				reactor->state = DC_REACTOR_WAITING_STATE;
				if (reactor->pending_list)
					reactor->state |= DC_REACTOR_SHUTDOWN_STATE;
				outstanding_items = reactor->outstanding_items;
				if (reactor->lock) dcpl_mutex_unlock(reactor->lock);
				break;
			} else {
				if (reactor->lock) dcpl_mutex_unlock(reactor->lock);
			}
			while (new_items != NULL) {
				dispatch_reactor_item(reactor, pop_item(&new_items), &results);
			}
			clock = dcpl_get_clock();
			while (reactor->timer_list) {
				next_date = reactor->timer_list->exp_date;
				if (next_date > clock) break;
				push_item(pop_item(&reactor->timer_list), &results);
			}
			timer_checked = DC_TRUE;
			if (results) {
				is_polling = DC_TRUE;
				// Make sure we don't wait if some timer items are ready
				next_date = clock;
			}
		}

		if (reactor->pending_list && !outstanding_items && !results && (!reactor->timer_list || !reactor->timer_list->next)) {
			reactor->status = DC_REACTOR_STOPPED_STATUS;
			break;
		}

		if (!reactor->read_list && !next_date && !reactor->write_list && !reactor->exception_list) {
			// Can only occur if results is empty.
			reactor->status = DC_REACTOR_DONE_STATUS;
			break;
		}

		ret = dcpl_reactor_wait(reactor, next_date, &results, &error_info);

		// Update the state to avoid spurious wake ups.
		if (reactor->lock) dcpl_mutex_lock(reactor->lock);
		reactor->state = DC_REACTOR_RUNNING_STATE;
		if (reactor->lock) dcpl_mutex_unlock(reactor->lock);

		// Handle errors
		if (ret < 0 && ret != DCPL_REACTOR_INTR) { // EINTR are ignored
			reactor->status = DC_REACTOR_ERROR_STATUS;
			set_reactor_error(reactor, &error_info, DPWS_ERR_INTERNAL_ERROR);
			ret = DPWS_ERR_INTERNAL_ERROR;
			break;
		}

		// Handle wait results
		if (ret == 0 && !is_polling) {
			// Wait returned on timeout, so add next timer item to result list
			push_item(pop_item(&reactor->timer_list), &results);
		}

		// Filter results: internal active items are executed immediately
		// and removed from the result list. Periodic timer items are
		// rescheduled
		ret = filter_results(reactor, &results, &outstanding_items, dpws);
		if (ret) {
			reactor->status = DC_REACTOR_ERROR_STATUS;
			break;
		}

	} while (results == NULL && reactor->status == DC_REACTOR_ACTIVE_STATUS);

	if (reactor->lock) dcpl_mutex_lock(reactor->lock);
	reactor->outstanding_items = outstanding_items;
	if (reactor->lock) dcpl_mutex_unlock(reactor->lock);

	switch (reactor->status) {
	case DC_REACTOR_ERROR_STATUS:
		delete_item_list(reactor, &results);
		delete_reactor(reactor);
		return ret;
	case DC_REACTOR_STOPPED_STATUS:
		delete_item_list(reactor, &results);
		delete_reactor(reactor);
		return DC_REACTOR_STOPPED_STATUS;
	case DC_REACTOR_DONE_STATUS:
		delete_reactor(reactor);
		return DC_REACTOR_DONE_STATUS;
	default:
		*active_list = results;
		return DPWS_OK;
	}
}

int dc_reactor_wait_single(struct reactor * reactor, struct reactor_item ** active_item, struct dpws * dpws, int32_t timeout)
{
	int ret = DPWS_OK;
	*active_item = NULL;
	if (!reactor->results)
		if ((ret = dc_reactor_wait(reactor, &reactor->results, dpws, timeout)))
			return ret;
	*active_item = pop_item(&reactor->results);
	return ret;
}

int dc_reactor_execute_item(struct reactor * reactor, struct reactor_item * item, struct dpws * dpws)
{
	int ret = DPWS_OK, ret1;
	// Any error occurring in the app callback must be handled in the application.
	// This may involve freeing resources associated to the item (e.g. socket).
	if (item->callback)
		ret = item->callback(dpws, item, item->callback_data);
	ret1 = return_reactor_item(reactor, item);
	if (!ret)
		ret = ret1;
	return ret;
}

int dc_reactor_execute_factory_item(struct reactor * reactor, struct reactor_item * item, struct dpws * dpws)
{
	int ret = DPWS_OK;
	if (item->status & DC_RI_FACTORY) {
		ret = dc_reactor_execute_item(reactor, item, dpws);
		if (!ret) {
			if (reactor->lock) dcpl_mutex_lock(reactor->lock);
			reactor->outstanding_items++;
			if (reactor->lock) dcpl_mutex_unlock(reactor->lock);
		}
	}
	return ret;
}

int dc_reactor_start(struct reactor * reactor, struct dpws * dpws, uint32_t timeout)
{
	int ret = DPWS_OK;
	struct reactor_item * results = NULL;
	struct reactor_item * item = NULL;
	if (timeout) {
		dc_reactor_stop(reactor, timeout);
	}
	while (1) {
		ret = dc_reactor_wait(reactor, &results, dpws, -1);
		if (ret) break;
		while ((item = pop_item(&results))) {
			if ((ret = dc_reactor_execute_item(reactor, item, dpws))) {
				delete_item_list(reactor, &results);
				delete_reactor(reactor);
				break;
			}
		}
	}
	return ret < 0 ? ret : DPWS_OK;
}

int dc_reactor_stop(struct reactor * reactor, uint32_t timeout)
{
	return dc_reactor_register_timer(reactor, DC_TRUE, timeout, 0, stop_callback, NULL, NULL, NULL);
}

int dc_reactor_shutdown(struct reactor * reactor, uint32_t timeout)
{
	return dc_reactor_register_timer(reactor, DC_TRUE, 0, 0, shutdown_callback, (void*)timeout, NULL, NULL);
}

struct reactor_item * dc_reactor_create_socket_item(struct reactor * reactor,
												   SOAP_SOCKET socket,
												   uint8_t status,
												   reactor_cbk callback,
												   void* callback_data,
												   reactor_del_cbk del_callback,
												   int16_t* href)
{
	struct reactor_item * item = NULL;
	item = create_reactor_item(reactor, status, callback, callback_data, del_callback);
	if (item)
		item->socket = socket;
	return item;
}

int dc_reactor_register_socket(struct reactor * reactor,
							   SOAP_SOCKET socket,
							   uint8_t status,
							   reactor_cbk callback,
							   void* callback_data,
							   reactor_del_cbk del_callback,
							   int16_t* href)
{
	struct reactor_item * item = NULL;
	if (socket == SOAP_INVALID_SOCKET)
		return DPWS_ERR_INVALID_PARAMETER;
	if (reactor_is_done(reactor))
		return DPWS_ERR_ILLEGAL_STATE;
	item = create_reactor_item(reactor, status, callback, callback_data, del_callback);
	if (!item)
		return DPWS_ERR_EOM;
	item->socket = socket;
	return register_reactor_item(reactor, item, DC_FALSE);
}

int dc_reactor_register_socket_with_timeout(struct reactor * reactor,
							   				SOAP_SOCKET socket,
							   				uint8_t status,
							   				int32_t timeout,
							   				reactor_cbk callback,
							   				void* callback_data,
										    reactor_del_cbk del_callback,
							   				int16_t* href)
{
	return dc_reactor_register_socket(reactor, socket, status, callback, callback_data, del_callback, href);
}

int dc_reactor_register_timer(struct reactor * reactor,
							  DC_BOOL internal,
							  uint32_t timeout,
							  uint32_t period,
							  reactor_cbk callback,
							  void* callback_data,
							  reactor_del_cbk del_callback,
							  int16_t* href)
{
	struct reactor_item * item = NULL;
	uint8_t status = DC_RI_TIMER | DC_RI_ONCE;
	if (reactor_is_done(reactor))
		return DPWS_ERR_ILLEGAL_STATE;
	if (internal)
		status |= DC_RI_INTERNAL;
	item = create_reactor_item(reactor, status, callback, callback_data, del_callback);
	if (!item)
		return DPWS_ERR_EOM;
	if (timeout)
		item->exp_date = dcpl_get_clock() + timeout;
	item->period = period;
	if (href) {	// TODO: generalize to socket items ?
		*href = item->href = createHandle(&reactor_handle_pool, DC_RI_HANDLE_TYPE, item);
		if (item->href < 0) {
			delete_reactor_item(reactor, item);
			return DPWS_ERR_EOM;
		}
		else
			checkoutHandle(&reactor_handle_pool, item->href);	// 2 uses hold
	}
	return register_reactor_item(reactor, item, DC_FALSE);
}

int dc_reactor_interrupt(struct reactor * reactor)
{
	DC_BOOL needs_wakeup = DC_FALSE;

	if (reactor->lock) dcpl_mutex_lock(reactor->lock);
	if (reactor->state == DC_REACTOR_WAITING_STATE) {
		needs_wakeup = DC_TRUE;
		reactor->state = DC_REACTOR_INTERRUPTED_STATE;
	}
	if (reactor->lock) dcpl_mutex_unlock(reactor->lock);
	if (needs_wakeup && dcpl_reactor_interrupt(reactor, NULL))
		return DPWS_ERR_INTERNAL_ERROR;
	return DPWS_OK;
}

int dc_reactor_item_invalidate(struct reactor_item * item)
{
	int ret = DPWS_OK;
	if (item->status == DC_RI_DONE)
		return DPWS_OK;
	item->status = DC_RI_DONE;
	if (item->del_callback) {
		// User-defined error: must be accounted for
		ret = item->del_callback(item, item->socket, item->callback_data);
	} else if (item->socket != SOAP_INVALID_SOCKET) {
		dcpl_closesocket(item->socket, NULL);
	}
	item->socket = SOAP_INVALID_SOCKET;
	item->callback_data = NULL;
	return ret;
}

int dc_reactor_item_set_socket(struct reactor_item * item, SOAP_SOCKET socket)
{
	item->socket = socket;
	return DPWS_OK;
}

int dc_reactor_item_get_socket(struct reactor_item * item, SOAP_SOCKET* socket)
{
	*socket = item->socket;
	return DPWS_OK;
}

int dc_reactor_unregister_item(int16_t href)
{
	return releaseHandle(&reactor_handle_pool, href);
}
