/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2109 $
|                     $Date: 2009-02-24 13:52:15 +0100 (mar, 24 fév 2009) $
\*============================================================================*/

/*******************************************************************************
*          Device Registry (server-side configuration & runtime data)          *
*******************************************************************************/
#ifndef REGISTRY_H
# define REGISTRY_H

#include "dcCOMN_Tools.h"
#include "dcDPWS_Dpws.h"
#include "dcDPWS_Memory.h"
#include "dc/dc_Dpws.h"
#include "dc/dc_Constants.h"
#include "dcCOMN_Handle.h"

/*----------------------------------------------------------------------------*\
 *                                  INTERNAL                                  *
\*----------------------------------------------------------------------------*/

/** Main configuration structure for the registry */
struct dpws_reg_config {
	uint32_t		boot_seq;	/**< The boot count used for WS-Discovery message sequencing */
	metadata_cbk	dev_metadata_hook;	/**< The user callback that is called whenever metadata version changes for a device */
	DC_BOOL			server_up;	/**< Indicates if the server has been started with dpws_server_init_XXX */

	/* toolkit parameters */
	char *			preferred_lang;	/**< The configuration parameter allowing to select the DPWS metadata language when several are available */
	DC_BOOL			bp1_1_compatibility;	/**< If DC_TRUE, WS-I BP1.1 compatibility is enabled for the toolkit. */

	/* backwards API compatibility toolkit parameters */
	DC_BOOL			tcp_listen;	/**< If DC_TRUE, the HTTP server will be started. */
	int				tcp_backlog;	/**< The number of TCP pending connection requests for the HTTP listener. */
};

/** Transport logical definition structure */
struct transport {
	char *			protocol_name;	/**< The describing name of the protocol */
	char *			scheme;	/**< The scheme for the transport that will be used to build service URLs */
};

struct invocation_list;	/**< forward declaration */

/** Network endpoint that defines the port on which a transport will listen */
struct network_endpoint {
	struct transport *		transport;	/**< The transport that manages the network endpoint */
	uint16_t				port;	/**< The IP port number for the network endpoint */
	struct invocation_list *		invoc_list;	/**< The list that contains a link to all the service reachable through this network endpoint */
};

DA_TYPED_DECL(sp,struct service_port *);	/**< Typed-array of service ports used to populate the \a invocation_list_t */

/** An ordered table of service port that allows to retrieve the invoked service
 * using dichotomy.
 */
typedef struct invocation_list {
	DA_TYPED(sp)		ports;	/**< Ordered array of service ports (alphabetic order of context_path) (slow for insertion, rather quick for retrieval with dichotomy).
								service ports are created for:
								 <ol><li>Devices (using the uuid for context path)
								 <li>Every service that is created automatically</li>
								 <li>User-created ports.</li>*/
	int					use_count;	/**< A use count hold by every invocation to know if an invocation list can be disposed or not */
	DC_BOOL				is_online;	/**< Flag to know if the list is reachable through the network or cloned or disposable */
	struct device *		off_device;	/**< A pointer to a device that was disabled causing this invocation list to be disposable and which handle should be released */
	struct network_endpoint * 	network_endpoint;	/**< The associated network endpoint if online */
} invocation_list_t;

/** Association structure between service & network endpoints. */
struct service_port {
	char * context_path;	/**< The context path that will be used to build the service URLs */
	struct network_endpoint * network_endpoint;	/**< A pointer to the network endpoint for which this service port was created */
	struct invocation_list * invoc_bind_list;	/**< transient field: only set when a port is invoked to be able to release the list */
	struct service_endpoint * service;	/**< A pointer to the service endpoint for which this service port was created */
};

/** The service structure.
 * Not named hosted service because the endpoint can be a device or a hosted service
 */
struct service_endpoint {
	short href;	/**< Stored in the structure because returned by an API. */
	char * service_id;	/**< The service ID defined by DPWS. An URI which is NULL for the device endpoint */
	char * hardware_id;
	char * compatible_id;
	short hrefServiceClass;	/**< The handle reference for the class of the service */
	struct device * device;	/**< The hosting device */
	DA_TYPED(href) service_ports;	/**< A typed-array that holds the bindings to several potential network endpoints */
	const void * user_data;	/**< API user free data that can be set through the configuration API and retrived in the service function */
	DC_BOOL subsc_manager;	/**< TRUE if the service is an event source that has received subscription requests */
};

/** Typed array for dispatch functions */
DA_TYPED_DECL(disp,dispatch_cbk);
/** Typed array for WSDL references */
DA_TYPED_DECL(winf,struct wsdl_info);

/** Service class structures that contain features common to its service instances */
struct service_class {
	char *			id;	/**< A user-friendly string identifier accessible through the configuration API. */
	DA_TYPED(pqn)	types;	/**< A prefixed QNames array of WS-Discovery types */
	DA_TYPED(winf)	wsdl_files;	/**< A \a struct wsdl_info array for DPWS metadata retrieval. */
	DA_TYPED(disp)	funcs;	/**< An array of dispatch functions. Generated or not by the skeleton generator, it dispatch request to the right function according to the action */
	fault_cbk		fault_func;	/**< An optional callback for listening faults not caused by a local invocation */
	event_end_cbk	event_end_hook;	/**< An optional callback for listening unexpected event subscription end */
	const void *	user_data;	/**< API user free data that can be set through the configuration API and retrived in the service function */
};

/** This structure is a configuration helper which values will be copied to the
 * device for runtime.
 */
struct device_model {
	char *			id;	/**< A user-friendly string identifier accessible through the configuration API. */

	DA_TYPED(lstr)	manufacturers;	/**< DPWS metadata */
	char *			manufacturer_url;	/**< DPWS metadata */
	DA_TYPED(lstr)	model_names;	/**< DPWS metadata */
	char *			model_number;	/**< DPWS metadata */
	char *			model_url;	/**< DPWS metadata */
	char *			presentation_url;	/**< DPWS metadata */

	DA_TYPED(href)	service_classes;	/**< Typed array for service class that will be used to create service instances if the automatic mode creation is used */
};

/** Status used to manage devices life-cycles */
typedef enum {
	INIT,		/**< Starting status. The object has been created. */
	OFFLINE,	/**< The device is configured an ready to be "published" on the network. */
	ONLINE,		/**< The device is reachable on the network. Granted even if helloes not send yet. In theory, helloes and byes could overlap. */
	CLONED,		/**< The device is a copy of an online device created to make update possible */
	REPLACED	/**< The device has been replaced by its clone so it has the same UUID and becomes unusable for the toolkit */
} device_status;

/**< Macro that returns true if the device is visible through the browsing API
 * so it cannot see clones or not-ready devices.
 */
#define API_VISIBLE_DEVICE(d) (d->status == OFFLINE || d->status == ONLINE)

/** TRUE if the device is online */
#define STARTED_DEVICE(d) (d->status == ONLINE)
/** TRUE if the device cannot be reached through the network */
#define STOPPED_DEVICE(d) (d->status != ONLINE)
/** TRUE if the device must be published through WS-Discovery */
#define DEVICE_VISIBLE(d) (d->status == ONLINE && d->href != hidden_device_href)

/** The device structure */
struct device {
	struct service_endpoint	* super;	/**< DPWS defines that a device is also a service so it holds types and others. */
	short			href;	/**< Stored in the structure because returned by an API. */
	unsigned short	local_id;	/**< A numeric local ID provided by the creator to be able to generate diversified but stable UUIDs for devices without persistency */
	device_status	status;	/**< The runtime status of the device. */

	char *			uuid;	/**< DPWS device UUID. Stored as a string to make serialization more efficient */
	char *			context_path;	/**< Transient field that must not be freed: will be copied to super->service_ports->automatically bound port */
	uint32_t		metadata_version;	/**< Mandatory WS-Discovery metadata that should be incremented every time device metadata changes. */
	DA_TYPED(str) 	scopes;	/**< WS-Discovery metadata used for filtering */
	DA_TYPED(href)	hosted_services;	/**< A handle array that contains the services hosted by the device */

	DA_TYPED(lstr)	manufacturers;	/**< DPWS metadata */
	char *			manufacturer_url;	/**< DPWS metadata */
	DA_TYPED(lstr)	model_names;	/**< DPWS metadata */
	char *			model_number;	/**< DPWS metadata */
	char *			model_url;	/**< DPWS metadata */
	char *			presentation_url;	/**< DPWS metadata that is often used for device HTTP page. */
	DA_TYPED(lstr)	friendly_names;	/**< DPWS metadata */
	char *			firmware_version;	/**< DPWS metadata */
	char *			serial_number;	/**< DPWS metadata */
	char *          device_category;
	char *          model_id;

	const void *	user_data;	/**< API user free data that can be set through the configuration API and retrived in the service function */
};

#ifdef __cplusplus
extern "C" {
#endif

/*----------------------------------------------------------------------------*\
 *                              CONFIGURATION API                             *
\*----------------------------------------------------------------------------*/

extern dcpl_mutex_t * reg_conf_lock;	/**< Configuration lock. */
extern handlePool_t registryPool;	/**< Registry objects handle pool. */

/** Sets an attribute value on a service instance object.
 * @param pService A pointer on the service endpoint structure.
 * @param att The attribute numeric identifier.
 * @param value The value to set.
 * @param reset If true, the current value will wil erased and replaced. If
 * false and multi-valued, the value is added to the list.
 * @return A DPWS error code.
 */
int set_service_endpoint_ptr_att(struct service_endpoint * pService, int att, const void * value, DC_BOOL reset);

/** Sets an attribute value on a service port object.
 * @param pServicePort A pointer on the service port structure.
 * @param att The attribute numeric identifier.
 * @param value The value to set.
 * @param reset If true, the current value will wil erased and replaced. If
 * false and multi-valued, the value is added to the list.
 * @return A DPWS error code.
 */
int set_service_port_ptr_att(struct service_port * pServicePort, int att, const void * value, DC_BOOL reset);

/** Sets an attribute value on a service class object.
 * @param pServiceClass A pointer on the service endpoint structure.
 * @param att The attribute numeric identifier.
 * @param value The value to set.
 * @param reset If true, the current value will wil erased and replaced. If
 * false and multi-valued, the value is added to the list.
 * @return A DPWS error code.
 */
int set_service_class_ptr_att(struct service_class * pServiceClass, int att, const void * value, DC_BOOL reset);

/** Sets an attribute value on a device model object.
 * @param pDeviceModel A pointer on the device model structure.
 * @param att The attribute numeric identifier.
 * @param value The value to set.
 * @param reset If true, the current value will wil erased and replaced. If
 * false and multi-valued, the value is added to the list.
 * @return A DPWS error code.
 */
int set_device_model_ptr_att(struct device_model * pDeviceModel, int att, const void * value, DC_BOOL reset);

/** Sets an attribute value on a device object.
 * @param pDevice A pointer on the device structure.
 * @param att The attribute numeric identifier.
 * @param value The value to set.
 * @param reset If true, the current value will wil erased and replaced. If
 * false and multi-valued, the value is added to the list.
 * @return A DPWS error code.
 */
int set_device_ptr_att(struct device * pDevice, int att, const void * value, DC_BOOL reset);

/** Sets an attribute value on the registry.
 * @param att The attribute numeric identifier.
 * @param value The value to set.
 * @param reset If true, the current value will wil erased and replaced. If
 * false and multi-valued, the value is added to the list.
 * @return A DPWS error code.
 */
int set_registry_config_ptr_att(int att, const void * value, DC_BOOL reset);

/** Get the value of an integer attribute of the registry.
 * @param att The attribute numeric identifier.
 * @param index The index of the attribute if multivalued or 0 else.
 * @return The retrieved value.
 */
unsigned long get_registry_config_int_att(int att, int index);

/** Get the value of a registry attribute of a pointer type.
 * @param att The attribute numeric identifier.
 * @param index The index of the attribute if multivalued or 0 else.
 * @return The retrieved value or NULL if not available.
 */
void * get_registry_config_ptr_att(int att, int index);

/** Get the value of a service class attribute.
 * @param pServiceClass A pointer on the service class structure.
 * @param att The attribute numeric identifier.
 * @param index The index of the attribute if multivalued or 0 else.
 * @return The retrieved value or NULL if not available.
 */
void * get_service_class_ptr_att(struct service_class * pServiceClass, int att, int index);

/** Get number of values for a service class attribute.
 * @param pServiceClass A pointer on the service class structure.
 * @param att The attribute numeric identifier.
 * @return The number of attribute values.
 */
int get_service_class_att_count(struct service_class * pServiceClass, int att);

/** Get the value of a device model attribute.
 * @param pDeviceModel A pointer on the device model structure.
 * @param att The attribute numeric identifier.
 * @param index The index of the attribute if multivalued or 0 else.
 * @return The retrieved value or NULL if not available.
 */
void * get_device_model_ptr_att(struct device_model * pDeviceModel, int att, int index);

/** Get number of values for a device model attribute.
 * @param pDeviceModel A pointer on the device model structure.
 * @param att The attribute numeric identifier.
 * @return The number of attribute values.
 */
int get_device_model_att_count(struct device_model * pDeviceModel, int att);

/** Get the value of a service endpoint attribute of integer type.
 * @param pService A pointer on the service endpoint structure.
 * @param att The attribute numeric identifier.
 * @param index The index of the attribute if multivalued or 0 else.
 * @return The retrieved value.
 */
unsigned long get_service_endpoint_int_att(struct service_endpoint * pService, int att, int index);

/** Get the value of a service endpoint attribute of pointer type.
 * @param pService A pointer on the service endpoint structure.
 * @param att The attribute numeric identifier.
 * @param index The index of the attribute if multivalued or 0 else.
 * @return The retrieved value or NULL if not available.
 */
void * get_service_endpoint_ptr_att(struct service_endpoint * pService, int att, int index);

/** Get number of values for a service endpoint attribute.
 * @param pService A pointer on the service endpoint structure.
 * @param att The attribute numeric identifier.
 * @return The number of attribute values.
 */
int get_service_endpoint_att_count(struct service_endpoint * pService, int att);

/** Get the value of a service port attribute.
 * @param pServicePort A pointer on the service port structure.
 * @param att The attribute numeric identifier.
 * @param index The index of the attribute if multivalued or 0 else.
 * @return The retrieved value or NULL if not available.
 */
void * get_service_port_ptr_att(struct service_port * pServicePort, int att, int index);

/** Get the value of a device attribute of integer type.
 * @param pDevice A pointer on the device structure.
 * @param att The attribute numeric identifier.
 * @param index The index of the attribute if multivalued or 0 else.
 * @return The retrieved value.
 */
unsigned long get_device_int_att(struct device * pDevice, int att, int index);

/** Get the value of a device attribute of pointer type.
 * @param pDevice A pointer on the device structure.
 * @param att The attribute numeric identifier.
 * @param index The index of the attribute if multivalued or 0 else.
 * @return The retrieved value or NULL if not available.
 */
void * get_device_ptr_att(struct device * pDevice, int att, int index);

/** Get number of values for a device attribute.
 * @param pDevice A pointer on the device structure.
 * @param att The attribute numeric identifier.
 * @return The number of attribute values.
 */
int get_device_att_count(struct device * pDevice, int att);


/*----------------------------------------------------------------------------*\
 *                                 MODULE API                                 *
\*----------------------------------------------------------------------------*/

extern struct dpws_reg_config registry_cfg;	/**< Module configuration. */
extern struct network_endpoint http_endpoint;	/**< The only currently supported network transport. Unique since toolkits listens on one unique HTTP port. */
extern short hidden_device_href;	/**< Private device that host endpoints. */

/** Initialize registry structures like mutexes.
 * @return A DPWS error code.
 */
int init_registry();

/** Resets the registry.
 * Created device are destroyed and configuration parameters reset.
 */
void clean_registry();

/** Cleans registry structures like mutexes.
 */
void uninit_registry();

#define DPWS_CLIENT_USE		0	/**< Toolkit mode is purely client (no listener) */
#define DPWS_DISCOVERY_USE	1	/**< Toolkit mode is in "discovery monitor" mode and listens to Hello & Byes */
#define DPWS_EVENTING_USE	2	/**< The toolkit mode is in "monitor" mode for Hello, Byes and eventind ends. For devices, the device is hidden and used only for hosting endpoints that will receive messages */
#define DPWS_DEVICE_USE		3	/**< full-fledged device or toolkit. */

/** Used by WS-Discovery to know if a device mustr be published.
 * @param pDevice A ponter on the device structure.
 * @return DPWS_DEVICE_USE (must be published) or  DPWS_EVENTING_USE (must not).
 */
int get_device_mode(struct device* pDevice);

/** Checks that mandatory configuration parameters with no default were set so
 * that the server loop is ready to be started.
 * @return DPWS_OK or a DPWS error code.
 */
int check_configuration();

/** Browses the invocation list of the reception network endpoint (unique
 * currently) to retrieve the service port and then the invoked service.
 * This function take a hold on the invocation list that should be released
 * using \a find_endpoint_port.
 * @param address The address coming from the WS-Addressing To header.
 * @return A pointer on the service port if retrieved or NULL else.
 */
struct service_port * find_endpoint_port(const char * address);

/** Releases the reference count held since the call to \a find_endpoint_port.
 * Should be called after invocation so the invocation list and then the device
 * can be freed if required.
 * @param endpoint A pointer on the held service port.
  */
void release_endpoint_port(struct service_port * endpoint);

/** Retrieves the service endpoint object for a service or service port.
 * @param href The service or service port handle reference.
 * @return A pointer on the service endpoint if retrieved or NULL else.
 */
struct service_endpoint * get_endpoint(short href);

/** Gives a read-only access to the array of WS-Discovery types for a service
 * @param service The pointer on the service endpoint.
 * @return A prefixed QNames array containing types.
 */
DA_TYPED(pqn) * get_service_types(struct service_endpoint * service);

/** Returns the array of  WS-Discovery types for a device.
 * @param dpws The DPWS runtime structure, used to retrieve the protocol version
 * @param device A pointer on the device.
 * @param[out] all_types A prefixed QNames array that will receive the readonly
 * content made of the device types plus the implicit wsdp:Device type. Content
 * should not be freed.
 * @param safe If true, no need to use the handle lock. Protected from the outside.
 * @return The provided prefixed QNames array containing types.
 */
DA_TYPED(pqn) * get_device_types(struct dpws * dpws, struct device * device, DA_TYPED(pqn) * all_types, DC_BOOL safe);

/** Retrieves the transient namespace table for a device's types to use in
 * WS-Discovery messages.
 * @param dpws The DPWS runtime structure that will be used for memory
 * allocation.
 * @param device A pointer on the device.
 * @return A GSOAP format namespace table allocated on the runtime structure.
 */
struct Namespace * get_device_discovery_ns_table(struct dpws * dpws, struct device * device);

/** Retrieves the transient namespace table for a device's hosted services types
 * to use in WS-MetadataExchange messages.
 * @param dpws The DPWS runtime structure that will be used for memory
 * allocation.
 * @param device A pointer on the device.
 * @return A GSOAP format namespace table allocated on the runtime structure.
 */
struct Namespace * get_device_metadata_ns_table(struct dpws * dpws, struct device * device);

/** Retrieves the number of WSDLs for a hosted service.
 * Used for metadata publication.
 * @param endpoint A pointer on the hosted service structure.
 * @return The number of WSDL for the hosted service that will be used to build
 * WS-MetadataExchange messages.
 */
int count_tns_wsdl(struct service_endpoint *endpoint);

/** Builds the transport address for a given service port.
 * @param dpws Used to allocate the returned buffer. If NULL, allocation is
 * performed on the heap.
 * @param host A string IP address (v4 or v6).
 * @return The built transport address.
 */
char* get_transport_address(struct dpws* dpws, const char * host, short hrefServicePort);

/** Builds the endpoint reference for a given service port.
 * @param dpws Used to allocate the returned buffer. If NULL, allocation is
 * performed on the heap.
 * @param[out] dest The endpoint reference that will be filled.
 * @param host A string IP address (v4 or v6).
 * @return The built endpoint reference.
 */
struct wsa_endpoint_ref * build_local_endpoint_ref(struct dpws* dpws, struct wsa_endpoint_ref * dest, const char * host, short service_port_href);

/** Schedules Hello messages for enabled devices when the server is started.
 * @return DPWS_OK or a DPWS error code.
 */
int advertize_enabled_devices();

/** Schedules Bye messages for enabled devices when the server is stopped.
 */
void shutdown_enabled_devices();

/** Browse callback for service ports */
typedef DC_BOOL (* service_port_cbk)(
	short * phref,	/**< Pointer on the service port handle reference */
	void * param	/**< The callback user data */
);

/** Browses service ports for a service (device or hosted service)
 * @param href The service handle reference.
 * @param hook The function to call back for every service port.
 * @param param Callback user data.
 * @return DPWS_OK or a DPWS error code.
 */
int browse_service_ports(short href, service_port_cbk hook, void * param);

#ifdef __cplusplus
}
#endif


/*----------------------------------------------------------------------------*\
 *                                MODULE CONSTANTS                            *
\*----------------------------------------------------------------------------*/

#define REGISTRY_POOL_NB_BUCKETS	32

#define DEVICE_INSTANCE_TYPE		1
#define SERVICE_ENDPOINT_TYPE		2
#define DEVICE_MODEL_TYPE			3
#define SERVICE_CLASS_TYPE 			4
#define SERVICE_PORT_TYPE 			5

#endif
