/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2254 $
|                     $Date: 2009-04-08 21:57:12 +0200 (mer, 08 avr 2009) $
\*============================================================================*/

#include "dc/dc_Dpws.h"
#include "dcDPWS_Memory.h"
#include "dcDPWS_Utils.h"
#include "dcCOMN_Tools.h"
#include "dcGSOAP_Runtime.h"

/*----------------------------------- Types ----------------------------------*/

typedef int (*token_parser_cbk)(struct dpws *, dyn_array_t *, char *);

#define DC_UTIL_STRDUP(dpws, s)	dpws ? DC_MSG_STRDUP(DC_MEM_API, dpws, s) : DC_STRDUP(DC_MEM_API, s)

/*------------------------- Static Functions prototypes ----------------------*/

static int parse_list(struct dpws * dpws, dyn_array_t * dest, char * source, token_parser_cbk token_parser);
static int parse_qname(struct dpws * dpws, DA_TYPED(qn) * dest, char * qname);
static int parse_string(struct dpws * dpws, DA_TYPED(str) * dest, char * source);
static int parse_qname_list(struct dpws * dpws, DA_TYPED(qn) * dest, char * source);
static int parse_token_list(struct dpws * dpws, DA_TYPED(str) * dest, char * source);
static int serialize_qname_list(struct soap* soap, const char * tag, DA_TYPED(qn) *qnames);
static int serialize_prefixed_qname_list(struct soap* soap, const char * tag, DA_TYPED(pqn) *qnames);
static int dpws_ignore_element(struct soap *soap);
static struct wsa_endpoint_ref * endpoint_ref_copy(struct dpws* dpws, struct wsa_endpoint_ref * dest, struct wsa_endpoint_ref * src);
static DC_BOOL strcasecmp_match(char **a, char **b);
static DC_BOOL strcmp_match(char **a, char **b);
static DC_BOOL ldap_match(char **a, char **b);
DC_BOOL _rfc2396_match(char **a, char **b);
static da_cmp_cbk get_scope_match_func(scope_match_rule_t match_rule);

/*----------------------------------------------------------------------------*\
 *                              PARSING UTILITIES                             *
\*----------------------------------------------------------------------------*/
static int parse_list(
    struct dpws * dpws,
    dyn_array_t * dest,
    char * source,
    token_parser_cbk token_parser
)
{
    char * p = source, * token = NULL;
    int ret = SOAP_OK;
    for (;;p++)
    {
        switch (*p)
        {
        case '\0':
            if (token)
                ret = token_parser(dpws, dest, token);
            return ret;
        case ' ':
        case '\t':
        case 0xA:
        case 0xD:
            if (token)
            {
                *p = '\0';
                ret = token_parser(dpws, dest, token);
                if (ret)
                    return ret;
                token = NULL;
            }
            break;
        default:
            if (!token)
                token = p;
        }
    }
}

static int parse_qname(struct dpws * dpws, DA_TYPED(qn) * dest, char * qname)
{
    struct qname * p;
    char * ns;
    int ret = DPWS_OK;

	DC_ERROR_ASSERT_ALLOC(p = DA_ADD(dest));
	DC_ERROR_ASSERT(ns = soap_get_ns_uri(dpws_dpws2soap(dpws), qname), SOAP_NAMESPACE);
	DC_ERROR_ASSERT_ALLOC(p->ns = DC_MSG_STRDUP(DC_MEM_UNKNOWN, dpws, ns));	// ns may not be available out of scope
    p->lname = soap_get_lname(dpws_dpws2soap(dpws), qname);	// the qname value should be available until soap_end
DC_FUNC_ERROR
	return ret;
}

static int parse_string(struct dpws * dpws, DA_TYPED(str) * dest, char * source)
{
    char ** p;
    int ret = DPWS_OK;

	DC_ERROR_ASSERT_ALLOC(p = DA_ADD(dest));
	DC_ERROR_ASSERT_ALLOC(*p = DC_MSG_STRDUP(DC_MEM_UNKNOWN, dpws, source));

DC_FUNC_ERROR
	return ret;
}

static int parse_qname_list(struct dpws * dpws, DA_TYPED(qn) * dest, char * source)
{
    return parse_list(dpws, (dyn_array_t *)dest, source, (token_parser_cbk)parse_qname);
}

static int parse_token_list(struct dpws * dpws, DA_TYPED(str) * dest, char * source)
{
    return parse_list(dpws, (dyn_array_t *)dest, source, (token_parser_cbk)&parse_string);
}

char* find_localized_entry(struct localized_string ** entries, int entry_nb, char *preferred_lang)
{
	int i, p_index = -1, d_index = -1, e_index = -1, ret = -1;
	char * lang;

	// Status (if more than one take the one without @lang or in english (first of the 2)
	for (i = 0; i < entry_nb; i++) {
		lang = entries[i]->lang;
		if (!lang)
			d_index = i;
		else if (preferred_lang && !strcmp(lang, preferred_lang))
			p_index = i;
		else if (!strncmp(lang, "en", 2))
			e_index = i;
	}
	if (p_index >= 0)	// take first the toolkit preferred language
		ret = p_index;
	else if (d_index >= 0)	// then the one without language
		ret = d_index;
	else if (e_index >= 0)	// then english
		ret = e_index;
	else if (entry_nb > 0)	// finally take first
		ret = 0;

	return ret < 0 ? NULL : entries[ret]->s;
}


/*----------------------------------------------------------------------------*\
 *                       CUSTOM GSOAP TYPES MARSHALLING                       *
\*----------------------------------------------------------------------------*/
void soap_serialize_qname_list(struct soap* soap, dyn_array_t *const* a) {}

void soap_default_qname_list(struct soap* soap, dyn_array_t ** a) {*a = NULL;}

static int serialize_qname_list(struct soap* soap, const char * tag, DA_TYPED(qn) *qnames)
{
	da_allocator_t daa;
    DA_TYPED(str) prefixes;
    int i, ret = DPWS_OK;

    DA_INIT(char*, &prefixes, DC_MEM_TRANSIENT, soap_get_da_allocator(soap, &daa), 3);

    // defines missing prefixes before soap_element_begin_out
    for (i = 0; i < qnames->nb; i++) {
		struct qname * qn = (struct qname *)GET_ENTRY(qnames, i);
		char ** pstr;
		DC_ERROR_ASSERT_ALLOC(pstr = DA_ADD(&prefixes));
		if (qn->ns && qn->lname) {
			*pstr = (char *)soap_check_prefix_definition(soap, qn->ns, NULL);
		} else {
			*pstr = NULL;
		}
	}

	soap_element_begin_out(soap, tag, -1, "");
    for (i = 0; i < prefixes.nb; i++)
    {
	    char * pfx;
		char * lname;
        if (i > 0)
        	soap_string_out(soap, " ", 0);
        pfx = *DA_GET(&prefixes, i);
		lname = DA_GET(qnames, i)->lname;
        if (pfx) {
        	soap_string_out(soap, pfx, 0);
        	soap_string_out(soap, ":", 0);
        }
		if (lname)
			soap_string_out(soap, lname, 0);
    }
    soap_element_end_out(soap, tag);
    soap_pop_namespace(soap);
    ret = soap->error;

DC_FUNC_ERROR
	return ret;
}

// when namespaces were defined at the top-level
static int serialize_prefixed_qname_list(struct soap* soap, const char * tag, DA_TYPED(pqn) *qnames)
{
    int i;
    char * pfx;

	soap_element_begin_out(soap, tag, -1, "");
    for (i = 0; i < qnames->nb; i++)
    {
        if (i > 0)
        	soap_string_out(soap, " ", 0);
        pfx = DA_GET(qnames, i)->prefix;
        if (pfx) {
        	soap_string_out(soap, pfx, 0);
        	soap_string_out(soap, ":", 0);
        }
       	soap_string_out(soap, DA_GET(qnames, i)->qname.lname, 0);
    }
    soap_element_end_out(soap, tag);
    return soap->error;
}

int soap_out_qname_list(struct soap* soap, const char * tag, int id, dyn_array_t *const* a, const char * type)
{
    dyn_array_t *qnames = *a;
    if (!qnames)
        soap_element_null(soap, tag, -1, "");
    else if (qnames->f_size == sizeof(struct qname))
    	serialize_qname_list(soap, tag, (DA_TYPED(qn)*)qnames);
    else
    	serialize_prefixed_qname_list(soap, tag, (DA_TYPED(pqn)*)qnames);
    return soap->error;
}

struct dynamic_array ** soap_in_qname_list(struct soap* soap, const char* tag, dyn_array_t **a, const char* type)
{
	DA_TYPED(qn) *qnames;
	da_allocator_t daa;
    if (*a)
        qnames = (DA_TYPED(qn) *)*a;
	else {
        qnames = (DA_TYPED(qn)*)soap_malloc(soap, sizeof(dyn_array_t));
        *a = (dyn_array_t *)qnames;
	}
    DA_INIT(qname_t, qnames, DC_MEM_TRANSIENT, soap_get_da_allocator(soap, &daa), DISCOVERY_ALLOC_INCREMENT);
    if (soap_element_begin_in(soap, tag, 0))
        return NULL;
	if (soap->body) {
		int ret = parse_qname_list(dpws_soap2dpws(soap), qnames, soap_string_in(soap, 1, -1, -1));
		if (ret) {
			soap->error = ret;
			return NULL;
		}
		if (soap_element_end_in(soap, tag))
			return NULL;
	}
    return a;
}

void soap_serialize_uri_list(struct soap* soap, dyn_array_t *const* a) {}

void soap_default_uri_list(struct soap* soap, dyn_array_t ** a) {*a = NULL;}

int soap_out_uri_list(struct soap* soap, const char * tag, int id, dyn_array_t *const* a, const char * type)
{
    int i;

    if (!*a)
        return soap_element_null(soap, tag, -1, "");

    soap_element_begin_out(soap, tag, -1, "");
    for (i = 0; i < (*a)->nb; i++)
    {
        if (i > 0)
            soap_string_out(soap, " ", 0);
       	soap_string_out(soap, *(char **)GET_ENTRY(*a, i), 0);
    }
    soap_element_end_out(soap, tag);

    return soap->error;
}

struct dynamic_array ** soap_in_uri_list(struct soap* soap, const char* tag, dyn_array_t **a, const char* type)
{
	DA_TYPED(str) *tab;
	da_allocator_t daa;
    if (*a)
        tab = (DA_TYPED(str) *)*a;
	else {
        tab = (DA_TYPED(str) *)soap_malloc(soap, sizeof(dyn_array_t));
        *a = (dyn_array_t *)tab;
	}

    DA_INIT(char *, tab, DC_MEM_TRANSIENT, soap_get_da_allocator(soap, &daa), DISCOVERY_ALLOC_INCREMENT);
    if (soap_element_begin_in(soap, tag, 0))
        return NULL;
	if (soap->body) {
		int ret = parse_token_list(dpws_soap2dpws(soap), tab, soap_string_in(soap, 1, -1, -1));
		if (ret) {
			soap->error = ret;
			return NULL;
		}
		if (soap_element_end_in(soap, tag))
			return NULL;
	}
    return a;
}


void soap_serialize_endpoint_ref(struct soap* soap, struct wsa_endpoint_ref *const*a) {}

void soap_default_endpoint_ref(struct soap* soap, struct wsa_endpoint_ref **a)
{
    *a = NULL;
}

int soap_out_endpoint_ref(struct soap *soap, const char *tag, int id, struct wsa_endpoint_ref *const*a, const char *type)
{
    struct wsa_endpoint_ref *p = *a;
    if (!p)
        return soap_element_null(soap, tag, -1, "");
    soap_element_begin_out(soap, tag, -1, "");
    soap_simple_element_out(soap, "wsa:Address", p->address);
    if (p->subscription_id)
    {
        soap_element_begin_out(soap, "wsa:ReferenceParameters", -1, "");
        soap_simple_element_out(soap, "wse:Identifier", p->subscription_id);
        soap_element_end_out(soap, "wsa:ReferenceParameters");
    }
    soap_element_end_out(soap, tag);
    return SOAP_OK;
}

static int dpws_ignore_element(struct soap *soap)
{
    if (!soap_peek_element(soap))
    {
        soap->peeked = 0;
        soap->level++;
        while (!dpws_ignore_element(soap))
            ;
        if (soap->error == SOAP_NO_TAG)
            soap->error = soap_element_end_in(soap, NULL);
    }
    return soap->error;
}

struct wsa_endpoint_ref ** soap_in_endpoint_ref(struct soap *soap, const char *tag, struct wsa_endpoint_ref **a, const char *type)
{
    struct wsa_endpoint_ref *p = *a;
    if (soap_element_begin_in(soap, tag, 0))
        return NULL;
    if (soap->body)
    {
        if (!p)
        {
            *a= p = (struct wsa_endpoint_ref*)soap_malloc(soap, sizeof(struct wsa_endpoint_ref));
            if (!p)
                return NULL;
            dpws_default_wsa_endpoint_ref(p);
        }
        if (wsa_simple_element_in(dpws_soap2dpws(soap), "wsa:Address", &p->address))
            return NULL;

        // ignore next sequence element
        if (!soap_element_begin_in(soap, "wsa:ReferenceProperties", 1))
        {
            if (soap_element_end_in(soap, "wsa:ReferenceProperties"))
                return NULL;
        }
        else if (soap->error != SOAP_TAG_MISMATCH && soap->error != SOAP_NO_TAG)
            return NULL;

        if (!soap_element_begin_in(soap, "wsa:ReferenceParameters", 1))
        {
            if (soap_simple_element_in(soap, "wse:Identifier", &p->subscription_id))
                return NULL;
            if (soap_element_end_in(soap, "wsa:ReferenceParameters"))
                return NULL;
        }
        else if (soap->error != SOAP_TAG_MISMATCH && soap->error != SOAP_NO_TAG)
            return NULL;

        // ignore next sequence element
        if (!soap_element_begin_in(soap, "wsa:PortType", 1))
        {
            if (soap_element_end_in(soap, "wsa:PortType"))
                return NULL;
        }
        else if (soap->error != SOAP_TAG_MISMATCH && soap->error != SOAP_NO_TAG)
            return NULL;

        // ignore next sequence element
        if (!soap_element_begin_in(soap, "wsa:ServiceName", 1))
        {
            if (soap_element_end_in(soap, "wsa:ServiceName"))
                return NULL;
        }
        else if (soap->error != SOAP_TAG_MISMATCH && soap->error != SOAP_NO_TAG)
            return NULL;

        // loop over optional elements (interoperability)
        for (;;)
        {
            soap->error = SOAP_TAG_MISMATCH;
            if (soap->error == SOAP_TAG_MISMATCH)
                soap->error = dpws_ignore_element(soap);
            if (soap->error == SOAP_NO_TAG)
                break;
            if (soap->error)
                return NULL;
        }
    }
    else
        return NULL;

    if (soap_element_end_in(soap, tag))
        return NULL;
    return a;
}
int check_wsa_version(struct dpws * dpws)
{
	struct soap * soap = dpws_dpws2soap(dpws);
	char * uri = soap_get_ns_uri(soap, soap->tag);
	if (dpws->wsa_version)
		return strcmp(dpws->wsa_version->wsa_uri, uri);
	else
		return set_wsa_version(dpws, uri);
}

int wsa_simple_element_in(struct dpws *dpws, const char *tag, char** value)
{
	int status;
	struct soap * soap = dpws_dpws2soap(dpws);
    if (!(status = soap_element_begin_in(soap, tag, 0)) && soap->body) {
		if (check_wsa_version(dpws))
			return SOAP_TAG_MISMATCH;
	    *value = soap_string_in(soap, 0, -1, -1);
	    status = soap_element_end_in(soap, tag);
    }
	return status;
}


/*----------------------------------------------------------------------------*\
 *                                 COPY UTILITIES                             *
\*----------------------------------------------------------------------------*/

static struct wsa_endpoint_ref * endpoint_ref_copy(struct dpws* dpws, struct wsa_endpoint_ref * dest, struct wsa_endpoint_ref * src)
{
    if (src && dest)
    {
    	DC_ERROR_ASSERT_NO_RC(dest->address = DC_UTIL_STRDUP(dpws, src->address));
		DC_ERROR_ASSERT_NO_RC((dest->subscription_id = DC_UTIL_STRDUP(dpws, src->subscription_id)) || !src->subscription_id);
#ifdef WITH_WSMAN
		DC_ERROR_ASSERT_NO_RC((dest->wsman_params.resource_uri = DC_UTIL_STRDUP(dpws, src->wsman_params.resource_uri)) || !src->wsman_params.resource_uri);
        if (src->wsman_params.nb_selectors > 0)
        {
	        int i;
	        DC_ERROR_ASSERT_NO_RC(dest->wsman_params.selectors = (struct wsman_selector*) (dpws ?
            		DC_MSG_MALLOC(DC_MEM_UNKNOWN, dpws, sizeof(struct wsman_selector) * src->wsman_params.nb_selectors) :
					DC_MALLOC(DC_MEM_API, sizeof(struct wsman_selector) * src->wsman_params.nb_selectors)));
            for (i = 0; i < src->wsman_params.nb_selectors; i++)
            {
            	DC_ERROR_ASSERT_NO_RC(dest->wsman_params.selectors[i].name = DC_UTIL_STRDUP(dpws, src->wsman_params.selectors[i].name));
            	DC_ERROR_ASSERT_NO_RC(dest->wsman_params.selectors[i].value = DC_UTIL_STRDUP(dpws, src->wsman_params.selectors[i].value));
            }
        }
        else {
        	dest->wsman_params.nb_selectors = 0;
        	dest->wsman_params.selectors = NULL;
        }
#endif
		dest->dpws_version = src->dpws_version;
    }
	return dest;

DC_FUNC_ERROR
	if (!dpws)
		dpws_endpoint_ref_content_free(dest);
	return NULL;
}

struct wsa_endpoint_ref * endpoint_ref_dup(struct dpws* dpws, struct wsa_endpoint_ref * src)
{
    struct wsa_endpoint_ref *dest = NULL;
    if (src)
    {
        dest = (struct wsa_endpoint_ref*)DC_MSG_MALLOC(DC_MEM_UNKNOWN, dpws, sizeof(struct wsa_endpoint_ref));
        endpoint_ref_copy(dpws, dest, src);
    }
    return dest;
}

struct wsa_endpoint_ref * dpws_endpoint_ref_copy(struct wsa_endpoint_ref * dest, struct wsa_endpoint_ref * src)
{
    return endpoint_ref_copy(NULL, dest, src);
}

struct wsa_endpoint_ref * dpws_endpoint_ref_dup(struct wsa_endpoint_ref * src)
{
    struct wsa_endpoint_ref *dest = NULL;
    if (src)
    {
        if (!(dest = (struct wsa_endpoint_ref*)DC_MALLOC(DC_MEM_API, sizeof(struct wsa_endpoint_ref))))
        	return NULL;
        dest = endpoint_ref_copy(NULL, dest, src);
    }
    return dest;
}

void dpws_free(void * buffer)
{
	DC_FREE(DC_MEM_API, buffer);
}

void dpws_endpoint_ref_content_free(struct wsa_endpoint_ref * src)
{
    if (src)
    {
        DC_FREE(DC_MEM_API, src->address);
        DC_FREE(DC_MEM_API, src->subscription_id);
        src->address = NULL;
        src->subscription_id = NULL;
#ifdef WITH_WSMAN
        DC_FREE(DC_MEM_API, src->wsman_params.resource_uri);
		src->wsman_params.resource_uri = NULL;
        if (src->wsman_params.nb_selectors > 0)
        {
	        int i;
            for (i = 0; i < src->wsman_params.nb_selectors; i++)
            {
            	DC_FREE(DC_MEM_API, src->wsman_params.selectors[i].name);
            	DC_FREE(DC_MEM_API, src->wsman_params.selectors[i].value);
            }
	        DC_FREE(DC_MEM_API, src->wsman_params.selectors);
			src->wsman_params.selectors = NULL;
	        src->wsman_params.nb_selectors = 0;
        }
#endif
    }
}

void dpws_endpoint_ref_free(struct wsa_endpoint_ref * src)
{
    if (src)
    {
        dpws_endpoint_ref_content_free(src);
        DC_FREE(DC_MEM_API, src);
    }
}

void dpws_endpoint_ref_array_free(struct wsa_endpoint_ref ** endpoints, int nb_endpoints)
{
    if (endpoints)
    {
        int i;
		for (i = 0; i < nb_endpoints; i++)
		{
			dpws_endpoint_ref_free(endpoints[i]);
		}
    }
}


/*----------------------------------------------------------------------------*\
 *                                SCOPE UTILITIES                             *
\*----------------------------------------------------------------------------*/

scope_match_rule_t parse_match_rule(struct dpws * dpws, char *uri)
{
    if (!uri)
        return WSD_MATCH_BY_RFC2396;
	else if (!strcmp(uri, dpws->protocols->wsd_match_by_uuid_uri))
        return WSD_MATCH_BY_UUID;
	else if (!strcmp(uri, dpws->protocols->wsd_match_by_ldap_uri))
        return WSD_MATCH_BY_LDAP;
	else if (!strcmp(uri, dpws->protocols->wsd_match_by_strcmp_uri))
        return WSD_MATCH_BY_STRCMP;
    else	// included rfc2396 match rule
        return WSD_MATCH_BY_RFC2396;
}

char * scope_match_string(struct dpws * dpws, scope_match_rule_t match_rule)
{
    switch (match_rule)
    {
    case WSD_MATCH_BY_RFC2396:
    default:
        return NULL;
    case WSD_MATCH_BY_UUID:
		return dpws->protocols->wsd_match_by_uuid_uri;
    case WSD_MATCH_BY_LDAP:
		return dpws->protocols->wsd_match_by_ldap_uri;
    case WSD_MATCH_BY_STRCMP :
		return dpws->protocols->wsd_match_by_strcmp_uri;
    }
}

static DC_BOOL strcasecmp_match(char **a, char **b)
{
    return strcasecmp(*a, *b) ? DC_FALSE : DC_TRUE;
}

static DC_BOOL strcmp_match(char **a, char **b)
{
    return strcmp(*a, *b) ? DC_FALSE : DC_TRUE;
}

static DC_BOOL ldap_match(char **a, char **b)
{
	return DC_FALSE;
}

DC_BOOL _rfc2396_match(char **a, char **b)
{
    return rfc2396_match(*a, *b);
}

static da_cmp_cbk get_scope_match_func(scope_match_rule_t match_rule)
{
    switch (match_rule)
    {
    case WSD_MATCH_BY_UUID:
        return (da_cmp_cbk)strcasecmp_match;
    case WSD_MATCH_BY_LDAP:
        return (da_cmp_cbk)ldap_match;
    case WSD_MATCH_BY_STRCMP :
        return (da_cmp_cbk)strcmp_match;
    case WSD_MATCH_BY_RFC2396:
    default:
        return (da_cmp_cbk)_rfc2396_match;
    }
}

DC_BOOL probe_match(dyn_array_t * types, DA_TYPED(str) * scopes, scope_match_rule_t match_rule,
	dyn_array_t *dev_types, DA_TYPED(str) *dev_scopes, da_cmp_cbk type_compare)
{
	char * default_scope = WSD_DEFAULT_SCOPE;
    da_cmp_cbk match_func = get_scope_match_func(match_rule);
    return (dev_scopes && dev_scopes->nb > 0
               ? da_is_included((dyn_array_t *)scopes, (dyn_array_t *)dev_scopes, match_func)
               : scopes == NULL || scopes->nb == 0 || (scopes->nb == 1 && match_func(GET_ENTRY(scopes, 0), &default_scope)))
               && da_is_included(types, dev_types, type_compare);	// will return true if types is empty
}

/*----------------------------------------------------------------------------*\
 *                           DATA CONSISTENCY TOOLS                           *
\*----------------------------------------------------------------------------*/

#ifdef DC_API_PARAM_CONTROL

DC_BOOL check_discovery_filter(discovery_filter_t * filter)
{
	int i;
	if (filter->types) {
		if (filter->nb_types <= 0) return DC_FALSE;
		for (i = 0; i < filter->nb_types; i++) {
			if (!filter->types[i].ns) return DC_FALSE;
			if (!filter->types[i].lname) return DC_FALSE;
		}
	} else
		if (filter->nb_types != 0) return DC_FALSE;
	if (filter->match_rule == WSD_MATCH_BY_LDAP) return DC_FALSE;
	if (filter->scopes) {
		if (filter->nb_scopes <= 0) return DC_FALSE;
		for (i = 0; i < filter->nb_scopes; i++) {
			if (!filter->scopes[i]) return DC_FALSE;
		}
	} else
		if (filter->nb_scopes != 0) return DC_FALSE;
	return DC_TRUE;
}
#endif
