/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2220 $
|                     $Date: 2009-03-25 10:28:16 +0100 (mer, 25 mar 2009) $
\*============================================================================*/

/** \page dyndepl Dynamic deployment
 * The dynamic deployment feature include:
 * <ul>
 * <li>Stack XML file configuration (dc_XMLConfiguration.h must be included),</li>
 * <li>Dynamic deployment service implementation (dc_Dyndepl.h must be included),</li>
 * <li>The dynamic deployment client helper (dc_DyndeplClient.h must be included),</li>
 * </ul>
 */

/***************************************************************************//**
* \file
 * API for the XML configuration feature.
 * The XML configuration feature is also the core for dynamic deployment since
 * it shares most of the dynamic deployment descriptor XML format.
*******************************************************************************/

#ifndef CONFIG_H
#define CONFIG_H

#include "dc/dc_Dpws.h"

#ifndef DC_RT_FMAC1	/* runtime functions declaration prefix */
# define DC_RT_FMAC1
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*----------------------------------------------------------------------------*\
 *                               API Error codes                              *
\*----------------------------------------------------------------------------*/

/** \name XML configuration errors
 * These code extend DPWS error codes.
 * @{
 */
#define DPWS_ERR_NO_LOADER_FOUND			-200	/**< The service loader could not be found for the implementation tag */
#define DPWS_ERR_NO_DISPATCH_FUNC			-201	/**< The service loader did not return the dispatch function for the service */
#define DPWS_ERR_NO_SERIALIZE_IMPL_FUNC		-202	/**< The service loader did not return the callback to serialize */
#define DPWS_ERR_INCORRECT_IMPL_TAG			-203	/**< The service loader could not parse the implementation tag */
#define DPWS_ERR_OPTIONAL_REF_MISSING		-204	/**< Not an error: the asked reference is missing but optional */
#define DPWS_ERR_MANDATORY_REF_MISSING		-205	/**< The asked mandatory reference is missing */
#define DPWS_ERR_COULD_NOT_BIND_IGNORE		-206	/**< The reference could not be bound but is ignorable */
#define DPWS_ERR_COULD_NOT_BIND_RETRY		-207	/**< The reference could not be bound: the service is supposed to retry later */
#define DPWS_ERR_COULD_NOT_BIND_FAIL		-208	/**< The reference could not be bound: the service is supposed to enter in fault mode */
#define DPWS_ERR_MULTIPLE_REFERENCE			-209	/**< The reference was bound to multiple endpoints which is not supported */
#define DPWS_ERR_STATIC_REFERENCE			-210	/**< The reference cannot be invalidated since it is static */
#define DPWS_ERR_LNAME_TOO_LONG				-211	/**< A local name in the configuration file exceeds the internal buffer */
#define DPWS_ERR_RESOURCE_ALREADY_EXIST		-212	/**< The resource already exists in the registry */
#define DPWS_ERR_SERVICE_NOT_FOUND			-213	/**< The hosted service was not found in the registry */
#define DPWS_ERR_SERVICE_CLASS_NOT_FOUND	-214	/**< The service class was not found in the registry */
#define DPWS_ERR_SUN_RESOURCE				-215	/**< The SUN resource could not be found. See SUN implementation error code */
// @}

/** \name XML grammatical errors
 * @{
 */
#define DPWS_ERR_TAG_MISMATCH			101	/**< The tag was not expected */
#define DPWS_ERR_EVENT_MISMATCH			102	/**< The EPX event was not expected */
#define DPWS_ERR_MISSING_PARTICLE		104	/**< An expected element is missing */
#define DPWS_ERR_TOO_MANY_OCCURENCES	105	/**< An element occurred too many times */
#define DPWS_ERR_UNEXPECTED_PARTICLE	106	/**< An expected element was encountered */
#define DPWS_ERR_MISSING_EXPECTED_ELT	107	/**< Not enough element occurrences */
#define DPWS_ERR_MISSING_REQUIRED_ATT	108 /**< A required attribute is missing */
#define DPWS_ERR_INVALID_ENUM			110	/**< A value does not belong to the enumeration */
// @}

/***************************************************************************//**
* \name Initialization API
* @{
*******************************************************************************/

/** Initialize the XML configuration features.
 * Must be called before any other call.
 * @return DPWS_OK or a DPWSCore error code if an error occurred.
 */
DC_RT_FMAC1 int dpws_config_init();

/** Shutdown the XML configuration features.
 * @return DPWS_OK or a DPWSCore error code if an error occurred.
 */
DC_RT_FMAC1 int dpws_config_shutdown();

// @}

/***************************************************************************//**
* \name Dynamic Loader API
* @{
*******************************************************************************/

/** This callback will be called when the service is instantiated by the dynamic
 * deployment service so that the implementation is able, for instance, to
 * create the user data for the service.
 * <b>Warning:</b> Note that the service was already created in the DC registry
 * but not in the "Dynamic Deployment" configuration so references and properties
 * are not usable yet.
 * @param href_endpoint The hosted service handle that was created and that can be
 * used by the loader to retrieve the implementation.
 * @return A DPWS error code.
 */
typedef int (*new_service_cbk) (short href_endpoint);

/** This callback will be called when the service is destroyed by the dynamic
 * deployment service so that the implementation is able, for instance, to free
 * the user data for the service.
 * @param href_endpoint The hosted service handle that was created and that can be
 * used by the loader to retrieve the implementation.
 * @return A DPWS error code.
 */
typedef int (*free_service_cbk) (short href_endpoint);

/** This callback will be called when the dynamic deployment service needs to get
 * back the implementation tag in order to return it in a response or make it
 * persistent.
 * @param href_sclass The service class handle that was passed initially to the
 * loader through the load callback.
 * @param szr_ctx The EPX serialization context for the implementation tag
 * restoration.
 * @return DPWS_OK or a DPWSCore error code if an error occurred.
 */
typedef int (*serialize_impl_cbk) (short href_sclass, void * szr_ctx);

/** This callback will be called when the service class is deleted by the
 * deployment srevice.
 * @param href_sclass The service class handle that was passed initially to the
 * loader through the load callback.
 * @return DPWS_OK or a DPWSCore error code if an error occured.
 */
typedef int (*unload_impl_cbk) (short href_sclass);

/** A structure that groups all callbacks a service class loader returns.
 */
struct scl_callbacks {
	new_service_cbk new_service;
	free_service_cbk free_service;
	serialize_impl_cbk serialize_impl;	// mandatory
	unload_impl_cbk unload_cbk;
};

/** This callback is exposed by a loader and called by the dynamic deployment
 * service when an implementation tag is parsed and service class is created.
 * @param href_sclass The service class handle that is being created by the
 * dynamic deployment service.
 * @param psr_ctx The callback is responsible for parsing the content until
 * the END_ELEMENT event of the implementation tag.
 * @param[out] p_dispatch_cbk The loader should return a DPWSCore dispatch
 * function. Only the serialization callback is mandatory.
 * @param[in,out] p_cbks The loader should fill the passed structured with
 * the expected callbacks.
 * @return DPWS_OK or a DPWSCore error code if an error occured.
 */
typedef	int (*load_cbk) (short href_sclass, void * psr_ctx, dispatch_cbk * p_dispatch_cbk, struct scl_callbacks * p_cbks);

/** Register a dynamic loader to the dynamic deployment service.
 * @param id The dynamic deployment descriptor implementation tag expanded name
 * which identifies the loader.
 * @param cbk The loader callback called by the dynamic deployment service when
 * an implementation tag is encountered.
 * @return DPWS_OK or a DPWSCore error code if an error occured.
 */
DC_RT_FMAC1 int dpws_register_loader(const struct qname * id, load_cbk cbk);

// @}

/***************************************************************************//**
* \name Reference API
* @{
*******************************************************************************/

/** To call the first time an EPR is needed for WS invocation.
 * @param dpws A pointer on the main DPWS API runtime data structure used to
 * perform potential WS-Discovery operations.
 * @param href_service The handle reference of the service endpoint to which
 * the binding reference is attached.
 * @param name The name of the binding.
 * @param[out] epr The return read-only EPR that must not be freed after use.
 * The EPR should be kept and used until an error occurs using it after what a
 * call to dpws_bind_reference may be required.
 * @return An API error code among which DPWS_ERR_COULD_NOT_BIND_IGNORE,
 * DPWS_ERR_COULD_NOT_BIND_RETRY and DPWS_ERR_COULD_NOT_BIND_FAIL should give
 * the caller the required behavior asked for the reference.
 */
DC_RT_FMAC1 int dpws_lookup_binding_ref(struct dpws * dpws, short href_service, const char * name, struct wsa_endpoint_ref ** epr);

/** To call when a previously retrieved EPR is invalid (invocation error).
 * @param dpws A pointer on the main DPWS API runtime data structure used to
 * perform potential WS-Discovery operations.
 * @param href_service The handle reference of the service endpoint to which
 * the binding reference is attached.
 * @param name The name of the binding.
 * @param[out] epr The return read-only EPR that must not be freed after use.
 * @param[out] epr The return read-only EPR that must not be freed after use.
 * The EPR should be kept and used until an error occurs using it after what a
 * call to dpws_bind_reference may be required.
 * @return An API error code among which DPWS_ERR_COULD_NOT_BIND_IGNORE,
 * DPWS_ERR_COULD_NOT_BIND_RETRY and DPWS_ERR_COULD_NOT_BIND_FAIL should give
 * the caller the required behavior asked for the reference.
 * DPWS_ERR_STATIC_REFERENCE will be returned if the binding reference is
 * defined as an EPR.
 */
DC_RT_FMAC1 int dpws_bind_reference(struct dpws * dpws, short href_service, const char * name, struct wsa_endpoint_ref ** epr);

// @}

/***************************************************************************//**
* \name Property API
* @{
*******************************************************************************/

/** Retrieves property values for a service.
 * @param href_service The service handle reference.
 * @param property The property name.
 * @param[in,out] values A string array provided by the caller to receive
 * results. Note that the returned strings should not be released (read-only).
 * @param[in,out] len Must contain initially the length of the string array.
 * When the function returns, it contains the number of results actually
 * retrieved.
 * @return DPWS_OK if the array supplied was large enough to receive results.
 * DPWS_ERR_MORE_RESULTS, if extra-results are available.
 */
DC_RT_FMAC1 int dpws_get_property_values(short href_service, char * property, char ** values, int * len);

// @}

/***************************************************************************//**
* \name XML Configuration persistence API
* @{
*******************************************************************************/

#define DPWS_CONFIG_DEFAULT_ID	"dpwscore.xml"	/**< A default name for the DPWSCore configuration "file" */

/** Load the DPWS registry from an EPX input stream.
 * This API should be used before the call to dpws_server_init.
 * @param psr_ctx The EPX parsing context.
 * @return DPWS_OK if no error occurred. Otherwise, a DPWS error code (negative
 * values) or a parsing error (positive) from gSOAP (I/O or syntactic error) or
 * SGXP (grammatical error defined in \a dcXTOOL_SchemaParsing.h).
 */
DC_RT_FMAC1 int dpws_load_config(void * psr_ctx);

/** Saves the DPWS registry into an EPX output stream.
 * @param szr_ctx The EPX serialization context.
 * @param inc_boot_seq If DC_TRUE, the boot sequence is incremented before.
 * @param as_fragment If DC_TRUE, the event stream is produced without start/end
 * document events.
 * @return 0 if no error occurred. Otherwise, an EPX implementation error code
 * (gSOAP error code).
 */
DC_RT_FMAC1 int dpws_save_config(void * szr_ctx, DC_BOOL inc_boot_seq, DC_BOOL as_fragment);

/** Load the DPWS registry using the SUN persistency layer.
 * This API should be used before the call to dpws_server_init.
 * @param dpws A DPWS runtime structure used for SUN serialization.
 * @param sun_in_stream Storage unit implementation-dependent opened input
 * stream. The stream will be closed automatically at the end of processing.
 * @return DPWS_OK if no error occurred. Otherwise, a DPWS error code (negative
 * values) or a parsing error (positive) from gSOAP (I/O or syntactic error) or
 * SGXP (grammatical error defined in \a dcXTOOL_SchemaParsing.h).
 */
DC_RT_FMAC1 int dpws_sun_load_config(struct dpws * dpws, void * sun_in_stream);

/** Saves the DPWS registry using the SUN persistency layer.
 * @param dpws A DPWS runtime structure used for SUN serialization.
 * @param sun_out_stream Storage unit implementation-dependent opened output
 * stream. The stream will be closed automatically at the end of processing.
 * @param inc_boot_seq If DC_TRUE, the boot sequence is incremented before.
 * @return 0 if no error occurred. Otherwise, an EPX implementation error code
 * (gSOAP error code).
 */
DC_RT_FMAC1 int dpws_sun_save_config(struct dpws * dpws, void * sun_out_stream, DC_BOOL inc_boot_seq);

// @}

#ifdef __cplusplus
}
#endif

#endif /*CONFIG_H*/
