/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2215 $
|                     $Date: 2009-03-19 15:44:46 +0100 (jeu, 19 mar 2009) $
\*============================================================================*/

/******************************************************************************\
 *                       Dynamic deployment data manager                      *
\******************************************************************************/

#include "dc/dc_Epx.h"
#include "dcCOMN_Tools.h"
#include "dcXCONF_Manager.h"
#include "dcCOMN_ListUtils.h"

/*----------------------------------- Types ----------------------------------*/

struct config_node {
	dcl_node_t n;
	void * config;
};

struct dynload_entry {
	struct qname platform_id;
	load_cbk cbk;
};

struct prop_grab_info {
	char * prop;
	char ** values;
	int i;
	int len;
};

typedef void (*free_cbk)(void *);

/*------------------------- Static Functions prototypes ----------------------*/

static int new_config(p_dcl_list_t list, void * config);
static DC_BOOL delete_config(p_dcl_list_t list, dcl_match_cbk_t match_cbk, void * cbk_data, free_cbk free_cbk);
static void delete_config_list(p_dcl_list_t list, free_cbk free_cbk);
static DC_BOOL check_reference(struct ws_binding_ref * b_ref, struct reference * ref);
static int service_match_cbk(struct config_node * node, short * href_service);
struct service_config * xconf_get_service(short href_service);
static void prefixed_qname_free_cbk(int mod, const da_allocator_t * p_alloc, struct prefixed_qname *pqname);
static void scope_free_cbk(int mod, const da_allocator_t * p_alloc, char **scope);
static void free_disc_hint_cbk(int mod, const da_allocator_t * p_alloc, struct discovery_hint * d_hint);
static void free_bref_cbk(int mod, const da_allocator_t * p_alloc, struct ws_binding_ref * bref);
static void free_pvalue_cbk(int mod, const da_allocator_t * p_alloc, struct property_value * pvalue);
static void delete_service_config(struct service_config * s_config);
static int sclass_match_cbk(struct config_node * node, short * href_sclass);
static struct service_class_config * get_service_class(short href_sclass);
struct service_class_config * xconf_get_service_class(short href_sclass);
static void free_ref_cbk(int mod, const da_allocator_t * p_alloc, struct reference * ref);
static void free_prop_cbk(int mod, const da_allocator_t * p_alloc, struct property * prop);
static DC_BOOL bref_match(struct ws_binding_ref * bref, char * name);
static DC_BOOL binding_match(struct reference * binding, char * name);
static DC_BOOL loader_match(struct dynload_entry * p_entry, struct qname * id);
static struct dynload_entry * get_loader(struct qname * id);
static void free_dynload_entry_cbk(int mod, const da_allocator_t * p_alloc, struct dynload_entry *p_entry);

/*----------------------------------- Data -----------------------------------*/

dcpl_mutex_t * xmlconf_lock = NULL;
dcl_list_t _services = {NULL, NULL}, _service_classes = {NULL, NULL};
p_dcl_list_t services = &_services, service_classes = &_service_classes;
DA_TYPED(de) loaders = DA_INITIALIZER(struct dynload_entry, DC_MEM_CONF_MANAGER, 1);

/*----------------------------------------------------------------------------*\
 *                              Common utilities                              *
\*----------------------------------------------------------------------------*/
static int new_config(p_dcl_list_t list, void * config)	// NOTE: allocated by the caller
{
	struct config_node * s_node = DC_MALLOC(DC_MEM_CONF_MANAGER, sizeof(struct config_node));
	if (!s_node)
		return DPWS_ERR_EOM;
	s_node->config = config;
	dcpl_mutex_lock(xmlconf_lock);
	dcl_add_last(list, &s_node->n);
	dcpl_mutex_unlock(xmlconf_lock);
	return DPWS_OK;
}

static DC_BOOL delete_config(p_dcl_list_t list, dcl_match_cbk_t match_cbk, void * cbk_data, free_cbk free_cbk)
{
	p_dcl_node_t c_node;
	DC_BOOL ret = DC_FALSE;

	dcpl_mutex_lock(xmlconf_lock);
	if ((c_node = dcl_find(list, match_cbk, cbk_data))) {
		dcl_remove(list, c_node);
		free_cbk(((struct config_node *)c_node)->config);
		DC_FREE(DC_MEM_CONF_MANAGER, c_node);
		ret = DC_TRUE;
	}
	dcpl_mutex_unlock(xmlconf_lock);

	return ret;
}

static void delete_config_list(p_dcl_list_t list, free_cbk free_cbk)
{
	p_dcl_node_t c_node, r_node;

	dcpl_mutex_lock(xmlconf_lock);
	c_node = list->first;
	while (c_node) {
		r_node = c_node;
		c_node = c_node->next;
		dcl_remove(list, r_node);
		free_cbk(((struct config_node *)r_node)->config);
		DC_FREE(DC_MEM_CONF_MANAGER, r_node);
	}
	dcpl_mutex_unlock(xmlconf_lock);
}

/*----------------------------------------------------------------------------*\
 *                       Service configuration management                     *
\*----------------------------------------------------------------------------*/
struct service_config * xconf_init_service()
{
	struct service_config * s_config = DC_MALLOC(DC_MEM_CONF_MANAGER, sizeof(struct service_config));
	if (s_config) {
		s_config->href = -1;
		DA_INIT(struct ws_binding_ref, &s_config->refs, DC_MEM_CONF_MANAGER, p_default_allocator, 1);
		DA_INIT(struct property_value, &s_config->property_values, DC_MEM_CONF_MANAGER, p_default_allocator, 1);
	}
	return s_config;
}

static DC_BOOL check_reference(struct ws_binding_ref * b_ref, struct reference * ref)
{
	if (strcmp(b_ref->name, ref->name))
		return DC_FALSE;
	else {
		b_ref->sclass_ref = ref;
		return DC_TRUE;
	}
}

int xconf_new_service(struct service_config * new_service)
{
	int i, j, ret = DPWS_OK;
	struct reference * ref;

	// Look for the service class in order to invoke the user data callback
	new_service->class = xconf_get_service_class((short)dpws_get_int_att(new_service->href, DPWS_INT_CLASS_HREF));
	if (new_service->class->cbks.new_service) {
		int ret = new_service->class->cbks.new_service(new_service->href);
		if (ret)
			return ret;
	}

	// Check references: loop on SC references first to detect mandatory reference missing
	if (new_service->refs.nb > 0 || new_service->property_values.nb > 0)
	{
		for(i = 0; i < new_service->class->refs.nb; i++) {
			ref = DA_GET(&new_service->class->refs, i);
			j = DA_BROWSE(&new_service->refs, check_reference, ref);
			if (j == new_service->refs.nb && ref->must_supply)
				return DPWS_ERR_MANDATORY_REF_MISSING;
		}
	}
	ret = new_config(services, new_service);

	return ret;
}

static int service_match_cbk(struct config_node * node, short * href_service)
{
	return *href_service == ((struct service_config *)node->config)->href;
}

struct service_config * xconf_get_service(short href_service)
{
	struct config_node * c_node;
	dcpl_mutex_lock(xmlconf_lock);
	c_node = (struct config_node *)dcl_find(services, (dcl_match_cbk_t)service_match_cbk, &href_service);
	dcpl_mutex_unlock(xmlconf_lock);
	return c_node ? (struct service_config *)c_node->config : NULL;
}

static void prefixed_qname_free_cbk(int mod, const da_allocator_t * p_alloc, struct prefixed_qname *pqname)
{
	p_alloc->free_cbk(mod, p_alloc->param, pqname->prefix);
	p_alloc->free_cbk(mod, p_alloc->param, pqname->qname.ns);
	p_alloc->free_cbk(mod, p_alloc->param, pqname->qname.lname);
}

static void scope_free_cbk(int mod, const da_allocator_t * p_alloc, char **scope)
{
	p_alloc->free_cbk(mod, p_alloc->param, *scope);
}

static void free_disc_hint_cbk(int mod, const da_allocator_t * p_alloc, struct discovery_hint * d_hint)
{
	DA_FREE(&d_hint->types, prefixed_qname_free_cbk);
	DA_FREE(&d_hint->scopes, scope_free_cbk);
}

static void free_bref_cbk(int mod, const da_allocator_t * p_alloc, struct ws_binding_ref * bref)
{
	p_alloc->free_cbk(mod, p_alloc->param, bref->name);
	dpws_endpoint_ref_free(bref->epr);
	dpws_release_proxy(bref->proxy);
	if (bref->wsd_hints) {
		DA_FREE(&bref->wsd_hints->hints, free_disc_hint_cbk);
		p_alloc->free_cbk(mod, p_alloc->param, bref->wsd_hints);
	}
}

static void free_pvalue_cbk(int mod, const da_allocator_t * p_alloc, struct property_value * pvalue)
{
	p_alloc->free_cbk(mod, p_alloc->param, pvalue->name);
	DC_FREE(mod, pvalue->value);
}

static void delete_service_config(struct service_config * s_config)
{
	// Invoke the user data callback
	if (s_config->class->cbks.free_service)
		s_config->class->cbks.free_service(s_config->href);
	xconf_free_service(s_config);
}

DC_BOOL xconf_delete_service(short href_service)
{
	return delete_config(services, (dcl_match_cbk_t)service_match_cbk, &href_service, (free_cbk)delete_service_config);
}

void xconf_free_service(struct service_config * service)
{
	DA_FREE(&service->refs, free_bref_cbk);
	DA_FREE(&service->property_values, free_pvalue_cbk);
	DC_FREE(DC_MEM_CONF_MANAGER, service);
}

void xconf_delete_services()
{
	delete_config_list(services, (free_cbk)delete_service_config);
}

/*----------------------------------------------------------------------------*\
 *                    Service class configuration management                  *
\*----------------------------------------------------------------------------*/
struct service_class_config * xconf_init_service_class()
{
	struct service_class_config * sc_config = DC_MALLOC(DC_MEM_CONF_MANAGER, sizeof(struct service_class_config));
	if (sc_config) {
		sc_config->href = -1;
		DA_INIT(struct reference, &sc_config->refs, DC_MEM_CONF_MANAGER, p_default_allocator, 1);
		DA_INIT(struct property, &sc_config->properties, DC_MEM_CONF_MANAGER, p_default_allocator, 1);
		sc_config->cbks.new_service = NULL;
		sc_config->cbks.free_service = NULL;
		sc_config->cbks.serialize_impl = NULL;
		sc_config->cbks.unload_cbk = NULL;
	}
	return sc_config;
}

int xconf_new_service_class(struct service_class_config * new_sclass, struct qname * impl_id, void * psr_ctx)
{
	int ret = DPWS_OK;
	struct dynload_entry * p_entry;
	dispatch_cbk dispatch_cbk;

	p_entry = get_loader(impl_id);
	if (!p_entry)
		return DPWS_ERR_NO_LOADER_FOUND;
	ret = p_entry->cbk(new_sclass->href, psr_ctx, &dispatch_cbk, &new_sclass->cbks);
	if (!ret) {
		if (!new_sclass->cbks.serialize_impl)
			return DPWS_ERR_NO_SERIALIZE_IMPL_FUNC;
		if (dispatch_cbk)
			dpws_set_ptr_att(new_sclass->href, DPWS_PTR_HANDLING_FUNCTION, (const void*)dispatch_cbk);
		else
			return DPWS_ERR_NO_DISPATCH_FUNC;
		ret = new_config(service_classes, new_sclass);
	}
	return ret;
}

static int sclass_match_cbk(struct config_node * node, short * href_sclass)
{
	return *href_sclass == ((struct service_class_config *)node->config)->href;
}

static struct service_class_config * get_service_class(short href_sclass)
{
	struct config_node * c_node;
	c_node = (struct config_node *)dcl_find(service_classes, (dcl_match_cbk_t)sclass_match_cbk, &href_sclass);
	return c_node ? (struct service_class_config *)c_node->config : NULL;
}

struct service_class_config * xconf_get_service_class(short href_sclass)
{
	struct service_class_config * sc_node;
	dcpl_mutex_lock(xmlconf_lock);
	sc_node = get_service_class(href_sclass);
	dcpl_mutex_unlock(xmlconf_lock);
	return sc_node;
}

static void free_ref_cbk(int mod, const da_allocator_t * p_alloc, struct reference * ref)
{
	p_alloc->free_cbk(mod, p_alloc->param, ref->name);
	prefixed_qname_free_cbk(mod, p_alloc, &ref->type);
}

static void free_prop_cbk(int mod, const da_allocator_t * p_alloc, struct property * prop)
{
	p_alloc->free_cbk(mod, p_alloc->param, prop->default_value);
	free_ref_cbk(mod, p_alloc, &prop->p);
}

DC_BOOL xconf_delete_service_class(short href_sclass)
{
	return delete_config(service_classes, (dcl_match_cbk_t)sclass_match_cbk, &href_sclass, (free_cbk)xconf_free_service_class);
}

void xconf_free_service_class(struct service_class_config * service_class)
{
	if (service_class->cbks.unload_cbk)
		service_class->cbks.unload_cbk(service_class->href);
	dpws_release_handle(service_class->href);
	DA_FREE(&service_class->refs, free_ref_cbk);
	DA_FREE(&service_class->properties, free_prop_cbk);
	DC_FREE(DC_MEM_CONF_MANAGER, service_class);
}

void xconf_delete_service_classes()
{
	delete_config_list(service_classes, (free_cbk)xconf_free_service_class);
}

/*----------------------------------------------------------------------------*\
 *                          Reference retrieval API                           *
\*----------------------------------------------------------------------------*/
static DC_BOOL bref_match(struct ws_binding_ref * bref, char * name) {
	return strcmp(bref->name, name) ? DC_FALSE : DC_TRUE;
}

struct ws_binding_ref * xconf_get_binding_ref(short href_service, const char * name)
{
	struct service_config * sc = xconf_get_service(href_service);
	int i = -1;

	if (sc)
		i = DA_BROWSE(&sc->refs, bref_match, (char *)name);

	return i >= 0 && i < sc->refs.nb ? DA_GET(&sc->refs, i) : NULL;
}

static DC_BOOL binding_match(struct reference * binding, char * name) {
	return strcmp(binding->name, name) ? DC_FALSE : DC_TRUE;
}

struct reference * xconf_get_binding(short href_service, const char * name)
{
	struct service_config * sc = xconf_get_service(href_service);
	int i = -1;

	if (sc)
		i = DA_BROWSE(&sc->class->refs, binding_match, (char *)name);

	return i >= 0 && i < sc->class->refs.nb ? DA_GET(&sc->class->refs, i) : NULL;
}

/*----------------------------------------------------------------------------*\
 *                          Dynamic Loader management                         *
\*----------------------------------------------------------------------------*/
DC_RT_FMAC1 int dpws_register_loader(const struct qname * id, load_cbk cbk)
{
	struct dynload_entry * p_entry;

	DC_CHECK_PARAM(id && cbk);

	p_entry = DA_ADD(&loaders);
	if (!p_entry)
		goto error;
	if ((!(p_entry->platform_id.ns = DC_STRDUP(DC_MEM_CONF_MANAGER, id->ns)) && id->ns)
		|| (!(p_entry->platform_id.lname = DC_STRDUP(DC_MEM_CONF_MANAGER, id->lname)) && id->lname))
		goto error;
	p_entry->cbk = cbk;
	return DPWS_OK;

error:
	return DPWS_ERR_EOM;
}

static DC_BOOL loader_match(struct dynload_entry * p_entry, struct qname * id) {
	return QNAME_EQUALS(p_entry->platform_id.ns, p_entry->platform_id.lname, id->ns, id->lname);
}

static struct dynload_entry * get_loader(struct qname * id) {
	int i = DA_BROWSE(&loaders, loader_match, id);
	return i == loaders.nb ? NULL : DA_GET(&loaders, i);
}

static void free_dynload_entry_cbk(int mod, const da_allocator_t * p_alloc, struct dynload_entry *p_entry)
{
	p_alloc->free_cbk(mod, p_alloc->param, p_entry->platform_id.ns);
	p_alloc->free_cbk(mod, p_alloc->param, p_entry->platform_id.lname);
}

void xconf_delete_loaders()
{
	DA_FREE(&loaders, free_dynload_entry_cbk);
}

/*----------------------------------------------------------------------------*\
 *                            Property access API                             *
\*----------------------------------------------------------------------------*/

DC_BOOL browse_prop_values(struct property_value * p_entry, struct prop_grab_info * g_info)
{
	if (!strcmp(g_info->prop, p_entry->name)) {
		if (g_info->i == g_info->len)
			return DC_TRUE;
		g_info->values[g_info->i] = p_entry->value;
		g_info->i++;
	}
	return DC_FALSE;
}

DC_RT_FMAC1 int dpws_get_property_values(short href_service, char * property, char ** values, int * len)
{
	struct prop_grab_info g_info;
	struct service_config * sc;

	DC_CHECK_PARAM(href_service >= 0 && property && values && len && *len > 0);

	sc = xconf_get_service(href_service);
	g_info.prop = property;
	g_info.values = values;
	g_info.len = *len;
	g_info.i = 0;

	if (!sc)
		return DPWS_ERR_NO_HANDLE_FOUND;

	if (DA_BROWSE(&sc->property_values, browse_prop_values, &g_info) != sc->property_values.nb)
		return DPWS_ERR_MORE_RESULTS;

	*len = g_info.i;

	return DPWS_OK;
}
