/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2007-2009 Odonata www.odonata.fr <<-                |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2213 $
|                     $Date: 2009-03-19 11:46:17 +0100 (jeu, 19 mar 2009) $
\*============================================================================*/
#include "dc/dc_FileX.h"
#include "dc/dc_Sun.h"
#include "dc/dc_Epx.h"
#include "dc/dc_Error.h"
#include "dcCOMN_Tools.h"

/*------------------------- Static Functions prototypes ----------------------*/

static size_t read_stream(struct soap* soap, char* buf, size_t size);
static int write_stream(struct soap* soap, const char* buf, size_t size);
static int pipe_epx_stream(void * pctx, void * sctx);

/*----------------------------------------------------------------------------*/

static size_t read_stream(struct soap* soap, char* buf, size_t size)
{
	sun_stream_read(soap->is, buf, (unsigned long *)&size);
    return size;
}

static int write_stream(struct soap* soap, const char* buf, size_t size)
{
    return sun_stream_write(soap->os, buf, (unsigned long *)&size) == SUN_EOS ? SOAP_EOF : SOAP_OK;
}

int xml_out_fopen(struct soap * soap, void * sun_out_stream)
{
	DC_CHECK_PARAM(soap && sun_out_stream);

    soap_init(soap);
    soap->fsend = write_stream;
   	soap->os = sun_out_stream;
    return SOAP_OK;
}

int xml_out_fclose(struct soap *soap)
{
	int ret;

	DC_CHECK_PARAM(soap);

	ret = sun_close(soap->os);
	soap_end(soap);
	soap_init(soap);
	return ret;
}

int xml_in_fopen(struct soap * soap, void * sun_in_stream)
{
	DC_CHECK_PARAM(soap && sun_in_stream);

    soap_init(soap);
    soap->frecv = read_stream;
    soap->is = sun_in_stream;
    return SOAP_OK;
}

int xml_in_fclose(struct soap *soap)
{
	int ret;

	DC_CHECK_PARAM(soap);

	ret = sun_close(soap->is);
	soap_end(soap);
	soap_init(soap);
	return ret;
}

static int pipe_epx_stream(void * pctx, void * sctx)
{
   int ret = EPX_OK;
   epx_event cev;

    /* Main loop: parsing events are directly transmitted to the serializer */
    for (;;)
    {
        switch(cev = epx_next(pctx))
        {
        case EPX_EVT_ERROR:
            ret = epx_get_parser_error(pctx);
            break;

		case EPX_EVT_IDLE: case EPX_EVT_END_DOCUMENT: case EPX_EVT_END_FRAGMENT:
            return SOAP_OK;	// natural exit

        case EPX_EVT_START_ELEMENT:
            ret = epx_start_element(sctx, epx_get_ns_uri(pctx), epx_get_lname(pctx));
            break;

        case EPX_EVT_PREFIX_DEFINITION:
            ret = epx_define_prefix(sctx, epx_get_ns_prefix(pctx), epx_get_ns_uri(pctx));
            break;

        case EPX_EVT_ATTRIBUTE:
			ret = epx_add_attribute(sctx, epx_get_ns_uri(pctx), epx_get_lname(pctx), epx_get_characters(pctx));
            break;

        case EPX_EVT_END_ELEMENT:
            ret = epx_end_element(sctx, epx_get_ns_uri(pctx), epx_get_lname(pctx));
            break;

        case EPX_EVT_CHARACTERS:
            ret = epx_put_characters(sctx, epx_get_characters(pctx));
            break;

        default:
        	break;

        }
        if (ret)
        	break;
    }
    return ret;
}

int xml_file_serialize(void * pctx, char * filename)
{
    void * sctx, *sun_out_stream;
    int s_options, ret = SOAP_OK;
    struct soap soap_out;

	DC_CHECK_PARAM(pctx);

	/* Uses fileX.h to prepare the soap structure for file writing */
    sun_out_stream = sun_write_init(NULL, filename, NULL);
	if (!sun_out_stream)
		return SOAP_EOF;
    ret = xml_out_fopen(&soap_out, sun_out_stream);
	if (ret)
		return ret;

	/* Serialization options setting */
    s_options = epx_get_serializer_options(NULL);
    s_options |= EPX_OPT_INDENT;	// we know the gSOAP implementation supports it

	/* Serializer creation. Note that the implementation is NULL since only gSOAP is
	 * currently implemented. The second parameter is used by the gSOAP implementation
	 * to avoid internal invokation.
	 */
    sctx = epx_new_serializer(NULL, &soap_out);
    if (!sctx) {
    	ret = SOAP_EOM;
    	goto exit;
    }

    /* Start serialization */
    if (epx_start_document(sctx, NULL, s_options)) {
		ret = epx_get_serializer_error(sctx);
		goto exit;
	}
    ret = pipe_epx_stream(pctx, sctx);
    if (ret)
    	goto exit;

	/* Ends serialization */
    ret = epx_end_document(sctx);

exit:
    xml_out_fclose(&soap_out);
    return ret;
}

int xml_file_parse(void * sctx, char * filename)
{
    void *pctx, *sun_in_stream;
    int p_options, ret = SOAP_OK;
    struct soap soap_in;

	DC_CHECK_PARAM(sctx && filename);

    sun_in_stream = sun_read_init(NULL, filename);
	if (!sun_in_stream)
		return SOAP_EOF;
	ret = xml_in_fopen(&soap_in, sun_in_stream);
	if (ret)
		return ret;

	/* Parsing options setting */
    p_options = epx_get_parser_options(NULL);
    p_options |= EPX_OPT_GENERATE_PREFIX_DEFINITIONS;

    pctx = epx_new_parser(NULL, &soap_in);
    if (!pctx) {
    	ret = SOAP_EOM;
    	goto exit;
    }

    /* Start serialization */
    if (epx_start_parsing(pctx, NULL, p_options)) {
		ret = epx_get_parser_error(pctx);
		goto exit;
	}
    ret = pipe_epx_stream(pctx, sctx);

exit:
    xml_in_fclose(&soap_in);
    return ret;
}

