/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2271 $
|                     $Date: 2009-04-16 19:00:04 +0200 (jeu, 16 avr 2009) $
\*============================================================================*/

/******************************************************************************\
 *                         EPX implementation on gSOAP                        *
\******************************************************************************/
#define EPX_GSOAP

#include <string.h>
#include "dc/dc_Epx.h"
#include "dcCOMN_Tools.h"
#include "dcGSOAP_Runtime.h"

/* Limitations:
 * - Mixed content are processed like simple content elements.
 */

/*----------------------------------- Types ----------------------------------*/

#define SOAP_LT (soap_wchar)(-2) /* XML character '<' */
#define SOAP_TT (soap_wchar)(-3) /* XML character '</' */
#define SOAP_GT (soap_wchar)(-4) /* XML character '>' */
#define SOAP_QT (soap_wchar)(-5) /* XML character '"' */
#define SOAP_AP (soap_wchar)(-6) /* XML character ''' */

#define soap_blank(c)		((c) >= 0 && (c) <= 32)
#define soap_notblank(c)	((c) > 32)

#define TAG_BUFFER_LEN	64

#define DEFAULT_PARSER_OPTIONS	EPX_OPT_COALESCING|EPX_OPT_IGNORE_COMMENTS|EPX_OPT_PARSER_STREAMING|EPX_OPT_GENERATE_PREFIX_DEFINITIONS

#define PARSER_CTX ((struct epx_parser_context *)parser_ctx)
#define SERIALIZER_CTX ((struct epx_serializer_context *)serializer_ctx)

struct epx_parser_context
{
    struct soap * soap;
    epx_event event;
    char * value;	// gSOAP allocation handle
    size_t offset;
    size_t length;
    int cur_att_idx;
    int att_nb;
    struct soap_nlist *cur_pfx_def;
    char generate_pfx;
    epx_boolean_t fragment;
    epx_boolean_t generate_fragment_pfx;
    unsigned int start_level;	// fragment parsing
};


#define DEFAULT_SERIALIZER_OPTIONS	EPX_OPT_GENERATE_PREFIXES|EPX_OPT_SERIALIZER_STREAMING|EPX_OPT_FRAGMENT_PARSING

struct epx_serializer_context
{
    struct soap * soap;
    int cdata;	// boolean flag
};

#define BOOLEAN2S(b) (b ? "true" : "false")


/*------------------------- Static Functions prototypes ----------------------*/

static const char* build_qname(struct epx_serializer_context * serializer_ctx, const char *ns_uri, const char *lname, epx_boolean_t alt_buf);
static int find_new_prefix(struct epx_parser_context * parser_ctx);
static int check_next_prefix(struct epx_parser_context * parser_ctx);
static int check_next_tag(struct epx_parser_context * parser_ctx);
static int element_end_in(struct soap *soap);
static size_t read_value_buffer(void * parser_ctx, char * dest, size_t buf_size);
static char * _get_qname_prefix(struct epx_parser_context * parser_ctx, char * qname);
static struct soap_attribute * get_attribute(struct soap * soap, int i);
static void get_atomic_value(struct epx_parser_context * parser_ctx);
static int flush_start_element(struct epx_serializer_context * szr_ctx, epx_boolean_t * end_element);
static int pututf8(struct soap *soap, register unsigned long c);
static int write_string_out(struct soap *soap, const char *buf, size_t len, int cdata);

/******************************************************************************/
/*                                Parsing API                                 */
/******************************************************************************/

int epx_get_parser_options(void * implementation)
{
    return DEFAULT_PARSER_OPTIONS;
}

int epx_is_parser_option_settable(void * implementation, int option)
{
    return option & (EPX_OPT_UTF8_OUTPUT|EPX_OPT_GENERATE_PREFIX_DEFINITIONS);
}

void * epx_new_parser(void * implementation, void * impl_data)
{
    struct epx_parser_context * ctx = (struct epx_parser_context*)DC_MALLOC(DC_MEM_TRANSIENT, sizeof(struct epx_parser_context));
    if (ctx) {
	    ctx->soap = (struct soap *)impl_data;
	    ctx->event = EPX_EVT_IDLE;
	    ctx->value = NULL;
	    ctx->cur_att_idx = -1;
	    ctx->att_nb = -1;
	    ctx->cur_pfx_def = NULL;
		ctx->fragment = EPX_FALSE;
	    ctx->start_level = 0;
	    epx_set_parsing_options(ctx, DEFAULT_PARSER_OPTIONS);
    }
    return ctx;
}

int epx_start_parsing(void * parser_ctx, void * dummy_source, int options)
{
	DC_CHECK_PARAM_RETURN(parser_ctx, EPX_ERR_INVALID_PARAMETER);

    return epx_set_parsing_options(parser_ctx, options);
}

int epx_set_parsing_options(void * parser_ctx, int options)
{
	DC_CHECK_PARAM_RETURN(parser_ctx, EPX_ERR_INVALID_PARAMETER);

    if (options & EPX_OPT_UTF8_OUTPUT)
        PARSER_CTX->soap->mode |= SOAP_C_UTFSTRING;
    else
        PARSER_CTX->soap->mode &= ~SOAP_C_UTFSTRING;
    PARSER_CTX->soap->mode &= ~SOAP_C_MBSTRING;
    PARSER_CTX->generate_pfx = options & EPX_OPT_GENERATE_PREFIX_DEFINITIONS ? 1 : 0;
    return EPX_OK;
}

int epx_get_parsing_options(void * parser_ctx)
{
	int options = DEFAULT_PARSER_OPTIONS;

	DC_CHECK_PARAM_RETURN(parser_ctx, EPX_ERR_INVALID_PARAMETER);

	if (PARSER_CTX->soap->mode & SOAP_C_UTFSTRING)
		options |= EPX_OPT_UTF8_OUTPUT;	// else is default
	if (!PARSER_CTX->generate_pfx)
		options &= ~EPX_OPT_GENERATE_PREFIX_DEFINITIONS;	// else is default
	return options;
}

int epx_start_fragment_parsing(void * parser_ctx, epx_boolean_t gen_all_pfx_events)
{
	DC_CHECK_PARAM_RETURN(parser_ctx, EPX_ERR_INVALID_PARAMETER);

	PARSER_CTX->start_level = PARSER_CTX->soap->level;
	PARSER_CTX->generate_fragment_pfx = gen_all_pfx_events;
	PARSER_CTX->fragment = EPX_TRUE;
	if (PARSER_CTX->event == EPX_EVT_IDLE)
		PARSER_CTX->event = EPX_EVT_START_DOCUMENT;	// in case of parsing initialized in a classic gSOAP parsing
	return EPX_OK;
}

int epx_reset_fragment_parsing(void * parser_ctx)
{
	DC_CHECK_PARAM_RETURN(parser_ctx, EPX_ERR_INVALID_PARAMETER);

	PARSER_CTX->fragment = EPX_FALSE;
	PARSER_CTX->event = EPX_EVT_END_ELEMENT;
	return EPX_OK;
}

int epx_get_parser_error(void * parser_ctx)
{
	DC_CHECK_PARAM_RETURN(parser_ctx, EPX_ERR_INVALID_PARAMETER);

	return PARSER_CTX->soap->error;
}

void epx_delete_parser(void * parser_ctx)
{
	DC_CHECK_PARAM_RETURN(parser_ctx,);

	// NOTE: Not the EPX role to free implementation data memory that it did not use
	if (PARSER_CTX->value)
		soap_dealloc(PARSER_CTX->soap, PARSER_CTX->value);	// NULL means free all !
	DC_FREE(DC_MEM_TRANSIENT, PARSER_CTX);
}

static const char* build_qname(struct epx_serializer_context * serializer_ctx, const char *ns_uri, const char *lname, epx_boolean_t alt_buf)
{
    const char *t;
    char *buf = alt_buf ? serializer_ctx->soap->tmpbuf : serializer_ctx->soap->tag;
    size_t l, buf_len = alt_buf ? 1024 : SOAP_TAGLEN;
    if (!ns_uri)
        return lname;
    t = soap_check_prefix_definition(serializer_ctx->soap, ns_uri, NULL);
    l = strlen(t);
	if (l > 0) {
		strcpy(buf, t);
		buf[l++] = ':';
	}
    strncpy(buf + l, lname, buf_len - l - 1);
    return buf;
}

/** Moves to the next not overloaded prefix */
static int find_new_prefix(struct epx_parser_context * parser_ctx)
{
	struct soap_nlist * n;
	for (; parser_ctx->cur_pfx_def; parser_ctx->cur_pfx_def = parser_ctx->cur_pfx_def->next) {
		for (n = parser_ctx->soap->nlist; n != parser_ctx->cur_pfx_def && n->id != parser_ctx->cur_pfx_def->id; n = n->next);
		if (n == parser_ctx->cur_pfx_def)	// no previous definition
			return EPX_TRUE;
	}
	return EPX_FALSE;
}

static int check_next_prefix(struct epx_parser_context * parser_ctx)
{
    parser_ctx->cur_pfx_def = (parser_ctx->cur_pfx_def == NULL) ? parser_ctx->soap->nlist : parser_ctx->cur_pfx_def->next;

	if (parser_ctx->cur_pfx_def
			&& (parser_ctx->cur_pfx_def->level == (parser_ctx->soap->level - 1)
				|| (parser_ctx->fragment && parser_ctx->generate_fragment_pfx && (parser_ctx->soap->level - 1) == parser_ctx->start_level && find_new_prefix(parser_ctx)))
		)	// when a fragment starts, one needs prefix definition at upper levels
		return EPX_TRUE;
	else {
       	parser_ctx->cur_pfx_def = NULL;
       	return EPX_FALSE;
	}
}

// returns EPX_TRUE if an event is set
static int check_next_tag(struct epx_parser_context * parser_ctx)
{
	int ret = EPX_FALSE;
    if (!soap_peek_element(parser_ctx->soap))
    {
        parser_ctx->soap->peeked = 0;
        parser_ctx->soap->level++;
        parser_ctx->att_nb = -1;
        parser_ctx->cur_pfx_def = NULL;
        parser_ctx->event = EPX_EVT_START_ELEMENT;
        ret = EPX_TRUE;
    }
    else if (parser_ctx->soap->error == SOAP_NO_TAG)	// end-element taq or characters
    {
    	soap_wchar c;
    	c = soap_get(parser_ctx->soap);
    	if (c == SOAP_LT || c == SOAP_TT)// distinguish end_element
    	{
	    	soap_unget(parser_ctx->soap, c);
            if (parser_ctx->fragment && parser_ctx->soap->level <= parser_ctx->start_level)    // fragment parsing check if no new element
        	    parser_ctx->event = EPX_EVT_END_FRAGMENT;
            else if (parser_ctx->soap->level == 0)
            	parser_ctx->event = EPX_EVT_END_DOCUMENT;
            else
			{
				element_end_in(parser_ctx->soap);
				parser_ctx->event = EPX_EVT_END_ELEMENT;
			}
			parser_ctx->soap->error = SOAP_OK;
    	}
		else
		{
	    	soap_unget(parser_ctx->soap, c);
    	    parser_ctx->event = EPX_EVT_CHARACTERS;
		}
	    ret = EPX_TRUE;
    }
	else if (parser_ctx->soap->error == SOAP_EOF && parser_ctx->soap->level == 0)
       	parser_ctx->event = EPX_EVT_END_DOCUMENT;
    else
	    parser_ctx->event = EPX_EVT_ERROR;
    return ret;
}

// gSOAP runtime feature rewriting (2 remove the pop from soap_element_end_in)
static int element_end_in(struct soap *soap)
{
    register soap_wchar c;
    register char *s;
    register int n = 0;
    if (soap->peeked)
    {
        if (soap->error == SOAP_NO_TAG)
            soap->error = SOAP_OK;
        if (*soap->tag)
            n++;
        soap->peeked = 0;
    }
    do
    {
        while (((c = soap_get(soap)) != SOAP_TT))
        {
            if ((int)c == EOF)
                return soap->error = SOAP_EOF;
            if (c == SOAP_LT)
                n++;
            else if (c == '/')
            {
                c = soap_get(soap);
                if (c == SOAP_GT)
                    n--;
                else
                    soap_unget(soap, c);
            }
        }
    }
    while (n--);

    s = soap->tag;
    while (soap_notblank(c = soap_getutf8(soap)))
        *s++ = (char)c;
    *s = '\0';
    if ((int)c == EOF)
        return soap->error = SOAP_EOF;
    while (soap_blank(c))
        c = soap_get(soap);
    if (c != SOAP_GT)
        return soap->error = SOAP_SYNTAX_ERROR;

    return SOAP_OK;
}

epx_event epx_next(void * parser_ctx)
{
	DC_CHECK_PARAM_RETURN(parser_ctx, EPX_EVT_ERROR);

	// Free gSOAP allocated temporary data
	if (PARSER_CTX->value)
	{
		soap_dealloc(PARSER_CTX->soap, PARSER_CTX->value);	// NULL means free all !
		PARSER_CTX->value = NULL;
	}
	else if (PARSER_CTX->event == EPX_EVT_CHARACTERS)
	{	// skip value
		register soap_wchar c = 0;
		do
			c = soap_get(PARSER_CTX->soap);
		while (c != SOAP_TT && (int)c != EOF);
		soap_unget(PARSER_CTX->soap, c);
	}

    switch (PARSER_CTX->event)
    {
    case EPX_EVT_IDLE:
        if (!soap_peek_element(PARSER_CTX->soap))
            PARSER_CTX->event = EPX_EVT_START_DOCUMENT;
        else
            PARSER_CTX->event = EPX_EVT_ERROR;
        break;

    case EPX_EVT_START_DOCUMENT:
    case EPX_EVT_CHARACTERS:
        check_next_tag(PARSER_CTX);
        break;

    case EPX_EVT_END_DOCUMENT:
    case EPX_EVT_END_FRAGMENT:
        PARSER_CTX->event = EPX_EVT_IDLE;
        break;

    case EPX_EVT_PREFIX_DEFINITION:
        if (!check_next_prefix(PARSER_CTX))
        {
            if (epx_get_att_nb(parser_ctx) > 0)
            	PARSER_CTX->event = EPX_EVT_ATTRIBUTE;
            else {
            	if (PARSER_CTX->soap->body)
            		check_next_tag(PARSER_CTX);
            	else
            		PARSER_CTX->event = EPX_EVT_END_ELEMENT;
            }
        }
        break;

    case EPX_EVT_ATTRIBUTE:
        if (++PARSER_CTX->cur_att_idx == PARSER_CTX->att_nb)
        {
        	if (PARSER_CTX->soap->body)
        		check_next_tag(PARSER_CTX);
        	else
        		PARSER_CTX->event = EPX_EVT_END_ELEMENT;
        }
        break;

    case EPX_EVT_END_ELEMENT:
        PARSER_CTX->soap->level--;
        soap_pop_namespace(PARSER_CTX->soap);
    	check_next_tag(PARSER_CTX);
        break;

    case EPX_EVT_START_ELEMENT:
        // check is there are some prefix definition
        if (PARSER_CTX->generate_pfx && check_next_prefix((struct epx_parser_context*)parser_ctx))
            PARSER_CTX->event = EPX_EVT_PREFIX_DEFINITION;
        else if (epx_get_att_nb(parser_ctx) > 0)
        	PARSER_CTX->event = EPX_EVT_ATTRIBUTE;
        else {
        	if (PARSER_CTX->soap->body)
        		check_next_tag(PARSER_CTX);
        	else
        		PARSER_CTX->event = EPX_EVT_END_ELEMENT;
        }

        break;

    default:
        PARSER_CTX->event = EPX_EVT_ERROR;
    }

    return PARSER_CTX->event;
}

epx_event epx_get_event(void * parser_ctx)
{
	DC_CHECK_PARAM_RETURN(parser_ctx, EPX_EVT_ERROR);

	return PARSER_CTX->event;
}

char * epx_get_characters(void * parser_ctx)
{
	DC_CHECK_PARAM_RETURN(parser_ctx, NULL);

	switch (PARSER_CTX->event)
    {
    case EPX_EVT_CHARACTERS:
    	if (!PARSER_CTX->value) {
			PARSER_CTX->value = soap_string_in(PARSER_CTX->soap, 1, -1, -1);
			PARSER_CTX->offset = 0;
			PARSER_CTX->length = strlen(PARSER_CTX->value);
    	}
		return PARSER_CTX->value;
    case EPX_EVT_ATTRIBUTE:
    	return epx_get_att_value(parser_ctx, PARSER_CTX->cur_att_idx);
    default:
		return NULL;
    }
}

static size_t read_value_buffer(void * parser_ctx, char * dest, size_t buf_size)
{
    size_t copy_size, diff = PARSER_CTX->length - PARSER_CTX->offset;
    copy_size = buf_size < diff ? buf_size : diff;
    memcpy(dest, PARSER_CTX->value + PARSER_CTX->offset, copy_size);
    PARSER_CTX->offset += copy_size;
    return copy_size;
}

size_t epx_read_characters(void * parser_ctx, char * dest, size_t buf_size)
{
	DC_CHECK_PARAM_RETURN(parser_ctx && dest && buf_size > 0, EPX_EVT_ERROR);

	if (PARSER_CTX->event != EPX_EVT_CHARACTERS || !epx_get_characters(parser_ctx))
        return EPX_ERR_NOT_SUPPORTED;
    return read_value_buffer(parser_ctx, dest, buf_size);
}

// missing required feature written using gSOAP runtime data
static char * _get_qname_prefix(struct epx_parser_context * parser_ctx, char * qname)
{
    char * colon = strchr(qname, ':'), *pfx = NULL;
    // search is preferred to copy to avoid allocation
    if (colon)
    {
    	size_t p_len = colon - qname;
	    if (!strncmp(qname, XML_NS_PREFIX, p_len))
	        pfx = XML_NS_PREFIX;
	    else if (!strncmp(qname, XMLNS_NS_PREFIX, p_len))
	        pfx = XMLNS_NS_PREFIX;
	    else
	    {
	        struct soap_nlist *np = parser_ctx->soap->nlist;

	        while (np && (strncmp(np->id, qname, p_len) || np->id[p_len]))
	            np = np->next;
	        if (np)
	        	pfx = np->id;
	    }
    }
    return pfx;
}

char * epx_get_ns_prefix(void * parser_ctx)
{
	DC_CHECK_PARAM_RETURN(parser_ctx, NULL);

    switch (PARSER_CTX->event)
    {
    case EPX_EVT_START_ELEMENT:
    case EPX_EVT_END_ELEMENT:
        return _get_qname_prefix(PARSER_CTX, PARSER_CTX->soap->tag);
    case EPX_EVT_ATTRIBUTE:
    	return epx_get_att_ns_prefix(parser_ctx, PARSER_CTX->cur_att_idx);
    case EPX_EVT_PREFIX_DEFINITION:
        return PARSER_CTX->cur_pfx_def->id;
    default:
        return NULL;
    }
}

char * epx_get_ns_uri(void * parser_ctx)
{
	DC_CHECK_PARAM_RETURN(parser_ctx, NULL);

	switch (PARSER_CTX->event)
    {
    case EPX_EVT_START_ELEMENT:
    case EPX_EVT_END_ELEMENT:
        return soap_get_ns_uri(PARSER_CTX->soap, PARSER_CTX->soap->tag);
    case EPX_EVT_ATTRIBUTE:
        return epx_get_att_ns_uri(parser_ctx, PARSER_CTX->cur_att_idx);
    case EPX_EVT_PREFIX_DEFINITION:
        if (PARSER_CTX->cur_pfx_def->index >= 0)
        {
            struct Namespace *ns = &PARSER_CTX->soap->local_namespaces[PARSER_CTX->cur_pfx_def->index];
            return (char *)(ns->out ? ns->out : ns->ns);
        }
        else
            return PARSER_CTX->cur_pfx_def->ns;
    default:
        return NULL;
    }
}

char * epx_get_lname(void * parser_ctx)
{
	DC_CHECK_PARAM_RETURN(parser_ctx, NULL);

	switch (PARSER_CTX->event)
    {
    case EPX_EVT_START_ELEMENT:
    case EPX_EVT_END_ELEMENT:
        return soap_get_lname(PARSER_CTX->soap, PARSER_CTX->soap->tag);
    case EPX_EVT_ATTRIBUTE:
        return epx_get_att_lname(parser_ctx, PARSER_CTX->cur_att_idx);
    default:
        return NULL;
    }
}

int epx_get_att_nb(void * parser_ctx)
{
    struct soap_attribute *tp;

    DC_CHECK_PARAM_RETURN(parser_ctx, EPX_ERR_INVALID_PARAMETER);

	if (PARSER_CTX->att_nb < 0)
    {
        PARSER_CTX->cur_att_idx = 0;
        PARSER_CTX->att_nb = 0;
        for (tp = PARSER_CTX->soap->attributes; tp; tp = tp->next)
        {
            if (tp->visible)
                PARSER_CTX->att_nb++;
        }
    }
    return PARSER_CTX->att_nb;
}

// missing required feature written using gSOAP runtime data
static struct soap_attribute * get_attribute(struct soap * soap, int i)
{
    struct soap_attribute *tp;
    int count = 0;

    for (tp = soap->attributes; tp; tp = tp->next)
    {
		if (tp->visible) {
			if (count == i)
				return tp;
            count++;
		}
    }
    return NULL;
}

char * epx_get_att_ns_uri(void * parser_ctx, int index)
{
    struct soap_attribute * att;

    DC_CHECK_PARAM_RETURN(parser_ctx && index >= 0, NULL);

    att = get_attribute(PARSER_CTX->soap, index);
	if (att)
	{
		char * colon = (char*) strchr(att->name, ':');
		return colon ? soap_get_namespace_uri(PARSER_CTX->soap, att->name, colon - att->name) : NULL;
	}
	else
		return NULL;
}

char * epx_get_att_ns_prefix(void * parser_ctx, int index)
{
    struct soap_attribute * att;

    DC_CHECK_PARAM_RETURN(parser_ctx && index >= 0, NULL);

	att = get_attribute(PARSER_CTX->soap, index);
    return att ? _get_qname_prefix(PARSER_CTX, att->name) : NULL;
}

char * epx_get_att_lname(void * parser_ctx, int index)
{
    struct soap_attribute * att;

    DC_CHECK_PARAM_RETURN(parser_ctx && index >= 0, NULL);

    att = get_attribute(PARSER_CTX->soap, index);
    return att ? soap_get_lname(PARSER_CTX->soap, att->name) : NULL;
}

char * epx_get_att_value(void * parser_ctx, int index)
{
    struct soap_attribute * att;

    DC_CHECK_PARAM_RETURN(parser_ctx && index >= 0, NULL);

    att = get_attribute(PARSER_CTX->soap, index);
    return att ? att->value : NULL;
}

char * epx_get_prefix_uri(void * parser_ctx, char * prefix)
{
    DC_CHECK_PARAM_RETURN(parser_ctx && prefix, NULL);
    return soap_get_namespace_uri(PARSER_CTX->soap, prefix, 0);
}

char * epx_get_uri_prefix(void * parser_ctx, char * uri)
{
    DC_CHECK_PARAM_RETURN(parser_ctx && uri, NULL);
	return (char *)soap_check_prefix_definition(PARSER_CTX->soap, uri, NULL);
}

/*----------------------------------------------------------------------------*\
 *                             Typed parsing API                              *
\*----------------------------------------------------------------------------*/

unsigned char * epx_get_base64_binary(void * parser_ctx, size_t * length)
{
	int n;

    DC_CHECK_PARAM_RETURN(parser_ctx && length, NULL);

	if (!PARSER_CTX->value)
	{
	    switch (PARSER_CTX->event)
	    {
	    case EPX_EVT_CHARACTERS:
			PARSER_CTX->value = (char*)soap_getbase64(PARSER_CTX->soap, &n, 1);
			PARSER_CTX->offset = 0;
			break;
	    case EPX_EVT_ATTRIBUTE:
			PARSER_CTX->value = (char *)soap_base642s(PARSER_CTX->soap, epx_get_att_value(parser_ctx, PARSER_CTX->cur_att_idx), NULL, -1, &n);
			break;
	    default:
			return NULL;
	    }
	    *length = PARSER_CTX->length = (size_t)n;
	}
	return (unsigned char*) PARSER_CTX->value;
}

size_t epx_read_base64_binary(void * parser_ctx, unsigned char * dest, size_t length)
{
    size_t l;

    DC_CHECK_PARAM_RETURN(parser_ctx && dest && length > 0, 0);

    if (PARSER_CTX->event != EPX_EVT_CHARACTERS || !epx_get_base64_binary(parser_ctx, &l))
        return EPX_ERR_NOT_SUPPORTED;
    return read_value_buffer(parser_ctx, (char*)dest, length);
}

unsigned char * epx_get_hex_binary(void * parser_ctx, size_t * length)
{
	int n;

    DC_CHECK_PARAM_RETURN(parser_ctx && length, NULL);

	if (!PARSER_CTX->value)
	{
	    switch (PARSER_CTX->event)
	    {
	    case EPX_EVT_CHARACTERS:
			PARSER_CTX->value = (char*)soap_gethex(PARSER_CTX->soap, &n);
			PARSER_CTX->offset = 0;
			break;
	    case EPX_EVT_ATTRIBUTE:
			PARSER_CTX->value = (char *)soap_hex2s(PARSER_CTX->soap, epx_get_att_value(parser_ctx, PARSER_CTX->cur_att_idx), NULL, -1, &n);
			break;
	    default:
			return NULL;
	    }
	    *length = PARSER_CTX->length = (size_t)n;
	}
	return (unsigned char*)PARSER_CTX->value;
}

size_t epx_read_hex_binary(void * parser_ctx, unsigned char * dest, size_t length)
{
    size_t l;

    DC_CHECK_PARAM_RETURN(parser_ctx && dest && length > 0, 0);

    if (PARSER_CTX->event != EPX_EVT_CHARACTERS || !epx_get_hex_binary(parser_ctx, &l))
        return EPX_ERR_NOT_SUPPORTED;
    return read_value_buffer(parser_ctx, (char*)dest, length);
}

static void get_atomic_value(struct epx_parser_context * parser_ctx)
{
	if (!parser_ctx->value)
	{
	    switch (parser_ctx->event)
	    {
	    case EPX_EVT_CHARACTERS:
	    	parser_ctx->value = (char *)soap_value(parser_ctx->soap);
			break;
	    case EPX_EVT_ATTRIBUTE:
	    	parser_ctx->value = epx_get_att_value(parser_ctx, parser_ctx->cur_att_idx);
			break;
	    default:
	    	break;
	    }
	}
}

int epx_get_boolean(void * parser_ctx, epx_boolean_t * b)
{
    DC_CHECK_PARAM_RETURN(parser_ctx && b, EPX_ERR_INVALID_PARAMETER);

    get_atomic_value(PARSER_CTX);
	if (!PARSER_CTX->value)
		return EPX_ERR_INVALID_REPRESENTATION;
	else if (!strncmp(PARSER_CTX->value, "true", 4) || !strncmp(PARSER_CTX->value, "1", 1))
		*b = EPX_TRUE;
	else if (!strncmp(PARSER_CTX->value, "false", 5) || !strncmp(PARSER_CTX->value, "0", 1))
		*b = EPX_FALSE;

	return EPX_OK;
}

int epx_get_long(void * parser_ctx, long * l)
{
    DC_CHECK_PARAM_RETURN(parser_ctx && l, EPX_ERR_INVALID_PARAMETER);

    get_atomic_value(PARSER_CTX);
	if (soap_s2long(PARSER_CTX->soap, PARSER_CTX->value, l))
		return EPX_ERR_INVALID_REPRESENTATION;
	return EPX_OK;
}

int epx_get_ulong(void * parser_ctx, unsigned long * ul)
{
    DC_CHECK_PARAM_RETURN(parser_ctx && ul, EPX_ERR_INVALID_PARAMETER);

    get_atomic_value(PARSER_CTX);
	if (soap_s2unsignedLong(PARSER_CTX->soap, PARSER_CTX->value, ul))
		return EPX_ERR_INVALID_REPRESENTATION;
	return EPX_OK;
}

int epx_get_float(void * parser_ctx, float * f)
{
    DC_CHECK_PARAM_RETURN(parser_ctx && f, EPX_ERR_INVALID_PARAMETER);

    get_atomic_value(PARSER_CTX);
	if (soap_s2float(PARSER_CTX->soap, PARSER_CTX->value, f))
		return EPX_ERR_INVALID_REPRESENTATION;
	return EPX_OK;
}

int epx_get_double(void * parser_ctx, double * d)
{
    DC_CHECK_PARAM_RETURN(parser_ctx && d, EPX_ERR_INVALID_PARAMETER);

	get_atomic_value(PARSER_CTX);
	if (soap_s2double(PARSER_CTX->soap, PARSER_CTX->value, d))
		return EPX_ERR_INVALID_REPRESENTATION;
	return EPX_OK;
}

int epx_get_qname(void * parser_ctx, epx_qname_t * qn)
{
    DC_CHECK_PARAM_RETURN(parser_ctx && qn, EPX_ERR_INVALID_PARAMETER);

    get_atomic_value(PARSER_CTX);
	qn->ns = soap_get_ns_uri(PARSER_CTX->soap, PARSER_CTX->value);
	qn->lname = soap_get_lname(PARSER_CTX->soap, PARSER_CTX->value);
	return EPX_OK;
}

/******************************************************************************/
/*                             Serialization API                              */
/******************************************************************************/

int epx_get_serializer_options(void * implementation)
{
    return DEFAULT_SERIALIZER_OPTIONS;
}

int epx_is_serializer_option_settable(void * implementation, int option)
{
    return option & (EPX_OPT_UTF8_INPUT|EPX_OPT_INDENT);
}

void * epx_new_serializer(void * implementation, void * impl_data)
{
    struct epx_serializer_context * ctx = (struct epx_serializer_context*) DC_MALLOC(DC_MEM_TRANSIENT, sizeof(struct epx_serializer_context));
    if (ctx) {
	    ctx->soap = (struct soap *)impl_data;
	    ctx->cdata = 0;
	    ctx->soap->type[0] = 0;
	    //	soap_begin(soap);
	    //	soap_begin_send(soap);
    }
    return ctx;
}

int epx_get_serializer_error(void * serializer_ctx)
{
    DC_CHECK_PARAM_RETURN(serializer_ctx, EPX_ERR_INVALID_PARAMETER);

	return SERIALIZER_CTX->soap->error;
}

void epx_delete_serializer(void * serializer_ctx)
{
    DC_CHECK_PARAM_RETURN(serializer_ctx,);

	// NOTE: Not the EPX role to free implementation data memory that it did not use
	DC_FREE(DC_MEM_TRANSIENT, SERIALIZER_CTX);
}

int epx_start_document(void * serializer_ctx, void * dummy_dest, int options)
{
    DC_CHECK_PARAM_RETURN(serializer_ctx, EPX_ERR_INVALID_PARAMETER);

	if (options & EPX_OPT_UTF8_INPUT)
        SERIALIZER_CTX->soap->mode |= SOAP_C_UTFSTRING;
    else
        SERIALIZER_CTX->soap->mode &= ~SOAP_C_UTFSTRING;
    SERIALIZER_CTX->soap->mode &= ~SOAP_C_MBSTRING;
    if (options & EPX_OPT_INDENT)
        SERIALIZER_CTX->soap->mode |= SOAP_XML_INDENT;
    else
        SERIALIZER_CTX->soap->mode &= ~SOAP_XML_INDENT;
    return EPX_OK;
}

static int flush_start_element(struct epx_serializer_context * szr_ctx, epx_boolean_t * end_element)
{
	int ret = SOAP_OK;
	if (szr_ctx->soap->type[0]) {
		const char * tag = build_qname(szr_ctx, szr_ctx->soap->msgbuf[0] ? szr_ctx->soap->msgbuf : NULL, szr_ctx->soap->type, EPX_FALSE);	// NOTE: misappropriation of buffers
		if (!(ret = soap_element(szr_ctx->soap, tag, -1, NULL))) {
			if (end_element) {
				ret = soap_element_start_end_out(szr_ctx->soap, tag);	// empty tag managed
				*end_element = EPX_TRUE;
				szr_ctx->soap->body = 0;
			}
			else
				ret = soap_element_start_end_out(szr_ctx->soap, NULL);
		}
		szr_ctx->soap->type[0] = 0;	// watchguard
	}
	return ret;
}

int epx_end_document(void * serializer_ctx)
{
    DC_CHECK_PARAM_RETURN(serializer_ctx, EPX_ERR_INVALID_PARAMETER);

    //	soap_end_send(SERIALIZER_CTX->soap);
    //	soap_end(SERIALIZER_CTX->soap);
    return EPX_OK;
}

int epx_start_element(void * serializer_ctx, char * ns_uri, char * local_name)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx && local_name, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    if (!ret)
    {
		if (ns_uri)
    		strncpy(SERIALIZER_CTX->soap->msgbuf, ns_uri, 1023);
		else
			SERIALIZER_CTX->soap->msgbuf[0] = 0;
    	strncpy(SERIALIZER_CTX->soap->type, local_name, TAG_BUFFER_LEN - 1);
    }
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_define_prefix(void * serializer_ctx, char * prefix, char * ns_uri)
{
    struct soap * soap = SERIALIZER_CTX->soap;

    DC_CHECK_PARAM_RETURN(serializer_ctx && ns_uri, EPX_ERR_INVALID_PARAMETER);

    strcpy(soap->tmpbuf, "xmlns");
    if (ns_uri)
    {
        strcat(soap->tmpbuf, ":");
        strcat(soap->tmpbuf, prefix);
    }
    soap_set_attr(soap, soap->tmpbuf, ns_uri);
    soap_push_namespace(soap, prefix, ns_uri);	// should have used soap_push_ns instead if was not static
    return soap->error ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_add_attribute(void * serializer_ctx, char * ns_uri, char * local_name, char * value)
{
    DC_CHECK_PARAM_RETURN(serializer_ctx && local_name, EPX_ERR_INVALID_PARAMETER);

    return value
    ? (soap_set_attr(SERIALIZER_CTX->soap, build_qname(SERIALIZER_CTX, ns_uri, local_name, EPX_FALSE), value)
    		? EPX_ERR_IMPLEMENTATION : EPX_OK)
    : EPX_ERR_INVALID_PARAMETER;
}

int epx_end_element(void * serializer_ctx, char * ns_uri, char * local_name)
{
	epx_boolean_t b = EPX_FALSE;
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx && local_name, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, &b);
    if (!ret) {
    	if (!b)
    		ret = soap_element_end_out(SERIALIZER_CTX->soap, build_qname(SERIALIZER_CTX, ns_uri, local_name, EPX_FALSE));
    	soap_pop_namespace(SERIALIZER_CTX->soap);
    }
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

// gSOAP runtime feature rewriting (UTF8 was encoded as a character entity except if canonical XML...)
static int pututf8(struct soap *soap, register unsigned long c)
{
    char tmp[16];
    if (c > 0 && c < 0x80)
    {
        *tmp = (char)c;
        return soap_send_raw(soap, tmp, 1);
    }
    else
    {
        register char *t = tmp;
        if (c < 0x0800)
            *t++ = (char)(0xC0 | ((c >> 6) & 0x1F));
        else
        {
            if (c < 0x010000)
                *t++ = (char)(0xE0 | ((c >> 12) & 0x0F));
            else
            {
                if (c < 0x200000)
                    *t++ = (char)(0xF0 | ((c >> 18) & 0x07));
                else
                {
                    if (c < 0x04000000)
                        *t++ = (char)(0xF8 | ((c >> 24) & 0x03));
                    else
                    {
                        *t++ = (char)(0xFC | ((c >> 30) & 0x01));
                        *t++ = (char)(0x80 | ((c >> 24) & 0x3F));
                    }
                    *t++ = (char)(0x80 | ((c >> 18) & 0x3F));
                }
                *t++ = (char)(0x80 | ((c >> 12) & 0x3F));
            }
            *t++ = (char)(0x80 | ((c >> 6) & 0x3F));
        }
        *t++ = (char)(0x80 | (c & 0x3F));
        *t = '\0';
    }
    return soap_send(soap, tmp);
}

// gSOAP runtime feature rewriting (to be able to work on a buffer and not a string, plus strange entity use...)
static int write_string_out(struct soap *soap, const char *buf, size_t len, int cdata)
{
    register const char *t, *s = buf;
    register soap_wchar c;
    register soap_wchar mask = 0xFFFFFF80UL;

    if (soap->mode & SOAP_C_UTFSTRING)
        mask = 0;
    t = s;
    while ((c = *t++) && (size_t)(t - buf) <= len)
    {
        switch (c)
        {
            // NOTE : Canonical XML processing removed from soap_string_out
            // eol & quote escaping has been removed (what's the use in an element ?)
        	// but gt escaping is kept for symmetry's sake
        case '&':
            if (!cdata)
            {
                if (soap_send_raw(soap, s, t - s - 1) || soap_send_raw(soap, "&amp;", 5))
                    return soap->error;
                s = t;
            }
            break;
        case '<':
            if (!cdata)
            {
                if (soap_send_raw(soap, s, t - s - 1) || soap_send_raw(soap, "&lt;", 4))
                    return soap->error;
                s = t;
            }
            break;
        case '>':
            if (!cdata)
            {
                if (soap_send_raw(soap, s, t - s - 1) || soap_send_raw(soap, "&gt;", 4))
                    return soap->error;
                s = t;
            }
            break;
        default:
            if (c & mask)
            {
                if (soap_send_raw(soap, s, t - s - 1) || pututf8(soap, (unsigned char)c))
                    return soap->error;
                s = t;
            }
        }
    }
    return soap_send_raw(soap, s, t - s - 1);
}

int epx_put_characters(void * serializer_ctx, char* value)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx && value, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    if (!ret && value)
    	ret = write_string_out(SERIALIZER_CTX->soap, value, -1, SERIALIZER_CTX->cdata);
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_write_characters(void * serializer_ctx, char* buffer, size_t length)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx && buffer && length > 0, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    if (!ret)
    	ret = write_string_out(SERIALIZER_CTX->soap, buffer, length, SERIALIZER_CTX->cdata);
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_start_cdata(void * serializer_ctx)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    SERIALIZER_CTX->cdata = 1;
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_end_cdata(void * serializer_ctx)
{
    DC_CHECK_PARAM_RETURN(serializer_ctx, EPX_ERR_INVALID_PARAMETER);

    SERIALIZER_CTX->cdata = 0;
    return EPX_OK;
}

int epx_comment(void * serializer_ctx, char * value)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx && value, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_ERR_NOT_SUPPORTED;
}

/*----------------------------------------------------------------------------*\
 *                          Typed serialization API                           *
\*----------------------------------------------------------------------------*/

int epx_write_base64_binary(void * serializer_ctx, unsigned char * buffer, size_t length)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx && buffer && length > 0, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    if (!ret)
    	ret = soap_putbase64(SERIALIZER_CTX->soap, buffer, (int)length) ? EPX_ERR_IMPLEMENTATION : EPX_OK;
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_write_hex_binary(void * serializer_ctx, unsigned char * buffer, size_t length)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx && buffer && length > 0, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    if (!ret)
    	ret = soap_puthex(SERIALIZER_CTX->soap, buffer, (int)length) ? EPX_ERR_IMPLEMENTATION : EPX_OK;
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_put_boolean(void * serializer_ctx, epx_boolean_t b)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    if (!ret)
    	ret = soap_string_out(SERIALIZER_CTX->soap, BOOLEAN2S(b), 0);

    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_put_long(void * serializer_ctx, long l)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    if (!ret)
    	ret = soap_string_out(SERIALIZER_CTX->soap, soap_long2s(SERIALIZER_CTX->soap, l), 0);
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_put_ulong(void * serializer_ctx, unsigned long ul)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    if (!ret)
    	ret = soap_string_out(SERIALIZER_CTX->soap, soap_unsignedLong2s(SERIALIZER_CTX->soap, ul), 0);
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_put_float(void * serializer_ctx, float f)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    if (!ret)
    	ret = soap_string_out(SERIALIZER_CTX->soap, soap_float2s(SERIALIZER_CTX->soap, f), 0);
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_put_double(void * serializer_ctx, double d)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    if (!ret)
    	ret = soap_string_out(SERIALIZER_CTX->soap, soap_double2s(SERIALIZER_CTX->soap, d), 0);
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_put_qname(void * serializer_ctx, epx_qname_t * qn)
{
    int ret;

    DC_CHECK_PARAM_RETURN(serializer_ctx && qn, EPX_ERR_INVALID_PARAMETER);

    ret = flush_start_element(SERIALIZER_CTX, NULL);
    if (!ret)
    	ret = soap_string_out(SERIALIZER_CTX->soap, build_qname(SERIALIZER_CTX, qn->ns, qn->lname, EPX_FALSE), 0);
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_add_base64_attribute(void * serializer_ctx, char * ns_uri, char * local_name, unsigned char * buffer, size_t length)
{
	int ret = SOAP_OK;
	char * value;

	DC_CHECK_PARAM_RETURN(serializer_ctx && local_name && buffer && length > 0, EPX_ERR_INVALID_PARAMETER);

	value = soap_s2base64(SERIALIZER_CTX->soap, buffer, NULL, (int)length);
	if (value)
		ret = soap_set_attr(SERIALIZER_CTX->soap, build_qname(SERIALIZER_CTX, ns_uri, local_name, EPX_FALSE), value);
	if (!ret)
		soap_dealloc(SERIALIZER_CTX->soap, value);
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_add_hex_attribute(void * serializer_ctx, char * ns_uri, char * local_name, unsigned char * buffer, size_t length)
{
	int ret = SOAP_OK;
	char * value;

	DC_CHECK_PARAM_RETURN(serializer_ctx && local_name && buffer && length > 0, EPX_ERR_INVALID_PARAMETER);

	value = (char*)soap_s2hex(SERIALIZER_CTX->soap, buffer, NULL, (int)length);
	if (value)
		ret = soap_set_attr(SERIALIZER_CTX->soap, build_qname(SERIALIZER_CTX, ns_uri, local_name, EPX_FALSE), value);
	if (!ret)
		soap_dealloc(SERIALIZER_CTX->soap, value);
    return ret ? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_add_boolean_attribute(void * serializer_ctx, char * ns_uri, char * local_name, epx_boolean_t b)
{
	DC_CHECK_PARAM_RETURN(serializer_ctx && local_name, EPX_ERR_INVALID_PARAMETER);

	return soap_set_attr(SERIALIZER_CTX->soap,
			build_qname(SERIALIZER_CTX, ns_uri, local_name, EPX_FALSE), BOOLEAN2S(b))
			? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_add_long_attribute(void * serializer_ctx, char * ns_uri, char * local_name, long l)
{
	DC_CHECK_PARAM_RETURN(serializer_ctx && local_name, EPX_ERR_INVALID_PARAMETER);

	return soap_set_attr(SERIALIZER_CTX->soap,
			build_qname(SERIALIZER_CTX, ns_uri, local_name, EPX_FALSE), soap_long2s(SERIALIZER_CTX->soap, l))
			? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_add_ulong_attribute(void * serializer_ctx, char * ns_uri, char * local_name, unsigned long ul)
{
	DC_CHECK_PARAM_RETURN(serializer_ctx && local_name, EPX_ERR_INVALID_PARAMETER);

	return soap_set_attr(SERIALIZER_CTX->soap,
			build_qname(SERIALIZER_CTX, ns_uri, local_name, EPX_FALSE), soap_unsignedLong2s(SERIALIZER_CTX->soap, ul))
			? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_add_float_attribute(void * serializer_ctx, char * ns_uri, char * local_name, float f)
{
	DC_CHECK_PARAM_RETURN(serializer_ctx && local_name, EPX_ERR_INVALID_PARAMETER);

	return soap_set_attr(SERIALIZER_CTX->soap,
			build_qname(SERIALIZER_CTX, ns_uri, local_name, EPX_FALSE), soap_float2s(SERIALIZER_CTX->soap, f))
			? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_add_double_attribute(void * serializer_ctx, char * ns_uri, char * local_name, double d)
{
	DC_CHECK_PARAM_RETURN(serializer_ctx && local_name, EPX_ERR_INVALID_PARAMETER);

	return soap_set_attr(SERIALIZER_CTX->soap,
			build_qname(SERIALIZER_CTX, ns_uri, local_name, EPX_FALSE), soap_double2s(SERIALIZER_CTX->soap, d))
			? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

int epx_add_qname_attribute(void * serializer_ctx, char * ns_uri, char * local_name, epx_qname_t * qn)
{
	DC_CHECK_PARAM_RETURN(serializer_ctx && local_name && qn, EPX_ERR_INVALID_PARAMETER);
	return soap_set_attr(SERIALIZER_CTX->soap,
			build_qname(SERIALIZER_CTX, ns_uri, local_name, EPX_FALSE),
			build_qname(SERIALIZER_CTX, qn->ns, qn->lname, EPX_TRUE))
			? EPX_ERR_IMPLEMENTATION : EPX_OK;
}

