/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2007-2009 Odonata www.odonata.fr <<-                |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2091 $
|                     $Date: 2009-02-17 15:38:44 +0100 (mar, 17 fév 2009) $
\*============================================================================*/
/******************************************************************************\
 *                      Very limited schema guided parsing                    *
\******************************************************************************/
#include <string.h>

#include "dcXTOOL_SchemaParsing.h"


/*------------------------- Static Functions prototypes ----------------------*/

static int parse_sequence(void * psr_ctx, struct elt_step * complex_type, int nb_steps);
static int process_att_events(void * psr_ctx, struct att_step * atts, int nb_att_steps);
static int parse_simple_content(void * psr_ctx, struct att_step * atts, int nb_att_steps, value_cbk cbk, void * user_data);

/*----------------------------------------------------------------------------*\
 *                                   Parsing                                  *
\*----------------------------------------------------------------------------*/

static int parse_sequence(void * psr_ctx, struct elt_step * complex_type, int nb_steps)
{
	xs_model_group prev_mg = XS_MG_SEQUENCE;
	unsigned int cur_elt_count = 0;
	int prev_mg_nb, ret, i = 0;
	char * cur_ns_uri = NULL, * cur_lname = NULL;
	epx_event evt = epx_get_event(psr_ctx);	// elt event has been consumed by attribute event processing

	for (;;)
	{
		// Next element event
		switch (evt)
		{
			case EPX_EVT_START_ELEMENT:
			if (i == nb_steps)
				return SGXP_ERR_UNEXPECTED_PARTICLE;
			break; // process elt
			case EPX_EVT_END_ELEMENT:
			// check no required element
			if (i == nb_steps)
				goto exit;
			else if (cur_elt_count < complex_type[i].min_occurs)
				return SGXP_ERR_MISSING_EXPECTED_ELT;
			for (i++ ;i < nb_steps && complex_type[i].min_occurs == 0; i++);
			if (i == nb_steps)
				goto exit;
			else
				return SGXP_ERR_MISSING_PARTICLE;
			case EPX_EVT_ERROR:
				return epx_get_parser_error(psr_ctx);
			default:	// ignore other elements
				evt = epx_next(psr_ctx);
				continue;
		}

		cur_ns_uri = epx_get_ns_uri(psr_ctx);
		cur_lname = epx_get_lname(psr_ctx);
		prev_mg = complex_type[i].mg;
		prev_mg_nb = complex_type[i].mg_nb;

		// Analyse if element matches content model
		if (complex_type[i].lname && QNAME_NOT_EQUALS_WILDCARD(cur_ns_uri, cur_lname, complex_type[i].ns_uri, complex_type[i].lname))
		{
			do {
				// check that the minimum number required were found for the previous step
				if (cur_elt_count < complex_type[i].min_occurs)
					return SGXP_ERR_MISSING_EXPECTED_ELT;

				// shift to the next matching elt entry
				i++;
				cur_elt_count = 0;
				if (i == nb_steps)
					return SGXP_ERR_UNEXPECTED_PARTICLE;

				// check that an ended choice has been completed
				if (prev_mg == XS_MG_CHOICE && (complex_type[i].mg != XS_MG_CHOICE || complex_type[i].mg_nb != prev_mg_nb))
					return SGXP_ERR_MISSING_PARTICLE;
			}
			while (QNAME_NOT_EQUALS_WILDCARD(cur_ns_uri, cur_lname, complex_type[i].ns_uri, complex_type[i].lname));
		}

		// verify too numerous elements
		cur_elt_count++;
		if (cur_elt_count > complex_type[i].max_occurs)
			return SGXP_ERR_TOO_MANY_OCCURENCES;

		// process
		// NOTE: all events inside the element must be consumed till the end element event
		if (complex_type[i].cbk) {
			if (complex_type[i].simple_content && epx_next(psr_ctx) != EPX_EVT_CHARACTERS)	// works if no comments or PI (SOAP) and coalescing
				ret = SGXP_ERR_EVENT_MISMATCH;
			else if ((ret = complex_type[i].cbk(psr_ctx, cur_ns_uri, cur_lname, complex_type[i].user_data)))
				return ret;
			else if (complex_type[i].simple_content && epx_next(psr_ctx) != EPX_EVT_END_ELEMENT)	// works if no comments or PI (SOAP) and coalescing
				ret = SGXP_ERR_EVENT_MISMATCH;
		}

		// skip following if choice
		while (complex_type[i].lname && complex_type[i].mg == XS_MG_CHOICE && complex_type[i].mg_nb == prev_mg_nb) {
			i++;
			cur_elt_count = 0;
		}
		evt = epx_next(psr_ctx);
	}
exit:
	return EPX_OK;
}

static int process_att_events(void * psr_ctx, struct att_step * atts, int nb_att_steps)
{
	int i, ret = EPX_OK, nb_mandatory = 0;
	char * att_ns, * att_lname;

	// Count mandatory attributes
	for (i = 0; i < nb_att_steps; i++)
	{
		if (atts[i].use == XS_ATT_REQUIRED)
			nb_mandatory++;
	}

	while (epx_next(psr_ctx) == EPX_EVT_ATTRIBUTE)
	{
		att_ns = epx_get_ns_uri(psr_ctx);
		att_lname = epx_get_lname(psr_ctx);
		for (i = 0; i < nb_att_steps; i++)
		{
			if (QNAME_EQUALS_WILDCARD(att_ns, att_lname, atts[i].ns_uri, atts[i].lname))
			{
				if (atts[i].cbk) {
					if ((ret = atts[i].cbk(psr_ctx, att_ns, att_lname, atts[i].user_data)))
						return ret;
				}
				if (atts[i].use == XS_ATT_REQUIRED)
					nb_mandatory--;
				break;
			}
		}
	}
	// if not found
	if (nb_mandatory > 0)
		return SGXP_ERR_MISSING_REQUIRED_ATT;

	return ret;
}

int sgxp_check_start_tag(void * psr_ctx, char * ns_uri, char * lname, DC_BOOL required)
{
	int ret = EPX_OK;
	if (epx_next(psr_ctx) != EPX_EVT_START_ELEMENT)
		ret = required ? SGXP_ERR_MISSING_EXPECTED_ELT : SGXP_ERR_EVENT_MISMATCH;

	if (!ret && QNAME_NOT_EQUALS(ns_uri, lname, epx_get_ns_uri(psr_ctx), epx_get_lname(psr_ctx)))
		ret = required ? SGXP_ERR_MISSING_EXPECTED_ELT : SGXP_ERR_TAG_MISMATCH;

	return ret;
}

static int parse_simple_content(void * psr_ctx, struct att_step * atts, int nb_att_steps, value_cbk cbk, void * user_data)
{
	int ret = EPX_OK;
	if (cbk)
	{
		if (epx_get_event(psr_ctx) == EPX_EVT_CHARACTERS)
				cbk(psr_ctx, epx_get_ns_uri(psr_ctx), epx_get_lname(psr_ctx), user_data);
		else
			ret = SGXP_ERR_EVENT_MISMATCH;
		if (epx_next(psr_ctx) != EPX_EVT_END_ELEMENT)
			ret = SGXP_ERR_EVENT_MISMATCH;
	}
	else if (epx_get_event(psr_ctx) != EPX_EVT_END_ELEMENT)	// attribute reading should have consumed the END_ELEMENT if empty
		ret = SGXP_ERR_EVENT_MISMATCH;
	return ret;
}

int sgxp_parse_simple_content(void * psr_ctx, struct att_step * atts, int nb_att_steps, value_cbk cbk, void * user_data)
{
	int ret = process_att_events(psr_ctx, atts, nb_att_steps);
	return ret ? ret : parse_simple_content(psr_ctx, atts, nb_att_steps, cbk, user_data);
}

int sgxp_parse_complex_content(void * psr_ctx, struct att_step * atts, int nb_att_steps, struct elt_step * complex_type, int nb_steps)
{
	int ret = process_att_events(psr_ctx, atts, nb_att_steps);
	if (!ret)
		ret = parse_sequence(psr_ctx, complex_type, nb_steps);
	return ret;
}

int sgxp_parse_simple_element(void * psr_ctx, char * ns_uri, char * lname, DC_BOOL required, struct att_step * atts, int nb_att_steps, value_cbk cbk, void * user_data)
{
	int ret = !sgxp_check_start_tag(psr_ctx, ns_uri, lname, required)
				&& (ret = process_att_events(psr_ctx, atts, nb_att_steps));
	return ret ? ret : parse_simple_content(psr_ctx, atts, nb_att_steps, cbk, user_data);
}

int sgxp_parse_complex_element(void * psr_ctx, char * ns_uri, char * lname, DC_BOOL required, struct att_step * atts, int nb_att_steps, struct elt_step * complex_type, int nb_steps)
{
	int ret = !sgxp_check_start_tag(psr_ctx, ns_uri, lname, required)
				&& (ret = process_att_events(psr_ctx, atts, nb_att_steps));
	return parse_sequence(psr_ctx, complex_type, nb_steps);
}

int sgxp_parse_list(void * psr_ctx, char * list, sgxp_token_parser_cbk token_parser, void * user_data)
{
    char * p = list, * token = NULL;
    int ret = EPX_OK;
    for (;;p++)
    {
        switch (*p)
        {
        case '\0':
            if (token)
                ret = token_parser(psr_ctx, token, user_data);
            return ret;
        case ' ':
        case '\t':
        case 0xA:
        case 0xD:
            if (token)
            {
                *p = '\0';	// ! string is modified (optimization)
                ret = token_parser(psr_ctx, token, user_data);
                if (ret)
                    return ret;
                token = NULL;
            }
            break;
        default:
            if (!token)
                token = p;
        }
    }
}
