package fuku.eb4j;

import fuku.eb4j.io.EBFile;
import fuku.eb4j.io.BookInputStream;
import fuku.eb4j.util.ByteUtil;
import fuku.eb4j.util.CompareUtil;

/**
 * 単一検索語検索クラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class SingleWordSearcher implements Searcher {

    /** 前方一致検索を示す定数 */
    static final int WORD = 0;
    /** 後方一致検索を示す定数 */
    static final int ENDWORD = 1;
    /** 完全一致検索を示す定数 */
    static final int EXACTWORD = 2;
    /** 条件検索を示す定数 */
    static final int KEYWORD = 3;
    /** 複合検索を示す定数 */
    static final int MULTI = 4;

    /** 最大インデックス深さ */
    private static final int MAX_INDEX_DEPTH = 6;

    /** 項目の配置スタイル */
    private static final int VARIABLE = 0;
    /** 項目の配置スタイル */
    private static final int FIXED = 1;

    /** 副本 */
    private SubBook _sub = null;
    /** インデックススタイル */
    private IndexStyle _style = null;
    /** 現在の検索種別 */
    private int _type = 0;

    /** 検索語 */
    private byte[] _word = null;
    /** 検索キー */
    private byte[] _canonical = null;
    /** 検索するファイル */
    private EBFile _file = null;

    /** キャッシュ */
    private byte[] _cache = new byte[BookInputStream.PAGE_SIZE];
    /** キャシュのページ位置 */
    private long _cachePage = 0L;
    /** キャシュのオフセット位置 */
    private int _off = 0;

    /** データのページ位置 */
    private long _page = 0L;
    /** データのページID */
    private int _pageID = 0;
    /** エントリのサイズ */
    private int _entryLength = 0;
    /** エントリの配置方法 */
    private int _entryArrangement = 0;
    /** エントリの数 */
    private int _entryCount = 0;
    /** エントリのインデックス */
    private int _entryIndex = 0;
    /** グループエントリ内であることを示すフラグ */
    private boolean _inGroupEntry = false;
    /** 比較結果 */
    private int _comparison = -1;

    /** キーワード検索用見出し位置 */
    private long _keywordHeading = 0L;


    /**
     * コンストラクタ。
     *
     * @param sub 副本
     * @param style インデックススタイル
     * @param type 検索種別
     * @see SingleWordSearcher#WORD
     * @see SingleWordSearcher#ENDWORD
     * @see SingleWordSearcher#EXACTWORD
     * @see SingleWordSearcher#KEYWORD
     * @see SingleWordSearcher#MULTI
     */
    SingleWordSearcher(SubBook sub, IndexStyle style, int type) {
        super();
        _sub = sub;
        _file = sub.getTextFile();
        _style = style;
        _type = type;
    }


    /**
     * 検索語を設定します。
     *
     * @param word 検索語
     */
    private void _setWord(byte[] word) {
        _word = word;
        _canonical = new byte[_word.length];
        System.arraycopy(_word, 0, _canonical, 0, _word.length);

        if (_sub.getBook().getCharCode() == Book.CHARCODE_ISO8859_1) {
            _style.fixWordLatin(_canonical);
        } else {
            _style.fixWord(_canonical);
        }

        if (_style.getIndexID() != 0x70 && _style.getIndexID() != 0x90) {
            System.arraycopy(_canonical, 0, _word, 0, _word.length);
        }

        // 後方検索の場合、反転する
        if (_type == ENDWORD) {
            if (_sub.getBook().getCharCode() == Book.CHARCODE_ISO8859_1) {
                ByteUtil.reverseWordLatin(_word);
                ByteUtil.reverseWordLatin(_canonical);
            } else {
                ByteUtil.reverseWord(_word);
                ByteUtil.reverseWord(_canonical);
            }
        }
    }

    /**
     * キーとパターンを比較します。
     *
     * @param canonical キー
     * @param pattern パターン
     * @return キーがパターンと同じ場合:0、
     *         キーがパターンより大きい場合:1以上、
     *         キーがパターンより小さい場合:-1以下
     */
    private int _compareToCanonical(byte[] canonical, byte[] pattern) {
        boolean exact = false;
        if (_type == EXACTWORD || _type == KEYWORD || _type == MULTI) {
            exact = true;
        }
        return CompareUtil.compareToCanonical(canonical, pattern, exact);
    }

    /**
     * 検索語とパターンを比較します。
     *
     * @param word 検索語
     * @param pattern パターン
     * @return 検索語がパターンと同じ場合:0、
     *         検索語がパターンより大きい場合:1以上、
     *         検索語がパターンより小さい場合:-1以下
     */
    private int _compareToWord(byte[] word, byte[] pattern) {
        boolean exact = false;
        if (_type == EXACTWORD || _type == KEYWORD || _type == MULTI) {
            exact = true;
        }
        int ret = 0;
        if (_sub.getBook().getCharCode() == Book.CHARCODE_ISO8859_1) {
            ret = CompareUtil.compareToLatin(word, pattern, exact);
        } else {
            ret = CompareUtil.compareTo(word, pattern, exact);
        }
        return ret;
    }

    /**
     * 検索を行います。
     *
     * @param word 検索語
     * @exception EBException 前処理中にエラーが発生した場合
     */
    void search(byte[] word) throws EBException {
        _setWord(word);
        _page = _style.getStartPage();

        BookInputStream bis = _file.getInputStream();
        try {
            long nextPage = _page;
            int depth;
            for (depth=0; depth<MAX_INDEX_DEPTH; depth++) {
                // データをキャッシュへ読み込む
                bis.seek(_page, 0);
                bis.readFully(_cache, 0, _cache.length);
                _cachePage = _page;

                _pageID = _cache[0] & 0xff;
                _entryLength = _cache[1] & 0xff;
                if (_entryLength == 0) {
                    _entryArrangement = VARIABLE;
                } else {
                    _entryArrangement = FIXED;
                }
                _entryCount = ByteUtil.getInt2(_cache, 2);
                _off = 4;

                // リーフインデックスに達っしたらループ終了
                if (_isLeafLayer(_pageID)) {
                    break;
                }

                // 次のレベルのインデックスを取得する
                byte[] b = new byte[_entryLength];
                for (_entryIndex=0; _entryIndex<_entryCount; _entryIndex++) {
                    if (_off + _entryLength + 4 > BookInputStream.PAGE_SIZE) {
                        throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                    }
                    System.arraycopy(_cache, _off, b, 0, b.length);
                    _off += _entryLength;
                    if (_compareToCanonical(_canonical, b) <= 0) {
                        nextPage = ByteUtil.getLong4(_cache, _off);
                        break;
                    }
                    _off += 4;
                }
                if (_entryIndex >= _entryCount || nextPage == _page) {
                    _comparison = -1;
                    return;
                }
                _page = nextPage;
            }

            // インデックス深さのチェック
            if (depth == MAX_INDEX_DEPTH) {
                throw new EBException(EBException.UNEXP_FILE, _file.getPath());
            }
        } finally {
            bis.close();
        }
        _entryIndex = 0;
        _comparison = 1;
        _inGroupEntry = false;
    }

    /**
     * 次の検索結果を返します。
     *
     * @return 検索結果 (次の検索結果がない場合null)
     * @exception EBException 検索中にエラーが発生した場合
     */
    public Result getNextResult() throws EBException {
        if (_comparison < 0) {
            return null;
        }

        while (true) {
            // キャッシュとデータのページが異なれば読み込む
            if (_cachePage != _page) {
                BookInputStream bis = _file.getInputStream();
                try {
                    bis.seek(_page, 0);
                    bis.readFully(_cache, 0, _cache.length);
                } finally {
                    bis.close();
                }
                _cachePage = _page;

                if (_entryIndex == 0) {
                    _pageID = _cache[0] & 0xff;
                    _entryLength = _cache[1] & 0xff;
                    if (_entryLength == 0) {
                        _entryArrangement = VARIABLE;
                    } else {
                        _entryArrangement = FIXED;
                    }
                    _entryCount = ByteUtil.getInt2(_cache, 2);
                    _entryIndex = 0;
                    _off = 4;
                }
            }

            if (!_isLeafLayer(_pageID)) {
                // リーフインデックスでなければ例外
                throw new EBException(EBException.UNEXP_FILE, _file.getPath());
            }

            if (!_hasGroupEntry(_pageID)) {
                // グループエントリなし
                while (_entryIndex < _entryCount) {
                    if (_entryArrangement == VARIABLE) {
                        if (_off + 1 > BookInputStream.PAGE_SIZE) {
                            throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                        }
                        _entryLength = _cache[_off] & 0xff;
                        _off++;
                    }

                    if (_off + _entryLength + 12 > BookInputStream.PAGE_SIZE) {
                        throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                    }

                    byte[] b = new byte[_entryLength];
                    System.arraycopy(_cache, _off, b, 0, b.length);
                    _off += _entryLength;

                    _comparison = _compareToWord(_word, b);
                    Result result = null;
                    if (_comparison == 0) {
                        // 本文/見出し位置の取得
                        long tPage = ByteUtil.getLong4(_cache, _off);
                        int tOff = ByteUtil.getInt2(_cache, _off+4);
                        long hPage = ByteUtil.getLong4(_cache, _off+6);
                        int hOff = ByteUtil.getInt2(_cache, _off+10);
                        result = new Result(_sub, hPage, hOff, tPage, tOff);
                    }

                    _entryIndex++;
                    _off += 12;

                    if (result != null) {
                        return result;
                    }

                    if (_comparison < 0) {
                        return null;
                    }
                }
            } else {
                // グループエントリあり
                while (_entryIndex < _entryCount) {
                    if (_off + 2 > BookInputStream.PAGE_SIZE) {
                        throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                    }
                    int groupID = _cache[_off] & 0xff;
                    Result result = null;
                    if (groupID == 0x00) {
                        // シングルエントリ
                        _entryLength = _cache[_off+1] & 0xff;
                        if (_off + _entryLength + 14 > BookInputStream.PAGE_SIZE) {
                            throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                        }

                        byte[] b = new byte[_entryLength];
                        System.arraycopy(_cache, _off+2, b, 0, b.length);
                        _off += _entryLength + 2;

                        _comparison = _compareToWord(_canonical, b);
                        if (_comparison == 0 && _compareToWord(_word, b) == 0) {
                            // 本文/見出し位置の取得
                            long tPage = ByteUtil.getLong4(_cache, _off);
                            int tOff = ByteUtil.getInt2(_cache, _off+4);
                            long hPage = ByteUtil.getLong4(_cache, _off+6);
                            int hOff = ByteUtil.getInt2(_cache, _off+10);
                            result = new Result(_sub, hPage, hOff, tPage, tOff);
                        }
                        _off += 12;
                        _inGroupEntry = false;
                    } else if (groupID == 0x80) {
                        // グループエントリの開始
                        _entryLength = _cache[_off+1] & 0xff;
                        byte[] b = new byte[_entryLength];
                        if (_type == KEYWORD) {
                            if (_off + _entryLength + 12 > BookInputStream.PAGE_SIZE) {
                                throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                            }

                            System.arraycopy(_cache, _off+6, b, 0, b.length);
                            _off += _entryLength + 6;
                            _comparison = _compareToWord(_word, b);
                            long hPage = ByteUtil.getLong4(_cache, _off);
                            int hOff = ByteUtil.getInt2(_cache, _off+4);
                            _keywordHeading =
                                BookInputStream.getPosition(hPage, hOff);
                            _off += 6;
                        } else if (_type == MULTI) {
                            if (_off + _entryLength + 6 > BookInputStream.PAGE_SIZE) {
                                throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                            }

                            System.arraycopy(_cache, _off+6, b, 0, b.length);
                            _comparison = _compareToWord(_word, b);
                            _off += _entryLength + 6;
                        } else {
                            if (_off + _entryLength + 4 > BookInputStream.PAGE_SIZE) {
                                throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                            }

                            System.arraycopy(_cache, _off+4, b, 0, b.length);
                            _comparison = _compareToWord(_canonical, b);
                            _off += _entryLength + 4;
                        }
                        _inGroupEntry = true;
                    } else if (groupID == 0xc0) {
                        // グループエントリの要素
                        if (_type == KEYWORD) {
                            if (_off + 7 > BookInputStream.PAGE_SIZE) {
                                throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                            }

                            if (_comparison == 0 && _inGroupEntry) {
                                // 本文/見出し位置の取得
                                long tPage = ByteUtil.getLong4(_cache, _off+1);
                                int tOff = ByteUtil.getInt2(_cache, _off+5);
                                result = new Result(_sub, _keywordHeading, tPage, tOff);
                                _keywordHeading =
                                    _sub.getNextHeadingPosition(_keywordHeading);
                            }
                            _off += 7;
                        } else if (_type == MULTI) {
                            if (_off + 13 > BookInputStream.PAGE_SIZE) {
                                throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                            }

                            if (_comparison == 0 && _inGroupEntry) {
                                // 本文/見出し位置の取得
                                long tPage = ByteUtil.getLong4(_cache, _off+1);
                                int tOff = ByteUtil.getInt2(_cache, _off+5);
                                long hPage = ByteUtil.getLong4(_cache, _off+7);
                                int hOff = ByteUtil.getInt2(_cache, _off+11);
                                result = new Result(_sub, hPage, hOff, tPage, tOff);
                            }
                            _off += 13;
                        } else {
                            _entryLength = _cache[_off+1] & 0xff;
                            if (_off + _entryLength + 14 > BookInputStream.PAGE_SIZE) {
                                throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                            }

                            byte[] b = new byte[_entryLength];
                            System.arraycopy(_cache, _off+2, b, 0, b.length);
                            _off += _entryLength + 2;
                            if (_comparison == 0 && _inGroupEntry
                                && _compareToWord(_word, b) == 0) {
                                // 本文/見出し位置の取得
                                long tPage = ByteUtil.getLong4(_cache, _off);
                                int tOff = ByteUtil.getInt2(_cache, _off+4);
                                long hPage = ByteUtil.getLong4(_cache, _off+6);
                                int hOff = ByteUtil.getInt2(_cache, _off+10);
                                result = new Result(_sub, hPage, hOff, tPage, tOff);
                            }
                            _off += 12;
                        }
                    } else {
                        // 未知のID
                        throw new EBException(EBException.UNEXP_FILE, _file.getPath());
                    }

                    _entryIndex++;

                    if (result != null) {
                        return result;
                    }

                    if (_comparison < 0) {
                        return null;
                    }
                }
            }

            // 次ページが存在すれば続行、存在しなければ終了
            if (_isLayerEnd(_pageID)) {
                _comparison = -1;
                break;
            }
            _page++;
            _entryIndex = 0;
        }
        return null;
    }

    /**
     * 指定されたページが最下層かどうかを判別します。
     *
     * @param id ページID
     * @return 最下層である場合はtrue、そうでない場合はfalse
     */
    private boolean _isLeafLayer(int id) {
        if ((id & 0x80) == 0x80) {
            return true;
        }
        return false;
    }

    /**
     * 指定されたページが階層開始ページかどうかを判別します。
     *
     * @param id ページID
     * @return 階層開始ページである場合はtrue、そうでない場合はfalse
     */
    private boolean _isLayerStart(int id) {
        if ((id & 0x40) == 0x40) {
            return true;
        }
        return false;
    }

    /**
     * 指定されたページが階層終了ページかどうかを判別します。
     *
     * @param id ページID
     * @return 階層終了ページである場合はtrue、そうでない場合はfalse
     */
    private boolean _isLayerEnd(int id) {
        if ((id & 0x20) == 0x20) {
            return true;
        }
        return false;
    }

    /**
     * 指定されたページがグループエントリを含んでいるかどうか判別します。
     *
     * @param id ページID
     * @return グループエントリを含んでいる場合はtrue、そうでない場合はfalse
     */
    private boolean _hasGroupEntry(int id) {
        if ((id & 0x10) == 0x10) {
            return true;
        }
        return false;
    }
}

// end of SingleWordSearcher.java
