package fuku.eb4j.io;

import java.io.*;
import java.util.zip.*;

import fuku.eb4j.EBException;
import fuku.eb4j.util.ByteUtil;

/**
 * EBZIP形式の書籍入力ストリームクラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class EBZipInputStream
    extends BookInputStream implements EBZipConstants {

    /**
     * コンストラクタ。
     *
     * @param info ファイル情報
     * @exception EBException 入出力エラーが発生した場合
     */
    EBZipInputStream(FileInfo info) throws EBException {
        super(info);
        open();
        // スライス単位でキャッシュする
        _cache = new byte[_info.sliceSize];
    }


    /**
     * EBZIP形式のファイル情報を初期化します。
     *
     * @exception EBException 入出力エラーが発生した場合
     */
    protected void initFileInfo() throws EBException {
        try {
            _info.realFileSize = _stream.length();
        } catch (IOException e) {
            throw new EBException(EBException.FAILED_READ_FILE, _info.file.getPath(), e);
        }

        // ヘッダの読み込み
        byte[] b = new byte[EBZIP_HEADER_SIZE];
        readRawFully(b, 0, b.length);

        _info.zipLevel = b[5] & 0x0f;
        _info.sliceSize = PAGE_SIZE << _info.zipLevel;
        _info.fileSize = ByteUtil.getLong4(b, 10);
        _info.zipCrc = ByteUtil.getLong4(b, 14);

        if (_info.fileSize < (1<<16)) {
            _info.zipIndexSize = 2;
        } else if (_info.fileSize < (1<<24)) {
            _info.zipIndexSize = 3;
        } else {
            _info.zipIndexSize = 4;
        }

        // 妥当性の検証
        String str = new String(b, 0, 5);
        if (!str.equals("EBZip")
            || (b[5] >>> 4) != EBFile.FORMAT_EBZIP
            || _info.sliceSize > (PAGE_SIZE << EBZIP_MAX_LEVEL)) {
            throw new EBException(EBException.UNEXP_FILE, _info.file.getPath());
        }
        super.initFileInfo();
    }

    /**
     * このファイルの圧縮レベルを返します。
     *
     * @return 圧縮レベル
     */
    public int getLevel() {
        return _info.zipLevel;
    }

    /**
     * このファイルのCRCを返します。
     *
     * @return CRC
     */
    public long getCRC() {
        return _info.zipCrc;
    }

    /**
     * EBZIP形式のファイルから最大lenバイトのデータをバイト配列に読み込みます。
     *
     * @param b データの読み込み先のバッファ
     * @param off データの開始オフセット
     * @param len 読み込まれる最大バイト数
     * @return バッファに読み込まれたバイトの合計数
     *         (ストリームの終わりに達してデータがない場合は-1)
     * @exception EBException 入出力エラーが発生した場合
     */
    public int read(byte[] b, int off, int len) throws EBException {
        int rlen = 0;
        while (rlen < len) {
            if (_info.fileSize <= _filePos) {
                // ストリームの終わり
                if (rlen == 0) {
                    // データを読み込んでいなければ-1
                    return -1;
                } else {
                    // データを読み込んでいればバイト数を返す
                    return rlen;
                }
            }
            // キャッシュの作成
            if (_cachePos < 0
                || _filePos < _cachePos
                || _cachePos + _info.sliceSize <= _filePos) {
                // キャッシュのデータ位置
                // _filePosの位置が含まれるスライスの先頭位置
                _cachePos = _filePos - (_filePos % _info.sliceSize);

                // 圧縮されたスライスのインデックスデータの位置
                // (スライスオフセット * インデックスサイズ) + ヘッダサイズ
                long pos = _filePos / _info.sliceSize * _info.zipIndexSize
                    + EBZIP_HEADER_SIZE;
                try {
                    _stream.seek(pos);
                } catch (IOException e) {
                    throw new EBException(EBException.FAILED_SEEK_FILE, _info.file.getPath(), e);
                }
                byte[] buf = new byte[_info.zipIndexSize*2];
                readRawFully(buf, 0, buf.length);

                // スライス位置の取得
                long slicePos = 0L;
                long nextSlicePos = 0L;
                switch (_info.zipIndexSize) {
                    case 2:
                        slicePos = ByteUtil.getInt2(buf, 0);
                        nextSlicePos = ByteUtil.getInt2(buf, 2);
                        break;
                    case 3:
                        slicePos = ByteUtil.getInt3(buf, 0);
                        nextSlicePos = ByteUtil.getInt3(buf, 3);
                        break;
                    case 4:
                        slicePos = ByteUtil.getLong4(buf, 0);
                        nextSlicePos = ByteUtil.getLong4(buf, 4);
                        break;
                    default:
                        break;
                }

                // 圧縮されたスライスのサイズ
                int sliceSize = (int)(nextSlicePos - slicePos);
                if (sliceSize <= 0 || _info.sliceSize < sliceSize) {
                    return -1;
                }

                // 圧縮スライスをデコードしてキャッシュに読み込む
                try {
                    _stream.seek(slicePos);
                } catch (IOException e) {
                    throw new EBException(EBException.FAILED_SEEK_FILE, _info.file.getPath(), e);
                }
                _decode(sliceSize);
            }

            // キャッシュからデータの取得
            int n = (int)(_info.sliceSize - (_filePos % _info.sliceSize));
            if (len - rlen < n) {
                n = len - rlen;
            }
            if (_info.fileSize - _filePos < n) {
                n = (int)(_info.fileSize - _filePos);
            }
            int p = (int)(_filePos % _info.sliceSize);
            System.arraycopy(_cache, p, b, off+rlen, n);
            rlen += n;
            _filePos += n;
        }
        return rlen;
    }

    /**
     * 復号化します。
     *
     * @param size 圧縮スライスサイズ
     * @exception EBException 入出力エラーが発生した場合
     */
    private void _decode(int size) throws EBException {
        if (size == _info.sliceSize) {
            // 圧縮されていないのでそのままキャッシュに読み込む
            readRawFully(_cache, 0, size);
        } else {
            byte[] b = new byte[size];
            Inflater inf = new Inflater();
            try {
                // 圧縮されたスライスをキャッシュに展開する
                readRawFully(b, 0, size);
                inf.setInput(b, 0, size);
                inf.inflate(_cache, 0, _info.sliceSize);
            } catch (DataFormatException e) {
                throw new EBException(EBException.UNEXP_FILE, _info.file.getPath(), e);
            } finally {
                inf.end();
            }
        }
    }
}

// end of EBZipInputStream.java
