package fuku.eb4j.util;

/**
 * 比較ユーティリティクラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class CompareUtil {

    /**
     * コンストラクタ。
     *
     */
    private CompareUtil() {
        super();
    }


    /**
     * キーとパターンをバイト値で比較します。
     *
     * @param key キー
     * @param pattern パターン
     * @param exact 完全一致で比較するか前方一致で比較するか
     *              (前方一致の場合、キーがパターンの前方に一致すれば同じとみなす)
     * @return キーがパターンと同じ場合は0、
     *         キーがパターンより大きい場合は1以上、
     *         キーがパターンより小さい場合は-1以下
     */
    public static int compareToCanonical(byte[] key, byte[] pattern, boolean exact) {
        int klen = key.length;
        int plen = pattern.length;
        int kByte, pByte;
        for (int i=0; i<klen; i++) {
            if (i >= plen) {
                return key[i] & 0xff;
            }
            if (key[i] == '\0') {
                if (exact) {
                    while (i < plen && (pattern[i] == '\0')) {
                        i++;
                    }
                    return i - plen;
                } else {
                    return 0;
                }
            }

            kByte = key[i] & 0xff;
            pByte = pattern[i] & 0xff;
            if (kByte != pByte) {
                return kByte - pByte;
            }
        }
        if (klen < plen && exact) {
            return - (pattern[klen] & 0xff);
        }
        return 0;
    }

    /**
     * キーとパターンをISO 8859-1文字セットで比較します。
     *
     * @param key キー (ISO 8859-1)
     * @param pattern パターン (ISO 8859-1)
     * @param exact 完全一致で比較するか前方一致で比較するか
     *              (前方一致の場合、キーがパターンの前方に一致すれば同じとみなす)
     * @return キーがパターンと同じ場合は0、
     *         キーがパターンより大きい場合は1以上、
     *         キーがパターンより小さい場合は-1以下
     */
    public static int compareToLatin(byte[] key, byte[] pattern, boolean exact) {
        int klen = key.length;
        int plen = pattern.length;
        int kByte, pByte;
        for (int i=0; i<klen; i++) {
            if (i >= plen) {
                return key[i] & 0xff;
            }
            if (key[i] == '\0') {
                if (exact) {
                    while (i < plen && (pattern[i] == ' ' || pattern[i] == '\0')) {
                        i++;
                    }
                    return i - plen;
                } else {
                    return 0;
                }
            }

            kByte = key[i] & 0xff;
            pByte = pattern[i] & 0xff;
            if (kByte != pByte) {
                return kByte - pByte;
            }
        }
        if (klen < plen && exact) {
            return - (pattern[klen] & 0xff);
        }
        return 0;
    }

    /**
     * キーとパターンをJIS X 0208文字セットで比較します。
     *
     * @param key キー (JIS X 0208)
     * @param pattern パターン (JIS X 0208)
     * @param exact 完全一致で比較するか前方一致で比較するか
     *              (前方一致の場合、キーがパターンの前方に一致すれば同じとみなす)
     * @return キーがパターンと同じ場合は0、
     *         キーがパターンより大きい場合は1以上、
     *         キーがパターンより小さい場合は-1以下
     */
    public static int compareTo(byte[] key, byte[] pattern, boolean exact) {
        int klen = key.length;
        int plen = pattern.length;
        int kByte, pByte;
        for (int i=0; i<klen; i+=2) {
            if (i >= plen) {
                return key[i] & 0xff;
            }
            if (key[i] == '\0') {
                if (exact) {
                    return - (pattern[i] & 0xff);
                } else {
                    return 0;
                }
            }
            if (i+1 >= klen || i+1 >= plen) {
                return (key[i] & 0xff) - (pattern[i] & 0xff);
            }

            kByte = key[i] & 0xff;
            pByte = pattern[i] & 0xff;
            if (kByte != pByte) {
                return kByte - pByte;
            }
            kByte = key[i+1] & 0xff;
            pByte = pattern[i+1] & 0xff;
            if (kByte != pByte) {
                return kByte - pByte;
            }
        }
        if (klen < plen && exact) {
            return - (pattern[klen] & 0xff);
        }
        return 0;
     }
}

// end of CompareUtil.java
