package fuku.player;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;
import javax.sound.sampled.*;
import javax.sound.midi.*;
import javax.swing.*;

/**
 * 音声再生クラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class AudioPlayerApplet extends JApplet
    implements ActionListener, LineListener, MetaEventListener {

    /** 再生 */
    private static final int PLAY = 0;
    /** 一時停止 */
    private static final int PAUSE = 1;
    /** 停止 */
    private static final int STOP = 2;

    /** アイコンファイル名 */
    private static final String[] ICON = {
        "fuku/player/resources/play16x16.png",
        "fuku/player/resources/pause16x16.png",
        "fuku/player/resources/stop16x16.png"
    };
    /** アイコンのサイズ */
    private static final Dimension ICON_SIZE = new Dimension(16, 16);

    /** アイコン */
    private ImageIcon[] _icon = null;

    /** データロードフラグ */
    private boolean _loaded = false;
    /** オーディオクリップ */
    private Clip _clip = null;
    /** シーケンサ */
    private Sequencer _sequencer = null;

    /** 再生ボタン */
    private JButton _playBtn = null;
    /** 停止ボタン */
    private JButton _stopBtn = null;

    /** 再生開始位置 */
    private int _pos = 0;
    /** リセットフラグ */
    private boolean _reset = false;


    /**
     * このアプレットを初期化します。
     *
     */
    public void init() {
        // アイコンの作成
        _icon = new ImageIcon[ICON.length];
        for (int i=0; i<ICON.length; i++) {
            URL url = getClass().getClassLoader().getResource(ICON[i]);
            _icon[i] = new ImageIcon(url);
        }

        // ボタンの作成
        _playBtn = new JButton(_icon[PLAY]);
        _playBtn.setMargin(new Insets(0, 0, 0, 0));
        _playBtn.setPreferredSize(ICON_SIZE);
        _playBtn.setMaximumSize(ICON_SIZE);
        _playBtn.setMinimumSize(ICON_SIZE);
        _playBtn.addActionListener(this);
        _stopBtn = new JButton(_icon[STOP]);
        _stopBtn.setMargin(new Insets(0, 0, 0, 0));
        _stopBtn.setPreferredSize(ICON_SIZE);
        _stopBtn.setMaximumSize(ICON_SIZE);
        _stopBtn.setMinimumSize(ICON_SIZE);
        _stopBtn.addActionListener(this);

        // 配置
        getContentPane().setLayout(new BorderLayout(0, 0));
        getContentPane().add(_playBtn, BorderLayout.WEST);
        getContentPane().add(_stopBtn, BorderLayout.EAST);
    }

    /**
     * オーディオデバイスを取得します。
     *
     */
    private void _setup() {
        if (_loaded) {
            return;
        }
        // ラインまたはデバイスの取得
        URL url = null;
        try {
            url = new URL(getDocumentBase(), getParameter("file"));
            AudioInputStream ais = AudioSystem.getAudioInputStream(url);
            DataLine.Info info = new DataLine.Info(Clip.class, ais.getFormat());
            _clip = (Clip)AudioSystem.getLine(info);
            _clip.addLineListener(this);
            _clip.open(ais);
        } catch (MalformedURLException e) {
            System.err.println(e.getMessage());
        } catch (Exception e1) {
            try {
                _sequencer = MidiSystem.getSequencer();
                _sequencer.addMetaEventListener(this);
                _sequencer.open();
                _sequencer.setSequence(MidiSystem.getSequence(url));
            } catch (InvalidMidiDataException imde) {
                System.err.println("Unsupported audio file.");
            } catch (Exception e2) {
                System.err.println(e2.getMessage());
            }
        }
        _loaded = true;
    }

    /**
     * アプレットが開始されると呼び出されます。
     *
     */
    public void start() {
    }

    /**
     * アプレットが停止されると呼び出されます。
     *
     */
    public void stop() {
    }

    /**
     * すべてのリソースを開放します。
     *
     */
    public void destroy() {
        getContentPane().removeAll();
        if (_clip != null) {
            _clip.removeLineListener(this);
            if (_clip.isRunning()) {
                _clip.stop();
            }
            if (_clip.isOpen()) {
                _clip.close();
            }
            _clip = null;
        }
        if (_sequencer != null) {
            _sequencer.removeMetaEventListener(this);
            if (_sequencer.isRunning()) {
                _sequencer.stop();
            }
            if (_sequencer.isOpen()) {
                _sequencer.close();
            }
            _sequencer = null;
        }
    }

    /**
     * アクションが発生すると呼び出されます。
     *
     * @param evt アクションイベント
     */
    public void actionPerformed(ActionEvent evt) {
        _setup();
        Object obj = evt.getSource();
        if (obj instanceof JButton) {
            JButton btn = (JButton)obj;
            if (obj.equals(_playBtn)) {
                if (_clip != null) {
                    if (_clip.isRunning()) {
                        _clip.stop();
                    } else {
                        _clip.setFramePosition(_pos);
                        _clip.start();
                    }
                } else if (_sequencer != null) {
                    if (_sequencer.isRunning()) {
                        _sequencer.stop();
                        _playBtn.setIcon(_icon[PLAY]);
                    } else {
                        _sequencer.start();
                        _playBtn.setIcon(_icon[PAUSE]);
                    }
                }
            } else if (obj.equals(_stopBtn)) {
                if (_clip != null) {
                    if (_clip.isRunning()) {
                        _clip.stop();
                    }
                } else if (_sequencer != null) {
                    if (_sequencer.isRunning()) {
                        _sequencer.stop();
                    }
                    _playBtn.setIcon(_icon[PLAY]);
                }
                _reset = true;
            }
        }
    }

    /**
     * ラインの状態が変化すると呼び出されます。
     *
     * @param evt ラインイベント
     */
    public void update(LineEvent evt) {
        if (_clip == null) {
            return;
        }
        LineEvent.Type type = evt.getType();
        if (type.equals(LineEvent.Type.START)) {
            _playBtn.setIcon(_icon[PAUSE]);
        } else if (type.equals(LineEvent.Type.STOP)) {
            if (_reset) {
                _pos = 0;
                _reset = false;
            } else if (_pos >= _clip.getFramePosition()) {
                // 一時停止など開始位置が途中に設定されている場合で
                // 位置に変化がない、または巻戻されている場合は
                // 開始位置を最初に設定
                _pos = 0;
            } else {
                _pos = _clip.getFramePosition();
                if (_pos < 0 || _pos >= _clip.getFrameLength()) {
                    // 異常な値であれば開始位置を最初に設定
                    _pos = 0;
                }
            }
            _playBtn.setIcon(_icon[PLAY]);
        }
    }

    /**
     * Sequencerが処理中のSequenceからMetaMessageを検出し処理した場合に呼び出されます。
     *
     * @param msg メタメッセージ
     */
    public void meta(MetaMessage msg) {
        if (_sequencer == null) {
            return;
        }
        if (msg.getType() == 47) {
            _playBtn.setIcon(_icon[PLAY]);
        }
    }
}

// end of AudioPlayerApplet.java
