package fuku.webbook;

import java.awt.Color;
import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import javax.servlet.*;
import javax.servlet.http.*;

import fuku.eb4j.SubBook;
import fuku.eb4j.EBException;
import fuku.eb4j.util.ImageUtil;

/**
 * リソースデータサーブレットクラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class ResourceServlet extends HttpServlet {

    /** PNG */
    private static final int PNG = 0;
    /** JPEG */
    private static final int JPEG = 1;
    /** WAVE */
    private static final int WAVE = 2;
    /** MIDI */
    private static final int MIDI = 3;
    /** MPEG */
    private static final int MPEG = 4;

    /** メディアタイプ */
    private static final String[] MEDIA_TYPE = {
        ".png", ".jpeg", ".wav", ".mid", ".mpeg"
    };

    /** MIMEタイプ */
    private static final String[] MIME_TYPE = {
        "image/png", "image/jpeg", "audio/x-wav", "audio/midi", "video/mpeg"
    };

    /**
     * GETリクエストの処理。
     *
     * @param req クライアントからのリクエスト
     * @param res クライアントへ返すレスポンス
     * @exception ServletException GETに相当するリクエストが処理できない場合
     * @exception IOException GETリクエストの処理中に入出力エラーが発生した場合
     */
    public void doGet(HttpServletRequest req, HttpServletResponse res)
        throws ServletException, IOException {
        doPost(req, res);
    }

    /**
     * POSTリクエストの処理。
     *
     * @param req クライアントからのリクエスト
     * @param res クライアントへ返すレスポンス
     * @exception ServletException POSTに相当するリクエストが処理できない場合
     * @exception IOException POSTリクエストの処理中に入出力エラーが発生した場合
     */
    public void doPost(HttpServletRequest req, HttpServletResponse res)
        throws ServletException, IOException {
        String path = req.getPathInfo();
        if (path == null || path.length() <= 0) {
            return;
        }

        // メディアタイプの設定
        int type = 0;
        for (type=0; type<MEDIA_TYPE.length; type++) {
            if (path.endsWith(MEDIA_TYPE[type])) {
                res.setContentType(MIME_TYPE[type]);
                break;
            }
        }
        if (type >= MEDIA_TYPE.length) {
            res.sendError(HttpServletResponse.SC_NOT_FOUND);
            return;
        }
        ServletOutputStream out = res.getOutputStream();

        // 要求対象のファイル
        File file = new File(path);
        String name = file.getName();
        // サーブレットコンテキストから書籍管理オブジェクトを取得
        WebBook webbook = (WebBook)getServletContext().getAttribute("webbook");

        SubBook sub = webbook.getSubBook(req.getRemoteHost(),
                                         file.getParentFile().getName());
        if (sub == null) {
            res.sendError(HttpServletResponse.SC_NOT_FOUND);
            return;
        }

        File cacheFile = null;
        boolean cacheFlag = false;
        switch (type) {
            case JPEG:
                cacheFlag = webbook.isImageCache();
                break;
            case WAVE:
            case MIDI:
                cacheFlag = webbook.isSoundCache();
                break;
            default:
                int prefix = name.charAt(0);
                if (prefix == 'N' || prefix == 'W') {
                    cacheFlag = webbook.isGaijiCache();
                } else if (prefix == 'M' || prefix == 'C') {
                    cacheFlag = webbook.isImageCache();
                }
                break;
        }
        if (cacheFlag) {
            // キャッシュがあれば出力
            cacheFile = new File(webbook.getCacheDirectory(), path);
            if (cacheFile.canRead()) {
                _send(out, cacheFile);
                return;
            }
        }

        // データの読み込み
        int idx1, idx2, idx3;
        long pos1, pos2;
        byte[] b = null;
        File f = null;
        try {
            switch (type) {
                case PNG:
                    int prefix = name.charAt(0);
                    if (prefix == 'N' || prefix == 'W') {
                        // 外字
                        boolean narrow = false;
                        if (prefix == 'N') {
                            narrow = true;
                        }
                        idx1 = name.indexOf('-');
                        idx2 = name.indexOf("_F-", idx1+1);
                        idx3 = name.indexOf("_B-", idx2+3);
                        int code = Integer.parseInt(name.substring(idx1+1, idx2), 16);
                        int fore = Integer.parseInt(name.substring(idx2+3, idx3), 16);
                        int back = Integer.parseInt(name.substring(idx3+3, name.length()-4), 16);
                        b = _getFontImage(sub, narrow, code, fore, back);
                    } else if (prefix == 'M') {
                        // モノクロ画像
                        idx1 = name.indexOf('-');
                        idx2 = name.indexOf("_W-", idx1+1);
                        idx3 = name.indexOf("_H-", idx2+3);
                        pos1 = Long.parseLong(name.substring(idx1+1, idx2), 16);
                        int width = Integer.parseInt(name.substring(idx2+3, idx3), 16);
                        int height = Integer.parseInt(name.substring(idx3+3, name.length()-4), 16);
                        b = _getMonoImage(sub, pos1, width, height);
                    } else if (prefix == 'C') {
                        // カラー画像
                        pos1 = Long.parseLong(name.substring(2, name.length()-4), 16);
                        b = _getColorImage(sub, pos1, type);
                    }
                    break;
                case JPEG:
                    pos1 = Long.parseLong(name.substring(2, name.length()-5), 16);
                    b = _getColorImage(sub, pos1, type);
                    break;
                case WAVE:
                case MIDI:
                    idx1 = name.indexOf('-');
                    idx2 = name.indexOf("_E-", idx1+1);
                    pos1 = Long.parseLong(name.substring(idx1+1, idx2), 16);
                    pos2 = Long.parseLong(name.substring(idx2+3, name.length()-4), 16);
                    b = _getSound(sub, type, pos1, pos2);
                    break;
                case MPEG:
                    f = sub.getMovieFile(name.substring(0, name.length()-5));
                    break;
                default:
                    break;
            }
        } catch (NumberFormatException e) {
            f = null;
            b = null;
        }

        // 応答の出力
        if (f != null) {
            _send(out, f);
        } else if (b != null) {
            // キャッシュの保存
            _store(cacheFile, b);
            out.write(b, 0, b.length);
        } else {
            res.sendError(HttpServletResponse.SC_NOT_FOUND);
        }
    }

    /**
     * 外字イメージを返します。
     *
     * @param sub 副本
     * @param narrow 半角/全角フラグ
     * @param code 外字のコード
     * @param fore 前景色
     * @param back 背景色
     * @return PNGデータ
     */
    private byte[] _getFontImage(SubBook sub, boolean narrow,
                                 int code, int fore, int back) {
        byte[] data = null;
        int width = 0;
        try {
            if (narrow) {
                if (!sub.getFont().hasNarrowFont()) {
                    return null;
                }
                data = sub.getFont().getNarrowFont(code);
                width = sub.getFont().getNarrowFontWidth();
            } else {
                if (!sub.getFont().hasWideFont()) {
                    return null;
                }
                data = sub.getFont().getWideFont(code);
                width = sub.getFont().getWideFontWidth();
            }
        } catch (EBException e) {
            log("failed to load font image", e);
            return null;
        }
        if (data == null) {
            return null;
        }
        int height = sub.getFont().getFontHeight();
        return ImageUtil.bitmapToPNG(data, width, height,
                                     new Color(fore), new Color(back), true, 9);
    }

    /**
     * モノクル画像を返します。
     *
     * @param sub 副本
     * @param pos 画像の位置
     * @param width 画像の幅
     * @param height 画像の高さ
     * @return PNGデータ
     */
    private byte[] _getMonoImage(SubBook sub, long pos, int width, int height) {
        byte[] data = null;
        try {
            data = sub.getGraphicData().getMonoGraphic(pos, width, height);
        } catch (EBException e) {
            log("failed to load mono image", e);
            return null;
        }
        if (data == null) {
            return null;
        }
        return ImageUtil.bitmapToPNG(data, width, height,
                                     Color.BLACK, Color.WHITE, false, 9);
    }

    /**
     * カラー画像を返します。
     *
     * @param sub 副本
     * @param pos 画像の位置
     * @param type メディアタイプ
     * @return イメージデータ
     */
    private byte[] _getColorImage(SubBook sub, long pos, int type) {
        byte[] data = null;
        try {
            data = sub.getGraphicData().getColorGraphic(pos);
        } catch (EBException e) {
            log("failed to load color image", e);
            return null;
        }
        if (data == null) {
            return null;
        }
        if (type == PNG) {
            return ImageUtil.dibToPNG(data, 9);
        }
        return data;
    }

    /**
     * 音声を返します。
     *
     * @param sub 副本
     * @param start 開始位置
     * @param end 終了位置
     * @return 音声データ
     */
    private byte[] _getSound(SubBook sub, int type, long start, long end) {
        byte[] data = null;
        try {
            switch (type) {
                case WAVE:
                    data = sub.getSoundData().getWaveSound(start, end);
                    break;
                case MIDI:
                    data = sub.getSoundData().getMidiSound(start, end);
                    break;
                default:
                    break;
            }
        } catch (EBException e) {
            log("failed to load sound data", e);
            return null;
        }
        return data;
    }

    /**
     * データをファイルに保存します。
     *
     * @param file ファイル
     * @param data データ
     */
    private void _store(File file, byte[] data) {
        if (file == null || data == null) {
            return;
        }
        if (!file.getParentFile().exists()) {
            file.getParentFile().mkdirs();
        }
        FileChannel channel = null;
        try {
            FileOutputStream fos = new FileOutputStream(file);
            channel = fos.getChannel();
            channel.write(ByteBuffer.wrap(data));
        } catch (IOException e) {
            log("failed to store file", e);
        } finally {
            if (channel != null) {
                try {
                    channel.close();
                } catch (IOException e) {
                }
            }
        }
    }

    /**
     * ファイルの内容を出力します。
     *
     * @param out 出力ストリーム
     * @param file 出力ファイル
     */
    private void _send(OutputStream out, File file) {
        BufferedInputStream bis = null;
        byte[] b = new byte[8192];
        int n = 0;
        try {
            bis = new BufferedInputStream(new FileInputStream(file));
            while ((n=bis.read(b, 0, b.length)) >= 0) {
                out.write(b, 0, n);
            }
        } catch (IOException e) {
            log("failed to send data", e);
        } finally {
            if (bis != null) {
                try {
                    bis.close();
                } catch (IOException e) {
                }
            }
        }
    }
}

// end of ResourceServlet.java
