package fuku.webbook;

import java.io.*;
import java.net.*;
import java.util.*;

import fuku.eb4j.Book;
import fuku.eb4j.SubBook;
import fuku.eb4j.ExtFont;
import fuku.eb4j.Searcher;
import fuku.eb4j.Result;
import fuku.eb4j.EBException;

/**
 * 書籍管理クラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class WebBook {

    /** 前方一致検索 */
    private static final int WORD = 0;
    /** 後方一致検索 */
    private static final int ENDWORD = 1;
    /** 完全一致検索 */
    private static final int EXACT = 2;
    /** 条件検索 */
    private static final int KEYWORD = 3;

    /** 検索種別の文字列表現 */
    public static final String[] SEARCH_NAME = {
        "前方一致検索",
        "後方一致検索",
        "完全一致検索",
        "条件検索"
    };

    /** リソースディレクトリ名 */
    private static final String RES_DIR = "resources/";

    /** すべての副本 */
    private SubBook[] _subs = null;

    /** 設定 */
    private WebBookConfig _config = null;


    /**
     * コンストラクタ。
     *
     */
    public WebBook() {
        super();
    }


    /**
     * このオブジェクトを設定します。
     *
     * @param conf このオブジェクトの定義
     * @exception EBException
     */
    public void setConfig(WebBookConfig conf) throws EBException {
        _subs = null;
        _config = conf;

        String[] bookList = conf.getBookList();
        String[] appendixList = conf.getAppendixList();
        if (bookList != null) {
            ArrayList subList = new ArrayList(8);
            // 書籍の取得
            int len = bookList.length;
            for (int i=0; i<len; i++) {
                Book book = new Book(bookList[i], appendixList[i]);
                SubBook[] sb = book.getSubBooks();
                int sblen = sb.length;
                for (int j=0; j<sblen; j++) {
                    // 外字の設定
                    if (sb[j].getFont(ExtFont.FONT_16).hasFont()) {
                        sb[j].setFont(ExtFont.FONT_16);
                    }
                    subList.add(sb[j]);
                }
            }
            if (!subList.isEmpty()) {
                _subs = (SubBook[])subList.toArray(new SubBook[0]);
            }
        }
    }

    /**
     * 外字キャッシュの有無を返します。
     *
     * @return キャッシュの有無
     */
    public boolean isGaijiCache() {
        if (_config == null) {
            return false;
        }
        return _config.isGaijiCache();
    }

    /**
     * 画像キャッシュの有無を返します。
     *
     * @return キャッシュの有無
     */
    public boolean isImageCache() {
        if (_config == null) {
            return false;
        }
        return _config.isImageCache();
    }

    /**
     * 音声キャッシュの有無を返します。
     *
     * @return キャッシュの有無
     */
    public boolean isSoundCache() {
        if (_config == null) {
            return false;
        }
        return _config.isSoundCache();
    }

    /**
     * キャッシュディレクトリを返します。
     *
     * @return キャッシュディレクトリ
     */
    public File getCacheDirectory() {
        if (_config == null) {
            return null;
        }
        return _config.getCacheDirectory();
    }

    /**
     * 指定された副本へのアクセスが許可されているかどうかを判定します。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 副本名
     * @return 許可されている場合はtrue、そうでない場合はfalse
     */
    public boolean isAllowed(String host, String name) {
        int idx = _getSubBookIndex(host, name);
        if (idx < 0) {
            return false;
        }
        return true;
    }

    /**
     * 指定されたホストからアクセス可能な副本の一覧を返します。
     *
     * @param host ホスト名またはIPアドレス
     * @return 副本の一覧
     */
    public SubBook[] getSubBooks(String host) {
        ArrayList list = new ArrayList();
        if (_subs != null) {
            for (int i=0; i<_subs.length; i++) {
                String path = _subs[i].getBook().getPath();
                if (_config.isAllowed(host, path)) {
                    list.add(_subs[i]);
                }
            }
        }
        return (SubBook[])list.toArray(new SubBook[0]);
    }

    /**
     * 指定された副本を返します。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 副本名
     * @return 副本
     */
    public SubBook getSubBook(String host, String name) {
        int idx = _getSubBookIndex(host, name);
        if (idx < 0) {
            return null;
        }
        return _subs[idx];
    }

    /**
     * 指定された副本のインデックス番号を返します。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 副本名
     * @return インデックス番号
     */
    private int _getSubBookIndex(String host, String name) {
        if (_subs == null || name == null) {
            return -1;
        }
        int len = _subs.length;
        for (int i=0; i<len; i++) {
            if (_subs[i].getName().equals(name)) {
                String path = _subs[i].getBook().getPath();
                if (_config.isAllowed(host, path)) {
                    return i;
                } else {
                    return -1;
                }
            }
        }
        return -1;
    }

    /**
     * フックを作成します。
     *
     * @param index 副本のインデックス
     */
    private HTMLHook _createHTMLHook(int index) {
        String href = null;
        try {
            href = "detail.jsp?book="
                + URLEncoder.encode(_subs[index].getName(), "UTF-8");
        } catch (UnsupportedEncodingException e) {
        }
        HTMLHook hook =
            new HTMLHook(_subs[index],  href,
                         RES_DIR + _subs[index].getName(),
                         _config.getURNRedirectURL());
        hook.setForegroundColor(_config.getForegroundColor());
        hook.setBackgroundColor(_config.getBackgroundColor());
        hook.setAnchorColor(_config.getAnchorColor());
        hook.setKeywordColor(_config.getKeywordColor());
        return hook;
    }

    /**
     * 指定された検索方法で検索を行います。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 検索対象の副本名 (nullまたは空文字列の場合はすべての副本)
     * @param word 検索語
     * @param method 検索方法
     * @param max 最大検索結果数
     * @return 検索結果 (検索結果がない場合はnull)
     */
    public Result[][] search(String host, String name,
                             String word, int method, int max) {
        if (_subs == null || max <= 0) {
            return null;
        }
        SubBook[] subs = null;
        if (name == null || name.length() <= 0) {
            subs = getSubBooks(host);
            if (subs.length <= 0) {
                return null;
            }
        } else {
            subs = new SubBook[1];
            subs[0] = getSubBook(host, name);
            if (subs[0] == null) {
                return null;
            }
        }

        int len = subs.length;
        Result[][] results = new Result[len][];
        ArrayList list = new ArrayList(max);
        for (int i=0; i<len; i++) {
            try {
                Searcher searcher = null;
                switch (method) {
                    case WORD:
                        searcher = subs[i].searchWord(word);
                        break;
                    case ENDWORD:
                        searcher = subs[i].searchEndword(word);
                        break;
                    case EXACT:
                        searcher = subs[i].searchExactword(word);
                        break;
                    case KEYWORD:
                        word = word.replace('\t', ' ');
                        word = word.replace((char)0x3000, ' ');
                        StringTokenizer st = new StringTokenizer(word);
                        int tokens = st.countTokens();
                        String[] key = new String[tokens];
                        for (int j=0; j<tokens; j++) {
                            key[j] = st.nextToken();
                        }
                        searcher = subs[i].searchKeyword(key);
                        break;
                    default:
                        return null;
                }
                for (int j=0; j<max; j++) {
                    Result result = searcher.getNextResult();
                    if (result == null) {
                        break;
                    }
                    if (_duplicateCheck(list, result)) {
                        j--;
                    } else {
                        list.add(result);
                    }
                }
                results[i] = (Result[])list.toArray(new Result[0]);
                list.clear();
            } catch (EBException e) {
                results[i] = new Result[0];
            }
        }
        return results;
    }

    /**
     * 複合検索を行います。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 検索対象の副本名
     * @param multiIndex 複合検索のインデックス
     * @param keys 検索キーワード
     * @return 検索結果
     */
    public Result[][] search(String host, String name, int multiIndex, String[] keys) {
        int idx = _getSubBookIndex(host, name);
        if (idx < 0 || !_subs[idx].hasMultiSearch()) {
            return null;
        }

        Result[][] results = new Result[1][];
        ArrayList list = new ArrayList(10);
        try {
            Searcher searcher = _subs[idx].searchMulti(multiIndex, keys);
            Result result = searcher.getNextResult();
            while (result != null) {
                if (!_duplicateCheck(list, result)) {
                    list.add(result);
                }
                result = searcher.getNextResult();
            }
            results[0] = (Result[])list.toArray(new Result[0]);
        } catch (EBException e) {
            results[0] = new Result[0];
        } catch (IllegalArgumentException e) {
            results[0] = new Result[0];
        }
        return results;
    }

    /**
     * 検索結果の重複をチェックします。
     *
     * @param list 検索結果のリスト
     * @param result 検索結果
     * @return 検索結果が重複している場合はtrue、そうでない場合はfalse
     */
    private boolean _duplicateCheck(List list, Result result) {
        long pos1 = result.getTextPosition();
        int size = list.size();
        for (int i=0; i<size; i++) {
            Result item = (Result)list.get(i);
            if (pos1 == item.getTextPosition()) {
                return true;
            }
        }
        return false;
    }

    /**
     * 見出しを返します。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 対象の副本名
     * @param result 検索結果
     * @return 見出し
     */
    public String getHeading(String host, String name, Result result) {
        return getHeading(host, name, result.getHeadingPosition());
    }

    /**
     * 見出しを返します。
     *
     * @param name 対象の副本名
     * @param host ホスト名またはIPアドレス
     * @param pos 見出しの位置
     * @return 見出し
     */
    public String getHeading(String host, String name, long pos) {
        int idx = _getSubBookIndex(host, name);
        if (idx < 0) {
            return "";
        }
        String heading = null;
        try {
            HTMLHook hook = _createHTMLHook(idx);
            hook.setForegroundColor(_config.getAnchorColor());
            hook.setInline(false);
            hook.setApplet(false);
            hook.setHeading(true);
            heading = (String)_subs[idx].getHeading(pos, hook);
        } catch (EBException e) {
        }
        if (heading == null) {
            heading = "";
        }
        return heading;
    }

    /**
     * 本文を返します。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 対象の副本名
     * @param result 検索結果
     * @param inline 画像のインライン表示有無
     * @param applet アプレットの使用有無
     * @return 本文
     */
    public String getText(String host, String name, Result result,
                          boolean inline, boolean applet) {
        return getText(host, name, result.getTextPosition(), inline, applet);
    }

    /**
     * 本文を返します。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 対象の副本名
     * @param pos 本文の位置
     * @param inline 画像のインライン表示有無
     * @param applet アプレットの使用有無
     * @return 本文
     */
    public String getText(String host, String name, long pos,
                          boolean inline, boolean applet) {
        int idx = _getSubBookIndex(host, name);
        if (idx < 0) {
            return "";
        }
        String text = null;
        try {
            HTMLHook hook = _createHTMLHook(idx);
            hook.setInline(inline);
            hook.setApplet(applet);
            hook.setHeading(false);
            text = (String)_subs[idx].getText(pos, hook);
        } catch (EBException e) {
        }
        if (text == null) {
            text = "";
        }
        return text;
    }

    /**
     * メニュー情報を返します。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 対象の副本名
     * @return メニュー情報
     */
    public String getMenu(String host, String name) {
        int idx = _getSubBookIndex(host, name);
        if (idx < 0) {
            return "";
        }
        String text = null;
        try {
            HTMLHook hook = _createHTMLHook(idx);
            hook.setInline(false);
            hook.setApplet(false);
            hook.setHeading(false);
            text = (String)_subs[idx].getMenu(hook);
        } catch (EBException e) {
        }
        if (text == null) {
            text = "";
        }
        return text;
    }

    /**
     * 著作権情報を返します。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 対象の副本名
     * @return 著作権情報
     */
    public String getCopyright(String host, String name) {
        int idx = _getSubBookIndex(host, name);
        if (idx < 0) {
            return "";
        }
        String text = null;
        try {
            HTMLHook hook = _createHTMLHook(idx);
            hook.setInline(true);
            hook.setApplet(false);
            hook.setHeading(false);
            text = (String)_subs[idx].getCopyright(hook);
        } catch (EBException e) {
        }
        if (text == null) {
            text = "";
        }
        return text;
    }

    /**
     * 候補一覧を返します。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 対象の副本名
     * @param multiIndex 複合検索のインデックス
     * @param entryIndex エントリのインデックス
     * @return 候補一覧
     */
    public String getCandidate(String host, String name,
                               int multiIndex, int entryIndex) {
        int idx = _getSubBookIndex(host, name);
        if (idx < 0) {
            return "";
        }
        String text = null;
        try {
            text = (String)_subs[idx].getCandidate(multiIndex,
                                                   entryIndex,
                                                   new ListupHook(_subs[idx]));
        } catch (EBException e) {
        } catch (IllegalArgumentException e) {
        }
        if (text == null) {
            text = "";
        }
        return text;
    }

    /**
     * 書籍情報を返します。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 対象の副本名
     * @return 書籍情報
     */
    public String getInfo(String host, String name) {
        int idx = _getSubBookIndex(host, name);
        if (idx < 0) {
            return "";
        }
        StringBuffer buf = new StringBuffer();
        Book book = _subs[idx].getBook();

        final String[] TRCLASS = {"odd", "even"};
        int tr = 0;

        buf.append("<TABLE class=\"main\">");
        buf.append("<TR class=\"header main\">");
        buf.append("<TH colspan=\"2\" class=\"header\">");
        buf.append(_subs[idx].getTitle());
        buf.append("</TH></TR>");

        // 書籍の種類
        buf.append("<TR class=\"").append(TRCLASS[tr]).append(" main\">");
        tr = (tr + 1) % 2;
        buf.append("<TD class=\"infoL\">書籍の種類：</TD>");
        String text = null;
        if (book.getBookType() == Book.DISC_EB) {
            text = "EB/EBG/EBXA/EBXA-C/S-EBXA";
        } else if (book.getBookType() == Book.DISC_EPWING) {
            text = "EPWING V" + book.getVersion();
        } else {
            text = "unknown";
        }
        buf.append("<TD class=\"infoR\">").append(text).append("</TD></TR>");

        // 文字セット
        buf.append("<TR class=\"").append(TRCLASS[tr]).append(" main\">");
        tr = (tr + 1) % 2;
        buf.append("<TD class=\"infoL\">文字セット：</TD>");
        switch (book.getCharCode()) {
            case Book.CHARCODE_ISO8859_1:
                text = "ISO 8859-1";
                break;
            case  Book.CHARCODE_JISX0208:
                text = "JIS X 0208";
                break;
            case Book.CHARCODE_JISX0208_GB2312:
                text = "JIS X 0208 + GB 2312";
                break;
            default:
                text = "unknown";
        }
        buf.append("<TD class=\"infoR\">").append(text).append("</TD></TR>");

        // 検索方式
        buf.append("<TR class=\"").append(TRCLASS[tr]).append(" main\">");
        tr = (tr + 1) % 2;
        buf.append("<TD class=\"infoL\" valign=\"top\">検索方式：</TD>");
        StringBuffer list = new StringBuffer();
        if (_subs[idx].hasWordSearch()) {
            list.append("前方一致検索<BR>");
        }
        if (_subs[idx].hasEndwordSearch()) {
            list.append("後方一致検索<BR>");
        }
        if (_subs[idx].hasExactwordSearch()) {
            list.append("完全一致検索<BR>");
        }
        if (_subs[idx].hasKeywordSearch()) {
            list.append("条件検索<BR>");
        }
        if (_subs[idx].hasMultiSearch()) {
            list.append("複合検索<BR>");
        }
        if (_subs[idx].hasMenu()) {
            list.append("メニュー<BR>");
        }
        if (_subs[idx].hasCopyright()) {
            list.append("著作権表示<BR>");
        }
        buf.append("<TD class=\"infoR\">").append(list).append("</TD></TR>");

        // 外字サイズ
        list.delete(0, list.length());
        for (int i=0; i<4; i++) {
            ExtFont font = _subs[idx].getFont(i);
            if (font.hasFont()) {
                list.append(Integer.toString(font.getFontHeight())).append(" ");
            }
        }
        if (list.length() > 0) {
            buf.append("<TR class=\"").append(TRCLASS[tr]).append(" main\">");
            tr = (tr + 1) % 2;
            buf.append("<TD class=\"infoL\" valign=\"top\">外字サイズ：</TD>");
            buf.append("<TD class=\"infoR\">").append(list).append("</TD></TR>");
        }

        // 半角外字
        ExtFont font = _subs[idx].getFont();
        if (font.hasNarrowFont()) {
            buf.append("<TR class=\"").append(TRCLASS[tr]).append(" main\">");
            tr = (tr + 1) % 2;
            buf.append("<TD class=\"infoL\" valign=\"top\">半角外字コード：</TD>");
            list.delete(0, list.length());
            list.append("0x");
            int code = font.getNarrowFontStart();
            String hex = Integer.toHexString(code).toUpperCase();
            int len = 4 - hex.length();
            if (len > 0) {
                for (int j=0; j<len; j++) {
                    list.append('0');
                }
            }
            list.append(hex).append(" 〜 0x");
            code = font.getNarrowFontEnd();
            hex = Integer.toHexString(code).toUpperCase();
            len = 4 - hex.length();
            if (len > 0) {
                for (int j=0; j<len; j++) {
                    list.append('0');
                }
            }
            list.append(hex);
            try {
                list.append(" (<A href=\"detail.jsp?view=font&book=");
                list.append(URLEncoder.encode(_subs[idx].getName(), "UTF-8"));
                list.append("&type=narrow\" target=\"detailFrame\">一覧</A>)");
            } catch (UnsupportedEncodingException e) {
            }
            buf.append("<TD class=\"infoR\">").append(list).append("</TD></TR>");
        }

        // 全角外字
        if (font.hasWideFont()) {
            buf.append("<TR class=\"").append(TRCLASS[tr]).append(" main\">");
            tr = (tr + 1) % 2;
            buf.append("<TD class=\"infoL\" valign=\"top\">全角外字コード：</TD>");
            list.delete(0, list.length());
            list.append("0x");
            int code = font.getWideFontStart();
            String hex = Integer.toHexString(code).toUpperCase();
            int len = 4 - hex.length();
            if (len > 0) {
                for (int j=0; j<len; j++) {
                    list.append('0');
                }
            }
            list.append(hex).append(" 〜 0x");
            code = font.getWideFontEnd();
            hex = Integer.toHexString(code).toUpperCase();
            len = 4 - hex.length();
            if (len > 0) {
                for (int j=0; j<len; j++) {
                    list.append('0');
                }
            }
            list.append(hex);
            try {
                list.append(" (<A href=\"detail.jsp?view=font&book=");
                list.append(URLEncoder.encode(_subs[idx].getName(), "UTF-8"));
                list.append("&type=wide\" target=\"detailFrame\">一覧</A>)");
            } catch (UnsupportedEncodingException e) {
            }
            buf.append("<TD class=\"infoR\">").append(list).append("</TD></TR>");
        }

        // 複合検索
        if (_subs[idx].hasMultiSearch()) {
            int count = _subs[idx].getMultiCount();
            for (int i=0; i<count; i++) {
                buf.append("<TR class=\"empty\"><TD colspan=\"2\"></TD></TR>");
                buf.append("<TR class=\"header main\">");
                buf.append("<TH colspan=\"2\" class=\"header\">");
                buf.append(_subs[idx].getMultiTitle(i));
                buf.append("</TH></TR>");

                int entryCount = _subs[idx].getMultiEntryCount(i);
                tr = 0;
                for (int j=0; j<entryCount; j++) {
                    buf.append("<TR class=\"").append(TRCLASS[tr]).append(" main\">");
                    tr = (tr + 1) % 2;
                    buf.append("<TD class=\"infoL\">");
                    buf.append("ラベル ").append(Integer.toString(j+1)).append("：<BR>");
                    buf.append("候補：");
                    buf.append("</TD>");
                    buf.append("<TD class=\"infoR\">");
                    buf.append(_subs[idx].getMultiEntryLabel(i, j)).append("<BR>");
                    if (_subs[idx].hasMultiEntryCandidate(i, j)) {
                        buf.append("有");
                    } else {
                        buf.append("無");
                    }
                    buf.append("</TD></TR>");
                }
            }
        }

        buf.append("</TABLE>");
        return buf.toString();
    }

    /**
     * 外字リストを返します。
     *
     * @param host ホスト名またはIPアドレス
     * @param name 対象の副本名
     * @param type 0:半角/1:全角
     * @return 外字リスト
     */
    public String getFontList(String host, String name, int type) {
        int idx = _getSubBookIndex(host, name);
        if (idx < 0) {
            return "";
        }
        StringBuffer buf = new StringBuffer();

        // 外字情報の取得
        ExtFont font = _subs[idx].getFont();
        int start = -1;
        int end = -1;
        boolean narrow = true;
        if (type == 0) {
            if (font.hasNarrowFont()) {
                narrow = true;
                start = font.getNarrowFontStart();
                end = font.getNarrowFontEnd();
            }
        } else {
            if (font.hasWideFont()) {
                narrow = false;
                start = font.getWideFontStart();
                end = font.getWideFontEnd();
            }
        }
        if (start == -1 || end == -1) {
            return buf.toString();
        }

        String src = RES_DIR + _subs[idx].getName() + "/";
        String prefix = null;
        if (narrow) {
            prefix = "N";
        } else {
            prefix = "W";
        }
        prefix += Integer.toString(_subs[idx].getFont().getFontHeight());

        buf.append("<TABLE class=\"main\">");
        // タイトル
        String text = null;
        if (type == 0) {
            text = "半角";
        } else {
            text = "全角";
        }
        buf.append("<TR class=\"header main\">");
        buf.append("<TH colspan=\"8\" class=\"header\">");
        buf.append(text).append("外字一覧 (");
        buf.append(_subs[idx].getTitle()).append(")");
        buf.append("</TH></TR>");

        int charcode = _subs[idx].getBook().getCharCode();
        int code = start & 0xfff8;
        int fore = _config.getForegroundColor().getRGB() | 0xff000000;
        int back = _config.getBackgroundColor().getRGB() | 0xff000000;
        String height = Integer.toString(_subs[idx].getFont().getFontHeight());
        String width = null;
        if (type == 0) {
            width = Integer.toString(_subs[idx].getFont().getNarrowFontWidth());
        } else {
            width = Integer.toString(_subs[idx].getFont().getWideFontWidth());
        }
        String alt;
        for (int i=code; i<=end; i+=8) {
            if (charcode != Book.CHARCODE_ISO8859_1
                && ((i & 0xff) < 0x20 || (i & 0xff) > 0x7f)) {
                continue;
            }
            buf.append("<TR class=\"odd main\">");
            for (int j=0; j<8; j++) {
                buf.append("<TD class=\"font1\">");
                code = i + j;
                if (code > end) {
                    alt = null;
                } else {
                    if (charcode == Book.CHARCODE_ISO8859_1
                        && ((code & 0xff) < 0x01 || (code & 0xff) > 0xfe)) {
                        alt = null;
                    } else if (charcode != Book.CHARCODE_ISO8859_1
                               && ((code & 0xff) < 0x21 || (code & 0xff) > 0x7e)) {
                        alt = null;
                    } else {
                        // 出力ファイル名
                        alt = prefix + "-" + Integer.toHexString(code).toUpperCase();
                    }
                }

                if (alt != null) {
                    buf.append("<IMG class=\"gaiji\" src=\"");
                    buf.append(src).append(alt);
                    buf.append("_F-");
                    buf.append(Integer.toHexString(fore).substring(2).toUpperCase());
                    buf.append("_B-");
                    buf.append(Integer.toHexString(back).substring(2).toUpperCase());
                    buf.append(".png\"");
                    buf.append(" width=\"").append(width).append("\"");
                    buf.append(" height=\"").append(height).append("\"");
                    buf.append(" alt=\"[").append(alt).append("]\">");
                }
                buf.append("</TD>");
            }
            buf.append("</TR>");
            buf.append("<TR class=\"even main\">");
            for (int j=0; j<8; j++) {
                buf.append("<TD class=\"font2\">");
                code = i + j;
                if (code <= end) {
                    if (charcode == Book.CHARCODE_ISO8859_1
                        && ((code & 0xff) < 0x01 || (code & 0xff) > 0xfe)) {
                    } else if (charcode != Book.CHARCODE_ISO8859_1
                               && ((code & 0xff) < 0x21 || (code & 0xff) > 0x7e)) {
                    } else {
                        // コード
                        buf.append("0x").append(Integer.toHexString(code).toUpperCase());
                    }
                }
                buf.append("</TD>");
            }
            buf.append("</TR>");
        }

        buf.append("</TABLE>");
        return buf.toString();
    }
}

// end of WebBook.java
