package fuku.eb4j.tool;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.text.*;
import java.util.*;
import java.util.zip.*;

import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;

import fuku.eb4j.Book;
import fuku.eb4j.SubBook;
import fuku.eb4j.ExtFont;
import fuku.eb4j.EBException;
import fuku.eb4j.Version;
import fuku.eb4j.io.EBFile;
import fuku.eb4j.io.BookInputStream;
import fuku.eb4j.io.EBZipInputStream;
import fuku.eb4j.io.EBZipConstants;
import fuku.eb4j.util.ByteUtil;

/**
 * Ҥΰ/ĥץࡣ
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.3
 */
public final class EBZip implements EBZipConstants {

    /** ץ֥̾ */
    private static final String _PROGRAM = "fuku.eb4j.tool.EBZip";

    /** ǥեɤ߹ߥǥ쥯ȥ */
    private static final String DEFAULT_BOOK_DIR = ".";
    /** ǥեȽϥǥ쥯ȥ */
    private static final String DEFAULT_OUTPUT_DIR = ".";

    /** 񤭶ػߥ⡼*/
    private static final int OVERWRITE_NO = 0;
    /** 񤭻礻⡼*/
    private static final int OVERWRITE_QUERY = 1;
    /** 񤭥⡼*/
    private static final int OVERWRITE_FORCE = 2;

    /** ̥⡼ */
    private static final int ACTION_ZIP = 0;
    /** ⡼ */
    private static final int ACTION_UNZIP = 1;
    /** ⡼ */
    private static final int ACTION_INFO = 2;

    /** Ψɽѥեޥå */
    private static final DecimalFormat FMT = new DecimalFormat("##0.0'%'");

    /** ޥɥ饤󥪥ץ */
    private static final LongOpt[] LONGOPT = {
        new LongOpt("force-overwrite", LongOpt.NO_ARGUMENT, null, 'f'),
        new LongOpt("no-overwrite", LongOpt.NO_ARGUMENT, null, 'n'),
        new LongOpt("information", LongOpt.NO_ARGUMENT, null, 'i'),
        new LongOpt("keep", LongOpt.NO_ARGUMENT, null, 'k'),
        new LongOpt("level", LongOpt.REQUIRED_ARGUMENT, null, 'l'),
        new LongOpt("gzip", LongOpt.REQUIRED_ARGUMENT, null, 'g'),
        new LongOpt("outpur-directory", LongOpt.REQUIRED_ARGUMENT, null, 'o'),
        new LongOpt("quiet", LongOpt.NO_ARGUMENT, null, 'q'),
        new LongOpt("skip-content", LongOpt.REQUIRED_ARGUMENT, null, 's'),
        new LongOpt("subbook", LongOpt.REQUIRED_ARGUMENT, null, 'S'),
        new LongOpt("compress", LongOpt.NO_ARGUMENT, null, 'z'),
        new LongOpt("uncompress", LongOpt.NO_ARGUMENT, null, 'u'),
        new LongOpt("test", LongOpt.NO_ARGUMENT, null, 't'),
        new LongOpt("help", LongOpt.NO_ARGUMENT, null, 'h'),
        new LongOpt("version", LongOpt.NO_ARGUMENT, null, 'v'),
    };


    /** ɤ߹ߥǥ쥯ȥ */
    private static String _bookDir = DEFAULT_BOOK_DIR;
    /** ǥ쥯ȥ */
    private static String _outDir = DEFAULT_OUTPUT_DIR;
    /** оܤΥꥹ */
    private static String[] _subbooks = null;
    /** ˡ */
    private static int _overwrite = OVERWRITE_QUERY;
    /** EBZIP̥٥ */
    private static int _level = EBZIP_DEFAULT_LEVEL;
    /** GZIP̥٥ */
    private static int _gzip = GZIP_DEFAULT_LEVEL;
    /** ꥸʥեݻե饰 */
    private static boolean _keep = false;
    /** ϥå޻ߥե饰 */
    private static boolean _quiet = false;
    /** ̥ƥȥ⡼ɥե饰 */
    private static boolean _test = false;
    /** ̵ե饰 */
    private static boolean _skipFont = false;
    /** ̵ե饰 */
    private static boolean _skipSound = false;
    /** ̵ե饰 */
    private static boolean _skipGraphic = false;
    /** ư̵ե饰 */
    private static boolean _skipMovie = false;


    /**
     * ᥤ᥽åɡ
     *
     * @param args ޥɹ԰
     */
    public static void main(String[] args) {
        Getopt g = new Getopt(_PROGRAM, args, "fnikl:g:o:qs:S:zuthv", LONGOPT);
        int action = ACTION_ZIP;
        StringTokenizer st = null;
        List list = new ArrayList(4);
        int c;
        while ((c=g.getopt()) != -1) {
            switch (c) {
                case 'f':
                    _overwrite = OVERWRITE_FORCE;
                    break;
                case 'n':
                    _overwrite = OVERWRITE_NO;
                    break;
                case 'i':
                    action = ACTION_INFO;
                    break;
                case 'k':
                    _keep = true;
                    break;
                case 'l':
                    String level = g.getOptarg();
                    try {
                        _level = Integer.parseInt(level);
                    } catch (NumberFormatException e) {
                        System.err.println(_PROGRAM
                                           + ": invalid compression level `"
                                           + level + "'");
                        System.exit(1);
                    }
                    if (_level > EBZIP_MAX_LEVEL || _level < 0) {
                        System.err.println(_PROGRAM
                                           + ": invalid compression level `"
                                           + level + "'");
                        System.exit(1);
                    }
                    break;
                case 'g':
                    String gzip = g.getOptarg();
                    try {
                        _gzip = Integer.parseInt(gzip);
                    } catch (NumberFormatException e) {
                        System.err.println(_PROGRAM
                                           + ": invalid gzip level `"
                                           + gzip + "'");
                        System.exit(1);
                    }
                    if (_gzip > GZIP_MAX_LEVEL || _gzip < 0) {
                        System.err.println(_PROGRAM
                                           + ": invalid gzip level `"
                                           + gzip + "'");
                        System.exit(1);
                    }
                    break;
                case 'o':
                    _outDir = g.getOptarg();
                    break;
                case 'q':
                    _quiet = true;
                    break;
                case 's':
                    st = new StringTokenizer(g.getOptarg(), ",");
                    while (st.hasMoreTokens()) {
                        String skip = st.nextToken().trim().toLowerCase();
                        if (skip.equals("font")) {
                            _skipFont = true;
                        } else if (skip.equals("sound")) {
                            _skipSound = true;
                        } else if (skip.equals("graphic")) {
                            _skipGraphic = true;
                        } else if (skip.equals("movie")) {
                            _skipMovie = true;
                        } else {
                            System.err.println(_PROGRAM
                                               + ": invalid content name `"
                                               + skip + "'");
                            System.exit(1);
                        }
                    }
                    break;
                case 'S':
                    st = new StringTokenizer(g.getOptarg(), ",");
                    while (st.hasMoreTokens()) {
                        list.add(st.nextToken().trim().toLowerCase());
                    }
                    break;
                case 'z':
                    action = ACTION_ZIP;
                    break;
                case 'u':
                    action = ACTION_UNZIP;
                    break;
                case 't':
                    _test = true;
                    break;
                case 'h':
                    _usage(0);
                    break;
                case 'v':
                    _version();
                    break;
                default:
                    _usage(1);
            }
        }
        if (!list.isEmpty()) {
            _subbooks = (String[])list.toArray(new String[0]);
        }

        int idx = g.getOptind();
        if (idx+1 == args.length) {
            _bookDir = args[idx];
        } else if (idx+1 < args.length) {
            System.err.println(_PROGRAM + ": too many arguments");
            _usage(1);
        }

        EBZip ebzip = new EBZip();
        try {
            ebzip._exec(action);
        } catch (EBException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        }
    }


    /**
     * ˡɽޤ
     *
     * @param status λơ
     */
    private static void _usage(int status) {
        if (status != 0) {
            System.out.println("Try `java " + _PROGRAM + " --help' for more information");
        } else {
            System.out.println("Usage: java " + _PROGRAM + " [option...] [book-directory]");
            System.out.println("");
            System.out.println("Options:");
            System.out.println("  -f, --force-overwrite      force overwrite of output files");
            System.out.println("  -n, --no-overwrite         don't overwrite output files");
            System.out.println("  -i, --information          list information of compressed files");
            System.out.println("  -k, --keep                 keep (don't delete) original files");
            System.out.println("  -l INTEGER, --level INTEGER");
            System.out.println("                             compression level; 0.." + EBZIP_MAX_LEVEL);
            System.out.println("                             (default: " + EBZIP_DEFAULT_LEVEL + ")");
            System.out.println("  -g INTEGER, --gzip INTEGER");
            System.out.println("                             gzip level; 0.." + GZIP_MAX_LEVEL);
            System.out.println("                             (default: " + GZIP_DEFAULT_LEVEL + ")");
            System.out.println("  -o DIRECTORY, --output-directory DIRECTORY");
            System.out.println("                             output files under DIRECTORY");
            System.out.println("                             (default: " + DEFAULT_OUTPUT_DIR + ")");
            System.out.println("  -q, --quiet                suppress all warnings");
            System.out.println("  -s TYPE[,TYPE], --skip-content TYPE[,TYPE]");
            System.out.println("                             skip content; font, graphic, sound or movie");
            System.out.println("                             (default: none is skipped)");
            System.out.println("  -S SUBBOOK[,SUBBOOK], --subbook SUBBOOK[,SUBBOOK]");
            System.out.println("                             target subbook");
            System.out.println("                             (default: all subbook)");
            System.out.println("  -z, --compress             compress files");
            System.out.println("  -u, --uncompress           uncompress files");
            System.out.println("  -t, --test                 only check for input files");
            System.out.println("  -h, --help                 display this help and exit");
            System.out.println("  -v, --version              output version information and exit");
            System.out.println("");
            System.out.println("Argument:");
            System.out.println("  book-directory             top directory of a book");
            System.out.println("                             (default: " + DEFAULT_BOOK_DIR + ")");
            System.out.println("");
            System.out.println("Report bugs to <" + Version.EMAIL + ">.");
        }
        System.exit(status);
    }

    /**
     * Сɽޤ
     *
     */
    private static void _version() {
        System.out.println(_PROGRAM + " " + Version.VERSION);
        System.out.println(Version.COPYRIGHT);
        System.out.println("All right reserved.");
        System.exit(0);
    }


    /**
     * 󥹥ȥ饯
     *
     */
    public EBZip() {
        super();
    }


    /**
     * ޥɤ¹Ԥޤ
     *
     * @param action ޥɤư
     * @exception EBException Ҥν㳰ȯ
     */
    private void _exec(int action) throws EBException {
        Book book = new Book(_bookDir);
        File root = new File(_bookDir);
        SubBook[] sub = book.getSubBooks();
        EBFile file = null;
        for (int i=0; i<sub.length; i++) {
            if (_subbooks != null) {
                boolean show = false;
                String dir = sub[i].getName().toLowerCase();
                for (int j=0; j<_subbooks.length; j++) {
                    if (_subbooks[j].equals(dir)) {
                        show = true;
                        break;
                    }
                }
                if (!show) {
                    continue;
                }
            }
            if (book.getBookType() == Book.DISC_EB) {
                file = sub[i].getTextFile();
                _act(action, file);
                if (action == ACTION_UNZIP
                    && file.getFormat() == EBFile.FORMAT_SEBXA) {
                    // SEBXḀե饰κ
                    _fixSEBXA(_getOutFile(file, ".org"));
                }
            } else {
                // ʸե
                file = sub[i].getTextFile();
                _act(action, file);
                if (file.getName().compareToIgnoreCase("honmon2") == 0) {
                    // ե
                    if (!_skipSound) {
                        file = sub[i].getSoundFile();
                        if (file != null) {
                            _act(action, file);
                        }
                    }
                    if (!_skipGraphic) {
                        file = sub[i].getGraphicFile();
                        if (file != null) {
                            if (action == ACTION_ZIP) {
                                _copy(file);
                            } else {
                                _act(action, file);
                            }
                        }
                    }
                }
                // ե
                if (!_skipFont) {
                    for (int j=0; j<4; j++) {
                        ExtFont font = sub[i].getFont(j);
                        if (font.hasWideFont()) {
                            file = font.getWideFontFile();
                            _act(action, file);
                        }
                        if (font.hasNarrowFont()) {
                            file = font.getNarrowFontFile();
                            _act(action, file);
                        }
                    }
                }
                // ưե
                if (!_skipMovie && action != ACTION_INFO) {
                    File[] files = sub[i].getMovieFileList();
                    if (files != null) {
                        for (int j=0; j<files.length; j++) {
                            _copy(_getOutFile(files[j], null), files[j]);
                        }
                    }
                }
            }
        }
        if (book.getBookType() == Book.DISC_EB) {
            try {
                file = new EBFile(root, "language", EBFile.FORMAT_PLAIN);
                _act(action, file);
            } catch (EBException e) {
            }
            file = new EBFile(root, "catalog", EBFile.FORMAT_PLAIN);
            if (action == ACTION_ZIP) {
                _copy(file);
            } else {
                _act(action, file);
            }
        } else {
            file = new EBFile(root, "catalogs", EBFile.FORMAT_PLAIN);
            if (action == ACTION_ZIP) {
                _copy(file);
            } else {
                _act(action, file);
            }
        }
    }

    /**
     * ꤵ줿¹Ԥޤ
     *
     * @param action 
     * @param file ե
     */
    private void _act(int action, EBFile file) {
        switch (action) {
            case ACTION_ZIP:
                _zip(file);
                break;
            case ACTION_UNZIP:
                _unzip(file);
                break;
            case ACTION_INFO:
                _info(file);
                break;
            default:
        }
    }

    /**
     * ꤵ줿ե򰵽̤ޤ
     *
     * @param file ե
     */
    private void _zip(EBFile file) {
        if (!_test) {
            _mkdir(file);
        }

        File f = _getOutFile(file, ".ebz");
        if (!_quiet) {
            // ե̾ν
            System.out.println("==> compress " + file.getPath() + " <==");
            System.out.println("output to " + f.getPath());
        }

        if (f.equals(file.getFile())) {
            if (!_quiet) {
                System.out.println("the input and output files are the same, skipped.");
                System.out.println("");
            }
            return;
        }

        if (!_test) {
            if (!_isOverwrite(f)) {
                return;
            }
        }

        BookInputStream bis = null;
        FileChannel channel = null;
        try {
            bis = file.getInputStream();

            int sliceSize = BookInputStream.PAGE_SIZE << _level;
            long fileSize = bis.getFileSize();

            int indexSize = 0;
            if (fileSize < (1<<16)) {
                indexSize = 2;
            } else if (fileSize < (1<<24)) {
                indexSize = 3;
            } else {
                indexSize = 4;
            }

            /*
             * Original File:
             *   +-----------------+-----------------+-...-+-------+
             *   |     slice 1     |     slice 2     |     |slice N| [EOF]
             *   |                 |                 |     |       |
             *   +-----------------+-----------------+-...-+-------+
             *        slice size        slice size            odds
             *   <-------------------- file size ------------------>
             *
             * Compressed file:
             *   +------+---------+...+---------+---------+----------+...+-
             *   |Header|index for|   |index for|index for|compressed|   |
             *   |      | slice 1 |   | slice N |   EOF   |  slice 1 |   |
             *   +------+---------+...+---------+---------+----------+...+-
             *             index         index     index
             *             size          size      size
             *          <---------  index length --------->
             *
             *     total_slices = N = (file_size + slice_size - 1) / slice_size
             *     index_length = (N + 1) * index_size
             */
            int totalSlice = (int)((fileSize + sliceSize - 1) / sliceSize);
            long indexLength = (totalSlice + 1) * indexSize;

            byte[] in = new byte[sliceSize];
            byte[] out = new byte[sliceSize+1024];
            long slicePos = EBZIP_HEADER_SIZE + indexLength;

            // إåȥǥåΥߡǡ񤭹
            if (!_test) {
                channel = new FileOutputStream(f).getChannel();
                Arrays.fill(out, 0, out.length, (byte)0);
                long i;
                for (i=slicePos; i>=sliceSize; i=i-sliceSize) {
                    channel.write(ByteBuffer.wrap(out, 0, sliceSize));
                }
                if (i > 0) {
                    channel.write(ByteBuffer.wrap(out, 0, (int)i));
                }
            }

            long inTotalLength = 0;
            long outTotalLength = 0;
            int interval = 1024 >>> _level;
            Adler32 crc32 = new Adler32();
            Deflater def = new Deflater(_gzip);
            for (int i=0; i<totalSlice; i++) {
                // 饤ǡɤ߹
                bis.seek(inTotalLength);
                int inLen = bis.read(in, 0, in.length);
                if (inLen < 0) {
                    System.err.println(_PROGRAM
                                       + ": failed to read the file: "
                                       + f.getPath());
                    return;
                } else if (inLen == 0) {
                    System.err.println(_PROGRAM + ": unexpected EOF: "
                                       + f.getPath());
                    return;
                } else if (inLen != in.length
                           && inTotalLength + inLen != fileSize) {
                    System.err.println(_PROGRAM + ": unexpected EOF: "
                                       + f.getPath());
                    return;
                }

                // CRCι
                crc32.update(in, 0, inLen);

                // ǽ饤ǥ饤ʤ0
                if (inLen < sliceSize) {
                    Arrays.fill(in, inLen, in.length, (byte)0);
                    inLen = sliceSize;
                }
                // 饤򰵽
                def.reset();
                def.setInput(in, 0, inLen);
                def.finish();
                int outLen = 0;
                while (!def.needsInput()) {
                    int n = def.deflate(out, outLen, out.length-outLen);
                    outLen += n;
                }
                // ̥饤ꥸʥ礭ϥꥸʥ񤭹
                if (outLen >= sliceSize) {
                    System.arraycopy(in, 0, out, 0, sliceSize);
                    outLen = sliceSize;
                }

                // ̤饤ǡν񤭹
                if (!_test) {
                    // եɲ
                    channel.position(channel.size());
                    channel.write(ByteBuffer.wrap(out, 0, outLen));
                }

                // ǥåκ
                long nextPos = slicePos + outLen;
                switch (indexSize) {
                    case 2:
                        out[0] = (byte)((slicePos >>> 8) & 0xff);
                        out[1] = (byte)(slicePos & 0xff);
                        out[2] = (byte)((nextPos >>> 8) & 0xff);
                        out[3] = (byte)(nextPos & 0xff);
                        break;
                    case 3:
                        out[0] = (byte)((slicePos >>> 16) & 0xff);
                        out[1] = (byte)((slicePos >>> 8) & 0xff);
                        out[2] = (byte)(slicePos & 0xff);
                        out[3] = (byte)((nextPos >>> 16) & 0xff);
                        out[4] = (byte)((nextPos >>> 8) & 0xff);
                        out[5] = (byte)(nextPos & 0xff);
                        break;
                    case 4:
                        out[0] = (byte)((slicePos >>> 24) & 0xff);
                        out[1] = (byte)((slicePos >>> 16) & 0xff);
                        out[2] = (byte)((slicePos >>> 8) & 0xff);
                        out[3] = (byte)(slicePos & 0xff);
                        out[4] = (byte)((nextPos >>> 24) & 0xff);
                        out[5] = (byte)((nextPos >>> 16) & 0xff);
                        out[6] = (byte)((nextPos >>> 8) & 0xff);
                        out[7] = (byte)(nextPos & 0xff);
                        break;
                    default:
                }

                // ǥåν񤭹
                if (!_test) {
                    channel.position(EBZIP_HEADER_SIZE+i*indexSize);
                    channel.write(ByteBuffer.wrap(out, 0, indexSize*2));
                }

                inTotalLength += inLen;
                outTotalLength += outLen + indexSize;
                slicePos = nextPos;

                // Ľɽ
                if (!_quiet && (i%interval) + 1 == interval) {
                    double rate = (double)(i + 1) / (double)totalSlice * 100.0;
                    System.out.println(FMT.format(rate) + " done ("
                                       + inTotalLength + " / "
                                       + fileSize + " bytes)");
                }
            }
            def.end();

            // إåκ
            out[0] = (byte)'E';
            out[1] = (byte)'B';
            out[2] = (byte)'Z';
            out[3] = (byte)'i';
            out[4] = (byte)'p';
            out[5] = (byte)((EBFile.FORMAT_EBZIP << 4) | (_level & 0x0f));
            out[6] = (byte)0;
            out[7] = (byte)0;
            out[8] = (byte)0;
            out[9] = (byte)0;
            out[10] = (byte)((fileSize >>> 24) & 0xff);
            out[11] = (byte)((fileSize >>> 16) & 0xff);
            out[12] = (byte)((fileSize >>> 8) & 0xff);
            out[13] = (byte)(fileSize & 0xff);
            long crc = crc32.getValue();
            out[14] = (byte)((crc >>> 24) & 0xff);
            out[15] = (byte)((crc >>> 16) & 0xff);
            out[16] = (byte)((crc >>> 8) & 0xff);
            out[17] = (byte)(crc & 0xff);
            long mtime = System.currentTimeMillis();
            out[18] = (byte)((mtime >>> 24) & 0xff);
            out[19] = (byte)((mtime >>> 16) & 0xff);
            out[20] = (byte)((mtime >>> 8) & 0xff);
            out[21] = (byte)(mtime & 0xff);

            // إåν񤭹
            if (!_test) {
                channel.position(0);
                channel.write(ByteBuffer.wrap(out, 0, EBZIP_HEADER_SIZE));
            }

            // ̤ɽ
            outTotalLength += EBZIP_HEADER_SIZE + indexSize;
            if (!_quiet) {
                System.out.println("completed (" + fileSize
                                   + " / " + fileSize + " bytes)");
                if (inTotalLength != 0) {
                    double rate = (double)(outTotalLength) / (double)bis.getRealFileSize() * 100.0;
                    System.out.println(bis.getRealFileSize() + " -> "
                                       + outTotalLength + " bytes ("
                                       + FMT.format(rate) + ")");
                }
            }
        } catch (EBException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        } catch (FileNotFoundException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        } catch (IOException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        } catch (SecurityException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        } finally {
            bis.close();
            if (channel != null) {
                try {
                    channel.close();
                } catch (IOException e) {
                }
            }
        }
        // ꥸʥեκ
        if (!_test && !_keep) {
            _delete(file.getFile());
        }
        if (!_quiet) {
            System.out.println("");
        }
    }

    /**
     * ꤵ줿եषޤ
     *
     * @param file ե
     */
    private void _unzip(EBFile file) {
        // ̵̥եϤΤޤޥԡ
        if (file.getFormat() == EBFile.FORMAT_PLAIN) {
            _copy(file);
            return;
        }

        if (!_test) {
            _mkdir(file);
        }

        String suffix = null;
        if (file.getFormat() != EBFile.FORMAT_EBZIP) {
            suffix = ".org";
        }
        File f = _getOutFile(file, suffix);
        if (!_quiet) {
            // ե̾ν
            System.out.println("==> uncompress " + file.getPath() + " <==");
            System.out.println("output to " + f.getPath());
        }

        if (f.equals(file.getFile())) {
            if (!_quiet) {
                System.out.println("the input and output files are the same, skipped.");
                System.out.println("");
            }
            return;
        }

        if (!_test) {
            if (!_isOverwrite(f)) {
                return;
            }
        }

        BookInputStream bis = null;
        FileChannel channel = null;
        try {
            bis = file.getInputStream();
            byte[] b = new byte[bis.getSliceSize()];
            if (!_test) {
                channel = new FileOutputStream(f).getChannel();
            }
            long totalLength = 0;
            int totalSlice = (int)((bis.getFileSize()
                                    + bis.getSliceSize() - 1)
                                   / bis.getSliceSize());
            Adler32 crc32 = new Adler32();
            for (int i=0; i<totalSlice; i++) {
                // ǡɤ߹
                bis.seek(totalLength);
                int n = bis.read(b, 0, b.length);
                if (n < 0) {
                    System.err.println(_PROGRAM
                                       + ": failed to read the file: "
                                       + f.getPath());
                    return;
                } else if (n == 0) {
                    System.err.println(_PROGRAM + ": unexpected EOF: "
                                       + f.getPath());
                    return;
                } else if (n != b.length
                           && totalLength + n != bis.getFileSize()) {
                    System.err.println(_PROGRAM + ": unexpected EOF: "
                                       + f.getPath());
                    return;
                }
                // CRCι
                if (bis instanceof EBZipInputStream) {
                    crc32.update(b, 0, n);
                }
                // ǡν񤭹
                if (!_test) {
                    channel.write(ByteBuffer.wrap(b, 0, n));
                }
                totalLength += n;

                // Ľɽ
                if (!_quiet && (i%1024) + 1 == 1024) {
                    double rate = (double)(i + 1) / (double)totalSlice * 100.0;
                    System.out.println(FMT.format(rate) + " done ("
                                       + totalLength + " / "
                                       + bis.getFileSize() + " bytes)");
                }
            }
            // ̤ɽ
            if (!_quiet) {
                System.out.println("completed (" + bis.getFileSize()
                                   + " / " + bis.getFileSize() + " bytes)");
                System.out.println(bis.getRealFileSize() + " -> "
                                   + totalLength + " bytes");
            }

            // CRCγǧ
            if (bis instanceof EBZipInputStream) {
                if (crc32.getValue() != ((EBZipInputStream)bis).getCRC()) {
                    System.err.println(_PROGRAM + ": CRC error: " + f.getPath());
                    return;
                }
            }
        } catch (EBException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        } catch (FileNotFoundException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        } catch (IOException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        } catch (SecurityException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        } finally {
            bis.close();
            if (channel != null) {
                try {
                    channel.close();
                } catch (IOException e) {
                }
            }
        }
        // ꥸʥեκ
        if (!_test && !_keep) {
            _delete(file.getFile());
        }
        if (!_quiet) {
            System.out.println("");
        }
    }

    /**
     * ꤵ줿եξϤޤ
     *
     * @param file ե
     */
    private void _info(EBFile file) {
        // ե̾ν
        System.out.println("==> " + file.getPath() + " <==");
        BookInputStream bis = null;
        try {
            bis = file.getInputStream();
        } catch (EBException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
            System.out.println("");
            return;
        } finally {
            bis.close();
        }

        // ե륵Ψν
        StringBuffer buf = new StringBuffer();
        String text = null;
        switch (file.getFormat()) {
            case EBFile.FORMAT_PLAIN:
                buf.append(bis.getFileSize());
                buf.append(" bytes (not compressed)");
                break;
            case EBFile.FORMAT_EBZIP:
                int level = ((EBZipInputStream)bis).getLevel();
                text = "ebzip level " + level + " compression)";
                break;
            case EBFile.FORMAT_SEBXA:
                text = "S-EBXA compression)";
                break;
            default:
                text = "EPWING compression)";
        }
        if (text != null) {
            long size = bis.getFileSize();
            long real = bis.getRealFileSize();
            buf.append(Long.toString(size)).append(" -> ");
            buf.append(Long.toString(real)).append(" bytes (");
            if (size == 0) {
                System.out.print("empty original file, ");
            } else {
                double rate = (double)real / (double)size * 100.0;
                buf.append(FMT.format(rate));
                buf.append(", ");
            }
            buf.append(text);
        }
        System.out.println(buf.toString());
        System.out.println("");
    }

    /**
     * ꤵ줿ե뤫S-EBXA̾ޤ
     *
     * @param file ե
     */
    private void _fixSEBXA(File file) {
        if (!_quiet) {
            System.out.println("==> fix " + file.getPath() + " <==");
        }

        FileChannel channel = null;
        boolean err = false;
        try {
            channel = new RandomAccessFile(file, "rw").getChannel();

            // ǥåڡ˥ޥåԥ
            MappedByteBuffer buf = channel.map(FileChannel.MapMode.READ_WRITE,
                                               0, BookInputStream.PAGE_SIZE);

            // 0x12/0x22Υǥåμ
            int indexCount = buf.get(1) & 0xff;
            int removeCount = 0;
            int inOff = 16;
            int outOff = 16;
            for (int i=0; i<indexCount; i++) {
                int index = buf.get(inOff) & 0xff;
                if (index == 0x21 || index == 0x22) {
                    removeCount++;
                } else {
                    if (inOff != outOff) {
                        for (int j=0; j<16; j++) {
                            buf.put(outOff+j, buf.get(inOff+j));
                        }
                    }
                    outOff += 16;
                }
                inOff += 16;
            }
            for (int i=0; i<removeCount; i++) {
                for (int j=0; j<16; j++) {
                    buf.put(outOff+j, (byte)0);
                }
                outOff += 16;
            }
            buf.force();
        } catch (FileNotFoundException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
            err = true;
        } catch (IOException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
            err = true;
        } catch (SecurityException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
            err = true;
        } finally {
            if (channel != null) {
                try {
                    channel.close();
                } catch (IOException e) {
                }
            }
        }
        if (!_quiet) {
            if (!err) {
                System.out.println("complated");
            }
            System.out.println("");
        }
    }

    /**
     * եФϥե֤ޤ
     *
     * @param file ե
     * @param suffix ĥ
     * @return ϥե
     */
    private File _getOutFile(EBFile file, String suffix) {
        return _getOutFile(file.getFile(), suffix);
    }

    /**
     * եФϥե֤ޤ
     *
     * @param file ե
     * @param suffix ĥ
     * @return ϥե
     */
    private File _getOutFile(File file, String suffix) {
        String bookDir = null;
        String inFile = null;
        try {
            bookDir = new File(_bookDir).getCanonicalPath();
            inFile = file.getCanonicalPath();
        } catch (IOException e) {
            System.err.println(_PROGRAM + ": can't get canonical path");
            System.exit(1);
        }

        String fname = inFile.substring(bookDir.length());
        if (fname.length() > 4) {
            String s = fname.substring(fname.length()-4);
            if (s.compareToIgnoreCase(".ebz") == 0) {
                fname = fname.substring(0, fname.length()-4);
            } else if (s.compareToIgnoreCase(".org") == 0) {
                fname = fname.substring(0, fname.length()-4);
            }
        }
        if (suffix != null) {
            fname += suffix;
        }
        return new File(_outDir, fname);
    }

    /**
     * եνǥ쥯ȥޤ
     *
     * @param file ե
     */
    private void _mkdir(EBFile file) {
        _mkdir(_getOutFile(file, null));
    }

    /**
     * եνǥ쥯ȥޤ
     *
     * @param file ե
     */
    private void _mkdir(File file) {
        File dir = file.getParentFile();
        if (!dir.exists()) {
            try {
                dir.mkdirs();
            } catch (SecurityException e) {
                System.err.println(_PROGRAM + ": can't create directory ("
                                   + e.getMessage() + ")");
                System.exit(1);
            }
        }
    }

    /**
     * ե˥ԡޤ
     *
     * @param file ե
     */
    private void _copy(EBFile file) {
        _copy(file.getFile(), _getOutFile(file, null));
    }

    /**
     * ϥեϥե˥ԡޤ
     *
     * @param file1 ϥե
     * @param file2 ϥե
     */
    private void _copy(File file1, File file2) {
        if (!_test) {
            _mkdir(file2);
        }
        if (!_quiet) {
            // ե̾ν
            System.out.println("==> copy " + file1.getPath() + " <==");
            System.out.println("output to " + file2.getPath());
        }

        if (file1.equals(file2)) {
            if (!_quiet) {
                System.out.println("the input and output files are the same, skipped.");
                System.out.println("");
            }
            return;
        }

        if (_test) {
            if (!_quiet) {
                System.out.println("");
            }
            return;
        }

        if (!_isOverwrite(file2)) {
            return;
        }

        FileChannel in = null;
        FileChannel out = null;
        try {
            in = new FileInputStream(file1).getChannel();
            out = new FileOutputStream(file2).getChannel();
            in.transferTo(0, (int)in.size(), out);
            // ̤ɽ
            if (!_quiet) {
                System.out.println("completed (" + in.size()
                                   + " / " + out.size() + " bytes)");
            }
        } catch (FileNotFoundException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        } catch (IOException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        } catch (SecurityException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        } finally {
            if (in != null) {
                try {
                    in.close();
                } catch (IOException e) {
                }
            }
            if (out != null) {
                try {
                    out.close();
                } catch (IOException e) {
                }
            }
        }
        // ꥸʥեκ
        if (!_keep) {
            _delete(file1);
        }
        if (!_quiet) {
            System.out.println("");
        }
    }

    /**
     * եޤ
     *
     * @param file ե
     */
    private void _delete(File file) {
        try {
            if (!file.delete()) {
                System.err.println(_PROGRAM
                                   + ": failed to delete the file: "
                                   + file.getPath());
            }
        } catch (SecurityException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        }
    }

    /**
     * 񤭤γǧԤޤ
     *
     * @param file ե
     * @return 񤭤ԤtrueǤʤfalse
     */
    private boolean _isOverwrite(File file) {
        if (!file.exists()) {
            return true;
        }
        if (_overwrite == OVERWRITE_NO) {
            if (!_quiet) {
                System.err.println("already exists, skip the file");
                System.err.println("");
            }
            return false;
        } else if (_overwrite == OVERWRITE_QUERY) {
            while (true) {
                System.err.println("");
                System.err.println("the file already exists: " + file.getPath());
                System.err.print("do you wish to overwrite (y or n)? ");
                BufferedReader br = null;
                try {
                    br = new BufferedReader(new InputStreamReader(System.in));
                    String line = br.readLine();
                    if (line != null) {
                        line = line.trim();
                        if (line.compareToIgnoreCase("y") == 0) {
                            break;
                        } else if (line.compareToIgnoreCase("n") == 0) {
                            System.err.println("");
                            return false;
                        }
                    }
                } catch (IOException e) {
                }
            }
            System.err.println("");
        }
        return true;
    }
}

// end of EBZip.java
