/*
 * $Id: ExcelEditor.java,v 1.1 2005/06/12 11:12:39 velvet Exp $
 * 
 * Copyright 2000-2005 The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package jp.sourceforge.excel.plugin;

import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedList;
import java.util.List;

import jp.sourceforge.poi.util.PoiUtil;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFPalette;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.MultiPageEditorPart;


public class ExcelEditor extends MultiPageEditorPart implements
        IResourceChangeListener {
    final protected static PoiUtil poiUtil = PoiUtil.getPoiUtil();

    boolean dirty;

    private Table tables[];

    protected HSSFWorkbook workbook;

    /**
     * Creates a multi-page editor example.
     */
    public ExcelEditor() {
        super();
        ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
        dirty = false;
    }

    void createExcelPage() throws IOException, CoreException {

        IEditorInput input = getEditorInput();
        if (input instanceof IFileEditorInput) {
            IFileEditorInput fileInput = (IFileEditorInput) input;
            IFile ifile = fileInput.getFile();

            InputStream in = null;
            try {
                in = ifile.getContents();
                POIFSFileSystem fs = null;
                fs = new POIFSFileSystem(in);
                workbook = new HSSFWorkbook(fs);
                HSSFPalette palette = workbook.getCustomPalette();

                List colNamesList = new LinkedList();

                for (char c = 'A'; c <= 'Z'; c++)
                    colNamesList.add(new String(new char[] { c }));

                for (char c0 = 'A'; c0 <= 'H'; c0++)
                    for (char c1 = 'A'; c1 <= 'Z'; c1++)
                        colNamesList.add(new String(new char[] { c0, c1 }));

                for (char c = 'A'; c <= 'V'; c++)
                    colNamesList.add(new String(new char[] { 'I', c }));

                tables = new Table[workbook.getNumberOfSheets()];
                for (int i = 0; i < workbook.getNumberOfSheets(); i++) {
                    final HSSFSheet sheet = workbook.getSheetAt(i);
                    String sheetName = workbook.getSheetName(i);
                    final Table table = new Table(getContainer(), SWT.SINGLE
                            | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
                    table.setHeaderVisible(true);

                    for (int j = 0; j < colNamesList.size(); j++) {
                        TableColumn col = new TableColumn(table, SWT.LEFT);
                        col.setText((String) colNamesList.get(j));
                        col.setWidth(50);
                    }

                    for (int row = sheet.getFirstRowNum(); row <= sheet
                            .getLastRowNum(); row++) {
                        HSSFRow rowValue = sheet.getRow(row);
                        if (rowValue == null)
                            continue;
                        TableItem item = new TableItem(table, SWT.NULL);
                        List colValuesList = new LinkedList();
                        for (short col = 0; col < rowValue.getFirstCellNum(); col++)
                            colValuesList.add("");
                        for (short col = rowValue.getFirstCellNum(); col <= rowValue
                                .getLastCellNum(); col++) {
                            HSSFCell cell = rowValue.getCell(col);
                            colValuesList.add(cell == null ? "" : poiUtil
                                    .getCellValueAsString(cell));
                            if (cell != null) {
                                // FIXME F܂삵܂
                                short cellColor = cell.getCellStyle()
                                        .getFillBackgroundColor();
                                //System.out.println(cellColor);
                                HSSFColor bgcolor = (HSSFColor) PoiUtil.colorMap
                                        .get(new Short(cellColor));
                                if (bgcolor == null)
                                    bgcolor = new HSSFColor.WHITE();
                                short triplet[] = bgcolor.getTriplet();
                                item.setBackground(col, new Color(item
                                        .getDisplay(), triplet[0], triplet[1],
                                        triplet[2]));

                            }
                        }

                        String data[] = (String[]) colValuesList
                                .toArray(new String[colValuesList.size()]);
                        item.setText(data);

                    }

                    final TableEditor tableEditor = new TableEditor(table);
                    tableEditor.grabHorizontal = true;

                    table.addMouseListener(new MouseAdapter() {
                        public void mouseDown(MouseEvent e) {

                            final int index = table.getSelectionIndex();
                            if (index == -1) {
                                return;
                            }
                            table.setSelection(new int[0]);
                            TableItem item = table.getItem(index);
                            Point point = new Point(e.x, e.y);

                            for (int i = 0; i < table.getColumnCount(); i++) {
                                final int row = i;
                                // IꂽWZ̈Ɋ܂܂Ă邩𔻒
                                if (item.getBounds(i).contains(point)) {
                                    final int column = i;
                                    final Text text = new Text(table, SWT.NONE);
                                    text.setText(item.getText(i));
                                    text.addFocusListener(new FocusAdapter() {
                                        public void focusLost(FocusEvent e) {
                                            setSheetValue(sheet, row, index,
                                                    text.getText());
                                            setDirty(true);

                                            TableItem item = tableEditor
                                                    .getItem();
                                            item
                                                    .setText(column, text
                                                            .getText());
                                            text.dispose();

                                        }
                                    });
                                    tableEditor.setEditor(text, item, i);
                                    text.setFocus();
                                    text.selectAll();
                                }
                            }
                        }
                    });
                    tables[i] = table;
                    int index = addPage(table);
                    setPageText(index, sheetName);

                }
            } finally {
                try {
                    if (in != null)
                        in.close();
                } catch (IOException e) {
                    // 
                }
            }

        }

        //getEditorInput();
    }

    /**
     * Creates the pages of the multi-page editor.
     */
    protected void createPages() {
        try {
            createExcelPage();
        } catch (CoreException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * The <code>MultiPageEditorPart</code> implementation of this
     * <code>IWorkbenchPart</code> method disposes all nested editors.
     * Subclasses may extend.
     */
    public void dispose() {
        ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
        super.dispose();
    }

    /**
     * Saves the multi-page editor's document.
     */
    public void doSave(IProgressMonitor monitor) {
        //getEditor(0).doSave(monitor);
        IEditorInput input = getEditorInput();

        if (input instanceof IFileEditorInput) {
            IFileEditorInput fileInput = (IFileEditorInput) input;
            IFile file = fileInput.getFile();
            ChunkedBufferByteArrayOutputStream out = new ChunkedBufferByteArrayOutputStream();
            try {
                workbook.write(out);
                out.close();
                ChunkedBufferByteArrayInputStream stream = new ChunkedBufferByteArrayInputStream(
                        out);

                file.setContents(stream, true, true, monitor);
            } catch (IOException e) {
                throw new RuntimeException(e);
            } catch (CoreException e) {
                throw new RuntimeException(e);
            }

        }
    }

    /**
     * Saves the multi-page editor's document as another file. Also updates the
     * text for page 0's tab, and updates this multi-page editor's input to
     * correspond to the nested editor's.
     */
    public void doSaveAs() {
        // FIXME had do nothing...
        IEditorPart editor = getEditor(0);
        editor.doSaveAs();
        setPageText(0, editor.getTitle());
        setInput(editor.getEditorInput());
    }

    /*
     * (non-Javadoc) Method declared on IEditorPart
     */
    public void gotoMarker(IMarker marker) {
        setActivePage(0);
        IDE.gotoMarker(getEditor(0), marker);
    }

    /**
     * The <code>MultiPageEditorExample</code> implementation of this method
     * checks that the input is an instance of <code>IFileEditorInput</code>.
     */
    public void init(IEditorSite site, IEditorInput editorInput)
            throws PartInitException {
        if (!(editorInput instanceof IFileEditorInput))
            throw new PartInitException(
                    "Invalid Input: Must be IFileEditorInput");
        super.init(site, editorInput);
    }

    /*
     * (non-Javadoc) Method declared on IEditorPart.
     */
    public boolean isSaveAsAllowed() {
        return true;
    }

    protected void setDirty(boolean dirty) {
        if (!this.dirty && dirty) {
            setPartName("*" + getPartName());
        }
        this.dirty = dirty;
    }

    public boolean isDirty() {
        return dirty;
    }

    /**
     * Closes all project files on project close.
     */
    public void resourceChanged(final IResourceChangeEvent event) {
        if (event.getType() == IResourceChangeEvent.PRE_CLOSE) {
            Display.getDefault().asyncExec(new Runnable() {
                public void run() {
                    IWorkbenchPage[] pages = getSite().getWorkbenchWindow()
                            .getPages();
                    //					for (int i = 0; i < pages.length; i++) {
                    //						if (((FileEditorInput) editor.getEditorInput())
                    //								.getFile().getProject().equals(
                    //										event.getResource())) {
                    //							IEditorPart editorPart = pages[i].findEditor(editor
                    //									.getEditorInput());
                    //							pages[i].closeEditor(editorPart, true);
                    //						}
                    //					}
                }
            });
        }
    }

    public void setSheetValue(HSSFSheet sheet, int col, int row, String value) {
        HSSFRow rowValue = sheet.getRow(row);
        if (rowValue == null)
            return;

        HSSFCell cell = rowValue.getCell((short) col);
        if (cell == null) {
            cell = rowValue.createCell((short) col);
            cell.setCellValue(value);
        } else {
            String cellValue = poiUtil.getCellValueAsString(cell);

            if (cellValue == null || !cellValue.equals(value)) {
                cell.setCellValue(value);
            }
        }

    }
}
