/* 
 * Copyright (C) 2008 cocha,hozumi
 * http:
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 	
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "EcoDecoTooL.h"

#define SAFE_RELEASE(p) { if(p!=NULL) { (p)->Release(); (p)=NULL; } }



static const GUID CLSID_EcoDecoInPlace  = { 0x6e94d0bb, 0x0567, 0x4ef6, { 0xac, 0x5d, 0x7b, 0xf2, 0x9b, 0xf1, 0x17, 0xe3 } };
static const GUID IID_IEcoDecoInterface = { 0x1bb79127, 0xe01d, 0x4f40, { 0x92, 0x57, 0xbb, 0xc6, 0x22, 0x43, 0x7f, 0xb7 } };


static const GUID CLSID_EcoDecoWriter         = { 0x4b1abfc2, 0xe95f, 0x4cb3, { 0xbe, 0xc8, 0x65, 0x7e, 0x6d, 0x89, 0x6a, 0x70 } };
static const GUID IID_IEcoDecoWriterInterface = { 0xfb4a5a46, 0xbb7, 0x4c75, { 0xad, 0xa1, 0xe0, 0x24, 0x1e, 0xc9, 0x4e, 0x9e } };


static const GUID CLSID_EcoDecoWav     = { 0xefbf44b7, 0xbb1f, 0x48c8, { 0xba, 0x59, 0xeb, 0x98, 0x3, 0x57, 0x9b, 0x1e } };
static const GUID IID_IWavConInterface = { 0xbda4cab0, 0x2eb2, 0x42cb, { 0xa6, 0x31, 0xe8, 0x9e, 0x82, 0x35, 0x2d, 0xd1} };


static const GUID CLSID_EcoDecoMp3Dec            = { 0xb335bfbf, 0xa6c5, 0x4418, { 0x89, 0x56, 0x58, 0x60, 0xb9, 0x3e, 0x2b, 0xc4 } };
static const GUID IID_IMp3DecInterface           = { 0xa658a66f, 0x63a9, 0x42a1, { 0x8a, 0x1e, 0x94, 0xad, 0xc8, 0x9a, 0xa3, 0x60 } };
static const GUID CLSID_EcoDecoFraunhoferWriter  = { 0xf714a0fa, 0xb683, 0x4185, { 0x9b, 0xeb, 0xd4, 0x54, 0xd3, 0xb1, 0xcf, 0x7d } };
static const GUID IID_IFraunhoferWriterInterface = { 0x993828f1, 0xd8ca, 0x4f9d, { 0xac, 0xd6, 0x17, 0xa5, 0x81, 0xb1, 0x59, 0x7c } };
static const GUID CLSID_EcoDecoLameWriter        = { 0x0b76d3b5, 0x01fd, 0x48d0, { 0x9b, 0xb4, 0x33, 0x8b, 0x46, 0xd9, 0x83, 0xa0 } };
static const GUID IID_ILameWriterInterface       = { 0x40f60711, 0x37bb, 0x4d40, { 0x93, 0xd7, 0x13, 0xa7, 0x64, 0x87, 0x1a, 0xae } };


static const GUID CLSID_aoTuvDec            = { 0x733b2bde, 0x28e8, 0x4dbb, { 0xba, 0x68, 0x36, 0x7d, 0x45, 0xc0, 0x77, 0xf6 } };
static const GUID IID_IaoTuvDecInterface    = { 0x4559b935, 0x767f, 0x4549, { 0xb3, 0x25, 0x7b, 0x49, 0x3c, 0x4c, 0xaf, 0x49 } };
static const GUID CLSID_aoTuvWriter         = { 0x1afacf98, 0x9baa, 0x4545, { 0x90, 0x99, 0x75, 0xe5, 0x51, 0x9c, 0xa0, 0xbd } };
static const GUID IID_IaoTuVWriterInterface = { 0xbd7daec2, 0x193e, 0x4a6a, { 0xb9, 0xd3, 0x47, 0x6d, 0x8c, 0x92, 0x9b, 0x35 } };


static const GUID CLSID_AC3Filter         = { 0xA753A1EC, 0x973E, 0x4718, { 0xAF, 0x8E, 0xA3, 0xF5, 0x54, 0xD4, 0x5C, 0x44 } };
static const GUID CLSID_CoreAAC           = { 0x6AC7C19E, 0x8CA0, 0x4E3D, { 0x9A, 0x9F, 0x28, 0x81, 0xDE, 0x29, 0xE0, 0xAC } };
static const GUID CLSID_TrueAudioDecoder  = { 0x680EF885, 0x41CF, 0x4F81, { 0xAA, 0x06, 0xB7, 0x24, 0xBC, 0x96, 0x3A, 0x31 } };
static const GUID CLSID_RealAudioDecoder  = { 0x941A4793, 0xA705, 0x4312, { 0x8D, 0xFC, 0xC1, 0x1C, 0xA0, 0x5F, 0x39, 0x7E } };


static const GUID CLSID_CDDAReader        = { 0x54A35221, 0x2C8D, 0x4A31, { 0xA5, 0xDF, 0x6D, 0x80, 0x98, 0x47, 0xE3, 0x93 } };
static const GUID CLSID_DCBassSource      = { 0xABE7B1D9, 0x4B3E, 0x4ACD, { 0xA0, 0xD1, 0x92, 0x61, 0x1D, 0x3A, 0x44, 0x92 } };


static const GUID CLSID_Mp3Parser         = { 0x13CEBFE0, 0x256B, 0x44BA, { 0xA2, 0x7B, 0x9F, 0x85, 0xCB, 0xDB, 0x97, 0x2D } } ;
static const GUID CLSID_AC3Parser         = { 0x280a3020, 0x86cf, 0x11d1, { 0xab, 0xe6, 0x00, 0xa0, 0xc9, 0x05, 0xf3, 0x75 } };
static const GUID CLSID_WAVEParser        = { 0xD51BD5A1, 0x7548, 0x11CF, { 0xA5, 0x20, 0x00, 0x80, 0xC7, 0x7E, 0xF5, 0x8A } };
static const GUID CLSID_FlvSplitter       = { 0x47E792CF, 0x0BBE, 0x4F7A, { 0x85, 0x9C, 0x19, 0x4B, 0x07, 0x68, 0x65, 0x0A } };
static const GUID CLSID_MatroskaSplitter  = { 0x149D2E01, 0xC32E, 0x4939, { 0x80, 0xF6, 0xC0, 0x7B, 0x81, 0x01, 0x5A, 0x7A } };
static const GUID CLSID_MP4Splitter       = { 0x61F47056, 0xE400, 0x43D3, { 0xAF, 0x1E, 0xAB, 0x7D, 0xFF, 0xD4, 0xC4, 0xAD } };
static const GUID CLSID_MpaSplitter       = { 0x0E9D4BF7, 0xCBCB, 0x46C7, { 0xBD, 0x80, 0x4E, 0xF2, 0x23, 0xA3, 0xDC, 0x2B } };
static const GUID CLSID_OggSplitter       = { 0x9FF48807, 0xE133, 0x40AA, { 0x82, 0x6F, 0x9B, 0x29, 0x59, 0xE5, 0x23, 0x2D } };
static const GUID CLSID_RealMediaSplitter = { 0xE21BE468, 0x5C18, 0x43EB, { 0xB0, 0xCC, 0xDB, 0x93, 0xA8, 0x47, 0xD7, 0x69 } };


static const GUID MEDIASUBTYPE_AAC    = { 0x000000ff, 0x0000, 0x0010, { 0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71 } };
static const GUID MEDIASUBTYPE_MP3    = { 0x00000055, 0x0000, 0x0010, { 0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71 } };
static const GUID MEDIASUBTYPE_Vorbis = { 0xcddca2d5, 0x6d75, 0x4f98, { 0x84, 0x0e, 0x73, 0x7b, 0xed, 0xd5, 0xc6, 0x3b } };

static const GUID g_aGuidAudioDecoder[] =
{
   CLSID_EcoDecoMp3Dec, CLSID_CoreAAC, CLSID_aoTuvDec, CLSID_TrueAudioDecoder, CLSID_AC3Filter, CLSID_ACMWrapper
};

static const GUID g_aGuidPushFilter[] =
{
   CLSID_CDDAReader, CLSID_DCBassSource
};

static const GUID g_aGuidPullFilter[] = 
{

   CLSID_MpaSplitter, CLSID_MPEG1Splitter, CLSID_AC3Parser, CLSID_WAVEParser, CLSID_MMSPLITTER, CLSID_AviSplitter,
   CLSID_FlvSplitter, CLSID_MatroskaSplitter, CLSID_MP4Splitter, CLSID_OggSplitter, CLSID_RealMediaSplitter
};

CDirectShow::CDirectShow()
{
   m_pGraphBuilder = NULL;
   m_pMediaEventEx = NULL;
   m_pMediaControl = NULL;

   m_nRawFileType = 0;
   m_llTotalBytes = 0;
   m_pEcoDecoInterface = NULL;

   ::ZeroMemory(&m_InputFormat, sizeof(m_InputFormat));
   ::ZeroMemory(&m_OutputFormat, sizeof(m_OutputFormat));

   pDShowTool = new CDShowTool();
}

CDirectShow::~CDirectShow()
{
   Release();

   if(pDShowTool != NULL)
   {
      delete pDShowTool;
      pDShowTool = NULL;
   }
}

void CDirectShow::Release()
{
   if(m_pMediaControl != NULL)
   {
      m_pMediaControl->Stop();
      SAFE_RELEASE(m_pMediaControl);
   }

   if(m_pMediaEventEx != NULL)
   {
      m_pMediaEventEx->SetNotifyWindow(NULL, 0, 0);
      SAFE_RELEASE(m_pMediaEventEx);
   }

   SAFE_RELEASE(m_pEcoDecoInterface);
   SAFE_RELEASE(m_pMediaEventEx);
   SAFE_RELEASE(m_pMediaControl);
   SAFE_RELEASE(m_pGraphBuilder);
}

int CDirectShow::PlayEnd(void)
{
   if(m_pMediaEventEx == NULL)
      return 0;

   long lEventCode;

   m_pMediaEventEx->WaitForCompletion(10, &lEventCode);

   if(lEventCode == EC_ERRORABORT)
      return EC_ERRORABORT;

   if(lEventCode == EC_COMPLETE)
      return EC_COMPLETE;

   return 0;
}

int CDirectShow::GetRawFileType(void)
{
   return m_nRawFileType;
}

int CDirectShow::GetPercent(void)
{
   if(m_pEcoDecoInterface != NULL && m_llTotalBytes > 0)
   {
      LONGLONG llTransformedBytes;
      m_pEcoDecoInterface->GetTransformedBytes(&llTransformedBytes);

      return (int)(llTransformedBytes * 100 / m_llTotalBytes);
   }

   return -1;
}

void CDirectShow::GetFormat(WAVEFORMATEX *pIn, WAVEFORMATEX *pOut)
{
   ::CopyMemory(pIn, &m_InputFormat, sizeof(m_InputFormat));
   ::CopyMemory(pOut, &m_OutputFormat, sizeof(m_OutputFormat));
}

IPin* CDirectShow::GetAudioOutputPin(IBaseFilter *pFilter)
{  

   int nPinCount;
   IPin *pPin = NULL;

   nPinCount = pDShowTool->GetPinCount(pFilter, PINDIR_OUTPUT);

   for(int i=0;i<nPinCount;i++)
   {
      pPin = pDShowTool->GetPin(pFilter, PINDIR_OUTPUT, i);

      if(pPin != NULL)
      {
         AM_MEDIA_TYPE *pAm = NULL;
         ULONG cFetched;
         IEnumMediaTypes *pEnumMediaTypes = NULL;
         pPin->EnumMediaTypes(&pEnumMediaTypes);

         if(pEnumMediaTypes != NULL)
         {
            pEnumMediaTypes->Reset();
            pEnumMediaTypes->Next(1, &pAm, &cFetched);

            if(pAm != NULL)
            {
               if(pAm->majortype == MEDIATYPE_Audio)
               {
                  FreeMediaType(*pAm);
                  pAm = NULL;

                  SAFE_RELEASE(pEnumMediaTypes);

                  return pPin;
               }

               FreeMediaType(*pAm);
               pAm = NULL;
            }

            SAFE_RELEASE(pEnumMediaTypes);
         }

         SAFE_RELEASE(pPin);
      }
   }

   return NULL;
}

IBaseFilter *CDirectShow::ConnectAudio(IBaseFilter *pFilter)
{  
   

   HRESULT hr;
   int nArraySize;
   IPin *pInPin = NULL;
   IPin *pOutPin = NULL;

   IBaseFilter *pNextFilter = NULL;

   pOutPin = GetAudioOutputPin(pFilter);
   if(pOutPin == NULL)
      return NULL;

   
   nArraySize = sizeof(g_aGuidAudioDecoder) / sizeof(g_aGuidAudioDecoder[0]);

   for(int i=0;i<nArraySize;i++)
   {  

      pNextFilter = pDShowTool->AddFilter(m_pGraphBuilder, g_aGuidAudioDecoder[i]);

      if(pNextFilter != NULL)
      {
         int nPinCount;

         nPinCount = pDShowTool->GetPinCount(pNextFilter, PINDIR_INPUT);

         for(int j=0;j<nPinCount;j++)
         {
            pInPin = pDShowTool->GetPin(pNextFilter, PINDIR_INPUT, j);
            if(pInPin != NULL)
            {
               hr = m_pGraphBuilder->ConnectDirect(pOutPin, pInPin, NULL);
               if(SUCCEEDED(hr))
               {
                  SAFE_RELEASE(pOutPin);
                  SAFE_RELEASE(pInPin);
                  return pNextFilter;
               }

               SAFE_RELEASE(pInPin);
            }
         }

         m_pGraphBuilder->RemoveFilter(pNextFilter);
         SAFE_RELEASE(pNextFilter);
      }
   }

   return NULL;
}

HRESULT CDirectShow::RenderWmvRaw(WCHAR *pwInputFileName, IBaseFilter *pRenderFilter)
{  
   

   const GUID CLSID_WMSource = { 0x6B6D0800, 0x9ADA, 0x11D0, { 0xA5, 0x20, 0x00, 0xA0, 0xD1, 0x01, 0x29, 0xC0 } };

   HRESULT hr;
   IBaseFilter *pSourceFilter = NULL;

   
   ::CoCreateInstance(CLSID_WMSource, NULL, CLSCTX_INPROC_SERVER, IID_IBaseFilter, (void**)&pSourceFilter);

   if(pSourceFilter != NULL)
   {
      IFileSourceFilter *pFileSourceFilter = NULL;

      pSourceFilter->QueryInterface(IID_IFileSourceFilter, (void**)&pFileSourceFilter);

      if(pFileSourceFilter != NULL)
      {
         hr = pFileSourceFilter->Load(pwInputFileName, NULL);

         if(FAILED(hr))
         {
            m_pGraphBuilder->RemoveFilter(pSourceFilter);
            SAFE_RELEASE(pSourceFilter);
            return E_FAIL;
         }
      }

      m_pGraphBuilder->RemoveFilter(pSourceFilter);
      SAFE_RELEASE(pSourceFilter);
   }

   
   pSourceFilter = pDShowTool->AddSourceFilter(m_pGraphBuilder, CLSID_WMAsfReader, pwInputFileName);

   if(pSourceFilter == NULL)
      return E_FAIL;

   
   hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pSourceFilter, pRenderFilter, NULL);
   if(FAILED(hr))
   {
      m_pGraphBuilder->RemoveFilter(pSourceFilter);
      SAFE_RELEASE(pSourceFilter);
      return E_FAIL;
   }

   SAFE_RELEASE(pSourceFilter);
   return S_OK;
}

HRESULT CDirectShow::RenderWma(WCHAR *pwInputFileName, IBaseFilter *pRenderFilter)
{  
   

   const GUID CLSID_WMSource           = { 0x6B6D0800, 0x9ADA, 0x11D0, { 0xA5, 0x20, 0x00, 0xA0, 0xD1, 0x01, 0x29, 0xC0 } };
   const GUID CLSID_WMAudioDecoderDMO  = { 0x2EEB4ADF, 0x4578, 0x4D10, { 0xBC, 0xA7, 0xBB, 0x95, 0x5F, 0x56, 0x32, 0x0A } };
   const GUID CLSID_WMSpeechDecoderDMO = { 0x874131CB, 0x4ECC, 0x443B, { 0x89, 0x48, 0x74, 0x6B, 0x89, 0x59, 0x5D, 0x20 } };

   HRESULT hr;
   IBaseFilter *pSourceFilter = NULL;
   IBaseFilter *pDmoFilter = NULL;

   
   hr = ::CoCreateInstance(CLSID_WMSource, NULL, CLSCTX_INPROC_SERVER, IID_IBaseFilter, (void**)&pSourceFilter);
   if(pSourceFilter != NULL)
   {
      IFileSourceFilter *pFileSourceFilter = NULL;

      pSourceFilter->QueryInterface(IID_IFileSourceFilter, (void**)&pFileSourceFilter);
      if(pFileSourceFilter != NULL)
      {
         hr = pFileSourceFilter->Load(pwInputFileName, NULL);
         SAFE_RELEASE(pFileSourceFilter);

         if(FAILED(hr))
            goto error;
      }

      m_pGraphBuilder->RemoveFilter(pSourceFilter);
      SAFE_RELEASE(pSourceFilter);
   }

   
   pSourceFilter = pDShowTool->AddSourceFilter(m_pGraphBuilder, CLSID_WMAsfReader, pwInputFileName);
   if(pSourceFilter == NULL)
      goto error;

   pDmoFilter = pDShowTool->AddDmoFilter(m_pGraphBuilder, DMOCATEGORY_AUDIO_DECODER, CLSID_WMAudioDecoderDMO);
   if(pDmoFilter == NULL)
      goto error;

   
   hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pSourceFilter, pDmoFilter, NULL);
   if(FAILED(hr))
   {  

      hr = m_pGraphBuilder->RemoveFilter(pDmoFilter);
      if(FAILED(hr))
         goto error;

      SAFE_RELEASE(pDmoFilter);

      pDmoFilter = pDShowTool->AddDmoFilter(m_pGraphBuilder, DMOCATEGORY_AUDIO_DECODER, CLSID_WMSpeechDecoderDMO);
      if(pDmoFilter == NULL)
         goto error;

      
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pSourceFilter, pDmoFilter, NULL);
      if(FAILED(hr))
      {
         hr = m_pGraphBuilder->RemoveFilter(pDmoFilter);
         if(FAILED(hr))
            goto error;

         SAFE_RELEASE(pDmoFilter);

         
         pDmoFilter = ConnectAudio(pSourceFilter);
         if(pDmoFilter == NULL)
            goto error;
      }
   }

   
   hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pDmoFilter, pRenderFilter, NULL);
   if(FAILED(hr))
      goto error;

   SAFE_RELEASE(pDmoFilter);
   SAFE_RELEASE(pSourceFilter);

   return S_OK;

error:

   SAFE_RELEASE(pDmoFilter);
   SAFE_RELEASE(pSourceFilter);

   return E_FAIL;
}

HRESULT CDirectShow::RenderAudio(WCHAR *pwInputFileName, IBaseFilter *pRenderFilter)
{
   HRESULT hr;
   int nArraySize;
   IBaseFilter *pSourceFilter = NULL;
   IBaseFilter *pPullFilter = NULL;
   IBaseFilter *pDecodeFilter = NULL;

   
   hr = RenderWma(pwInputFileName, pRenderFilter);
   if(SUCCEEDED(hr))
      return S_OK;

   
   pSourceFilter = pDShowTool->AddSourceFilter(m_pGraphBuilder, CLSID_AsyncReader, pwInputFileName);
   if(pSourceFilter != NULL)
   {
      
      nArraySize = sizeof(g_aGuidPullFilter) / sizeof(g_aGuidPullFilter[0]);

      for(int i=0;i<nArraySize;i++)
      {
         pPullFilter = pDShowTool->AddFilter(m_pGraphBuilder, g_aGuidPullFilter[i]);
         if(pPullFilter == NULL) continue;

         
         hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pSourceFilter, pPullFilter, NULL);
         if(SUCCEEDED(hr))
         {
            
            hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pPullFilter, pRenderFilter, NULL);
            if(SUCCEEDED(hr))
            {
               SAFE_RELEASE(pSourceFilter);
               SAFE_RELEASE(pPullFilter);
               return S_OK;
            }

            
            if(g_aGuidPullFilter[i] == CLSID_RealMediaSplitter)
            {
               pDecodeFilter = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_RealAudioDecoder);
               hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pPullFilter, pDecodeFilter, NULL);
               if(FAILED(hr))
                  SAFE_RELEASE(pDecodeFilter);
            }

            if(pDecodeFilter == NULL)
               pDecodeFilter = ConnectAudio(pPullFilter);

            if(pDecodeFilter != NULL)
            {
               
               hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pDecodeFilter, pRenderFilter, NULL);
               if(SUCCEEDED(hr))
               {
                  SAFE_RELEASE(pSourceFilter);
                  SAFE_RELEASE(pPullFilter);
                  SAFE_RELEASE(pDecodeFilter);
                  return S_OK;
               }

               m_pGraphBuilder->RemoveFilter(pDecodeFilter);
               SAFE_RELEASE(pDecodeFilter);
            }
         }

         m_pGraphBuilder->RemoveFilter(pPullFilter);
         SAFE_RELEASE(pPullFilter);
      }
   
      m_pGraphBuilder->RemoveFilter(pSourceFilter);
      SAFE_RELEASE(pSourceFilter);
   }

   
   nArraySize = sizeof(g_aGuidPushFilter) / sizeof(g_aGuidPushFilter[0]);

   for(int i=0;i<nArraySize;i++)
   {
      pSourceFilter = pDShowTool->AddSourceFilter(m_pGraphBuilder, g_aGuidPushFilter[i], pwInputFileName);
      if(pSourceFilter == NULL) continue;

      if(g_aGuidPushFilter[i] == CLSID_CDDAReader)
      {  

         pPullFilter = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_WAVEParser);
         if(pPullFilter != NULL)
         {
            hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pSourceFilter, pPullFilter, NULL);
            if(SUCCEEDED(hr))
            {
               hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pPullFilter, pRenderFilter, NULL);
               if(SUCCEEDED(hr))
               {
                  SAFE_RELEASE(pSourceFilter);
                  SAFE_RELEASE(pPullFilter);
                  return S_OK;
               }
            }

            m_pGraphBuilder->RemoveFilter(pPullFilter);
            SAFE_RELEASE(pPullFilter);
         }
      }
      else
      {  
         hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pSourceFilter, pRenderFilter, NULL);
         if(SUCCEEDED(hr))
         {
            SAFE_RELEASE(pSourceFilter);
            return S_OK;
         }
      }

      m_pGraphBuilder->RemoveFilter(pSourceFilter);
      SAFE_RELEASE(pSourceFilter);
   }

   return E_FAIL;
}

bool CDirectShow::ConvertToWav(HWND hWnd, WCHAR *pwInputFileName, WCHAR *pwOutputFileName, WAVEFORMATEX *pOutputFormat)
{
   HRESULT hr;
   WAVEFORMATEX wf;
   LONGLONG llDuration;
   IBaseFilter *pEcoDecoInPlace = NULL;
   IBaseFilter *pWavConFilter = NULL;
   IBaseFilter *pEcoDecoWriter = NULL;
   IMediaSeeking *pMediaSeeking = NULL;
   IWavConInterface *pWavConInterface = NULL;
   IEcoDecoWriterInterface *pEcoDecoWriterInterface = NULL;

   IBaseFilter *pEcoDecoMp3Dec = NULL;
   IMp3DecInterface *pMp3DecInterface = NULL;

   IBaseFilter *pEcoDecoaoTuVDec = NULL;
   IaoTuvDecInterface *paoTuvDecInterface = NULL;



   
   ::CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, IID_IGraphBuilder, (void **)&m_pGraphBuilder);
   if(m_pGraphBuilder == NULL) goto error;

   
   pEcoDecoInPlace = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoInPlace);
   if(pEcoDecoInPlace == NULL) goto error;

   if(pOutputFormat != NULL)
   {  

      
      pEcoDecoMp3Dec = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoMp3Dec);
      if(pEcoDecoMp3Dec != NULL)
      {
         hr = RenderAudio(pwInputFileName, pEcoDecoMp3Dec);
         if(SUCCEEDED(hr))
         {
            
            pEcoDecoMp3Dec->QueryInterface(IID_IMp3DecInterface, (void **)&pMp3DecInterface);
            if(pMp3DecInterface != NULL)
            {
               pMp3DecInterface->GetOutFormat(&wf);

               if(wf.nSamplesPerSec == pOutputFormat->nSamplesPerSec && wf.nChannels == pOutputFormat->nChannels)
               {  

                  if(pOutputFormat->wBitsPerSample == 8) 
                  {
                     



                     wf.wFormatTag = WAVE_FORMAT_IEEE_FLOAT;
                     wf.wBitsPerSample = 32;
                  }
                  else
                  {
                     wf.wFormatTag = pOutputFormat->wFormatTag;
                     wf.wBitsPerSample = pOutputFormat->wBitsPerSample;
                  }
               }
               else
               {  

                  


                  wf.wFormatTag = WAVE_FORMAT_IEEE_FLOAT;
                  wf.wBitsPerSample = 32;
               }

               
               wf.nChannels = pOutputFormat->nChannels;

               wf.cbSize = 0;
               wf.nBlockAlign = wf.nChannels * (wf.wBitsPerSample / 8);
               wf.nAvgBytesPerSec = wf.nSamplesPerSec * wf.nBlockAlign;

               
               hr = pMp3DecInterface->SetOutFormat(&wf);
               if(FAILED(hr))
               {
                  SAFE_RELEASE(pMp3DecInterface);
               }
               else
               {
                  
                  hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoMp3Dec, pEcoDecoInPlace, NULL);
                  if(FAILED(hr))
                     SAFE_RELEASE(pMp3DecInterface);
               }
            }
         }

         if(pMp3DecInterface == NULL)
         {
            
            SAFE_RELEASE(m_pGraphBuilder);
            ::CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, IID_IGraphBuilder, (void **)&m_pGraphBuilder);
            if(m_pGraphBuilder == NULL) goto error;

            
            pEcoDecoInPlace = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoInPlace);
            if(pEcoDecoInPlace == NULL) goto error;
         }
      }

      if(pMp3DecInterface == NULL)
      {
         
         pEcoDecoaoTuVDec = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_aoTuvDec);
         if(pEcoDecoaoTuVDec != NULL)
         {
            hr = RenderAudio(pwInputFileName, pEcoDecoaoTuVDec);
            if(SUCCEEDED(hr))
            {
               
               pEcoDecoaoTuVDec->QueryInterface(IID_IaoTuvDecInterface, (void **)&paoTuvDecInterface);
               if(paoTuvDecInterface != NULL)
               {
                  paoTuvDecInterface->GetOutFormat(&wf);

                  
                  if(wf.nSamplesPerSec == pOutputFormat->nSamplesPerSec && wf.nChannels == pOutputFormat->nChannels)
                  {  
                     hr = paoTuvDecInterface->SetOutBitsPerSample(WAVE_FORMAT_PCM, pOutputFormat->wBitsPerSample);
                  }
                  else
                  {  

                     hr = paoTuvDecInterface->SetOutBitsPerSample(WAVE_FORMAT_IEEE_FLOAT, 32);
                  }

                  if(FAILED(hr))
                  {
                     SAFE_RELEASE(paoTuvDecInterface);
                  }
                  else
                  {
                     
                     hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoaoTuVDec, pEcoDecoInPlace, NULL);
                     if(FAILED(hr))
                        SAFE_RELEASE(paoTuvDecInterface);
                  }
               }
            }
         }

         if(paoTuvDecInterface == NULL)
         {
            
            SAFE_RELEASE(m_pGraphBuilder);
            ::CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, IID_IGraphBuilder, (void **)&m_pGraphBuilder);
            if(m_pGraphBuilder == NULL) goto error;

            
            pEcoDecoInPlace = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoInPlace);
            if(pEcoDecoInPlace == NULL) goto error;
         }
      }
   }

   if(pMp3DecInterface == NULL && paoTuvDecInterface == NULL)
   {  

      
      hr = RenderAudio(pwInputFileName, pEcoDecoInPlace);
      if(FAILED(hr)) goto error;
   }

   
   pEcoDecoInPlace->QueryInterface(IID_IEcoDecoInterface, (void **)&m_pEcoDecoInterface);
   if(m_pEcoDecoInterface == NULL) goto error;

   
   pEcoDecoWriter = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoWriter);
   if(pEcoDecoWriter == NULL) goto error;

   
   pEcoDecoWriter->QueryInterface(IID_IEcoDecoWriterInterface, (void **)&pEcoDecoWriterInterface);
   if(pEcoDecoWriterInterface == NULL) goto error;

   
   m_pEcoDecoInterface->GetInFormat(&m_InputFormat);

   if(pOutputFormat == NULL)
   {  
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoInPlace, pEcoDecoWriter, NULL);
      if(FAILED(hr)) goto error;
   }
   else
   {
      if(m_InputFormat.wFormatTag == pOutputFormat->wFormatTag &&
         m_InputFormat.nSamplesPerSec == pOutputFormat->nSamplesPerSec &&
         m_InputFormat.wBitsPerSample == pOutputFormat->wBitsPerSample &&
         m_InputFormat.nChannels == pOutputFormat->nChannels)
      {  
         hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoInPlace, pEcoDecoWriter, NULL);
         if(FAILED(hr)) goto error;
      }
      else
      {  

         
         pWavConFilter = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoWav);
         if(pWavConFilter == NULL) goto error;

         
         hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoInPlace, pWavConFilter, NULL);
         if(FAILED(hr)) goto error;

         pWavConFilter->QueryInterface(IID_IWavConInterface, (void **)&pWavConInterface);
         if(pWavConInterface == NULL) goto error;

         pWavConInterface->SetQuality(4);
         pWavConInterface->SetOutFormat(pOutputFormat);

         
         hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pWavConFilter, pEcoDecoWriter, NULL);
         if(FAILED(hr)) goto error;
      }

      ::CopyMemory(&m_OutputFormat, pOutputFormat, sizeof(m_OutputFormat));
   }

   
   hr = pEcoDecoWriterInterface->SetFileName(pwOutputFileName);
   if(hr != S_OK) goto error;

   
   m_llTotalBytes = -1;

   m_pGraphBuilder->QueryInterface(IID_IMediaSeeking, (void **)&pMediaSeeking);
   if(pMediaSeeking != NULL)
   {
      hr = pMediaSeeking->GetDuration(&llDuration);
      if(SUCCEEDED(hr))
         m_llTotalBytes = llDuration * (m_InputFormat.wBitsPerSample/8) * m_InputFormat.nChannels * m_InputFormat.nSamplesPerSec / 10000000;
   }

   
   m_pGraphBuilder->QueryInterface(IID_IMediaEventEx, (void **)&m_pMediaEventEx);
   if(m_pMediaEventEx == NULL) goto error;

   if(hWnd != NULL)
   {
      hr = m_pMediaEventEx->SetNotifyWindow((OAHWND)hWnd, WM_GRAPHNOTIFY, 0);
      if(hr != NO_ERROR) goto error;
   }

   
   m_pGraphBuilder->QueryInterface(IID_IMediaControl, (void **)&m_pMediaControl);
   if(m_pMediaControl == NULL) goto error;

   
   hr = m_pMediaControl->Run();
   if(hr != NO_ERROR) goto error;


   SAFE_RELEASE(pEcoDecoMp3Dec);
   SAFE_RELEASE(pMp3DecInterface);
   SAFE_RELEASE(pEcoDecoaoTuVDec);
   SAFE_RELEASE(paoTuvDecInterface);

   SAFE_RELEASE(pMediaSeeking);
   SAFE_RELEASE(pEcoDecoInPlace);
   SAFE_RELEASE(pWavConInterface);
   SAFE_RELEASE(pWavConFilter);
   SAFE_RELEASE(pEcoDecoWriterInterface);
   SAFE_RELEASE(pEcoDecoWriter);

   return true;

error:

   SAFE_RELEASE(pEcoDecoMp3Dec);
   SAFE_RELEASE(pMp3DecInterface);
   SAFE_RELEASE(pEcoDecoaoTuVDec);
   SAFE_RELEASE(paoTuvDecInterface);

   SAFE_RELEASE(pMediaSeeking);
   SAFE_RELEASE(pEcoDecoInPlace);
   SAFE_RELEASE(pWavConInterface);
   SAFE_RELEASE(pWavConFilter);
   SAFE_RELEASE(pEcoDecoWriterInterface);
   SAFE_RELEASE(pEcoDecoWriter);

   Release();

   return false;
}

bool CDirectShow::ConvertToFraunhoferMP3(HWND hWnd, WCHAR *pwInputFileName, WCHAR *pwOutputFileName, int nCBRBitrate)
{
   int i, j;
   HRESULT hr;
   LONGLONG llDuration;
   IBaseFilter *pEcoDecoInPlace   = NULL;
   IBaseFilter *pWavConFilter     = NULL;
   IBaseFilter *pFraunhoferWriter = NULL;
   IMediaSeeking *pMediaSeeking   = NULL;
   IWavConInterface *pWavConInterface = NULL;
   IFraunhoferWriterInterface *pFraunhoferWriterInterface = NULL;

   
   ::CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, IID_IGraphBuilder, (void **)&m_pGraphBuilder);
   if(m_pGraphBuilder == NULL) goto error;

   
   pEcoDecoInPlace = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoInPlace);
   if(pEcoDecoInPlace == NULL) goto error;

   
   hr = RenderAudio(pwInputFileName, pEcoDecoInPlace);
   if(FAILED(hr)) goto error;

   
   pEcoDecoInPlace->QueryInterface(IID_IEcoDecoInterface, (void **)&m_pEcoDecoInterface);
   if(m_pEcoDecoInterface == NULL) goto error;

   
   m_pEcoDecoInterface->GetInFormat(&m_InputFormat);

   
   pFraunhoferWriter = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoFraunhoferWriter);
   if(pFraunhoferWriter == NULL) goto error;

   
   pFraunhoferWriter->QueryInterface(IID_IFraunhoferWriterInterface, (void **)&pFraunhoferWriterInterface);
   if(pFraunhoferWriterInterface == NULL) goto error;

   
   m_OutputFormat.wFormatTag = WAVE_FORMAT_PCM;
   m_OutputFormat.cbSize = 0;
   m_OutputFormat.wBitsPerSample = 16;

   m_OutputFormat.nChannels = m_InputFormat.nChannels;
   m_OutputFormat.nSamplesPerSec = m_InputFormat.nSamplesPerSec;
   m_OutputFormat.nBlockAlign = m_OutputFormat.nChannels * m_OutputFormat.wBitsPerSample / 8;
   m_OutputFormat.nAvgBytesPerSec = m_OutputFormat.nBlockAlign * m_OutputFormat.nSamplesPerSec;

   hr = pFraunhoferWriterInterface->CheckBitrate(&m_OutputFormat, nCBRBitrate);

   if(hr != S_OK)
   {  

      DWORD anSamplingRate[] = {8000, 11025, 12000, 16000, 22050, 24000, 32000, 44100, 48000};

      for(i=0;i<2;i++)
      {
         if(i==0)
            m_OutputFormat.nChannels = m_InputFormat.nChannels;
         else
         {
            if(m_InputFormat.nChannels == 2)
               m_OutputFormat.nChannels = 1;
            else
               m_OutputFormat.nChannels = 2;
         }

         for(j=0;j<8;j++)
         {  

            if(anSamplingRate[j] <= m_OutputFormat.nSamplesPerSec)
               continue;

            m_OutputFormat.nSamplesPerSec = anSamplingRate[j];
            m_OutputFormat.nBlockAlign = m_OutputFormat.nChannels * m_OutputFormat.wBitsPerSample / 8;
            m_OutputFormat.nAvgBytesPerSec = m_OutputFormat.nBlockAlign * m_OutputFormat.nSamplesPerSec;

            hr = pFraunhoferWriterInterface->CheckBitrate(&m_OutputFormat, nCBRBitrate);
            if(hr == S_OK)
               break;
         }

         if(hr == S_OK)
            break;

         for(j=7;j>=0;j--)
         {  

            if(anSamplingRate[j] >= m_OutputFormat.nSamplesPerSec)
               continue;

            m_OutputFormat.nSamplesPerSec = anSamplingRate[j];
            m_OutputFormat.nBlockAlign = m_OutputFormat.nChannels * m_OutputFormat.wBitsPerSample / 8;
            m_OutputFormat.nAvgBytesPerSec = m_OutputFormat.nBlockAlign * m_OutputFormat.nSamplesPerSec;

            hr = pFraunhoferWriterInterface->CheckBitrate(&m_OutputFormat, nCBRBitrate);
            if(hr == S_OK)
               break;
         }

         if(hr == S_OK)
            break;
      }
   }

   if(hr != S_OK)
      goto error;

   if(m_OutputFormat.nChannels == m_InputFormat.nChannels &&
      m_OutputFormat.wBitsPerSample == m_InputFormat.wBitsPerSample &&
      m_OutputFormat.nSamplesPerSec == m_InputFormat.nSamplesPerSec)
   {  

      
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoInPlace, pFraunhoferWriter, NULL);
      if(FAILED(hr)) goto error;

      
      hr = pFraunhoferWriterInterface->SetBitrate(nCBRBitrate);
      if(hr != S_OK) goto error;
   }
   else
   {  

      
      pWavConFilter = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoWav);
      if(pWavConFilter == NULL) goto error;

      
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoInPlace, pWavConFilter, NULL);
      if(FAILED(hr)) goto error;

      
      pWavConFilter->QueryInterface(IID_IWavConInterface, (void **)&pWavConInterface);
      if(pWavConInterface == NULL) goto error;

      pWavConInterface->SetQuality(4);
      pWavConInterface->SetOutFormat(&m_OutputFormat);

      
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pWavConFilter, pFraunhoferWriter, NULL);
      if(FAILED(hr)) goto error;

      
      hr = pFraunhoferWriterInterface->SetBitrate(nCBRBitrate);
      if(hr != S_OK) goto error;
   }

   
   hr = pFraunhoferWriterInterface->SetFileName(pwOutputFileName);
   if(hr != S_OK) goto error;

   
   m_llTotalBytes = -1;

   m_pGraphBuilder->QueryInterface(IID_IMediaSeeking, (void **)&pMediaSeeking);
   if(pMediaSeeking != NULL)
   {
      hr = pMediaSeeking->GetDuration(&llDuration);
      if(SUCCEEDED(hr))
         m_llTotalBytes = llDuration * (m_InputFormat.wBitsPerSample/8) * m_InputFormat.nChannels * m_InputFormat.nSamplesPerSec / 10000000;
   }

   
   m_pGraphBuilder->QueryInterface(IID_IMediaEventEx, (void **)&m_pMediaEventEx);
   if(m_pMediaEventEx == NULL) goto error;

   if(hWnd != NULL)
   {
      hr = m_pMediaEventEx->SetNotifyWindow((OAHWND)hWnd, WM_GRAPHNOTIFY, 0);
      if(hr != NO_ERROR) goto error;
   }

   
   m_pGraphBuilder->QueryInterface(IID_IMediaControl, (void **)&m_pMediaControl);
   if(m_pMediaControl == NULL) goto error;

   
   hr = m_pMediaControl->Run();
   if(hr != NO_ERROR) goto error;

   SAFE_RELEASE(pMediaSeeking);
   SAFE_RELEASE(pEcoDecoInPlace);
   SAFE_RELEASE(pFraunhoferWriterInterface);
   SAFE_RELEASE(pFraunhoferWriter);
   SAFE_RELEASE(pWavConInterface);
   SAFE_RELEASE(pWavConFilter);

   return true;

error:

   SAFE_RELEASE(pMediaSeeking);
   SAFE_RELEASE(pEcoDecoInPlace);
   SAFE_RELEASE(pFraunhoferWriterInterface);
   SAFE_RELEASE(pFraunhoferWriter);
   SAFE_RELEASE(pWavConInterface);
   SAFE_RELEASE(pWavConFilter);

   Release();

   return false;
}

bool CDirectShow::ConvertToLameMP3(HWND hWnd, WCHAR *pwInputFileName, WCHAR *pwOutputFileName, int nContainerType, int nCBRVBR)
{
   int i, j;
   HRESULT hr;
   LONGLONG llDuration;
   IBaseFilter *pEcoDecoInPlace   = NULL;
   IBaseFilter *pWavConFilter     = NULL;
   IBaseFilter *pLameWriter       = NULL;
   IMediaSeeking *pMediaSeeking   = NULL;
   IWavConInterface *pWavConInterface = NULL;
   ILameWriterInterface *pLameWriterInterface = NULL;

   LAME_CONFIG lameConfig;

   
   ::ZeroMemory(&lameConfig, sizeof(lameConfig));
   if(nContainerType == CONTAINER_MP3_LAME_CBR)
   {  
      lameConfig.nMode = 1;
      lameConfig.cbr.dwBitrate = nCBRVBR;
      lameConfig.nQuality = 0;
   }
   else
   {  
      lameConfig.nMode = 2;
      lameConfig.vbr.nVBRQuality = nCBRVBR;
      lameConfig.vbr.dwMaxBitrate = 320;
      lameConfig.vbr.dwMinBitrate = 8;
      lameConfig.nQuality = 0;
   }

   
   ::CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, IID_IGraphBuilder, (void **)&m_pGraphBuilder);
   if(m_pGraphBuilder == NULL) goto error;

   
   pEcoDecoInPlace = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoInPlace);
   if(pEcoDecoInPlace == NULL) goto error;

   
   hr = RenderAudio(pwInputFileName, pEcoDecoInPlace);
   if(FAILED(hr)) goto error;

   
   pEcoDecoInPlace->QueryInterface(IID_IEcoDecoInterface, (void **)&m_pEcoDecoInterface);
   if(m_pEcoDecoInterface == NULL) goto error;

   
   m_pEcoDecoInterface->GetInFormat(&m_InputFormat);

   
   pLameWriter = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoLameWriter);
   if(pLameWriter == NULL) goto error;

   
   pLameWriter->QueryInterface(IID_ILameWriterInterface, (void **)&pLameWriterInterface);
   if(pLameWriterInterface == NULL) goto error;

   
   m_OutputFormat.wFormatTag = WAVE_FORMAT_PCM;
   m_OutputFormat.cbSize = 0;
   m_OutputFormat.wBitsPerSample = 16;

   m_OutputFormat.nChannels = m_InputFormat.nChannels;
   m_OutputFormat.nSamplesPerSec = m_InputFormat.nSamplesPerSec;
   m_OutputFormat.nBlockAlign = m_OutputFormat.nChannels * m_OutputFormat.wBitsPerSample / 8;
   m_OutputFormat.nAvgBytesPerSec = m_OutputFormat.nBlockAlign * m_OutputFormat.nSamplesPerSec;

   hr = pLameWriterInterface->CheckBitrate(&m_OutputFormat, &lameConfig);

   if(hr != S_OK)
   {  

      DWORD anSamplingRate[] = {8000, 11025, 12000, 16000, 22050, 24000, 32000, 44100, 48000};

      for(i=0;i<2;i++)
      {
         if(i==0)
            m_OutputFormat.nChannels = m_InputFormat.nChannels;
         else
         {
            if(m_InputFormat.nChannels == 2)
               m_OutputFormat.nChannels = 1;
            else
               m_OutputFormat.nChannels = 2;
         }

         for(j=0;j<8;j++)
         {  

            if(anSamplingRate[j] <= m_OutputFormat.nSamplesPerSec)
               continue;

            m_OutputFormat.nSamplesPerSec = anSamplingRate[j];
            m_OutputFormat.nBlockAlign = m_OutputFormat.nChannels * m_OutputFormat.wBitsPerSample / 8;
            m_OutputFormat.nAvgBytesPerSec = m_OutputFormat.nBlockAlign * m_OutputFormat.nSamplesPerSec;

            hr = pLameWriterInterface->CheckBitrate(&m_OutputFormat, &lameConfig);
            if(hr == S_OK)
               break;
         }

         if(hr == S_OK)
            break;

         for(j=7;j>=0;j--)
         {  

            if(anSamplingRate[j] >= m_OutputFormat.nSamplesPerSec)
               continue;

            m_OutputFormat.nSamplesPerSec = anSamplingRate[j];
            m_OutputFormat.nBlockAlign = m_OutputFormat.nChannels * m_OutputFormat.wBitsPerSample / 8;
            m_OutputFormat.nAvgBytesPerSec = m_OutputFormat.nBlockAlign * m_OutputFormat.nSamplesPerSec;

            hr = pLameWriterInterface->CheckBitrate(&m_OutputFormat, &lameConfig);
            if(hr == S_OK)
               break;
         }

         if(hr == S_OK)
            break;
      }
   }

   if(hr != S_OK)
      goto error;

   if(m_OutputFormat.nChannels == m_InputFormat.nChannels &&
      m_OutputFormat.wBitsPerSample == m_InputFormat.wBitsPerSample &&
      m_OutputFormat.nSamplesPerSec == m_InputFormat.nSamplesPerSec)
   {  

      
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoInPlace, pLameWriter, NULL);
      if(FAILED(hr)) goto error;

      
      hr = pLameWriterInterface->SetOutFormat(&lameConfig);
      if(hr != S_OK) goto error;
   }
   else
   {  

      
      pWavConFilter = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoWav);
      if(pWavConFilter == NULL) goto error;

      
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoInPlace, pWavConFilter, NULL);
      if(FAILED(hr)) goto error;

      
      pWavConFilter->QueryInterface(IID_IWavConInterface, (void **)&pWavConInterface);
      if(pWavConInterface == NULL) goto error;

      pWavConInterface->SetQuality(4);
      pWavConInterface->SetOutFormat(&m_OutputFormat);

      
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pWavConFilter, pLameWriter, NULL);
      if(FAILED(hr)) goto error;

      
      hr = pLameWriterInterface->SetOutFormat(&lameConfig);
      if(hr != S_OK) goto error;
   }

   
   hr = pLameWriterInterface->SetFileName(pwOutputFileName);
   if(hr != S_OK) goto error;

   
   m_llTotalBytes = -1;

   m_pGraphBuilder->QueryInterface(IID_IMediaSeeking, (void **)&pMediaSeeking);
   if(pMediaSeeking != NULL)
   {
      hr = pMediaSeeking->GetDuration(&llDuration);
      if(SUCCEEDED(hr))
         m_llTotalBytes = llDuration * (m_InputFormat.wBitsPerSample/8) * m_InputFormat.nChannels * m_InputFormat.nSamplesPerSec / 10000000;
   }

   
   m_pGraphBuilder->QueryInterface(IID_IMediaEventEx, (void **)&m_pMediaEventEx);
   if(m_pMediaEventEx == NULL) goto error;

   if(hWnd != NULL)
   {
      hr = m_pMediaEventEx->SetNotifyWindow((OAHWND)hWnd, WM_GRAPHNOTIFY, 0);
      if(hr != NO_ERROR) goto error;
   }

   
   m_pGraphBuilder->QueryInterface(IID_IMediaControl, (void **)&m_pMediaControl);
   if(m_pMediaControl == NULL) goto error;

   
   hr = m_pMediaControl->Run();
   if(hr != NO_ERROR) goto error;

   SAFE_RELEASE(pMediaSeeking);
   SAFE_RELEASE(pEcoDecoInPlace);
   SAFE_RELEASE(pLameWriterInterface);
   SAFE_RELEASE(pLameWriter);
   SAFE_RELEASE(pWavConInterface);
   SAFE_RELEASE(pWavConFilter);

   return true;

error:

   SAFE_RELEASE(pMediaSeeking);
   SAFE_RELEASE(pEcoDecoInPlace);
   SAFE_RELEASE(pLameWriterInterface);
   SAFE_RELEASE(pLameWriter);
   SAFE_RELEASE(pWavConInterface);
   SAFE_RELEASE(pWavConFilter);

   Release();

   return false;
}

bool CDirectShow::ConvertToOgg(HWND hWnd, WCHAR *pwInputFileName, WCHAR *pwOutputFileName, int nContainerType, int nCBR, float fVBR)
{
   HRESULT hr;
   LONGLONG llDuration;
   IBaseFilter *pEcoDecoInPlace   = NULL;
   IBaseFilter *pWavConFilter     = NULL;
   IBaseFilter *paoTuvWriter      = NULL;
   IMediaSeeking *pMediaSeeking   = NULL;
   IWavConInterface *pWavConInterface = NULL;
   IaoTuVWriterInterface *paoTuvInterface = NULL;

   VORBISFORMAT vorbisFormat;

   
   ::ZeroMemory(&vorbisFormat, sizeof(vorbisFormat));
   if(nContainerType == CONTAINER_OGG_AOTUV_CBR || nContainerType == CONTAINER_OGG_LANCER_CBR)
   {  
      vorbisFormat.fQuality = -1.0f;
      vorbisFormat.nMinBitsPerSec = -1;
      vorbisFormat.nAvgBitsPerSec = nCBR;
      vorbisFormat.nMaxBitsPerSec = -1;
   }
   else
   {  
      vorbisFormat.fQuality = fVBR;
      vorbisFormat.nMinBitsPerSec = -1;
      vorbisFormat.nAvgBitsPerSec = -1;
      vorbisFormat.nMaxBitsPerSec = -1;
   }

   
   ::CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, IID_IGraphBuilder, (void **)&m_pGraphBuilder);
   if(m_pGraphBuilder == NULL) goto error;

   
   pEcoDecoInPlace = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoInPlace);
   if(pEcoDecoInPlace == NULL) goto error;

   
   hr = RenderAudio(pwInputFileName, pEcoDecoInPlace);
   if(FAILED(hr)) goto error;

   
   pEcoDecoInPlace->QueryInterface(IID_IEcoDecoInterface, (void **)&m_pEcoDecoInterface);
   if(m_pEcoDecoInterface == NULL) goto error;

   
   m_pEcoDecoInterface->GetInFormat(&m_InputFormat);

   if(nContainerType == CONTAINER_OGG_AOTUV_CBR || nContainerType == CONTAINER_OGG_AOTUV_VBR)
   {  
      paoTuvWriter = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_aoTuvWriter);
   }
   else
   {  
      
   }

   if(paoTuvWriter == NULL) goto error;

   
   paoTuvWriter->QueryInterface(IID_IaoTuVWriterInterface, (void **)&paoTuvInterface);
   if(paoTuvInterface == NULL) goto error;

   
   ::CopyMemory(&m_OutputFormat, &m_InputFormat, sizeof(m_OutputFormat));

   if(m_InputFormat.nSamplesPerSec <= 48000 && m_InputFormat.wBitsPerSample == 16)
   {  

      
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoInPlace, paoTuvWriter, NULL);
      if(FAILED(hr)) goto error;

      
      hr = paoTuvInterface->SetVorbisFormat(&vorbisFormat);
      if(hr != S_OK) goto error;
   }
   else
   {  

      
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoInPlace, paoTuvWriter, NULL);
      if(FAILED(hr)) goto error;

      
      hr = paoTuvInterface->SetVorbisFormat(&vorbisFormat);
      if(hr != S_OK) goto error;

      if(m_OutputFormat.nSamplesPerSec > 48000)
      {
         if(m_OutputFormat.nSamplesPerSec % 48000 == 0)
            m_OutputFormat.nSamplesPerSec = 48000;
         else
            m_OutputFormat.nSamplesPerSec = 44100;
      }

      m_OutputFormat.wFormatTag = WAVE_FORMAT_PCM;
      m_OutputFormat.wBitsPerSample = 16;
      m_OutputFormat.nBlockAlign = m_OutputFormat.nChannels * m_OutputFormat.wBitsPerSample / 8;
      m_OutputFormat.nAvgBytesPerSec = m_OutputFormat.nBlockAlign * m_OutputFormat.nSamplesPerSec;

      
      pWavConFilter = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoWav);
      if(pWavConFilter == NULL) goto error;

      
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pEcoDecoInPlace, pWavConFilter, NULL);
      if(FAILED(hr)) goto error;

      
      pWavConFilter->QueryInterface(IID_IWavConInterface, (void **)&pWavConInterface);
      if(pWavConInterface == NULL) goto error;

      pWavConInterface->SetQuality(4);
      pWavConInterface->SetOutFormat(&m_OutputFormat);

      
      hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pWavConFilter, paoTuvWriter, NULL);
      if(FAILED(hr)) goto error;

      
      hr = paoTuvInterface->SetVorbisFormat(&vorbisFormat);
      if(hr != S_OK) goto error;
   }

   
   hr = paoTuvInterface->SetFileName(pwOutputFileName);
   if(hr != S_OK) goto error;

   
   m_llTotalBytes = -1;

   m_pGraphBuilder->QueryInterface(IID_IMediaSeeking, (void **)&pMediaSeeking);
   if(pMediaSeeking != NULL)
   {
      hr = pMediaSeeking->GetDuration(&llDuration);
      if(SUCCEEDED(hr))
         m_llTotalBytes = llDuration * (m_InputFormat.wBitsPerSample/8) * m_InputFormat.nChannels * m_InputFormat.nSamplesPerSec / 10000000;
   }

   
   m_pGraphBuilder->QueryInterface(IID_IMediaEventEx, (void **)&m_pMediaEventEx);
   if(m_pMediaEventEx == NULL) goto error;

   if(hWnd != NULL)
   {
      hr = m_pMediaEventEx->SetNotifyWindow((OAHWND)hWnd, WM_GRAPHNOTIFY, 0);
      if(hr != NO_ERROR) goto error;
   }

   
   m_pGraphBuilder->QueryInterface(IID_IMediaControl, (void **)&m_pMediaControl);
   if(m_pMediaControl == NULL) goto error;

   
   hr = m_pMediaControl->Run();
   if(hr != NO_ERROR) goto error;

   SAFE_RELEASE(pMediaSeeking);
   SAFE_RELEASE(pEcoDecoInPlace);
   SAFE_RELEASE(paoTuvWriter);
   SAFE_RELEASE(pWavConInterface);
   SAFE_RELEASE(pWavConFilter);
   SAFE_RELEASE(paoTuvInterface);

   return true;

error:

   SAFE_RELEASE(pMediaSeeking);
   SAFE_RELEASE(pEcoDecoInPlace);
   SAFE_RELEASE(paoTuvWriter);
   SAFE_RELEASE(pWavConInterface);
   SAFE_RELEASE(pWavConFilter);
   SAFE_RELEASE(paoTuvInterface);

   Release();

   return false;
}

bool CDirectShow::ConvertRaw(HWND hWnd, WCHAR *pwInputFileName, WCHAR *pwOutputFileName)
{
   HRESULT hr;
   int nArraySize;
   IBaseFilter *pSourceFilter  = NULL;
   IBaseFilter *pPullFilter    = NULL;
   IBaseFilter *pEcoDecoWriter = NULL;
   IEcoDecoWriterInterface *pEcoDecoWriterInterface = NULL;

   
   ::CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, IID_IGraphBuilder, (void **)&m_pGraphBuilder);
   if(m_pGraphBuilder == NULL) goto error;

   
   pEcoDecoWriter = pDShowTool->AddFilter(m_pGraphBuilder, CLSID_EcoDecoWriter);
   if(pEcoDecoWriter == NULL) goto error;

   
   hr = RenderWmvRaw(pwInputFileName, pEcoDecoWriter);

   if(FAILED(hr))
   {   

      
      pSourceFilter = pDShowTool->AddSourceFilter(m_pGraphBuilder, CLSID_AsyncReader, pwInputFileName);
      if(pSourceFilter == NULL) goto error;

      nArraySize = sizeof(g_aGuidPullFilter) / sizeof(g_aGuidPullFilter[0]);

      for(int i=0;i<nArraySize;i++)
      {
         pPullFilter = pDShowTool->AddFilter(m_pGraphBuilder, g_aGuidPullFilter[i]);
         if(pPullFilter == NULL) continue;

         
         hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pSourceFilter, pPullFilter, NULL);
         if(SUCCEEDED(hr))
         {
            
            hr = pDShowTool->ConnectDirect(m_pGraphBuilder, pPullFilter, pEcoDecoWriter, NULL);
            if(SUCCEEDED(hr))
               break;
         }

         pDShowTool->Disconnect(m_pGraphBuilder, pSourceFilter, pPullFilter);
         m_pGraphBuilder->RemoveFilter(pPullFilter);
         SAFE_RELEASE(pPullFilter);
      }
   }

   
   pEcoDecoWriter->QueryInterface(IID_IEcoDecoWriterInterface, (void **)&pEcoDecoWriterInterface);
   if(pEcoDecoWriterInterface == NULL) goto error;

   
   m_nRawFileType = pEcoDecoWriterInterface->GetOutputMode();
   if(m_nRawFileType == CONTAINER_RAW)
      goto error;

   
   if(m_nRawFileType == CONTAINER_RAW_AAC)
      ::lstrcat(pwOutputFileName, TEXT(".aac"));
   else if(m_nRawFileType == CONTAINER_RAW_MP3)
      ::lstrcat(pwOutputFileName, TEXT(".mp3"));
   else if(m_nRawFileType == CONTAINER_RAW_MP2)
      ::lstrcat(pwOutputFileName, TEXT(".mp2"));
   else if(m_nRawFileType == CONTAINER_RAW_OGG)
      ::lstrcat(pwOutputFileName, TEXT(".ogg"));
   else if(m_nRawFileType == CONTAINER_RAW_WAV)
      ::lstrcat(pwOutputFileName, TEXT(".wav"));

   
   hr = pEcoDecoWriterInterface->SetFileName(pwOutputFileName);
   if(hr != S_OK) goto error;

   
   m_pGraphBuilder->QueryInterface(IID_IMediaEventEx, (void **)&m_pMediaEventEx);
   if(m_pMediaEventEx == NULL) goto error;

   if(hWnd != NULL)
   {
      hr = m_pMediaEventEx->SetNotifyWindow((OAHWND)hWnd, WM_GRAPHNOTIFY, 0);
      if(hr != NO_ERROR) goto error;
   }

   
   m_pGraphBuilder->QueryInterface(IID_IMediaControl, (void **)&m_pMediaControl);
   if(m_pMediaControl == NULL) goto error;

   
   hr = m_pMediaControl->Run();
   if(hr != NO_ERROR) goto error;

   SAFE_RELEASE(pEcoDecoWriterInterface);
   SAFE_RELEASE(pEcoDecoWriter);
   SAFE_RELEASE(pPullFilter);
   SAFE_RELEASE(pSourceFilter);

   return true;

error:

   SAFE_RELEASE(pEcoDecoWriterInterface);
   SAFE_RELEASE(pEcoDecoWriter);
   SAFE_RELEASE(pPullFilter);
   SAFE_RELEASE(pSourceFilter);

   return false;
}

