/**************************************************************************
 * Copyright (C) 2008 Cocha                                               *
 * http://sourceforge.jp/projects/ecodecotool/                            *
 *                                                                        *
 *  This Program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2, or (at your option)   *
 *  any later version.                                                    *
 *                                                                        *
 *  This Program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU General Public License     *
 *  along with GNU Make; see the file COPYING.  If not, write to          *
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. *
 *                                                                        *
 **************************************************************************/

#include "SourceApe.h"


CSourceApe::CSourceApe(IUnknown *pUnk, HRESULT *phr)
   : CSimplePushSource(L"Source", pUnk, CLSID_SourceApe, phr)
{  

   
   ::ZeroMemory(m_awInputFileName, sizeof(m_awInputFileName));
   m_pAPEDecompress = NULL;
   m_rtCurrent = 0;
   m_rtStop = INFINITE_SECOND;
}

CSourceApe::~CSourceApe()  
{  

   
   SAFE_DELETE(m_pAPEDecompress);
}  

HRESULT CSourceApe::FillBuffer(IMediaSample *pOutSample)
{  

   if(m_rtCurrent >= m_rtStop)
   {  
      return S_FALSE;
   }

   int nOutLength;
   REFERENCE_TIME rtEnd;
   BYTE *pbOutBuffer = NULL;
   pOutSample->GetPointer(&pbOutBuffer);

   int nRetVal, nBlocksRetrieved;

   
   nBlocksRetrieved = 0;
   nRetVal = m_pAPEDecompress->GetData((char *)pbOutBuffer, m_nBlockSize, &nBlocksRetrieved);

   if(nRetVal != ERROR_SUCCESS || nBlocksRetrieved == 0)
   {  
      return S_FALSE;
   }

   
   nOutLength = nBlocksRetrieved * m_outFormat.nBlockAlign;

   
   rtEnd = m_rtCurrent + (LONGLONG)nOutLength * ONE_SECOND / m_outFormat.nAvgBytesPerSec;

   if(rtEnd >= m_rtStop)
   {  

      nOutLength = nOutLength - (int)((rtEnd - m_rtStop) * m_outFormat.nAvgBytesPerSec / ONE_SECOND);
      rtEnd = m_rtStop;

      if(nOutLength < 0)
         nOutLength = 0;
   }

   
   pOutSample->SetTime(&m_rtCurrent, &rtEnd);
   pOutSample->SetActualDataLength(nOutLength);
   pOutSample->SetSyncPoint(FALSE);

   
   m_rtCurrent = rtEnd;

   return S_OK;
}

HRESULT CSourceApe::OnSeek(LONGLONG llNewPosition, LONGLONG llStopPosition)
{  

   if(m_pAPEDecompress == NULL)
      return E_NOTIMPL;

   if(llNewPosition >= 0)
   {  

      
      int nBlockOffset = (int)(llNewPosition * m_outFormat.nSamplesPerSec / ONE_SECOND);

      m_pAPEDecompress->Seek(nBlockOffset);

      m_rtCurrent = (REFERENCE_TIME)llNewPosition;
   }

   if(llStopPosition >= 0)
   {  
      m_rtStop = (REFERENCE_TIME)llStopPosition;
   }

   return S_OK;
}

HRESULT CSourceApe::OnConnectOutPin(CMediaType *pmtOut, int *pnOutBufferSize, REFERENCE_TIME *pTotalMediaTime)
{  

   if(m_pAPEDecompress == NULL)
      return E_FAIL;

   
   m_outFormat.nChannels       = m_pAPEDecompress->GetInfo(APE_INFO_CHANNELS);
   m_outFormat.nSamplesPerSec  = m_pAPEDecompress->GetInfo(APE_INFO_SAMPLE_RATE);
   m_outFormat.wBitsPerSample  = m_pAPEDecompress->GetInfo(APE_INFO_BITS_PER_SAMPLE);
   m_outFormat.nBlockAlign     = m_outFormat.nChannels * (m_outFormat.wBitsPerSample / 8);
   m_outFormat.nAvgBytesPerSec = m_outFormat.nSamplesPerSec * m_outFormat.nBlockAlign;
   m_outFormat.wFormatTag      = WAVE_FORMAT_PCM;
   m_outFormat.cbSize = 0;

   
   pmtOut->SetType(&MEDIATYPE_Audio);
   pmtOut->SetSubtype(&MEDIASUBTYPE_PCM);
   pmtOut->SetTemporalCompression(FALSE);
   pmtOut->SetSampleSize(0);
   pmtOut->SetFormatType(&FORMAT_WaveFormatEx);
   pmtOut->SetFormat((BYTE*)&m_outFormat, sizeof(m_outFormat));

   
   *pnOutBufferSize = 1024 * m_pAPEDecompress->GetInfo(APE_INFO_BLOCK_ALIGN) * 4;

   
   m_nBlockSize = 1024 * m_pAPEDecompress->GetInfo(APE_INFO_BLOCK_ALIGN);

   
   *pTotalMediaTime = (LONGLONG)m_pAPEDecompress->GetInfo(APE_INFO_TOTAL_BLOCKS) * ONE_SECOND / m_outFormat.nSamplesPerSec;

   return S_OK;
}

STDMETHODIMP CSourceApe::OnQueryInterface(REFIID riid, void ** ppv)
{
   if(riid == IID_IFileSourceFilter)
      return GetInterface((IFileSourceFilter *)this, ppv);

   return CSource::NonDelegatingQueryInterface(riid, ppv);
}

CUnknown * WINAPI CSourceApe::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{  

   ASSERT(phr);
    
   CSourceApe *pNewObject = new CSourceApe(punk, phr);
   if (pNewObject == NULL)
   {
      if(phr != NULL)
         *phr = E_OUTOFMEMORY;
   }

   return dynamic_cast<CUnknown *>(pNewObject);
}

STDMETHODIMP CSourceApe::Load(LPCOLESTR pszFileName, const AM_MEDIA_TYPE *pmt)
{  

   if(::PathFileExists(pszFileName) == FALSE)
      return E_FAIL;

   SAFE_DELETE(m_pAPEDecompress);

   
   int nRetVal = 0;
   m_pAPEDecompress = CreateIAPEDecompress(pszFileName, &nRetVal);
   if(m_pAPEDecompress == NULL)
      return E_FAIL;

   
   nRetVal = m_pAPEDecompress->Seek(0);
   if(nRetVal != ERROR_SUCCESS)
      return E_FAIL;

   ::lstrcpy(m_awInputFileName, pszFileName);
   return S_OK;
}

STDMETHODIMP CSourceApe::GetCurFile(LPOLESTR *ppszFileName, AM_MEDIA_TYPE *pmt)
{  

   CheckPointer(ppszFileName, E_POINTER);

   ::lstrcpy(*ppszFileName, m_awInputFileName);

   if(pmt != NULL)
   {
      ::ZeroMemory(pmt, sizeof(*pmt));
      pmt->majortype = MEDIATYPE_NULL;
      pmt->subtype = MEDIASUBTYPE_NULL;
   }

   return S_OK;
}

