/**************************************************************************
 * Copyright (C) 2008 Cocha                                               *
 * http://sourceforge.jp/projects/ecodecotool/                            *
 *                                                                        *
 *  This Program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2, or (at your option)   *
 *  any later version.                                                    *
 *                                                                        *
 *  This Program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU General Public License     *
 *  along with GNU Make; see the file COPYING.  If not, write to          *
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. *
 *                                                                        *
 **************************************************************************/

#include "DecodeFlac.h"


CDecodeFlac::CDecodeFlac(IUnknown *pUnk, HRESULT *phr)
   : CSimpleTransform(L"Flac Decoder", pUnk, CLSID_DecodeFlac, phr)
{  

   
   ::ZeroMemory(&m_outFormat, sizeof(m_outFormat));
   m_pDecoder = NULL;
   m_bMetaData = true;

   m_nOutBufferLength = 0;
   m_rtCurrent = 0;
}

CDecodeFlac::~CDecodeFlac()  
{  

   
   if(m_pDecoder != NULL)
   {
      FLAC__stream_decoder_delete(m_pDecoder);
      m_pDecoder = NULL;
   }
}  

FLAC__StreamDecoderReadStatus CDecodeFlac::read_callback(const FLAC__StreamDecoder *decoder, FLAC__byte buffer[], size_t *bytes, void *client_data)
{
   CDecodeFlac *pMyClass = (CDecodeFlac *)client_data;

   if( int(*bytes) < pMyClass->m_nInBufferLength )
   {
      ::CopyMemory(buffer, pMyClass->m_pInBufferPointer, *bytes);
      pMyClass->m_pInBufferPointer += *bytes;
      pMyClass->m_nInBufferLength -= *bytes;

      return FLAC__STREAM_DECODER_READ_STATUS_CONTINUE;
   }
   else
   {
      *bytes = pMyClass->m_nInBufferLength;
      ::CopyMemory(buffer, pMyClass->m_pInBufferPointer, *bytes);
 
      if(pMyClass->m_bMetaData == true)
        return FLAC__STREAM_DECODER_READ_STATUS_END_OF_STREAM;

      return FLAC__STREAM_DECODER_READ_STATUS_CONTINUE;
   }
}

void CDecodeFlac::metadata_callback(const FLAC__StreamDecoder *decoder, const FLAC__StreamMetadata *metadata, void *client_data)
{
   if(metadata->type == FLAC__METADATA_TYPE_STREAMINFO)
   {
      ((CDecodeFlac *)(client_data))->m_outFormat.nSamplesPerSec = (DWORD)metadata->data.stream_info.sample_rate;
      ((CDecodeFlac *)(client_data))->m_outFormat.wBitsPerSample = (WORD)metadata->data.stream_info.bits_per_sample;
      ((CDecodeFlac *)(client_data))->m_outFormat.nChannels = (WORD)metadata->data.stream_info.channels;

      ((CDecodeFlac *)(client_data))->m_nMaxBlockSize = metadata->data.stream_info.max_blocksize;

   }
}

FLAC__StreamDecoderWriteStatus CDecodeFlac::write_callback(const FLAC__StreamDecoder *decoder, const FLAC__Frame *frame, const FLAC__int32 *const buffer[], void *client_data) 
{
   CDecodeFlac *pMyClass = (CDecodeFlac *)client_data;

   int i, j;
   int nBlockSize = (int)frame->header.blocksize;
   int nChannels = (int)frame->header.channels;

   if(frame->header.bits_per_sample == 16)
   {
      short *ps = (short *)pMyClass->m_pOutBufferPointer;

      for(i=0;i<nBlockSize;i++)
      {
         for(j=0;j<nChannels;j++)
         {
            *ps = buffer[j][i];
            ps++;
         }
      }
   }
   else if(frame->header.bits_per_sample == 8)
   {
      BYTE *pb = pMyClass->m_pOutBufferPointer;

      for(i=0;i<nBlockSize;i++)
      {
         for(j=0;j<nChannels;j++)
         {
            *pb = buffer[j][i] + 0x80;
            pb++;
         }
      }
   }
   else if(frame->header.bits_per_sample == 24)
   {
      struct INT24
      {
         BYTE b[3];
      };

      INT24 *pn24 = (INT24 *)((CDecodeFlac *)(client_data))->m_pOutBufferPointer;

      for(i=0;i<nBlockSize;i++)
      {
         for(j=0;j<nChannels;j++)
         {
            *pn24 = *(INT24 *)(&buffer[j][i]);
            pn24++;
         }
      }
   }
   else
      return FLAC__STREAM_DECODER_WRITE_STATUS_ABORT;

   pMyClass->m_nOutBufferLength += (frame->header.blocksize * frame->header.channels * frame->header.bits_per_sample / 8);

	return FLAC__STREAM_DECODER_WRITE_STATUS_CONTINUE;
}

void CDecodeFlac::error_callback(const FLAC__StreamDecoder *decoder, FLAC__StreamDecoderErrorStatus status, void *client_data) 
{

}

HRESULT CDecodeFlac::OnStart(void)
{  
   m_rtCurrent = 0;
   return S_OK;
}

HRESULT CDecodeFlac::OnSeek(void)
{  
   m_rtCurrent = 0;
   return S_OK;
}

HRESULT CDecodeFlac::OnTransform(IMediaSample *pInSample, IMediaSample *pOutSample)
{  

   BYTE *pbInBuffer = NULL;
   BYTE *pbOutBuffer = NULL;

   
   pInSample->GetPointer(&pbInBuffer);

   
   pOutSample->GetPointer(&pbOutBuffer);

   
   m_pInBufferPointer = pbInBuffer;
   m_nInBufferLength = pInSample->GetActualDataLength();

   m_pOutBufferPointer = pbOutBuffer;
   m_nOutBufferLength = 0;

   
   FLAC__stream_decoder_process_single(m_pDecoder);

   
   pOutSample->SetSyncPoint(TRUE);
   pOutSample->SetDiscontinuity(FALSE);
   pOutSample->SetPreroll(FALSE);
   pOutSample->SetActualDataLength(m_nOutBufferLength);

   
   REFERENCE_TIME rtStop;

   rtStop = m_rtCurrent + (REFERENCE_TIME)((LONGLONG)m_nOutBufferLength * ONE_SECOND / m_outFormat.nAvgBytesPerSec);
   pOutSample->SetTime(&m_rtCurrent, &rtStop);
   m_rtCurrent = rtStop;

   return S_OK;
}

HRESULT CDecodeFlac::OnConnectInPin(const CMediaType *pmtIn)
{  

   if(pmtIn->majortype != MEDIATYPE_Audio)
      return S_FALSE;

   if(pmtIn->subtype != MEDIASUBTYPE_FLAC_FRAMED)
      return S_FALSE;

   if(pmtIn->formattype != FORMAT_WaveFormatEx)
      return S_FALSE;

   int nHeaderLength = ((WAVEFORMATEX *)pmtIn->pbFormat)->cbSize;
   if(nHeaderLength == 0)
      return S_FALSE;

   
   if(m_pDecoder != NULL)
   {
      FLAC__stream_decoder_delete(m_pDecoder);
      m_pDecoder = NULL;
   }

   
   m_pDecoder = FLAC__stream_decoder_new();
   if(m_pDecoder == NULL)
      return S_FALSE;

   FLAC__StreamDecoderInitStatus status;
	FLAC__stream_decoder_set_md5_checking(m_pDecoder, false);
   status = FLAC__stream_decoder_init_stream(m_pDecoder, read_callback, NULL, NULL, NULL, NULL, write_callback, metadata_callback, error_callback, this);
   if(status != FLAC__STREAM_DECODER_INIT_STATUS_OK)
   {
      FLAC__stream_decoder_delete(m_pDecoder);
      m_pDecoder = NULL;
      return S_FALSE;
   }

   ::ZeroMemory(&m_outFormat, sizeof(m_outFormat));

   
   m_bMetaData = true;
   m_pInBufferPointer = pmtIn->Format() + sizeof(WAVEFORMATEX);
   m_nInBufferLength = nHeaderLength;
   FLAC__stream_decoder_process_until_end_of_metadata(m_pDecoder);
   m_bMetaData = false;

   
   m_outFormat.cbSize = 0;
   m_outFormat.wFormatTag = WAVE_FORMAT_PCM;
   m_outFormat.nBlockAlign = m_outFormat.nChannels * (m_outFormat.wBitsPerSample / 8);
   m_outFormat.nAvgBytesPerSec = m_outFormat.nSamplesPerSec * m_outFormat.nBlockAlign;

   if(m_outFormat.nAvgBytesPerSec == 0 || m_nMaxBlockSize == 0)
      return S_FALSE;

   return S_OK;
}

HRESULT CDecodeFlac::OnConnectOutPin(const CMediaType *pmtIn, int nInBufferSize, CMediaType *pmtOut, int *pnOutBufferSize)
{  

   
   pmtOut->SetType(&MEDIATYPE_Audio);
   pmtOut->SetSubtype(&MEDIASUBTYPE_PCM);
   pmtOut->SetTemporalCompression(FALSE);
   pmtOut->SetSampleSize(0);
   pmtOut->SetFormatType(&FORMAT_WaveFormatEx);
   pmtOut->SetFormat((BYTE*)&m_outFormat, sizeof(m_outFormat));

   
   *pnOutBufferSize = m_nMaxBlockSize * m_outFormat.nBlockAlign * 12 / 10;

   return S_OK;
}

CUnknown * WINAPI CDecodeFlac::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{  

   ASSERT(phr);
    
   CDecodeFlac *pNewObject = new CDecodeFlac(punk, phr);
   if (pNewObject == NULL)
   {
      if(phr != NULL)
         *phr = E_OUTOFMEMORY;
   }

   return dynamic_cast<CUnknown *>(pNewObject);
}

