using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using MinorShift.Emuera.GameData;
using MinorShift.Emuera.Sub;
using MinorShift.Emuera.GameData.Variable;
using MinorShift.Emuera.GameData.Function;
using MinorShift.Emuera.GameProc;

namespace MinorShift.Emuera.GameData.Expression
{
    internal static class ExpressionParser
    {
        /// <summary>
        /// 現在の仕様では実数の演算は扱わない。
        /// </summary>
        /// <param name="st"></param>
        /// <returns></returns>
        public static double ReduceDoubleTerm(StringStream st)
        {
            TokenReader.SkipWhiteSpace(st);
            return TokenReader.ReadDouble(st);
        }

		/// <summary>
		/// カンマで区切られた引数を一括して取得。行端か)で終わる。
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static IOperandTerm[] ReduceArguments(StringStream st, bool startWithBracket, bool isDefine)
		{
			List<IOperandTerm> terms = new List<IOperandTerm>();
			TokenReader.SkipWhiteSpace(st);
			if ((startWithBracket) && (st.Current != '('))
				throw new ExeEE("渡された情報が間違っている");
			if (startWithBracket)
				st.ShiftNext();
			while (true)
			{
				TokenReader.SkipWhiteSpace(st);
				TokenType type = TokenReader.GetNextTokenType(st);
				if ((st.EOS) || (type == TokenType.EndOfLine))
					break;
				if (st.Current == ')')
					break;
                if (isDefine)
                    terms.Add(ReduceFuncExpressionTerm(st, new char[] { ',', ')', '=' }));
                else
                    terms.Add(ReduceFuncExpressionTerm(st, new char[] { ',', ')' }));
				TokenReader.SkipWhiteSpace(st);
                if (isDefine)
                {
                    if ((st.Current != ',' && st.Current != '=') || st.EOS || TokenReader.GetNextTokenType(st) == TokenType.EndOfLine)
                    {
                        if (terms[terms.Count - 1].GetOperandType() == typeof(Int64))
                            terms.Add(new NullTerm(0));
                        else
                            terms.Add(new NullTerm(""));
                        break;
                    }
                    if (st.Current == '=')
                    {
                        st.ShiftNext();
                        TokenReader.SkipWhiteSpace(st);
                        if (TokenReader.GetNextTokenType(st) == TokenType.Operator && st.Current != '+' && st.Current != '-')
                            throw new CodeEE("関数定義の引数には変数を指定してください");
                        terms.Add(reduceTerm(st, false, false, new char[] { ',', ')' }));
                    }
                    else
                    {
                        if (terms[terms.Count - 1].GetOperandType() == typeof(Int64))
                            terms.Add(new NullTerm(0));
                        else
                            terms.Add(new NullTerm(""));
                    }
                }
                if (st.Current != ',')
                    break;
                st.ShiftNext();
            }
			if (startWithBracket)
			{
				if (st.Current == ')')
					st.ShiftNext();
				else
					throw new CodeEE("括弧が閉じられていません");
			}
			IOperandTerm[] ret = new IOperandTerm[terms.Count];
			terms.CopyTo(ret);
			return ret;
		}

		/// <summary>
		/// 数式または文字列式。CALLの引数などを扱う。nullを返すことがある。
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
        public static IOperandTerm ReduceExpressionTerm(StringStream st, char[] endChar)
        {
            TokenReader.SkipWhiteSpace(st);
            if ((st.EOS) || (st.Current == ','))
                return null;
            IOperandTerm term = reduceTerm(st, false, false, endChar);
            return term;
        }

        public static IOperandTerm ReduceFuncExpressionTerm(StringStream st, char[] endChar)
        {
            TokenReader.SkipWhiteSpace(st);
            if ((st.EOS) || (st.Current == ','))
                return null;
            IOperandTerm term = reduceTerm(st, false, false, endChar);
            return term;
        }
        /// <summary>
        /// 単純文字列、書式付文字列、文字列式のうち、文字列式を取り扱う。
        /// </summary>
        /// <param name="st"></param>
        /// <returns></returns>
        public static IOperandTerm ReduceStringTerm(StringStream st, char[] endChar)
        {
            TokenReader.SkipWhiteSpace(st);
            if (st.EOS)
                return new SingleTerm("");
            IOperandTerm term = reduceTerm(st, true, false, endChar);
            if (term.GetOperandType() != typeof(string))
                throw new CodeEE("式の結果が文字列ではありません");
            return term;
        }

        public static IOperandTerm ReduceTernaryStringTerm(string str)
        {
            TermStack stack = new TermStack();
            StringStream st = new StringStream(str);

            string left = ReduceInnerTerms(st, new char[]{'?'}, true).Trim();
            stack.Add(reduceTerm(new StringStream(left), false, false, null));
            stack.Add(OperatorManager.ToOperatorType("?"));
            st.ShiftNext();
            TokenReader.SkipWhiteSpace(st);

            string right1 = ReduceInnerTerms(st, new char[]{'#'}, false).Trim(new char[]{' ','\t'});
            StringForm temp = new StringForm(right1);
            temp.Reduce();
            stack.Add(new SingleTerm(right1));
            stack.Add(OperatorManager.ToOperatorType("#"));
            st.ShiftNext();
            TokenReader.SkipWhiteSpace(st);

            string right2 = ReduceInnerTerms(st, new char[]{'\0'}, false).Trim(new char[] { ' ','\t' });
            temp = new StringForm(right2);
            temp.Reduce();
            stack.Add(new SingleTerm(right2));
            return stack.ReduceAll();
        }

        private static string ReduceInnerTerms(StringStream st, IList<char> endChar, bool isBoolTerm)
        {
            bool isEscape = false;
            bool stringTerm = false;
            int intTermCount = 0;
            int inOneTermCount = 0;
            StringBuilder sb = new StringBuilder();
            while ((!endChar.Contains(st.Current) || isEscape || intTermCount > 0 || stringTerm || inOneTermCount > 0) && !st.EOS)
            {
                if (st.Current == '\\' && !isEscape)
                {
                    isEscape = true;
                    st.ShiftNext();
                    continue;
                }
                if (isEscape)
                {
                    switch (st.Current)
                    {
                        case '{':
                            sb.Append("\\{");
                            break;
                        case '}':
                            sb.Append("\\}");
                            break;
                        case '%':
                            sb.Append("\\%");
                            break;
                        default:
                            sb.Append(st.Current);
                            break;
                    }
                    isEscape = false;
                    st.ShiftNext();
                    continue;
                }
                else if (st.Current == '\"')
                {
                    sb.Append("\"" + TokenReader.ReadStringWithDoubleQuotation(st) + "\"" );
                    continue;
                }
                else if (st.Current == '{' && !isBoolTerm)
                {
                    intTermCount++;
                }
                else if (st.Current == '}' && !isBoolTerm)
                {
                    if (intTermCount < 1)
                        throw new CodeEE("対応する\"{\"のない\"}\"です");
                    intTermCount--;
                }
                else if (st.Current == '%' && intTermCount == 0 && !isBoolTerm)
                {
                    stringTerm = !stringTerm;
                }
                else if (st.Current == '(' && isBoolTerm)
                {
                    inOneTermCount++;
                }
                else if (st.Current == ')' && isBoolTerm)
                {
                    if (inOneTermCount < 1)
                        throw new CodeEE("対応する\"(\"のない\")\"です");
                    inOneTermCount--;
                }
                sb.Append(st.Current);
                st.ShiftNext();
            }
            if (intTermCount > 0)
                throw new CodeEE("対応する\"}\"のない\"{\"です");
            if (inOneTermCount > 0)
                throw new CodeEE("対応する\")\"のない\"(\"です");
            return sb.ToString();
        }

        public static IOperandTerm ReduceVariableArgument(StringStream st, VariableCode varCode)
        {
            TermStack stack = new TermStack();
            TokenReader.SkipWhiteSpace(st);
            TokenType type = TokenReader.GetNextTokenType(st);
            if ((st.EOS) || (type == TokenType.EndOfLine) || (type == TokenType.EndOfExpression))
                throw new CodeEE("変数の:の後に引数がありません");
            else if (type == TokenType.Numeric)
            {
                stack.Add(TokenReader.ReadInt64(st, false));
            }
            else if (type == TokenType.Identifer)
            {
                string idStr = TokenReader.ReadSingleIdentifer(st);
                TokenReader.SkipWhiteSpace(st);
                if (st.Current == '(')
                {//関数
                    IOperandTerm[] args = ReduceArguments(st, true, false);
                    IOperandTerm mToken = FunctionMethodCreator.GetFunctionMethod(Process.Instance.LabelDic, idStr, args);
                    if (mToken == null)
                    {
                        if (!Program.AnalysisMode)
                            throw new CodeEE("\"" + idStr + "\"は解釈できない識別子です");
                        else
                        {
                            if (Process.Instance.tempDic.ContainsKey(idStr))
                                Process.Instance.tempDic[idStr]++;
                            else
                                Process.Instance.tempDic.Add(idStr, 1);
                        }
                    }
                    stack.Add(mToken);
                }
                else
                {//変数
                    VariableIdentifier id = VariableIdentifier.GetVariableId(idStr);
                    if (id == null)
                    {
                        if (ConstantData.isDefined(varCode, idStr))
                            stack.Add(idStr);
                        else
                            throw new CodeEE("\"" + idStr + "\"は解釈できない識別子です");
                    }
                    else
                        stack.Add(VariableParser.ReduceVariable(id, null, null, null));
                }
            }
            else if (type == TokenType.StringToken)
            {
                string str = TokenReader.ReadStringWithDoubleQuotation(st);
                if (!ConstantData.isDefined(varCode, str))
                    throw new CodeEE("\"" + str + "\"は定義されていない識別子です");
                stack.Add(str);
            }
            else if (st.Current == '(')
            {
                st.ShiftNext();
                stack.Add(reduceTerm(st, false, false, new char[]{')'}));
                if (st.Current != ')')
                    throw new CodeEE("括弧が閉じられていません");
                st.ShiftNext();
            }
			else if (type == TokenType.Operator)
			{
				throw new CodeEE("変数の引数の読み取り中に予期しない演算子を発見しました");
			}
			else
				throw new CodeEE("変数の引数の読み取り中に予期しない文字\'" + st.Current + "\'を発見しました");
            return stack.ReduceAll();
        }

        public static IOperandTerm ReduceIntegerTerm(StringStream st, char[] endChar)
        {
            IOperandTerm term = reduceTerm(st, false, false, endChar);
            if (term.GetOperandType() != typeof(Int64))
                throw new CodeEE("式の結果が数値ではありません");
            return term;
        }

		/// <summary>
		/// カンマで区切られたCASEの引数を一括して取得。行端で終わる。
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static CaseExpression[] ReduceCaseExpressions(StringStream st)
		{
			List<CaseExpression> terms = new List<CaseExpression>();
			TokenReader.SkipWhiteSpace(st);
			while (true)
			{
				TokenReader.SkipWhiteSpace(st);
				TokenType type = TokenReader.GetNextTokenType(st);
				if ((st.EOS) || (type == TokenType.EndOfLine))
					break;
				terms.Add(reduceCaseExpression(st));
				TokenReader.SkipWhiteSpace(st);
				if (st.Current != ',')
					break;
				st.ShiftNext();
			}
			CaseExpression[] ret = new CaseExpression[terms.Count];
			terms.CopyTo(ret);
			return ret;
		}

		private static CaseExpression reduceCaseExpression(StringStream st)
		{
			CaseExpression ret = new CaseExpression();
			TokenReader.SkipWhiteSpace(st);
			if (st.CurrentEqualTo("IS", StringComparison.OrdinalIgnoreCase))
			{
				st.Seek(2, System.IO.SeekOrigin.Current);
				TokenType type = TokenReader.GetNextTokenType(st);
				if(type == TokenType.WhiteSpace || type == TokenType.Operator)
				{
					ret.CaseType = CaseExpressionType.Is;
					TokenReader.SkipWhiteSpace(st);
					if(TokenReader.GetNextTokenType(st) != TokenType.Operator)
						throw new CodeEE("ISキーワードの後に演算子がありません");
					OperatorCode op = TokenReader.ReadOperator(st);
					if (!OperatorManager.IsBinary(op))
						throw new CodeEE("ISキーワードの後の演算子が2項演算子ではありません");
					ret.Operator = op;
					ret.LeftTerm = reduceTerm(st, false, false, new char[]{','});
					return ret;
                }
				st.Seek(-2, System.IO.SeekOrigin.Current);
			}
			ret.LeftTerm = reduceTerm(st, false, true, new char[]{','});

			if (st.CurrentEqualTo("TO", StringComparison.OrdinalIgnoreCase))
			{
				ret.CaseType = CaseExpressionType.To;
				st.Seek(2, System.IO.SeekOrigin.Current);
                ret.RightTerm = reduceTerm(st, false, true, new char[] { ',' });
				if (st.CurrentEqualTo("TO", StringComparison.OrdinalIgnoreCase))
					throw new CodeEE("TOキーワードが2度使われています");
				if (ret.LeftTerm.GetOperandType() != ret.RightTerm.GetOperandType())
					throw new CodeEE("TOキーワードの前後の型が一致していません");
				return ret;
			}
			ret.CaseType = CaseExpressionType.Normal;
			return ret;
		}

        private static IOperandTerm reduceTerm(StringStream st, bool endWithPercent, bool allowKeywordTo, IList<char> endChar)
        {
            TermStack stack = new TermStack();
            int termCount = 0;
            int ternaryCount = 0;
            OperatorCode formerOp = OperatorCode.NULL;
            while (true)
            {
                TokenReader.SkipWhiteSpace(st);
                TokenType type = TokenReader.GetNextTokenType(st);
                if ((st.EOS) || (type == TokenType.EndOfLine))
                    //|| (type == TokenType.EndOfExpression))
                    break;
                if (st.Current == '　')
                    throw new CodeEE("構文中に全角スペースが含まれています");
                if (endChar != null)
                {
                    if (endChar.Contains(st.Current))
                        break;
                    else if (type == TokenType.EndOfExpression)
                        throw new CodeEE("解釈できない文字\"" + st.Current.ToString() + "\"が含まれています");
                }
                else if (type == TokenType.EndOfExpression)
                {
                    if (st.Current == ')')
                        throw new CodeEE("対応する\"(\"のない\")\"です");
                    else
                        throw new CodeEE("解釈できない文字\""+ st.Current.ToString() + "\"が含まれています");
                }
                if ((endWithPercent) && (st.Current == '%'))
                    break;
                termCount++;
                if (type == TokenType.Numeric)
                {
                    stack.Add(TokenReader.ReadInt64(st, false));
                    continue;
                }
                if (type == TokenType.Identifer)
                {
					string idStr = TokenReader.ReadSingleIdentifer(st);
					int skip = TokenReader.SkipWhiteSpace(st);
					if (st.Current == '(')
					{//関数
						IOperandTerm[] args = ReduceArguments(st,true, false);
						IOperandTerm mToken = FunctionMethodCreator.GetFunctionMethod(Process.Instance.LabelDic, idStr, args);
                        if (mToken == null)
                        {
                            if (!Program.AnalysisMode)
                                throw new CodeEE("\"" + idStr + "\"は解釈できない識別子です");
                            else
                            {
                                if (Process.Instance.tempDic.ContainsKey(idStr))
                                    Process.Instance.tempDic[idStr]++;
                                else
                                    Process.Instance.tempDic.Add(idStr, 1);
                                continue;
                            }
                        }
						stack.Add(mToken);
					}
					else
					{//変数
						if (idStr.Equals("TO", StringComparison.OrdinalIgnoreCase))
						{
							if (allowKeywordTo)
							{
								st.Seek(-(skip + 2), System.IO.SeekOrigin.Current);
								break;
							}
							else
								throw new CodeEE("TOキーワードはここでは使用できません");
						}
                        else if 
						 (idStr.Equals("IS", StringComparison.OrdinalIgnoreCase))
                            throw new CodeEE("ISキーワードはここでは使用できません");
                        else
                            stack.Add(VariableParser.ReduceVariable(idStr, st));
					}
                    continue;
                }
                if (type == TokenType.Operator)
                {
                    OperatorCode op = TokenReader.ReadOperator(st);
                    if (formerOp == OperatorCode.Equal || formerOp == OperatorCode.Greater || formerOp == OperatorCode.Less
                        || formerOp == OperatorCode.GreaterEqual || formerOp == OperatorCode.LessEqual || formerOp == OperatorCode.NotEqual)
                    {
                        if (op == OperatorCode.Equal || op == OperatorCode.Greater || op == OperatorCode.Less
                        || op == OperatorCode.GreaterEqual || op == OperatorCode.LessEqual || op == OperatorCode.NotEqual)
                        {
                            if (LogicalLineParser.printWarningLoad != null)
                            {
                                LogicalLineParser.printWarningLoad("（構文上の注意）比較演算子が連続しています。", LogicalLineParser.CurrentLine, 0, false, false);
                            }
                        }
                    }
                    stack.Add(op);
                    formerOp = op;
                    if (OperatorManager.ToOperatorString(op) == "?")
                        ternaryCount++;
                    else if (OperatorManager.ToOperatorString(op) == "#" && ternaryCount > 0)
                        ternaryCount--;
                    else if (OperatorManager.ToOperatorString(op) == "#")
                        throw new CodeEE("対応する\'?\'がありません");
                    continue;
                }
                if (type == TokenType.StringToken)
                {
                    stack.Add(TokenReader.ReadStringWithDoubleQuotation(st));
                    continue;
				}
				if (type == TokenType.StringFormToken)
				{
					if (st.Current != '@')
						throw new ExeEE("おかしい");
					st.ShiftNext();
					if (st.Current != '\"')
						throw new CodeEE("@の使い方が不正です");
					st.ShiftNext();
					StringForm sf = new StringForm("");
					sf.Reduce(st, '\"');
					if (st.Current != '\"')
						throw new CodeEE("@\"で文字列が始まっていますが\"で閉じられていません");
					st.ShiftNext();
					stack.Add(new StringFormTerm(sf));
					continue;
				}

                if (type == TokenType.TernaryOperator)
                {
                    if (st.Current != '\\')
                        throw new ExeEE("おかしい");
                    st.ShiftNext();
                    if (st.Current != '@')
                        throw new ExeEE("おかしい");
                    st.ShiftNext();
                    string ternaryStr = resolveEscape(st);
                    if (ternaryStr == null)
                        throw new CodeEE("対応する\\@が見つかりません");

                    stack.Add(ReduceTernaryStringTerm(ternaryStr));
                    st.ShiftNext();
                    continue;
                }
                if (st.Current == '(')
                {
                    st.ShiftNext();
                    stack.Add(reduceTerm(st, false, false, new char[]{')'}));
                    if (st.Current != ')')
                        throw new CodeEE("括弧が閉じられていません");
                    st.ShiftNext();
                    continue;
                }
				throw new CodeEE("式の解析中に予期しない文字\'" + st.Current + "\'を発見しました");
            }
            if (ternaryCount > 0)
                throw new CodeEE("\'?\'と\'#\'の数が正しく対応していません");
            return stack.ReduceAll();
        }


        public static bool checkTernary(StringStream st)
        {
            int count = 0;
            while (!st.EOS)
            {
                TokenReader.SkipWhiteSpace(st);
                //三項演算子が来る前に%が来たらなしで確定
                if (st.Current == '%')
                    return false;
                if (st.Current == ',')
                {
                    //書式指定の","が来たので、次の特殊処理をする
                    count++;
                    st.ShiftNext();
                } 
                TokenType type = TokenReader.GetNextTokenType(st);
                if (type == TokenType.TernaryOperator)
                    return true;
                if (type == TokenType.Numeric)
                {
                    TokenReader.ReadInt64(st, false);
                    continue;
                }
                if (type == TokenType.Identifer)
                {
                    string idStr = TokenReader.ReadSingleIdentifer(st);
                    if (count == 2 && ((idStr.ToUpper() == "LEFT") || (idStr.ToUpper() == "RIGHT")))
                        continue;
                    int skip = TokenReader.SkipWhiteSpace(st);
                    if (st.Current == '(')
                    {//関数
                        IOperandTerm[] args = ReduceArguments(st, true, false);
                    }
                    else
                    {//変数
                        VariableParser.ReduceVariable(idStr, st);
                    }
                    continue;
                }
                if (type == TokenType.Operator)
                {
                    OperatorCode op = TokenReader.ReadOperator(st);
                    continue;
                }
                if (type == TokenType.StringToken)
                {
                    TokenReader.ReadStringWithDoubleQuotation(st);
                    continue;
                }
                if (type == TokenType.StringFormToken)
                {
                    if (st.Current != '@')
                        throw new ExeEE("おかしい");
                    st.ShiftNext();
                    if (st.Current != '\"')
                        throw new CodeEE("@の使い方が不正です");
                    st.ShiftNext();
                    StringForm sf = new StringForm("");
                    sf.Reduce(st, '\"');
                    if (st.Current != '\"')
                        throw new CodeEE("@\"で文字列が始まっていますが\"で閉じられていません");
                    st.ShiftNext();
                    continue;
                }
                //ひっかからないなら次の文字へ
                st.ShiftNext();
            }        
            return false;
        }

        public static string resolveEscape(StringStream st)
        {
            bool isEscape = false;
            StringBuilder sb = new StringBuilder();
            while (!(st.Current == '@' && isEscape) && !st.EOS && (st.Current != '\0'))
            {
                if (st.Current == '\\' && !isEscape)
                {
                    isEscape = true;
                    st.ShiftNext();
                    continue;
                }
                if (isEscape)
                {
                    switch (st.Current)
                    {
                        case StringStream.EndOfString:
                            throw new CodeEE("エスケープ文字\\の後に文字がありません");
                        case '\n':
                            break;
                        case 's':
                            sb.Append(' ');
                            break;
                        case 'S':
                            sb.Append('　');
                            break;
                        case 't':
                            sb.Append('\t');
                            break;
                        case 'n':
                            sb.Append('\n');
                            break;
                        default:
                            sb.Append('\\');
                            sb.Append(st.Current);
                            break;
                    }
                    isEscape = false;
                    st.ShiftNext();
                    continue;
                }
                sb.Append(st.Current);
                st.ShiftNext();
            }
            if (!(isEscape && st.Current == '@'))
                return null;
            return sb.ToString();
        }

        /// <summary>
        /// 式解決用クラス
        /// </summary>
        private class TermStack
        {
            /// <summary>
            /// 次に来るべきものの種類。
            /// (前置)単項演算子か値待ちなら0、二項・三項演算子待ちなら1、値待ちなら2、++、--、!に対応する値待ちの場合は3。
            /// </summary>
            int state = 0;
            bool hasBefore = false;
            bool hasAfter = false;
            bool waitAfter = false;
            Stack<Object> stack = new Stack<Object>();
            public void Add(OperatorCode op)
            {
                if (state == 2 || state == 3)
                    throw new CodeEE("式が異常です");
                if (state == 0)
                {
                    if (!OperatorManager.IsUnary(op))
                        throw new CodeEE("式が異常です");
                    stack.Push(op);
                    if (op == OperatorCode.Plus || op == OperatorCode.Minus || op == OperatorCode.BitNot)
                        state = 2;
                    else
                        state = 3;
                    return;
                }
                if (state == 1)
                {
                    //後置単項演算子の場合は特殊処理へ
                    if (OperatorManager.IsUnaryAfter(op))
                    {
                        if (hasAfter)
                        {
                            hasAfter = false;
                            throw new CodeEE("後置の単項演算子が複数存在しています");
                        }
                        if (hasBefore)
                        {
                            hasBefore = false;
                            throw new CodeEE("インクリメント・デクリメントを前置・後置両方同時に使うことはできません");
                        }
                        stack.Push(op);
                        reduceUnaryAfter();
                        //前置単項演算子が処理を待っている場合はここで解決
                        if (waitAfter)
                            reduceUnary();
                        hasBefore = false;
                        hasAfter = true;
                        waitAfter = false;
                        return;
                    }
                    if (!OperatorManager.IsBinary(op) && !OperatorManager.IsTernary(op))
                        throw new CodeEE("式が異常です");
                    //先に未解決の前置演算子解決
                    if (waitAfter)
                        reduceUnary();
                    int priority = OperatorManager.GetPriority(op);
                    //直前の計算の優先度が同じか高いなら還元。
                    while (lastPriority() >= priority)
                    {
                        this.reduceLastThree();
                    }
                    stack.Push(op);
                    state = 0;
                    waitAfter = false;
                    hasBefore = false;
                    hasAfter = false;
                    return;
                }
                throw new CodeEE("式が異常です");
            }
            public void Add(Int64 i) { Add(new SingleTerm(i)); }
            public void Add(string s) { Add(new SingleTerm(s)); }
            public void Add(IOperandTerm term)
            {
                stack.Push(term);
                if (state == 1)
                    throw new CodeEE("式が異常です");
                if (state == 2)
                    waitAfter = true;
                if (state == 3)
                {
                    reduceUnary();
                    hasBefore = true;
                }
                state = 1;
                return;
            }

            private void reduceUnary()
            {
                if (stack.Count < 2)
                    throw new ExeEE("不正な時期の呼び出し");
                IOperandTerm operand = (IOperandTerm)stack.Pop();
                OperatorCode op = (OperatorCode)stack.Pop();
                Type retType = OperatorMethod.GetReturnType(operand.GetOperandType(), op);
                UnaryMethod method = OperatorMethod.GetUnaryMethod(op);
                if ((retType == typeof(void)) || (method == null))
                {
                    string errMes = "";
                    if (operand.GetOperandType() == typeof(Int64))
                        errMes += "数値型";
                    else if (operand.GetOperandType() == typeof(string))
                        errMes += "文字列型";
                    else
                        errMes += "不定型";
                    errMes += "に単項演算子\'" + OperatorManager.ToOperatorString(op) + "\'は適用できません";
                    throw new CodeEE(errMes);
                }
                if (operand is SingleTerm)
                {
                    stack.Push(method(operand, GetSingle));
                }
                else
                {
                    stack.Push(new UnaryExpressionTerm(operand, op, method));
                }
            }

            private void reduceUnaryAfter()
            {
                if (stack.Count < 2)
                    throw new ExeEE("不正な時期の呼び出し");
                OperatorCode op = (OperatorCode)stack.Pop();
                IOperandTerm operand = (IOperandTerm)stack.Pop();
                Type retType = OperatorMethod.GetReturnType(operand.GetOperandType(), op);
                UnaryAfterMethod method = OperatorMethod.GetUnaryAfterMethod(op);
                if (operand is VariableToken)
                {
                    if ((retType == typeof(void)) || (method == null))
                    {
                        string errMes = "";
                        if (operand.GetOperandType() == typeof(Int64))
                            errMes += "数値型";
                        else if (operand.GetOperandType() == typeof(string))
                            errMes += "文字列型";
                        else
                            errMes += "不定型";
                        errMes += "に後置単項演算子\'" + OperatorManager.ToOperatorString(op) + "\'は適用できません";
                        throw new CodeEE(errMes);
                    }
                    stack.Push(new UnaryAfterExpressionTerm(operand, op, method));
                }
                else
                {
                    throw new CodeEE("変数以外をインクリメントすることはできません");
                }
            }

            private int lastPriority()
            {
                if (stack.Count < 3)
                    return -1;
                object temp = (object)stack.Pop();
                OperatorCode opCode = (OperatorCode)stack.Peek();
                int priority = OperatorManager.GetPriority(opCode);
                stack.Push(temp);
                return priority;
            }

            public IOperandTerm ReduceAll()
            {
                if (stack.Count == 0)
                    return new SingleTerm(0);
                if (state != 1)
                    throw new CodeEE("式が異常です");
                //単項演算子の待ちが未解決の時はここで解決
                if (waitAfter)
                    reduceUnary();
                waitAfter = false;
                hasBefore = false;
                hasAfter = false;
                while (stack.Count > 1)
                {
                    reduceLastThree();
                }
                return (IOperandTerm)stack.Pop();
            }



            private void reduceLastThree()
            {
                if (stack.Count < 2)
                    throw new ExeEE("不正な時期の呼び出し");
                IOperandTerm right = (IOperandTerm)stack.Pop();//後から入れたほうが右側
                OperatorCode op = (OperatorCode)stack.Pop();
                IOperandTerm left = (IOperandTerm)stack.Pop();
                if (OperatorManager.IsTernary(op))
                {
                    if (stack.Count > 1)
                    {
                        reduceTernary(left, right, op);
                        return;
                    }
                    throw new CodeEE("式の数が不足しています");
                }
                if (!OperatorManager.IsBinary(op))
                    throw new ExeEE("異常な式を受け入れてしまっている");
                Type retType = OperatorMethod.GetReturnType(right.GetOperandType(), left.GetOperandType(), op);

                BinaryMethod method = OperatorMethod.GetBinaryMethod(op);
                if ((retType == typeof(void)) || (method == null))
                {
                    string errMes = "";
                    if (left.GetOperandType() == typeof(Int64))
                        errMes += "数値型と";
                    else if (left.GetOperandType() == typeof(string))
                        errMes += "文字列型と";
                    else
                        errMes += "不定型と";
                    if (right.GetOperandType() == typeof(Int64))
                        errMes += "数値型の";
                    else if (right.GetOperandType() == typeof(string))
                        errMes += "文字列型の";
                    else
                        errMes += "不定型の";
                    errMes += "演算に二項演算子\'" + OperatorManager.ToOperatorString(op) + "\'は適用できません";
                    throw new CodeEE(errMes);
                }
                if ((right is SingleTerm) && (left is SingleTerm))
                {
                    stack.Push(method(left, right, GetSingle));
                }
                else
                {
                    stack.Push(new BinaryExpressionTerm(left, right, op, method));
                }
			}

            private void reduceTernary(IOperandTerm left, IOperandTerm right, OperatorCode op)
            {
                OperatorCode newOp = (OperatorCode)stack.Pop();
                IOperandTerm newLeft = (IOperandTerm)stack.Pop();
                Type retType = OperatorMethod.GetReturnType(newLeft.GetOperandType(), left.GetOperandType(), right.GetOperandType(), op);

                TernaryMethod method = OperatorMethod.GetTernaryMethod(newOp);
                if ((retType == typeof(void)) || (method == null))
                {
                    string errMes = "";
                    if (retType == typeof(void))
                        errMes += "型が一致しません";
                    else
                        errMes += "演算子が存在しません";
                    throw new CodeEE(errMes);
                }
                if ((right is SingleTerm) && (left is SingleTerm) && (newLeft is SingleTerm))
                {
                    stack.Push(method(newLeft, left, right, GetSingle));
                }
                else
                {
                    stack.Push(new TernaryExpressionTerm(newLeft, left, right, newOp, method));
                }
            }

			SingleTerm GetSingle(IOperandTerm oprand)
			{
				return (SingleTerm)oprand;
			}
        }

    }
}