using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using MinorShift.Emuera.Sub;
using MinorShift.Emuera.GameData;
using MinorShift.Emuera.GameView;
using MinorShift.Emuera.GameData.Expression;
using MinorShift.Emuera.GameData.Variable;

namespace MinorShift.Emuera.GameProc
{

	internal sealed partial class Process
	{
		private sealed class ErbLoader
		{
			public ErbLoader(EmueraConsole main, VariableEvaluator var)
			{
				output = main;
				varData = var;
			}
			readonly EmueraConsole output;
			readonly VariableEvaluator varData;
			Dictionary<string, int> ignoredWarningCount = new Dictionary<string, int>();
			List<string> ignoredFNFWarningFileList = new List<string>();
			int ignoredFNFWarningCount = 0;

			int enabledLineCount = 0;
			int lineCount = 0;
			LabelDictionary labelDic;

            bool noError = true;

			/// <summary>
			/// 複数のファイルを読む
			/// </summary>
			/// <param name="filepath"></param>
			public bool LoadErbFiles(string erbDir, bool displayReport, LabelDictionary labelDictionary)
			{
				//1.713 labelDicをnewする位置を変更。
				//checkScript();の時点でExpressionPerserがProcess.instance.LabelDicを必要とするから。
				labelDic = labelDictionary;
				List<KeyValuePair<string, string>> erbFiles = Config.Instance.GetFiles(erbDir, "*.ERB");
				int first = Environment.TickCount;
                noError = true;
                try
                {
                    for (int i = 0; i < erbFiles.Count; i++)
                    {
                        string file = erbFiles[i].Value;
                        string filename = erbFiles[i].Key;
#if DEBUG
						if (Config.Instance.DisplayReport)
							output.PrintLine(filename + "読み込み中・・・" + (Environment.TickCount - first).ToString() + ":");
#else
                        if (displayReport)
                            output.PrintLine(filename + "読み込み中・・・");
#endif
                        System.Windows.Forms.Application.DoEvents();
                        loadErb(file, filename);
                        //読み込んだファイルのパスを記録
                        //一部ファイルの再読み込み時の処理用
                        loadedFileList.Add(filename);
                    }
                    checkScript();
                }
                catch (Exception e)
                {
                    System.Media.SystemSounds.Hand.Play();
                    output.PrintError("予期しないエラーが発生しました");
                    output.PrintError(e.GetType().ToString() + ":" + e.Message);
                    return false;
                }
				return noError;
			}

            /// <summary>
            /// 指定されたファイルを読み込む
            /// </summary>
            /// <param name="filename"></param>
            public bool loadErbs(List<string> path, LabelDictionary labelDictionary)
            {
                string fname;
                noError = true;
                labelDic = labelDictionary;
                foreach (string fpath in path)
                {
                    if (fpath.StartsWith(Program.ExeDir) && !Program.AnalysisMode)
                        fname = fpath.Substring(Program.ExeDir.Length + "erb\\".Length);
                    else
                        fname = fpath;
                    if (Program.AnalysisMode)
                        output.PrintLine(fname + "読み込み中・・・");
                    System.Windows.Forms.Application.DoEvents();
                    loadErb(fpath, fname);
                    if (!Program.AnalysisMode)
                        labelDic.checkLabelList(fname, loadedFileList);
                }
                checkScript();
                return noError;
            }

			/// <summary>
			/// ファイル一つを読む
			/// </summary>
			/// <param name="filepath"></param>
			private void loadErb(string filepath, string filename)
			{
				EraStreamReader eReader = new EraStreamReader();
                if (!eReader.Open(filepath, filename))
				{
                    output.PrintError(eReader.Filename + "のオープンに失敗しました");
					return;
				}
                try
                {
                    string line = null;
                    LogicalLine nextLine = new NullLine();
                    LogicalLine lastLine = new NullLine();
                    FunctionLabelLine lastLabelLine = null;
                    InvalidLabelLine lastInvalidLabelLine = null;
                    StringStream st = null;
                    bool skip = false;
					bool ifdebug = false;
					bool ifndebug = false;
                    string rowLine = null;
					ScriptPosition position = null;
                    int funcCount = 0;
                    if (Program.AnalysisMode)
                        output.Print("　");
                    while ((line = eReader.ReadLine()) != null)
                    {
                        lineCount++;
                        if (line.Length == 0)
                            continue;
                        rowLine = line;
                        st = new StringStream(line);
                        TokenReader.SkipWhiteSpace(st);
                        //命令の前に全角スペースが来た行は無視される(eramakerの仕様)
                        if (st.Current == ';' || st.Current == '　')
                        {
							if (st.CurrentEqualTo(";#;") && Program.DebugMode)
							{
								st.ShiftNext();
								st.ShiftNext();
								st.ShiftNext();
								TokenReader.SkipWhiteSpace(st);
							}
							else
							{
								if (!st.CurrentEqualTo(";!;"))
									continue;
								st.ShiftNext();
								st.ShiftNext();
								st.ShiftNext();
								TokenReader.SkipWhiteSpace(st);
								if (st.Current == ';')
									continue;
							}
						}
                        if (st.EOS)
                            continue;
                        //全置換はここで対応
                        if (Config.Instance.UseRenameFile && renameDic != null && ((rowLine.Trim().IndexOf("[[") == 0) && (rowLine.Trim().IndexOf("]]") == rowLine.Trim().Length - 2)))
                        {
                            line = st.Substring();
                            foreach (KeyValuePair<string, string> pair in renameDic)
                                line = line.Replace(pair.Key, pair.Value);
                            st = new StringStream(line);
                        }
                        position = new ScriptPosition(eReader.Filename, eReader.LineNo, rowLine);
                        //命令文の一部を[[]]置換できるのはよろしくないので、処理位置変更に対応
                        if (st.Current == '[' && st.Next != '[')
                        {
                            st.ShiftNext();
                            if (TokenReader.GetNextTokenType(st) != TokenType.Identifer)
                                output.PrintWarning("[]の使い方が不正です", position, 1);
                            string token = TokenReader.ReadSingleIdentifer(st);
                            if ((token == null) || (st.Current != ']'))
                                output.PrintWarning("[]の使い方が不正です", position, 1);
                            switch (token)
                            {
                                case "SKIPSTART":
                                    if (skip)
                                        output.PrintWarning("[SKIPSTART]が重複して使用されています", position, 1);
                                    skip = true;
                                    break;
                                case "SKIPEND":
									if (!skip)
										output.PrintWarning("[SKIPSTART]と対応しない[SKIPEND]です", position, 1);
									skip = false;
									break;
								case "IF_DEBUG":
									if (ifdebug)
										output.PrintWarning("[IF_DEBUG]が重複して使用されています", position, 1);
									else if (ifndebug)
										output.PrintWarning("[IF_DEBUG]と[IF_NDEBUG]が重複して使用されています", position, 1);
									else
										ifdebug = true;
									break;
								case "IF_NDEBUG":
									if (ifndebug)
										output.PrintWarning("[IF_NDEBUG]が重複して使用されています", position, 1);
									else if (ifdebug)
										output.PrintWarning("[IF_DEBUG]と[IF_NDEBUG]が重複して使用されています", position, 1);
									else
										ifndebug = true;
									break;
								case "ENDIF":
									if (!ifdebug && !ifndebug)
										output.PrintWarning("[IF_DEBUG]又は[IF_NDEBUG]と対応しない[ENDIF]です", position, 1);
									ifdebug = false;
									ifndebug = false;
									break;
                                default:
                                    output.PrintWarning("認識できないプリプロセッサです", position, 1);
                                    break;
                            }
                            st.ShiftNext();
                            if (!st.EOS)
                                output.PrintWarning("[" + token + "]の後ろは無視されます。", position, 1);
                            continue;
                        }
                        if ((skip) || (Program.DebugMode && ifndebug) || (!Program.DebugMode &&ifdebug))
                            continue;
                        //ここまでプリプロセッサ

                        if (st.Current == '#')
                        {
                            if (!(lastLine is FunctionLabelLine) && !(lastLine is InvalidLabelLine))
                            {
                                output.PrintWarning("関数宣言の直後以外で#行が使われています", position, 1);
                                continue;
                            }
                            st.ShiftNext();
                            string token = TokenReader.ReadSingleIdentifer(st);
                            if (token == null)
                            {
                                output.PrintWarning("解釈できない#行です", position, 1);
                                continue;
                            }
                            if (Config.Instance.IgnoreCase)
                                token = token.ToUpper();
                            if (lastLine is FunctionLabelLine)
                            {
                                FunctionLabelLine label = (FunctionLabelLine)lastLine;
                                switch (token)
                                {
                                    case "SINGLE":
                                        if (label.IsMethod)
                                            output.PrintWarning("式中関数では#SINGLEは機能しません", position, 1);
                                        else if (label.Depth != 0)
                                            output.PrintWarning("システム関数以外では#SINGLEは機能しません", position, 1);
                                        label.IsSingle = true;
                                        break;
                                    case "LATER":
                                        if (label.IsMethod)
                                            output.PrintWarning("式中関数では#LATERは機能しません", position, 1);
                                        else if (label.Depth != 0)
                                            output.PrintWarning("システム関数以外では#LATERは機能しません", position, 1);
                                        label.Priority++;
                                        break;
                                    case "PRI":
                                        if (label.IsMethod)
                                            output.PrintWarning("式中関数では#PRIは機能しません", position, 1);
                                        else if (label.Depth != 0)
                                            output.PrintWarning("システム関数以外では#PRIは機能しません", position, 1);
                                        label.Priority--;
                                        break;
                                    case "FUNCTION":
                                    case "FUNCTIONS":
                                        if (char.IsDigit(label.LabelName[0]))
                                        {
                                            output.PrintWarning("#" + token + "属性は関数名が数字で始まる関数には指定できません", position, 1);
                                            label.IsError = true;
                                            label.ErrMes = "関数名が数字で始まっています";
                                            break;
                                        }
                                        if (label.IsMethod)
                                        {
                                            if ((label.MethodType == typeof(Int64) && token == "FUNCTION") || (label.MethodType == typeof(string) && token == "FUNCTIONS"))
                                            {
                                                output.PrintWarning("関数" + label.LabelName + "にはすでに" + token + "が宣言されています(この行は無視されます)", position, 1);
                                                continue;
                                            }
                                            noError = false;
                                            if (label.MethodType == typeof(Int64) && token == "FUNCTIONS")
                                                output.PrintWarning("関数" + label.LabelName + "にはすでに#FUNCTIONが宣言されています", position, 2);
                                            else if (label.MethodType == typeof(string) && token == "FUNCTION")
                                                output.PrintWarning("関数" + label.LabelName + "にはすでに#FUNCTIONSが宣言されています", position, 2);
                                            break;
                                        }
                                        if (label.Depth == 0)
                                        {
                                            noError = false;
                                            output.PrintWarning("システム関数に#" + token + "が指定されています", position, 2);
                                            break;
                                        } 
                                        label.IsMethod = true;
                                        label.Depth = 0;
                                        if (token == "FUNCTIONS")
                                            label.MethodType = typeof(string);
                                        else
                                            label.MethodType = typeof(Int64);
                                        break;
                                    default:
                                        output.PrintWarning("解釈できない#行です", position, 1);
                                        break;
                                }
                            }
                            else
                            {
                                InvalidLabelLine label = (InvalidLabelLine)lastLine;
                                switch (token)
                                {
                                    case "SINGLE": label.IsSingle = true; break;
                                    case "LATER": label.Priority++; break;
                                    case "PRI": label.Priority--; break;
                                    case "FUNCTION":
                                    case "FUNCTIONS":

                                        label.IsMethod = true;
                                        label.Depth = 0;
                                        if (token == "FUNCTIONS")
                                            label.MethodType = typeof(string);
                                        else
                                            label.MethodType = typeof(Int64);
                                        break;
                                    default:
                                        output.PrintWarning("解釈できない#行です", position, 1);
                                        break;
                                }
                            }
                            continue;
                        }
						nextLine = LogicalLineParser.ParseLine(st, position, output, renameDic);

                        if (nextLine == null)
                            continue;
                        if (nextLine is InvalidLine)
                        {
                            noError = false;
                            output.PrintWarning(nextLine.ErrMes, position, 2);
                        }
                        else if (nextLine is InvalidLabelLine)
                        {
                            noError = false;
                            output.PrintWarning(nextLine.ErrMes, position, 2);
                            lastLabelLine = null;
                            lastInvalidLabelLine = (InvalidLabelLine)nextLine;
                        }
                        else if (nextLine is FunctionLabelLine)
                        {
                            FunctionLabelLine label = (FunctionLabelLine)nextLine;
                            labelDic.AddLabel(label);
                            lastLabelLine = label;
                            lastInvalidLabelLine = null;
                            if (labelDic.IsOverride(label) && (Config.Instance.WarnNormalFunctionOverloading || Program.AnalysisMode))
                            {
                                ScriptPosition pos = labelDic.GetLabel(label.LabelName).Position;
                                output.PrintWarning("関数@" + label.LabelName + "は既に定義(" + pos.Filename + "の" + pos.LineNo.ToString() + "行目)されています", position, 1);
                            }
                            funcCount++;
                            if (Program.AnalysisMode && ((funcCount % Config.Instance.PrintCPerLine) == 0))
                            {
                                output.NewLine();
                                output.Print("　");
                            }
                        }
                        else if (nextLine is GotoLabelLine)
                        {
                            GotoLabelLine gotoLabel = (GotoLabelLine)nextLine;
                            gotoLabel.SetParentFunction(lastLabelLine);
                            if (!labelDic.AddLabelDollar((GotoLabelLine)nextLine))
                            {
                                ScriptPosition pos = labelDic.GetLabelDollar(gotoLabel.LabelName, lastLabelLine).Position;
								output.PrintWarning("ラベル名$" + gotoLabel.LabelName + "は既に同じ関数内(" + pos.Filename + "の" + pos.LineNo.ToString() + "行目)で使用されています", position, 2);
                            }
                        }
                        if (lastLabelLine == null && lastInvalidLabelLine == null)
                            output.PrintWarning("関数が定義されるより前に行があります", position, 1);
                        else if (nextLine != null)
                            lastLine = addLine(nextLine, lastLine);
                        else
                            lastLine = null;
                    }
                    addLine(new NullLine(), lastLine);
					if (skip)
						output.PrintWarning("[SKIPSTART]に対応する[SKIPEND]がありません", position, 1);
					if (ifdebug)
						output.PrintWarning("[IF_DEBUG]に対応する[ENDIF]がありません", position, 1);
					if (ifndebug)
						output.PrintWarning("[IF_NDEBUG]に対応する[ENDIF]がありません", position, 1);
                }
                finally
				{
					eReader.Close();
				}
				return;
			}


			private LogicalLine addLine(LogicalLine nextLine, LogicalLine lastLine)
			{
				if (nextLine == null)
					return null;
				enabledLineCount++;
				//nextLine.PrevLine = lastLine;
				lastLine.NextLine = nextLine;
				return nextLine;
			}

			Dictionary<string, string> renameDic = null;
			internal void LoadEraExRenameFile(string filepath)
			{
				EraStreamReader eReader = new EraStreamReader();
				if ((!File.Exists(filepath)) || (!eReader.Open(filepath)))
				{
					return;
				}

				renameDic = new Dictionary<string, string>();
				string line = null;
				ScriptPosition pos = null;
                Regex reg = new Regex(@"\\,", RegexOptions.Compiled);
				try
				{
					while ((line = eReader.ReadLine()) != null)
					{
						if (line.Length == 0)
							continue;
						if (line.StartsWith(";"))
							continue;
                        string[] baseTokens = reg.Split(line);
                        if (!baseTokens[baseTokens.Length - 1].Contains(","))
                            continue;
                        string[] last = baseTokens[baseTokens.Length - 1].Split(',');
                        baseTokens[baseTokens.Length - 1] = last[0];
                        string[] tokens = new string[2];
                        tokens[0] = string.Join(",", baseTokens);
                        tokens[1] = last[1];
                        pos = new ScriptPosition(eReader.Filename, eReader.LineNo, line);
						//右がERB中の表記、左が変換先になる。
						string value = tokens[0].Trim();
						string key = string.Format("[[{0}]]", tokens[1].Trim());
						renameDic[key] = value;
						pos = null;
					}
				}
				catch (Exception e)
				{
					if (pos != null)
						throw new CodeEE(e.Message, pos);
					else
						throw new CodeEE(e.Message);

				}
				finally
				{
					eReader.Close();
				}

			}

			public bool useCallForm = false;
			/// <summary>
			/// 読込終わったファイルをチェックする
			/// </summary>
			private void checkScript()
			{
				int usedLabelCount = 0;
				int labelDepth = -1;
				List<FunctionLabelLine> labelList = labelDic.GetAllLabels();
				while (true)
				{
					labelDepth++;
					int countInDepth = 0;
					foreach (FunctionLabelLine label in labelList)
					{
                        if ((label.Depth != labelDepth) && !Program.AnalysisMode)
                            continue;
						usedLabelCount++;
						countInDepth++;
						checkFunctionWithCatch(label);
					}
                    if (countInDepth == 0 || Program.AnalysisMode)
                        break;
				}
				labelDepth = -1;
				List<string> ignoredFNCWarningFileList = new List<string>();
				int ignoredFNCWarningCount = 0;

				bool ignoreAll = false;
				switch (Config.Instance.FunctionNotCalledWarning)
				{
					case DisplayWarningFlag.IGNORE:
					case DisplayWarningFlag.LATER:
						ignoreAll = true;
						break;
				}
				if (useCallForm)
				{//callform系が使われたら全ての関数が呼び出されたとみなす。
					foreach (FunctionLabelLine label in labelList)
					{
						if (label.Depth != labelDepth)
							continue;
						checkFunctionWithCatch(label);
					}
				}
				else
				{
					foreach (FunctionLabelLine label in labelList)
					{
						if (label.Depth != labelDepth)
							continue;
						bool ignore = false;
						if (Config.Instance.FunctionNotCalledWarning == DisplayWarningFlag.ONCE)
						{
							string filename = label.Position.Filename.ToUpper();

							if (!string.IsNullOrEmpty(filename))
							{
								if (ignoredFNCWarningFileList.Contains(filename))
								{
									ignore = true;
								}
								else
								{
									ignore = false;
									ignoredFNCWarningFileList.Add(filename);
								}
							}
                            break;
						}
						if (ignoreAll || ignore)
							ignoredFNCWarningCount++;
						else
                            printWarning("関数@" + label.LabelName + "は定義されていますが一度も呼び出されません", label, 1, false, false);
						if (!Config.Instance.IgnoreUncalledFunction)
							checkFunctionWithCatch(label);
						else
						{
							if (!(label.NextLine is NullLine) && !(label.NextLine is FunctionLabelLine))
							{
								if (!label.NextLine.IsError)
								{
									label.NextLine.IsError = true;
									label.NextLine.ErrMes = "呼び出されないはずの関数が呼ばれた";
								}
							}
						}
					}
				}
                if (Program.AnalysisMode && (warningDic.Keys.Count > 0 || Process.Instance.tempDic.Keys.Count > 0))
                {
                    output.PrintError("・定義が見つからなかった関数: 他のファイルで定義されている場合はこの警告は無視できます");
                    if (warningDic.Keys.Count > 0)
                    {
                        output.PrintError("　○一般関数:");
                        foreach (string labelName in warningDic.Keys)
                        {
                            output.PrintError("　　" + labelName + ": " + warningDic[labelName].ToString() + "回");
                        }
                    }
                    if (Process.Instance.tempDic.Keys.Count > 0)
                    {
                        output.PrintError("　○文中関数:");
                        foreach (string labelName in Process.Instance.tempDic.Keys)
                        {
                            output.PrintError("　　" + labelName + ": " + Process.Instance.tempDic[labelName].ToString() + "回");
                        }
                    }
                }
                else
                {
                    if ((ignoredFNCWarningCount > 0) && (Config.Instance.DisplayWarningLevel <= 1) && (Config.Instance.FunctionNotCalledWarning != DisplayWarningFlag.IGNORE))
                        output.PrintError(string.Format("警告Lv1:定義された関数が一度も呼び出されていない事に関する警告を{0}件無視しました", ignoredFNCWarningCount));
                    if ((ignoredFNFWarningCount > 0) && (Config.Instance.DisplayWarningLevel <= 2) && (Config.Instance.FunctionNotCalledWarning != DisplayWarningFlag.IGNORE))
                        output.PrintError(string.Format("警告Lv2:定義されていない関数を呼び出した事に関する警告を{0}件無視しました", ignoredFNFWarningCount));
                    foreach (KeyValuePair<string, int> pair in ignoredWarningCount)
                    {
                        output.PrintError(string.Format("警告Lv{0}以上:{1}:{2}件のエラーを無視しました", Config.Instance.DisplayWarningLevel, pair.Key, pair.Value));
                    }
                }
                if (Config.Instance.DisplayReport)
                    output.PrintError(string.Format("全行数:{0}, 非コメント行数:{1}, 全関数合計:{2}, 被呼出関数合計:{3}", lineCount, enabledLineCount, labelDic.Count, usedLabelCount));
				if (Config.Instance.AllowFunctionOverloading && Config.Instance.WarnFunctionOverloading)
				{
					List<string> overloadedList = MinorShift.Emuera.GameData.Function.FunctionMethodCreator.GetOverloadedList(labelDic);
					if (overloadedList.Count > 0)
					{
						output.NewLine();
						output.PrintError("＊＊＊＊＊警告＊＊＊＊＊");
						foreach (string funcname in overloadedList)
						{
							output.Print("　システム関数\"" + funcname + "\"がユーザー定義関数によって上書きされています");
							output.NewLine();
						}
						output.Print("　上記の関数を利用するスクリプトは意図通りに動かない可能性があります");
						output.NewLine();
						output.NewLine();
						output.Print("　※この警告は該当する式中関数を利用しているEmuera専用スクリプト向けの警告です。");
						output.NewLine();
						output.Print("　eramaker用のスクリプトの動作には影響しません。");
						output.NewLine();
						output.Print("　今後この警告が不要ならばコンフィグの「システム関数が上書きされたとき警告を表示する」をOFFにして下さい。");
						output.NewLine();
                        output.PrintError("＊＊＊＊＊＊＊＊＊＊＊＊");
					}
				}
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="str"></param>
			/// <param name="line"></param>
			/// <param name="level">警告レベル.0:軽微なミス.1:無視できる行.2:行が実行されなければ無害.3:致命的</param>
			private void printWarning(string str, LogicalLine line, int level, bool isError, bool isBackComp)
			{
				if (isError)
				{
					line.IsError = true;
					line.ErrMes = str;
				}
				if (level < Config.Instance.DisplayWarningLevel && !Program.AnalysisMode)
					return;
                if (isBackComp && !Config.Instance.WarnBackCompatibility)
                    return;
				if ((line.Position != null) && (line.Position.Filename != null))
				{
					string filename = line.Position.Filename.ToUpper();
					if (!string.IsNullOrEmpty(filename) && !Program.AnalysisMode)
					{
						if (Config.Instance.IgnoreWarningFiles.Contains(filename))
						{
							if (ignoredWarningCount.ContainsKey(filename))
								ignoredWarningCount[filename]++;
							else
								ignoredWarningCount.Add(filename, 1);
							return;
						}
					}
				}
				output.PrintWarning(str, line.Position, level);
			}

            public Dictionary<string, Int64> warningDic = new Dictionary<string, Int64>();
			private void printFunctionNotFoundWarning(string str, LogicalLine line, int level, bool isError)
			{
                if (Program.AnalysisMode)
                {
                    if (warningDic.ContainsKey(str))
                        warningDic[str]++;
                    else
                        warningDic.Add(str, 1);
                    return;
                }
				if (isError)
				{
					line.IsError = true;
					line.ErrMes = str;
				}
				if (level < Config.Instance.DisplayWarningLevel)
					return;
				bool ignore = false;
				if (Config.Instance.FunctionNotFoundWarning == DisplayWarningFlag.IGNORE)
					ignore = true;
				else if (Config.Instance.FunctionNotFoundWarning == DisplayWarningFlag.DISPLAY)
					ignore = false;
				else if (Config.Instance.FunctionNotFoundWarning == DisplayWarningFlag.ONCE)
				{

					string filename = line.Position.Filename.ToUpper();
					if (!string.IsNullOrEmpty(filename))
					{
						if (ignoredFNFWarningFileList.Contains(filename))
						{
							ignore = true;
						}
						else
						{
							ignore = false;
							ignoredFNFWarningFileList.Add(filename);
						}
					}
				}
				if (ignore && !Program.AnalysisMode)
				{
					ignoredFNFWarningCount++;
					return;
				}
                printWarning(str, line, level, isError, false);
			}

			private void checkFunctionWithCatch(FunctionLabelLine label)
			{//ここでエラーを捕まえることは本来はないはず。ExeEE相当。
				try
				{
					checkFunction(label);
				}
				catch(Exception exc)
				{
                    System.Media.SystemSounds.Hand.Play();
                    string errmes = exc.Message;
					if(!(exc is EmueraException))
						errmes = exc.GetType().ToString() + ":" + errmes;
                    printWarning("@" + label.LabelName + " の解析中にエラー:" + errmes, label, 2, true, false);
					label.ErrMes = "ロード時に解析に失敗した関数が呼び出されました";
				}
			}

			private void checkFunction(FunctionLabelLine label)
			{
				System.Windows.Forms.Application.DoEvents();
				int depth = label.Depth;
				if (depth < 0)
					depth = -2;
				LogicalLine nextLine = label;
                List<InstructionLine> tempLineList = new List<InstructionLine>();
				string filename = null;
				if (label.Position.Filename != null)
					filename = label.Position.Filename.ToUpper();
				bool ignoreWarning = false;
				if (!string.IsNullOrEmpty(filename))
					ignoreWarning = Config.Instance.IgnoreWarningFiles.Contains(filename);
				//1周目/3周
				//引数の解析とか
                bool inMethod = label.IsMethod;
				while (true)
				{
					nextLine = nextLine.NextLine;
					if ((nextLine is NullLine) || (nextLine is FunctionLabelLine))
						break;
                    if (nextLine is InvalidLabelLine)
                    {
                        inMethod = ((InvalidLabelLine)nextLine).IsMethod;
                        continue;
                    }
                    if (!(nextLine is InstructionLine))
						continue;
					InstructionLine func = (InstructionLine)nextLine;
                    if (inMethod)
					{
						if ((func.Function & BuiltInFunctionCode.__METHOD_SAFE__) != BuiltInFunctionCode.__METHOD_SAFE__)
						{
                            printWarning(func.Function.ToString() + "命令は#FUNCTION中で使うことはできません", nextLine, 2, true, false);
							break;
						}
					}
					if (Config.Instance.NeedReduceArgumentOnLoad || Program.AnalysisMode)
						LogicalLineParser.SetArgumentTo(func, this.printWarning);
				}

				//2周目/3周
				//IF-ELSEIF-ENDIF、REPEAT-RENDの対応チェックなど
                //PRINTDATA系もここでチェック
				nextLine = label;
            secondLoop:
                bool mustBack = false;
				Stack<InstructionLine> nestStack = new Stack<InstructionLine>();
				InstructionLine pairLine = null;
				while (true)
				{
					nextLine = nextLine.NextLine;
					if ((nextLine is NullLine) || (nextLine is FunctionLabelLine))
						break;
                    if (nextLine is InvalidLabelLine)
                    {
                        mustBack = true;
                        break;
                    }
					if (!(nextLine is InstructionLine))
						continue;
					InstructionLine func = (InstructionLine)nextLine;
					pairLine = null;
                    InstructionLine baseFunc = nestStack.Count == 0 ? null : nestStack.Peek();
                    if (baseFunc != null)
                    {
                        if ((baseFunc.Function == BuiltInFunctionCode.PRINTDATA) || (baseFunc.Function == BuiltInFunctionCode.PRINTDATAL) || (baseFunc.Function == BuiltInFunctionCode.PRINTDATAW)
                            || (baseFunc.Function == BuiltInFunctionCode.PRINTDATAD) || (baseFunc.Function == BuiltInFunctionCode.PRINTDATADL) || (baseFunc.Function == BuiltInFunctionCode.PRINTDATADW))
                        {
                            if ((func.Function != BuiltInFunctionCode.DATA) && (func.Function != BuiltInFunctionCode.DATAFORM) && (func.Function != BuiltInFunctionCode.DATALIST) && (func.Function != BuiltInFunctionCode.ENDLIST) && (func.Function != BuiltInFunctionCode.ENDDATA))
                            {
                                printWarning("PRINTDATA構文に使用できない命令\'" + func.Function.ToString() + "\'が含まれています", func, 2, true, false);
                                continue;
                            }
                        }
                        else if (baseFunc.Function == BuiltInFunctionCode.DATALIST)
                        {
                            if ((func.Function != BuiltInFunctionCode.DATA) && (func.Function != BuiltInFunctionCode.DATAFORM) && (func.Function != BuiltInFunctionCode.ENDLIST))
                            {
                                printWarning("DATALIST構文に使用できない命令\'" + func.Function.ToString() + "\'が含まれています", func, 2, true, false);
                                continue;
                            }
                        }
                        else if ((baseFunc.Function == BuiltInFunctionCode.TRYCALLLIST) || (baseFunc.Function == BuiltInFunctionCode.TRYJUMPLIST) || (baseFunc.Function == BuiltInFunctionCode.TRYGOTOLIST))
                        {
                            if ((func.Function != BuiltInFunctionCode.FUNC) && (func.Function != BuiltInFunctionCode.ENDFUNC))
                            {
                                printWarning(baseFunc.Function.ToString() + "構文に使用できない命令" + func.Function.ToString() + "が含まれています", func, 2, true, false);
                                continue;
                            }
                        }
                        else if (baseFunc.Function == BuiltInFunctionCode.SELECTCASE)
                        {
                            if ((baseFunc.IfCaseList.Count == 0) && (func.Function != BuiltInFunctionCode.CASE) && (func.Function != BuiltInFunctionCode.CASEELSE) && (func.Function != BuiltInFunctionCode.ENDSELECT))
                            {
                                printWarning("SELECTCASE構文の分岐の外に命令" + func.Function.ToString() + "が含まれています", func, 2, true, false);
                                continue;
                            }
                        }
                    }
					switch (func.Function)
					{
						case BuiltInFunctionCode.REPEAT:
							foreach (InstructionLine iLine in nestStack)
							{
								if (iLine.Function == BuiltInFunctionCode.REPEAT)
								{
                                    printWarning("REPEAT文が入れ子にされています", func, 2, true, false);
									break;
								}
							}
							if (func.IsError)
								break;
							nestStack.Push(func);
							break;
						case BuiltInFunctionCode.IF:
							nestStack.Push(func);
							func.IfCaseList = new List<InstructionLine>();
							func.IfCaseList.Add(func);
							break;
						case BuiltInFunctionCode.SELECTCASE:
							nestStack.Push(func);
							func.IfCaseList = new List<InstructionLine>();
							break;
						case BuiltInFunctionCode.FOR:
						case BuiltInFunctionCode.WHILE:
						case BuiltInFunctionCode.TRYCGOTO:
						case BuiltInFunctionCode.TRYCJUMP:
						case BuiltInFunctionCode.TRYCCALL:
						case BuiltInFunctionCode.TRYCGOTOFORM:
						case BuiltInFunctionCode.TRYCJUMPFORM:
						case BuiltInFunctionCode.TRYCCALLFORM:
                        case BuiltInFunctionCode.DO:
							nestStack.Push(func);
							break;
						case BuiltInFunctionCode.BREAK:
						case BuiltInFunctionCode.CONTINUE:
							InstructionLine[] array = nestStack.ToArray();
							for (int i = 0; i < array.Length; i++)
							{
								if ((array[i].Function == BuiltInFunctionCode.REPEAT)
									|| (array[i].Function == BuiltInFunctionCode.FOR)
									|| (array[i].Function == BuiltInFunctionCode.WHILE)
                                    || (array[i].Function == BuiltInFunctionCode.DO))
								{
									pairLine = array[i];
									break;
								}
							}
							if (pairLine == null)
							{
                                printWarning("REPEAT, FOR, WHILE, DOの中以外で" + func.Function.ToString() + "文が使われました", func, 2, true, false);
								break;
							}
							func.JumpTo = pairLine;
							break;

						case BuiltInFunctionCode.ELSEIF:
						case BuiltInFunctionCode.ELSE:
							{
								//1.725 Stack<T>.Peek()はStackが空の時はnullを返す仕様だと思いこんでおりました。
                                InstructionLine ifLine = nestStack.Count == 0 ? null : nestStack.Peek();
								if ((ifLine == null) || (ifLine.Function != BuiltInFunctionCode.IF))
								{
                                    printWarning("IF～ENDIFの外で" + func.Function.ToString() + "文が使われました", func, 2, true, false);
									break;
								}
								if (ifLine.IfCaseList[ifLine.IfCaseList.Count - 1].Function == BuiltInFunctionCode.ELSE)
                                    printWarning("ELSE文より後で" + func.Function.ToString() + "文が使われました", func, 1, false, false);
								ifLine.IfCaseList.Add(func);
							}
							break;
						case BuiltInFunctionCode.ENDIF:
							{
								InstructionLine ifLine = nestStack.Count == 0 ? null : nestStack.Peek();
								if ((ifLine == null) || (ifLine.Function != BuiltInFunctionCode.IF))
								{
                                    printWarning("対応するIFの無いENDIF文です", func, 2, true, false);
									break;
								}
								foreach (InstructionLine ifelseifLine in ifLine.IfCaseList)
								{
									ifelseifLine.JumpTo = func;
                                    if (ifelseifLine.Argument == null)
                                        LogicalLineParser.SetArgumentTo(ifelseifLine, this.printWarning);
								}
								nestStack.Pop();
							}
							break;
						case BuiltInFunctionCode.CASE:
						case BuiltInFunctionCode.CASEELSE:
							{
								InstructionLine selectLine = nestStack.Count == 0 ? null : nestStack.Peek();
								if ((selectLine == null) || (selectLine.Function != BuiltInFunctionCode.SELECTCASE))
								{
                                    printWarning("SELECTCASE～ENDSELECTの外で" + func.Function.ToString() + "文が使われました", func, 2, true, false);
									break;
								}
								if ((selectLine.IfCaseList.Count > 0) &&
									(selectLine.IfCaseList[selectLine.IfCaseList.Count - 1].Function == BuiltInFunctionCode.CASEELSE))
                                    printWarning("CASEELSE文より後で" + func.Function.ToString() + "文が使われました", func, 1, false, false);
								selectLine.IfCaseList.Add(func);
							}
							break;
						case BuiltInFunctionCode.ENDSELECT:
							{
								InstructionLine selectLine = nestStack.Count == 0 ? null : nestStack.Peek();
								if ((selectLine == null) || (selectLine.Function != BuiltInFunctionCode.SELECTCASE))
								{
                                    printWarning("対応するSELECTCASEの無いENDSELECT文です", func, 2, true, false);
									break;
								}
								nestStack.Pop();
								selectLine.JumpTo = func;

                                if (selectLine.Argument == null)
                                    LogicalLineParser.SetArgumentTo(selectLine, this.printWarning);
								if (selectLine.IsError)
									break;
								IOperandTerm term = ((ExpressionArgument)selectLine.Argument).Term;
								if (term == null)
								{
                                    printWarning("SELECTCASEの引数がありません", selectLine, 2, true, false);
									break;
								}
								foreach (InstructionLine caseLine in selectLine.IfCaseList)
								{
									caseLine.JumpTo = func;
                                    if (caseLine.Argument == null)
                                        LogicalLineParser.SetArgumentTo(caseLine, this.printWarning);
									if (caseLine.IsError)
										continue;
									if (caseLine.Function == BuiltInFunctionCode.CASEELSE)
										continue;
									CaseExpression[] caseExps = ((CaseArgument)caseLine.Argument).CaseExps;
									if(caseExps.Length == 0)
                                        printWarning("CASEの引数がありません", caseLine, 1, true, false);

									foreach(CaseExpression exp in caseExps)
									{
										if (exp.GetOperandType() != term.GetOperandType())
                                            printWarning("CASEの引数の型がSELECTCASEと一致しません", caseLine, 2, true, false);
									}
									
								}
							}
							break;
						case BuiltInFunctionCode.REND:
                        case BuiltInFunctionCode.NEXT:
                        case BuiltInFunctionCode.WEND:
                        case BuiltInFunctionCode.LOOP:
                            BuiltInFunctionCode parentFunc = BuiltInFunctionManager.getParentFunc(func.Function);
                            if (parentFunc == BuiltInFunctionCode.__NULL__)
                                throw new ExeEE("何か変？");
							if ((nestStack.Count == 0)
                                || (nestStack.Peek().Function != parentFunc))
							{
                                printWarning("対応する" + parentFunc.ToString() + "の無い" + func.Function.ToString() + "文です", func, 2, true, false);
								break;
							}
							pairLine = nestStack.Pop();//REPEAT
							func.JumpTo = pairLine;
							pairLine.JumpTo = func;
							break;
						case BuiltInFunctionCode.CATCH:
							pairLine = nestStack.Count == 0 ? null : nestStack.Peek();
							if ((pairLine == null)
								|| ((pairLine.Function != BuiltInFunctionCode.TRYCGOTO)
								&& (pairLine.Function != BuiltInFunctionCode.TRYCCALL)
								&& (pairLine.Function != BuiltInFunctionCode.TRYCJUMP)
								&& (pairLine.Function != BuiltInFunctionCode.TRYCGOTOFORM)
								&& (pairLine.Function != BuiltInFunctionCode.TRYCCALLFORM)
								&& (pairLine.Function != BuiltInFunctionCode.TRYCJUMPFORM)))
							{
                                printWarning("対応するTRYC系命令がありません", func, 2, true, false);
								break;
							}
							pairLine = nestStack.Pop();//TRYC
							pairLine.JumpToEndCatch = func;//TRYCにCATCHの位置を教える
							nestStack.Push(func);
							break;
						case BuiltInFunctionCode.ENDCATCH:
							if ((nestStack.Count == 0)
								|| (nestStack.Peek().Function != BuiltInFunctionCode.CATCH))
							{
                                printWarning("対応するCATCHのないENDCATCHです", func, 2, true, false);
								break;
							}
							pairLine = nestStack.Pop();//CATCH
							pairLine.JumpToEndCatch = func;//CATCHにENDCATCHの位置を教える
							break;
                        case BuiltInFunctionCode.PRINTDATA:
                        case BuiltInFunctionCode.PRINTDATAL:
                        case BuiltInFunctionCode.PRINTDATAW:
                        case BuiltInFunctionCode.PRINTDATAD:
                        case BuiltInFunctionCode.PRINTDATADL:
                        case BuiltInFunctionCode.PRINTDATADW:
                            foreach (InstructionLine iLine in nestStack)
                            {
                                if (iLine.Function == BuiltInFunctionCode.PRINTDATA || iLine.Function == BuiltInFunctionCode.PRINTDATAL || iLine.Function == BuiltInFunctionCode.PRINTDATAW
                                    || iLine.Function == BuiltInFunctionCode.PRINTDATAD || iLine.Function == BuiltInFunctionCode.PRINTDATADL || iLine.Function == BuiltInFunctionCode.PRINTDATADW)
                                {
                                    printWarning("PRINTDATA系命令が入れ子にされています", func, 2, true, false);
                                    break;
                                }
                            }
                            if (func.IsError)
                                break;
                            func.dataList = new List<List<InstructionLine>>();
                            nestStack.Push(func);
                            break;
                        case BuiltInFunctionCode.DATALIST:
                            InstructionLine plist = (nestStack.Count == 0) ? null : nestStack.Peek();
                            if ((plist == null)
                               || (plist.Function != BuiltInFunctionCode.PRINTDATA && plist.Function != BuiltInFunctionCode.PRINTDATAL && plist.Function != BuiltInFunctionCode.PRINTDATAW
                                   && plist.Function != BuiltInFunctionCode.PRINTDATAD && plist.Function != BuiltInFunctionCode.PRINTDATADL && plist.Function != BuiltInFunctionCode.PRINTDATADW))
                            {
                                printWarning("対応するPRINTDATA系命令のないDATALISTです", func, 2, true, false);
                                break;
                            }
                            tempLineList = new List<InstructionLine>();
                            nestStack.Push(func);

                            break;
                        case BuiltInFunctionCode.ENDLIST:
                            if ((nestStack.Count == 0)
                                || (nestStack.Peek().Function != BuiltInFunctionCode.DATALIST))
                            {
                                printWarning("対応するDATALISTのないENDLISTです", func, 2, true, false);
                                break;
                            }
                            nestStack.Pop();
                            nestStack.Peek().dataList.Add(tempLineList);
                            break;
                        case BuiltInFunctionCode.DATA:
                        case BuiltInFunctionCode.DATAFORM:
                            InstructionLine pdata = (nestStack.Count == 0) ? null : nestStack.Peek();
                            if ((pdata == null) ||
                            (pdata.Function != BuiltInFunctionCode.PRINTDATA && pdata.Function != BuiltInFunctionCode.PRINTDATAL && pdata.Function != BuiltInFunctionCode.PRINTDATAW
                             && pdata.Function != BuiltInFunctionCode.PRINTDATAD && pdata.Function != BuiltInFunctionCode.PRINTDATADL && pdata.Function != BuiltInFunctionCode.PRINTDATADW && pdata.Function != BuiltInFunctionCode.DATALIST))
                            {
                                printWarning("対応するPRINTDATA系命令のない" + func.Function.ToString() + "です", func, 2, true, false);
                                break;
                            }
                            List<InstructionLine> iList = new List<InstructionLine>();
                            if (pdata.Function != BuiltInFunctionCode.DATALIST)
                            {
                                iList.Add(func);
                                pdata.dataList.Add(iList);
                            }
                            else
                                tempLineList.Add(func);
                            break;
                        case BuiltInFunctionCode.ENDDATA:
                            InstructionLine pdataline = (nestStack.Count == 0) ? null : nestStack.Peek();
                            if ((pdataline == null) ||
                                (pdataline.Function != BuiltInFunctionCode.PRINTDATA && pdataline.Function != BuiltInFunctionCode.PRINTDATAL && pdataline.Function != BuiltInFunctionCode.PRINTDATAW
                                 && pdataline.Function != BuiltInFunctionCode.PRINTDATAD && pdataline.Function != BuiltInFunctionCode.PRINTDATADL && pdataline.Function != BuiltInFunctionCode.PRINTDATADW))
                            {
                                printWarning("対応するPRINTDATA系命令のない" + func.Function.ToString() + "です", func, 2, true, false);
                                break;
                            }
                            if (pdataline.Function == BuiltInFunctionCode.DATALIST)
                                printWarning("DATALISTが閉じられていません", func, 2, true, false);
                            pdataline.JumpTo = func;
                            nestStack.Pop();
                            break;
                        case BuiltInFunctionCode.TRYCALLLIST:
                        case BuiltInFunctionCode.TRYJUMPLIST:
                        case BuiltInFunctionCode.TRYGOTOLIST:
                            foreach (InstructionLine iLine in nestStack)
                            {
                                if (iLine.Function == BuiltInFunctionCode.TRYCALLLIST || iLine.Function == BuiltInFunctionCode.TRYJUMPLIST || iLine.Function == BuiltInFunctionCode.TRYGOTOLIST)
                                {
                                    printWarning("TRYCALLLIST系命令が入れ子にされています", func, 2, true, false);
                                    break;
                                }
                            }
                            if (func.IsError)
                                break;
                            func.callList = new List<InstructionLine>();
                            nestStack.Push(func);
                            break;
                        case BuiltInFunctionCode.FUNC:
                            InstructionLine pFunc = (nestStack.Count == 0) ? null : nestStack.Peek();
                            if ((pFunc == null) ||
                                (pFunc.Function != BuiltInFunctionCode.TRYCALLLIST && pFunc.Function != BuiltInFunctionCode.TRYJUMPLIST && pFunc.Function != BuiltInFunctionCode.TRYGOTOLIST))
                            {
                                printWarning("対応するTRYCALLLIST系命令のない" + func.Function.ToString() + "です", func, 2, true, false);
                                break;
                            }
                            if (pFunc.Function == BuiltInFunctionCode.TRYGOTOLIST)
                            {
                                if (((SpCallformArgment)func.Argument).Args.Length != 0)
                                {
                                    printWarning("TRYGOTOLISTの呼び出し対象に引数が設定されています", func, 2, true, false);
                                    break;
                                }
                            }
                            useCallForm = true;
                            pFunc.callList.Add(func);
                           break;
                        case BuiltInFunctionCode.ENDFUNC:
                            InstructionLine pf = (nestStack.Count == 0) ? null : nestStack.Peek();
                            if ((pf == null) ||
                                (pf.Function != BuiltInFunctionCode.TRYCALLLIST && pf.Function != BuiltInFunctionCode.TRYJUMPLIST && pf.Function != BuiltInFunctionCode.TRYGOTOLIST))
                            {
                                printWarning("対応するTRYCALLLIST系命令のない" + func.Function.ToString() + "です", func, 2, true, false);
                                break;
                            }
                            pf.JumpTo = func;
                            nestStack.Pop();
                            break;
                        case BuiltInFunctionCode.NOSKIP:
                            foreach (InstructionLine iLine in nestStack)
                            {
                                if (iLine.Function == BuiltInFunctionCode.NOSKIP)
                                {
                                    printWarning("NOSKIP系命令が入れ子にされています", func, 2, true, false);
                                    break;
                                }
                            }
                            if (func.IsError)
                                break;
                            nestStack.Push(func);
                            break;
                        case BuiltInFunctionCode.ENDNOSKIP:
                            InstructionLine pfunc = (nestStack.Count == 0) ? null : nestStack.Peek();
                            if ((pfunc == null) ||
                                (pfunc.Function != BuiltInFunctionCode.NOSKIP))
                            {
                                printWarning("対応するNOSKIP系命令のない" + func.Function.ToString() + "です", func, 2, true, false);
                                break;
                            }
                            //エラーハンドリング用
                            pfunc.JumpTo = func;
                            func.JumpTo = pfunc;
                            nestStack.Pop();
                         break;
                    }

				}

				while (nestStack.Count != 0)
				{
					InstructionLine func = nestStack.Pop();
					string funcName = func.Function.ToString();
                    string funcMatch = BuiltInFunctionManager.getMatchFunction(func.Function);
                    if (func != null)
                        printWarning(funcName + "に対応する" + funcMatch + "が見つかりません", func, 2, true, false);
                    else
                        printWarning("ディフォルトエラー（Emuera設定漏れ）", func, 2, true, false);
                }
                if (mustBack)
                    goto secondLoop;

				//3周目/3周
				//フロー制御命令のジャンプ先を設定
				nextLine = label;
				LogicalLine jumpto = null;
				while (true)
				{
					nextLine = nextLine.NextLine;
					if ((nextLine is NullLine) || (nextLine is FunctionLabelLine))
						break;
					if (!(nextLine is InstructionLine))
						continue;
					if (nextLine.IsError)
						continue;
					InstructionLine func = (InstructionLine)nextLine;
					switch (func.Function)
					{
						case BuiltInFunctionCode.SIF:

							jumpto = func.NextLine;

							if ((jumpto == null) || (jumpto.NextLine == null) ||
							(jumpto is FunctionLabelLine) || (jumpto is NullLine))
                                printWarning("SIF文のステートメントがありません", nextLine, 2, true, false);
							else if (jumpto is InstructionLine)
							{
								InstructionLine sifFunc = (InstructionLine)jumpto;
								switch (sifFunc.Function)
								{
									case BuiltInFunctionCode.SIF:
									case BuiltInFunctionCode.IF:
									case BuiltInFunctionCode.ELSEIF:
									case BuiltInFunctionCode.ELSE:
									case BuiltInFunctionCode.ENDIF:
									case BuiltInFunctionCode.REPEAT:
									case BuiltInFunctionCode.REND:
									case BuiltInFunctionCode.FOR:
									case BuiltInFunctionCode.NEXT:
									case BuiltInFunctionCode.WHILE:
									case BuiltInFunctionCode.WEND:
									case BuiltInFunctionCode.CATCH:
									case BuiltInFunctionCode.ENDCATCH:
									case BuiltInFunctionCode.SELECTCASE:
									case BuiltInFunctionCode.CASE:
									case BuiltInFunctionCode.CASEELSE:
									case BuiltInFunctionCode.ENDSELECT:
                                        printWarning("SIF文の次の行を" + sifFunc.Function.ToString() + "文にすることはできません", nextLine, 2, true, false);
										break;
									default:
										func.JumpTo = func.NextLine.NextLine;
										break;
								}
							}
							else
								func.JumpTo = func.NextLine.NextLine;
							break;
						case BuiltInFunctionCode.GOTO://$ラベルへジャンプ
						case BuiltInFunctionCode.TRYGOTO:
                        case BuiltInFunctionCode.TRYCGOTO:
                            {
								string labelName = func.ArgumentStr.Trim();
								if (Config.Instance.IgnoreCase)
									labelName = labelName.ToUpper();
								jumpto = labelDic.GetLabelDollar(labelName, label);
								if ((jumpto == null) && (func.Function == BuiltInFunctionCode.GOTO))
                                    printWarning("指定されたラベル名\"$" + labelName + "\"は現在の関数内に存在しません", nextLine, 2, true, false);
                                else if (jumpto != null)
                                {
                                    func.JumpTo = jumpto;
                                }
                            }
							break;
                        case BuiltInFunctionCode.JUMP:
                        case BuiltInFunctionCode.TRYJUMP:
                        case BuiltInFunctionCode.TRYCJUMP:
						case BuiltInFunctionCode.CALL://関数に移動。移動元を記憶し、RETURNで帰る。
						case BuiltInFunctionCode.TRYCALL:
                        case BuiltInFunctionCode.TRYCCALL:
                            {
                                if (func.Argument == null)
                                    LogicalLineParser.SetArgumentTo(func, this.printWarning);
                                SpCallArgment callArg = (SpCallArgment)func.Argument;
								if (func.IsError)
									break;
                                string labelName = callArg.Str;
                                if (Config.Instance.IgnoreCase)
                                    labelName = labelName.ToUpper();
                                FunctionLabelLine targetLabel = labelDic.GetLabel(labelName);
								if ((targetLabel == null)
									&& ((func.Function == BuiltInFunctionCode.CALL)
									|| (func.Function == BuiltInFunctionCode.JUMP)))
								{
                                    if (!Program.AnalysisMode)
                                        printFunctionNotFoundWarning("指定された関数名\"@" + labelName + "\"は存在しません", nextLine, 2, true);
                                    else
                                        printFunctionNotFoundWarning(labelName, nextLine, 2, true);
                                    break;
								}
								if (targetLabel != null)
								{
									func.JumpTo = targetLabel;
									if (targetLabel.Depth < 0)
										targetLabel.Depth = depth + 1;
									checkArgs(targetLabel, callArg.Args, nextLine);
								}

                             }
                            break;
						case BuiltInFunctionCode.RESTART:
							func.JumpTo = label;
							break;

						case BuiltInFunctionCode.JUMPFORM:
						case BuiltInFunctionCode.CALLFORM:
						//case BuiltInFunctionCode.GOTOFORM:
						case BuiltInFunctionCode.TRYJUMPFORM:
						case BuiltInFunctionCode.TRYCALLFORM:
						//case BuiltInFunctionCode.TRYGOTOFORM:
                        case BuiltInFunctionCode.TRYCJUMPFORM:
                        case BuiltInFunctionCode.TRYCCALLFORM:
                        //case BuiltInFunctionCode.TRYCGOTOFORM:
                            useCallForm = true;
							break;
						case BuiltInFunctionCode.RETURNF:
							{
								if (!label.IsMethod)
								{
                                    printWarning("RETURNFは#FUCNTION以外では使用できません", nextLine, 2, true, false);
								}
								if ((func.Argument != null) && (!func.IsError))
								{
									ExpressionArgument expArg = (ExpressionArgument)func.Argument;
									IOperandTerm term = ((ExpressionArgument)func.Argument).Term;
									if (term != null)
									{
										if (label.MethodType != term.GetOperandType())
										{
											if (label.MethodType == typeof(Int64))
                                                printWarning("#FUCNTIONで始まる関数の戻り値に文字列型が指定されました", nextLine, 2, true, false);
											else if (label.MethodType == typeof(string))
                                                printWarning("#FUCNTIONSで始まる関数の戻り値に数値型が指定されました", nextLine, 2, true, false);
											else if (label.MethodType == typeof(string))
                                                printWarning("関数の型と戻り値の型が一致しません", nextLine, 2, true, false);
										}
										break;
									}
								}
							}
							break;
					}
				}
			}

			private bool checkArgs(FunctionLabelLine targetLabel, IOperandTerm[] args, LogicalLine nextLine)
			{
				if (args.Length == 0)
					return true;
				if (targetLabel.Arg.Length < args.Length)
				{
                    printWarning("引数の数(" + args.Length.ToString() + ")が関数\"@" + targetLabel.LabelName + "\"に設定された数(" + targetLabel.Arg.Length.ToString() +")を超えています", nextLine, 2, true, false);
					return false;
				}

				for (int i = 0; i < args.Length; i++)
				{
					if (args[i] == null)
						continue;
					if ((args[i].GetOperandType() == typeof(string)) && (targetLabel.Arg[i].GetOperandType() == typeof(Int64)))
					{
                        printWarning((i + 1).ToString() + "番目の引数の型が関数\"@" + targetLabel.LabelName + "\"に設定された引数の型と一致しません", nextLine, 2, true, false);
						return false;
					}
				}
				return true;
			}
		}
	}
}
