﻿using System;
using System.Collections.Generic;
using System.Text;
using MinorShift.Emuera.Sub;
using MinorShift.Emuera.GameProc;
using MinorShift.Emuera.GameData.Expression;

namespace MinorShift.Emuera.GameData.Variable
{
	//IndexOutOfRangeException, ArgumentOutOfRangeExceptionを投げることがある。VariableTermの方で処理すること。
	//引数は整数しか受け付けない。*.csvを利用した置換はVariableTermの方で処理すること
	internal abstract class VariableToken
	{
		protected VariableToken(VariableCode varCode, VariableData varData)
		{
			Code = varCode;
			VariableType = ((varCode & VariableCode.__INTEGER__) == VariableCode.__INTEGER__) ? typeof(Int64) : typeof(string);
			VarCodeInt = (int)(varCode & VariableCode.__LOWERCASE__);
			varName = varCode.ToString();
			this.varData = varData;
			IsPrivate = false;
		}

		public readonly VariableCode Code;
		public readonly int VarCodeInt;
		protected readonly VariableData varData;
		protected string varName;
		public Type VariableType { get; protected set; }
		public bool CanRestructure { get; protected set; }
		public string Name { get{return varName;} }


		//CodeEEにしているけど実際はExeEEかもしれない
		public virtual Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
		{ throw new CodeEE("整数型でない変数" + varName + "を整数型として呼び出しました"); }
		public virtual string GetStrValue(ExpressionMediator exm, Int64[] arguments)
		{ throw new CodeEE("文字列型でない変数" + varName + "を文字列型として呼び出しました"); }
        public virtual void SetValue(Int64 value, Int64[] arguments)
		{ throw new CodeEE("整数型でない変数" + varName + "を整数型として呼び出しました"); }
		public virtual void SetValue(string value, Int64[] arguments)
		{ throw new CodeEE("文字列型でない変数" + varName + "を文字列型として呼び出しました"); }
		public virtual Int64 PlusValue(Int64 value, Int64[] arguments)
		{ throw new CodeEE("整数型でない変数" + varName + "を整数型として呼び出しました"); }
		public virtual Int32 GetLength()
		{ throw new CodeEE("配列型でない変数" + varName + "の長さを取得しようとしました"); }
		public virtual Int32 GetLength(int dimension)
		{ throw new CodeEE("配列型でない変数" + varName + "の長さを取得しようとしました"); }
		public virtual object GetArray()
		{
			if (IsCharacterData)
				throw new CodeEE("キャラクタ変数" + varName + "を非キャラ変数として呼び出しました");
			throw new CodeEE("配列型でない変数" + varName + "の配列を取得しようとしました");
		}
		public virtual object GetArrayChara(int charano)
		{
			if (!IsCharacterData)
				throw new CodeEE("非キャラクタ変数" + varName + "をキャラ変数として呼び出しました");
			throw new CodeEE("配列型でない変数" + varName + "の配列を取得しようとしました");
		}

		public void throwOutOfRangeException(Int64[] arguments, Exception e)
		{
			CheckElement(arguments);
			throw e;
		}
		public virtual void CheckElement(Int64[] arguments) { }
		public virtual void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
		{
			CheckElement(arguments);
			return;
		}

		public int CodeInt
		{ get { return VarCodeInt; } }
		public VariableCode CodeFlag
		{ get { return Code & VariableCode.__UPPERCASE__; } }

		public bool IsNull
		{
			get
			{
				return Code == VariableCode.__NULL__;
			}
		}
		public bool IsCharacterData
		{
			get
			{
				return ((Code & VariableCode.__CHARACTER_DATA__) == VariableCode.__CHARACTER_DATA__);
			}
		}
		public bool IsInteger
		{
			get
			{
				return ((Code & VariableCode.__INTEGER__) == VariableCode.__INTEGER__);
			}
		}
		public bool IsString
		{
			get
			{
				return ((Code & VariableCode.__STRING__) == VariableCode.__STRING__);
			}
		}
		public bool IsArray1D
		{
			get
			{
				return ((Code & VariableCode.__ARRAY_1D__) == VariableCode.__ARRAY_1D__);
			}
		}
		public bool IsArray2D
		{
			get
			{
				return ((Code & VariableCode.__ARRAY_2D__) == VariableCode.__ARRAY_2D__);
			}
		}
		public bool IsArray3D
		{
			get
			{
				return ((Code & VariableCode.__ARRAY_3D__) == VariableCode.__ARRAY_3D__);
			}
		}
		public bool Readonly
		{
			get
			{
				return ((Code & VariableCode.__UNCHANGEABLE__) == VariableCode.__UNCHANGEABLE__);
			}
		}
		public bool IsCalc
		{
			get
			{
				return ((Code & VariableCode.__CALC__) == VariableCode.__CALC__);
			}
		}
		public bool IsLocal
		{
			get
			{
				return ((Code & VariableCode.__LOCAL__) == VariableCode.__LOCAL__);
			}
		}
		public bool IsPrivate{get; protected set;}

	}

	internal abstract class UserDefinedVariableToken : VariableToken
	{
		protected UserDefinedVariableToken(VariableCode varCode, string name, int size)
			: base(varCode, null)
		{
			varName = name;
			IsPrivate = true;
			this.size = size;
		}

		public abstract void SetDefault();
		protected int size;
		public override Int32 GetLength()
		{
			return size;
		}
			
		public override Int32 GetLength(int dimension)
		{
			if (dimension == 0)
				return size;
			throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
		}
		public override void CheckElement(Int64[] arguments)
		{
			//if (array == null)
			//	throw new ExeEE("プライベート変数" + varName + "の配列が用意されていない");
			if ((arguments[0] < 0) || (arguments[0] >= size))
				throw new CodeEE("配列変数" + varName + "の第１引数(" + arguments[0].ToString() + ")は配列の範囲外です");
		}
		public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
		{
			CheckElement(arguments);
			if ((index1 < 0) || (index1 > size))
				throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
			if ((index2 < 0) || (index2 > size))
				throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
		}
	}

	internal abstract class PrivateVariableToken : UserDefinedVariableToken
	{

		protected PrivateVariableToken(VariableCode varCode, string name, int size)
			: base(varCode, name, size)
		{
			IsPrivate = true;
		}
		public abstract void In();
		public abstract void Out();
        public abstract void UpdateGeneration();
	}
	
	internal abstract class LocalVariableToken : VariableToken
	{
		public LocalVariableToken(VariableCode varCode, VariableData varData, string subId, int size)
			: base(varCode, varData)
		{
			CanRestructure = false;
			this.subID = subId;
			this.size = size;
		}
        public abstract void SetDefault();
        public abstract void resize(int newSize);
		protected string subID;
		protected int size;
		public override Int32 GetLength()
		{
			return size;
		}
		public override Int32 GetLength(int dimension)
		{
			if (dimension == 0)
				return size;
			throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
		}
		public override void CheckElement(Int64[] arguments)
		{
			//if (array == null)
			//	throw new ExeEE("プライベート変数" + varName + "の配列が用意されていない");
			if ((arguments[0] < 0) || (arguments[0] >= size))
				throw new CodeEE("配列変数" + varName + "の第１引数(" + arguments[0].ToString() + ")は配列の範囲外です");
		}
		public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
		{
			CheckElement(arguments);
			if ((index1 < 0) || (index1 > size))
				throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
			if ((index2 < 0) || (index2 > size))
				throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
		}
	}



	//サブクラスの詳細はVariableData以外は知らなくてよい
	internal sealed partial class VariableData
	{
		#region 変数
		private sealed class IntVariableToken : VariableToken
		{
			public IntVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
				array = varData.DataInteger;
			}
			Int64[] array;
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[VarCodeInt];
			}

			public override void SetValue(Int64 value, Int64[] arguments)
			{
				array[VarCodeInt] = value;
			}

			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{
				array[VarCodeInt] += value;
				return array[VarCodeInt];
			}
		}

		private sealed class Int1DVariableToken : VariableToken
		{
			public Int1DVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
				array = varData.DataIntegerArray[VarCodeInt];
			}
			Int64[] array;
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0]];
			}

			public override void SetValue(Int64 value, Int64[] arguments)
			{
				array[arguments[0]] = value;
			}

			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{
				array[arguments[0]] += value;
				return array[arguments[0]];
			}
			public override Int32 GetLength()
			{ return array.Length; }
			public override Int32 GetLength(int dimension)
			{
				if (dimension == 0)
					return array.Length;
				throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
			}
			public override object GetArray() {return array;}
			
			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= array.Length))
					throw new CodeEE("配列変数" + varName + "の第１引数(" + arguments[0].ToString() + ")は配列の範囲外です");
			}
			public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
			{
				CheckElement(arguments);
				if ((index1 < 0) || (index1 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
				if ((index2 < 0) || (index2 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
			}
		}

		private sealed class Int2DVariableToken : VariableToken
		{
			public Int2DVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
				array = varData.DataIntegerArray2D[VarCodeInt];
			}
			Int64[,] array;
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0], arguments[1]];
			}

			public override void SetValue(Int64 value, Int64[] arguments)
			{
				array[arguments[0], arguments[1]] = value;
			}

			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{
				array[arguments[0], arguments[1]] += value;
				return array[arguments[0], arguments[1]];
			}
			public override Int32 GetLength()
			{ throw new CodeEE("2次元配列型変数" + varName + "の長さを取得しようとしました"); }
			public override Int32 GetLength(int dimension)
			{
				if ((dimension == 0) || (dimension == 1))
					return array.GetLength(dimension);
				throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
			}
			public override object GetArray() {return array;}
			
			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= array.GetLength(0)))
					throw new CodeEE("二次元配列" + varName + "の第１引数(" + arguments[0].ToString() + ")は配列の範囲外です");
				if ((arguments[1] < 0) || (arguments[1] >= array.GetLength(1)))
					throw new CodeEE("二次元配列" + varName + "の第２引数(" + arguments[1].ToString() + ")は配列の範囲外です");
			}
			public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
			{
				CheckElement(arguments);
				if ((index1 < 0) || (index1 > array.GetLength(1)))
					throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
				if ((index2 < 0) || (index2 > array.GetLength(1)))
					throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
			}
		}

		private sealed class Int3DVariableToken : VariableToken
		{
			public Int3DVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
				array = varData.DataIntegerArray3D[VarCodeInt];
			}
			Int64[, ,] array;
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0], arguments[1], arguments[2]];
			}

			public override void SetValue(Int64 value, Int64[] arguments)
			{
				array[arguments[0], arguments[1], arguments[2]] = value;
			}

			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{
				array[arguments[0], arguments[1], arguments[2]] += value;
				return array[arguments[0], arguments[1], arguments[2]];
			}
			public override Int32 GetLength()
			{ throw new CodeEE("3次元配列型変数" + varName + "の長さを取得しようとしました"); }
			public override Int32 GetLength(int dimension)
			{
				if ((dimension == 0) || (dimension == 1) || (dimension == 2))
					return array.GetLength(dimension);
				throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
			}
			public override object GetArray() {return array;}
			
			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= array.GetLength(0)))
					throw new CodeEE("三次元配列" + varName + "の第１引数(" + arguments[0].ToString() + ")は配列の範囲外です");
				if ((arguments[1] < 0) || (arguments[1] >= array.GetLength(1)))
					throw new CodeEE("三次元配列" + varName + "の第２引数(" + arguments[1].ToString() + ")は配列の範囲外です");
				if ((arguments[2] < 0) || (arguments[2] >= array.GetLength(2)))
					throw new CodeEE("三次元配列" + varName + "の第３引数(" + arguments[2].ToString() + ")は配列の範囲外です");
			}
			public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
			{
				CheckElement(arguments);
				if ((index1 < 0) || (index1 > array.GetLength(2)))
					throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
				if ((index2 < 0) || (index2 > array.GetLength(2)))
					throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
			}
		}

		private sealed class StrVariableToken : VariableToken
		{
			public StrVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
				array = varData.DataString;
			}
			string[] array;
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[VarCodeInt];
			}

			public override void SetValue(string value, Int64[] arguments)
			{
				array[VarCodeInt] = value;
			}
			
		}

		private sealed class Str1DVariableToken : VariableToken
		{
			public Str1DVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
				array = varData.DataStringArray[VarCodeInt];
			}
			string[] array;
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0]];
			}

			public override void SetValue(string value, Int64[] arguments)
			{
				array[arguments[0]] = value;
			}
			public override Int32 GetLength()
			{ return array.Length; }
			public override Int32 GetLength(int dimension)
			{
				if (dimension == 0)
					return array.Length;
				throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
			}
			public override object GetArray() {return array;}
			
			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= array.Length))
					throw new CodeEE("配列変数" + varName + "の第１引数(" + arguments[0].ToString() + ")は配列の範囲外です");
			}
			public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
			{
				CheckElement(arguments);
				if ((index1 < 0) || (index1 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
				if ((index2 < 0) || (index2 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
			}
		}

		private sealed class Str2DVariableToken : VariableToken
		{
			public Str2DVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
				array = varData.DataStringArray2D[VarCodeInt];
			}
			string[,] array;
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0], arguments[1]];
			}

			public override void SetValue(string value, Int64[] arguments)
			{
				array[arguments[0], arguments[1]] = value;
			}
			public override Int32 GetLength()
			{ throw new CodeEE("2次元配列型変数" + varName + "の長さを取得しようとしました"); }
			public override Int32 GetLength(int dimension)
			{
				if ((dimension == 0) || (dimension == 1))
					return array.GetLength(dimension);
				throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
			}
			public override object GetArray() {return array;}
			
			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= array.GetLength(0)))
					throw new CodeEE("二次元配列" + varName + "の第１引数(" + arguments[0].ToString() + ")は配列の範囲外です");
				if ((arguments[1] < 0) || (arguments[1] >= array.GetLength(1)))
					throw new CodeEE("二次元配列" + varName + "の第２引数(" + arguments[1].ToString() + ")は配列の範囲外です");
			}
			public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
			{
				CheckElement(arguments);
				if ((index1 < 0) || (index1 > array.GetLength(1)))
					throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
				if ((index2 < 0) || (index2 > array.GetLength(1)))
					throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
			}
		}

		private sealed class Str3DVariableToken : VariableToken
		{
			public Str3DVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
				array = varData.DataStringArray3D[VarCodeInt];
			}
			string[, ,] array;
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0], arguments[1], arguments[2]];
			}

			public override void SetValue(string value, Int64[] arguments)
			{
				array[arguments[0], arguments[1], arguments[2]] = value;
			}
			public override Int32 GetLength()
			{ throw new CodeEE("3次元配列型変数" + varName + "の長さを取得しようとしました"); }
			public override Int32 GetLength(int dimension)
			{
				if ((dimension == 0) || (dimension == 1) || (dimension == 2))
					return array.GetLength(dimension);
				throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
			}
			public override object GetArray() {return array;}
			
			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= array.GetLength(0)))
					throw new CodeEE("三次元配列" + varName + "の第１引数(" + arguments[0].ToString() + ")は配列の範囲外です");
				if ((arguments[1] < 0) || (arguments[1] >= array.GetLength(1)))
					throw new CodeEE("三次元配列" + varName + "の第２引数(" + arguments[1].ToString() + ")は配列の範囲外です");
				if ((arguments[2] < 0) || (arguments[2] >= array.GetLength(2)))
					throw new CodeEE("三次元配列" + varName + "の第３引数(" + arguments[2].ToString() + ")は配列の範囲外です");
			}
			public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
			{
				CheckElement(arguments);
				if ((index1 < 0) || (index1 > array.GetLength(2)))
					throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
				if ((index2 < 0) || (index2 > array.GetLength(2)))
					throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
			}
		}

		private sealed class CharaIntVariableToken : VariableToken
		{
			public CharaIntVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				return chara.DataInteger[VarCodeInt];
			}

			public override void SetValue(Int64 value, Int64[] arguments)
			{
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				chara.DataInteger[VarCodeInt] = value;
			}

			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				chara.DataInteger[VarCodeInt] += value;
				return chara.DataInteger[VarCodeInt];
			}
			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= varData.CharacterList.Count))
					throw new CodeEE("キャラクタ配列変数" + varName + "の第１引数(" + arguments[0].ToString() + ")はキャラ登録番号の範囲外です");
			}
		}

		private sealed class CharaInt1DVariableToken : VariableToken
		{
			public CharaInt1DVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;

			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				return chara.DataIntegerArray[VarCodeInt][arguments[1]];
			}

			public override void SetValue(Int64 value, Int64[] arguments)
			{
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				chara.DataIntegerArray[VarCodeInt][arguments[1]] = value;
			}

			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				chara.DataIntegerArray[VarCodeInt][arguments[1]] += value;
				return chara.DataIntegerArray[VarCodeInt][arguments[1]];
			}
			public override Int32 GetLength()
			{ return varData.Constant.CharacterIntArrayLength[VarCodeInt]; }
			public override Int32 GetLength(int dimension)
			{
				if (dimension == 0)
					return varData.Constant.CharacterIntArrayLength[VarCodeInt];
				throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
			}
			
			public override object GetArrayChara(int charano)
			{
				CharacterData chara = varData.CharacterList[charano];
				return chara.DataIntegerArray[VarCodeInt];
			}
			
			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= varData.CharacterList.Count))
					throw new CodeEE("キャラクタ配列変数" + varName + "の第１引数(" + arguments[0].ToString() + ")はキャラ登録番号の範囲外です");
				Int64[] array = varData.CharacterList[(int)arguments[0]].DataIntegerArray[VarCodeInt];
				if ((arguments[1] < 0) || (arguments[1] >= array.Length))
					throw new CodeEE("キャラクタ配列変数" + varName + "の第２引数(" + arguments[1].ToString() + ")は配列の範囲外です");
			}
			public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
			{
				CheckElement(arguments);
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				Int64[] array = chara.DataIntegerArray[VarCodeInt];
				if ((index1 < 0) || (index1 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
				if ((index2 < 0) || (index2 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
			}
		}


		private sealed class CharaStrVariableToken : VariableToken
		{
			public CharaStrVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
			}
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				return chara.DataString[VarCodeInt];
			}

			public override void SetValue(string value, Int64[] arguments)
			{
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				chara.DataString[VarCodeInt] = value;
			}
			
			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= varData.CharacterList.Count))
					throw new CodeEE("キャラクタ配列変数" + varName + "の第１引数(" + arguments[0].ToString() + ")はキャラ登録番号の範囲外です");
			}

		}

		private sealed class CharaStr1DVariableToken : VariableToken
		{
			public CharaStr1DVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
			}
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				return chara.DataStringArray[VarCodeInt][arguments[1]];
			}

			public override void SetValue(string value, Int64[] arguments)
			{
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				chara.DataStringArray[VarCodeInt][arguments[1]] = value;
			}
			public override Int32 GetLength()
			{ return varData.Constant.CharacterStrArrayLength[VarCodeInt]; }
			public override Int32 GetLength(int dimension)
			{
				if (dimension == 0)
					return varData.Constant.CharacterStrArrayLength[VarCodeInt];
				throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
			}
			
			
			public override object GetArrayChara(int charano)
			{
				CharacterData chara = varData.CharacterList[charano];
				return chara.DataStringArray[VarCodeInt];
			}
			
			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= varData.CharacterList.Count))
					throw new CodeEE("キャラクタ配列変数" + varName + "の第１引数(" + arguments[0].ToString() + ")はキャラ登録番号の範囲外です");
				string[] array = varData.CharacterList[(int)arguments[0]].DataStringArray[VarCodeInt];
				if ((arguments[1] < 0) || (arguments[1] >= array.Length))
					throw new CodeEE("キャラクタ配列変数" + varName + "の第２引数(" + arguments[1].ToString() + ")は配列の範囲外です");
			}
			public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
			{
				CheckElement(arguments);
				CharacterData chara = varData.CharacterList[(int)arguments[0]];
				string[] array = chara.DataStringArray[VarCodeInt];
				if ((index1 < 0) || (index1 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
				if ((index2 < 0) || (index2 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
			}
		}

		#endregion
		#region 定数
		private abstract class ConstantToken : VariableToken
		{
			public ConstantToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = true;
			}
			public override void SetValue(Int64 value, Int64[] arguments)
			{ throw new CodeEE("読み取り専用の変数" + varName + "に代入しようとしました"); }
			public override void SetValue(string value, Int64[] arguments)
			{ throw new CodeEE("読み取り専用の変数" + varName + "に代入しようとしました"); }
			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{ throw new CodeEE("読み取り専用の変数" + varName + "に代入しようとしました"); }
		}

		private sealed class IntConstantToken : ConstantToken
		{
			public IntConstantToken(VariableCode varCode, VariableData varData, Int64 i)
				: base(varCode, varData)
			{
				this.i = i;
			}
			Int64 i;
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return i;
			}
		}
		private sealed class StrConstantToken : ConstantToken
		{
			public StrConstantToken(VariableCode varCode, VariableData varData, string s)
				: base(varCode, varData)
			{
				this.s = s;
			}
			string s;
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				return s;
			}
		}
		private sealed class Int1DConstantToken : ConstantToken
		{
			public Int1DConstantToken(VariableCode varCode, VariableData varData, Int64[] array)
				: base(varCode, varData)
			{
				this.array = array;
			}
			Int64[] array = null;
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0]];
			}
			public override Int32 GetLength()
			{ return array.Length; }
			public override Int32 GetLength(int dimension)
			{
				if (dimension == 0)
					return array.Length;
				throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
			}
			public override object GetArray() {return array;}

			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= array.Length))
					throw new CodeEE("配列変数" + varName + "の第１引数(" + arguments[0].ToString() + ")は配列の範囲外です");
			}
			public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
			{
				CheckElement(arguments);
				if ((index1 < 0) || (index1 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
				if ((index2 < 0) || (index2 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
			}
		}

		private sealed class Str1DConstantToken : ConstantToken
		{
			public Str1DConstantToken(VariableCode varCode, VariableData varData, string[] array)
				: base(varCode, varData)
			{
				this.array = array;
			}
			string[] array = null;
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0]];
			}
			public override Int32 GetLength()
			{ return array.Length; }
			public override Int32 GetLength(int dimension)
			{
				if (dimension == 0)
					return array.Length;
				throw new CodeEE("配列型変数" + varName + "の存在しない次元の長さを取得しようとしました");
			}
			public override object GetArray() {return array;}
			
			public override void CheckElement(Int64[] arguments)
			{
				if ((arguments[0] < 0) || (arguments[0] >= array.Length))
					throw new CodeEE("配列変数" + varName + "の第１引数(" + arguments[0].ToString() + ")は配列の範囲外です");
			}
			public override void IsArrayRangeValid(Int64[] arguments, Int64 index1, Int64 index2, string funcName, Int64 i1, Int64 i2)
			{
				CheckElement(arguments);
				if ((index1 < 0) || (index1 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i1.ToString() + "引数(" + index1.ToString() + ")は配列" + varName + "の範囲外です");
				if ((index2 < 0) || (index2 > array.Length))
					throw new CodeEE(funcName + "命令の第" + i2.ToString() + "引数(" + index2.ToString() + ")は配列" + varName + "の範囲外です");
			}
		}

		#endregion
		#region 特殊処理

		private abstract class PseudoVariableToken : VariableToken
		{
			protected PseudoVariableToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
			}
			public override void SetValue(Int64 value, Int64[] arguments)
			{ throw new CodeEE("擬似変数" + varName + "に代入しようとしました"); }
			public override void SetValue(string value, Int64[] arguments)
			{ throw new CodeEE("擬似変数" + varName + "に代入しようとしました"); }
			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{ throw new CodeEE("擬似変数" + varName + "に代入しようとしました"); }
			public override Int32 GetLength()
			{ throw new CodeEE("擬似変数" + varName + "の長さを取得しようとしました"); }
			public override Int32 GetLength(int dimension)
			{ throw new CodeEE("擬似変数" + varName + "の長さを取得しようとしました"); }
			public override object GetArray()
			{ throw new CodeEE("擬似変数" + varName + "の配列を取得しようとしました"); }
		}


		private sealed class RandToken : PseudoVariableToken
		{
			public RandToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				Int64 i = arguments[0];
				if (i <= 0)
					throw new CodeEE("RANDの引数に0以下の値(" + i.ToString() + ")が指定されました");
				return exm.VEvaluator.GetNextRand(i);
			}
		}
		private sealed class CompatiRandToken : PseudoVariableToken
		{
			public CompatiRandToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				Int64 i = arguments[0];
				if (i == 0)
					return 0L;
				else if (i < 0)
					i = -i;
				return exm.VEvaluator.GetNextRand(32768) % i;//0-32767の乱数を引数で除算した余り
			}
		}

		private sealed class CHARANUM_Token : PseudoVariableToken
		{
			public CHARANUM_Token(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return varData.CharacterList.Count;
			}
		}

		private sealed class LASTLOAD_TEXT_Token : PseudoVariableToken
		{
			public LASTLOAD_TEXT_Token(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
			}
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				return varData.LastLoadText;
			}
		}

		private sealed class LASTLOAD_VERSION_Token : PseudoVariableToken
		{
			public LASTLOAD_VERSION_Token(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return varData.LastLoadVersion;
			}
		}

		private sealed class LASTLOAD_NO_Token : PseudoVariableToken
		{
			public LASTLOAD_NO_Token(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return varData.LastLoadNo;
			}
		}
		private sealed class LINECOUNT_Token : PseudoVariableToken
		{
			public LINECOUNT_Token(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return exm.Console.LineCount;
			}
		}
		
		private sealed class WINDOW_TITLE_Token : VariableToken
		{
			public WINDOW_TITLE_Token(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = false;
			}
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				return GlobalStatic.Console.GetWindowTitle();
			}
			public override void SetValue(string value, Int64[] arguments)
			{
				GlobalStatic.Console.SetWindowTitle(value);
			}
		}

        private sealed class MONEYLABEL_Token : VariableToken
        {
            public MONEYLABEL_Token(VariableCode varCode, VariableData varData)
                : base(varCode, varData)
            {
                CanRestructure = false;
            }
            public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
            {
                return Config.MoneyLabel;
            }
        }

		private sealed class EmptyStrToken : PseudoVariableToken
		{
			public EmptyStrToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = true;
			}
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				return "";
			}
		}
		private sealed class EmptyIntToken : PseudoVariableToken
		{
			public EmptyIntToken(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = true;
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return 0L;
			}
		}

		private sealed class Debug__FILE__Token : PseudoVariableToken
		{
			public Debug__FILE__Token(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = true;
			}
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				LogicalLine line = exm.Process.GetScaningLine();
				if ((line == null) || (line.Position == null))
					return "";
				return line.Position.Filename;
			}
		}

		private sealed class Debug__FUNCTION__Token : PseudoVariableToken
		{
			public Debug__FUNCTION__Token(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = true;
			}
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				LogicalLine line = exm.Process.GetScaningLine();
				if ((line == null) || (line.ParentLabelLine == null))
					return "";//システム待機中のデバッグモードから呼び出し
				return line.ParentLabelLine.LabelName;
			}
		}
		private sealed class Debug__LINE__Token : PseudoVariableToken
		{
			public Debug__LINE__Token(VariableCode varCode, VariableData varData)
				: base(varCode, varData)
			{
				CanRestructure = true;
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				LogicalLine line = exm.Process.GetScaningLine();
				if ((line == null) || (line.Position == null))
					return -1L;
				return line.Position.LineNo;
			}
		}
		#endregion
#region LOCAL
		

		private sealed class LocalInt1DVariableToken : LocalVariableToken
		{
			public LocalInt1DVariableToken(VariableCode varCode, VariableData varData, string subId, int size)
				: base(varCode, varData, subId, size)
			{
			}
			Int64[] array = null;
			
			public override void SetDefault()
			{
				if(array != null)
					Array.Clear(array, 0, size);
			}

			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				if(array == null)
					array = new Int64[size];
				return array[arguments[0]];
			}

			public override void SetValue(Int64 value, Int64[] arguments)
			{
				if(array == null)
					array = new Int64[size];
				array[arguments[0]] = value;
			}

			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{
				if(array == null)
					array = new Int64[size];
				array[arguments[0]] += value;
				return array[arguments[0]];
			}
			
			public override object GetArray()
			{
				if(array == null)
					array = new Int64[size];
				return array;
			}

            public override void resize(int newSize)
            {
                this.size = newSize;
                array = null;
            }
		}

		private sealed class LocalStr1DVariableToken : LocalVariableToken
		{
			public LocalStr1DVariableToken(VariableCode varCode, VariableData varData, string subId, int size)
				: base(varCode, varData, subId, size)
			{
			}
			string[] array = null;
			public override void SetDefault()
			{
				if(array != null)
					Array.Clear(array, 0, size);
			}

			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				if (array == null)
					array = new string[size];
				return array[arguments[0]];
			}

			public override void SetValue(string value, Int64[] arguments)
			{
				if (array == null)
					array = new string[size];
				array[arguments[0]] = value;
			}
			
			public override object GetArray()
			{
				if (array == null)
					array = new string[size];
				return array;
			}

            public override void resize(int newSize)
            {
                this.size = newSize;
                array = null;
            }
			
		}

#endregion
		#region userdef
		
		
		private sealed class UserDefinedInt1DVariableToken : UserDefinedVariableToken
		{
			public UserDefinedInt1DVariableToken(string name, int size)
				: base(VariableCode.VAR, name, size)
			{
				CanRestructure = false;
				array = new Int64[size];
			}
			Int64[] array = null;
			public override void SetDefault()
			{
				Array.Clear(array, 0, size);
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0]];
			}
            public override void SetValue(Int64 value, Int64[] arguments)
			{
				array[arguments[0]] = value;
			}
			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{
				array[arguments[0]] += value;
				return array[arguments[0]];
			}
			public override object GetArray() {return array;}

		}

		private sealed class UserDefinedStr1DVariableToken : UserDefinedVariableToken
		{
			public UserDefinedStr1DVariableToken(string name, int size)
				: base(VariableCode.PRIVATES, name, size)
			{
				CanRestructure = false;
				array = new string[size];
			}
			string[] array = null;
			public override void SetDefault()
			{
				Array.Clear(array, 0, size);
			}
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0]];
			}
			public override void SetValue(string value, Int64[] arguments)
			{
				array[arguments[0]] = value;
			}
			public override object GetArray() {return array;}

		}
		
		
		#region private
		private sealed class PrivateInt1DVariableToken : PrivateVariableToken
		{
			public PrivateInt1DVariableToken(string name, int size)
				: base(VariableCode.PRIVATE, name, size)
			{
				CanRestructure = false;
				arrayList = new List<Int64[]>();
			}
			readonly List<Int64[]> arrayList = null;
			Int64[] array = null;
            Int64[] newArray = null;
			int counter = 0;
			public override void SetDefault()
			{
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0]];
			}

			public override void SetValue(Int64 value, Int64[] arguments)
			{
                //再帰対策として、引数処理では一時配列を使用する
                if (newArray != null)
                    newArray[arguments[0]] = value;
                else
                    array[arguments[0]] = value;
			}

			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{
				array[arguments[0]] += value;
				return array[arguments[0]];
			}
			public override object GetArray() {return array;}

			public override void In()
			{
				counter++;
                //この時点で更新してしまうと再帰ができない
                newArray = new Int64[size];
				//array = new Int64[size];
				//arrayList.Add(array);
			}

            public override void UpdateGeneration()
            {
                //一時変数で処理したものをこの時点で反映
                array = newArray;
                arrayList.Add(array);
                newArray = null;
            }

			public override void Out()
			{
				counter--;
				arrayList.RemoveAt(arrayList.Count - 1);
				if(arrayList.Count > 0)
					array = arrayList[arrayList.Count -1];
				else
					array = null;
			}
		}

		private sealed class PrivateStr1DVariableToken : PrivateVariableToken
		{
			public PrivateStr1DVariableToken(string name, int size)
				: base(VariableCode.PRIVATES, name, size)
			{
				CanRestructure = false;
				arrayList = new List<string[]>();
			}
			int counter = 0;
			readonly List<string[]> arrayList = null;
			string[] array = null;
            string[] newArray = null;
			public override void SetDefault()
			{
			}
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
                return array[arguments[0]];
			}

			public override void SetValue(string value, Int64[] arguments)
			{
                if (newArray != null)
                    newArray[arguments[0]] = value;
                else
                    array[arguments[0]] = value;
			}
			public override object GetArray() {return array;}
			public override void In()
			{
				counter++;
                //この時点で配列を更新してしまうと再帰ができない
                newArray = new string[size];
                //array = new string[size];
                //arrayList.Add(array);
			}
            public override void UpdateGeneration()
            {
                //一時変数で処理したものをこの時点で反映
                array = newArray;
                arrayList.Add(array);
                newArray = null;
            }
            public override void Out()
			{
				counter--;
				arrayList.RemoveAt(arrayList.Count -1);
				if(arrayList.Count > 0)
					array = arrayList[arrayList.Count -1];
				else
					array = null;
			}
		}
		private sealed class PrivateStaticInt1DVariableToken : PrivateVariableToken
		{
			public PrivateStaticInt1DVariableToken(string name, int size)
				: base(VariableCode.PRIVATE, name, size)
			{
				CanRestructure = false;
				array = new Int64[size];
			}
			Int64[] array = null;
			public override void SetDefault()
			{
				Array.Clear(array, 0, size);
			}
			public override Int64 GetIntValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0]];
			}
            
            public override void SetValue(Int64 value, Int64[] arguments)
			{
				array[arguments[0]] = value;
			}

			public override Int64 PlusValue(Int64 value, Int64[] arguments)
			{
				array[arguments[0]] += value;
				return array[arguments[0]];
			}
			public override object GetArray() {return array;}
			public override void In()
			{
			}
			public override void Out()
			{
			}
            public override void UpdateGeneration()
            {
            }
		}

		private sealed class PrivateStaticStr1DVariableToken : PrivateVariableToken
		{
			public PrivateStaticStr1DVariableToken(string name, int size)
				: base(VariableCode.PRIVATES, name, size)
			{
				CanRestructure = false;
				array = new string[size];
			}
			string[] array = null;
			public override void SetDefault()
			{
				Array.Clear(array, 0, size);
			}
			public override string GetStrValue(ExpressionMediator exm, Int64[] arguments)
			{
				return array[arguments[0]];
			}

			public override void SetValue(string value, Int64[] arguments)
			{
				array[arguments[0]] = value;
			}
			public override object GetArray() {return array;}
			public override void In()
			{
			}
			public override void Out()
			{
			}
            public override void UpdateGeneration()
            {
            }
	}
		#endregion
		#endregion
	}
}