/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

#ifdef HAVE_RUBY_H

#include <ruby.h>

typedef struct {
  eh_connection_t *connection_backref;
  eh_strbuf_t strbuf_from_client;
  eh_method_t method;
  int reqbody_len;
  char *headers_str;
  char *filename;
  int http_version_minor;
} eh_cgiruby_extdata_t;

EXTERN VALUE ruby_errinfo;

#define TAG_RETURN      0x1
#define TAG_BREAK       0x2
#define TAG_NEXT        0x3
#define TAG_RETRY       0x4
#define TAG_REDO        0x5
#define TAG_RAISE       0x6
#define TAG_THROW       0x7
#define TAG_FATAL       0x8
#define TAG_MASK        0xf

static void
eh_error_pos(char **str)
{
  if (ruby_sourcefile) {
    if (ruby_sourceline == 0) {
      x_append_printf (str, "%s", ruby_sourcefile);
    }
    else {
      x_append_printf (str, "%s:%d", ruby_sourcefile, ruby_sourceline);
    }
  }
}


#define TRACE_MAX (TRACE_HEAD+TRACE_TAIL+5)
#define TRACE_HEAD 8
#define TRACE_TAIL 5

static void
eh_print_ruby_error (char **str)
{
  VALUE errat = Qnil;
  volatile VALUE eclass;
  VALUE estr;
  char *einfo;
  int elen;
  int state;
  
  if (NIL_P(ruby_errinfo)) return;
  
  errat = rb_funcall (ruby_errinfo, rb_intern ("backtrace"), 0);
  if (NIL_P (errat)) {
    if (ruby_sourcefile)
      x_append_printf (str, "%s:%d", ruby_sourcefile, ruby_sourceline);
    else
      x_append_printf (str, "%d", ruby_sourceline);
  }
  else {
    VALUE mesg = RARRAY (errat)->ptr[0];
    
    if (NIL_P (mesg))
      eh_error_pos (str);
    else {
      char *s;
      s = x_strndup (RSTRING (mesg)->ptr, RSTRING (mesg)->len);
      x_str_append (str, s);
      x_free (s);
    }
  }
  
  einfo = "";
  elen = 0;
  eclass = CLASS_OF (ruby_errinfo);
#ifdef __cplusplus
  estr = rb_protect ((VALUE (*)(...))rb_obj_as_string, ruby_errinfo, &state);
#else
  estr = rb_protect (rb_obj_as_string, ruby_errinfo, &state);
#endif
  if (state == 0) {
    einfo = RSTRING (estr)->ptr;
    elen = RSTRING (estr)->len;
  }
  if (eclass == rb_eRuntimeError && elen == 0) {
    x_append_printf (str, ": unhandled exception\n");
  }
  else {
    VALUE epath;
    
    epath = rb_class_path(eclass);
    if (elen == 0) {
      char *s;
      s = x_strndup (RSTRING (epath)->ptr, RSTRING (epath)->len);
      x_append_printf (str, ": %s\n", s);
      x_free (s);
    }
    else {
      char *tail  = NULL;
      int len = elen;
      char *s;
      
      if (RSTRING (epath)->ptr[0] == '#') epath = 0;
      if ((tail = strchr (einfo, '\n')) != NULL) {
	len = tail - einfo;
	tail++;         /* skip newline */
      }
      s = x_strndup (einfo, len);
      x_append_printf (str, ": %s", s);
      x_free (s);
      if (epath) {
	s = x_strndup (RSTRING (epath)->ptr, RSTRING (epath)->len);
	x_append_printf (str, " (%s)\n", s);
	x_free (s);
      }
      if (tail) {
	s = x_strndup (tail, elen - len - 1);
	x_append_printf (str, "%s\n", s);
      }
    }
  }
  
  if (!NIL_P(errat)) {
    int i;
    struct RArray *ep = RARRAY(errat);
    
    ep = RARRAY(errat);
    for (i=1; i<ep->len; i++) {
      if (TYPE(ep->ptr[i]) == T_STRING) {
	x_append_printf (str, "\tfrom %s\n", RSTRING(ep->ptr[i])->ptr);
      }
      if (i == TRACE_HEAD && ep->len > TRACE_MAX) {
	x_append_printf (str, "\t ... %ld levels...\n",
			 ep->len - TRACE_HEAD - TRACE_TAIL);
	i = ep->len - TRACE_TAIL;
      }
    }
  }
}

static void
eh_set_env_variables (char **envp)
{
  char **p;
  for (p = envp; *p; p++) {
    char *s;
#ifdef DEBUG_ENV
    eh_log (EH_LOG_INFO, "setting %s", *p);
#endif
    s = strchr (*p, '=');
    if (s == NULL)
      continue;
    *s++ = '\0';
    setenv (*p, s, 1);
  }
}

static void
eh_unset_env_variables (char **envp)
{
  char **p;
#ifdef DEBUG_ENV
  eh_log (EH_LOG_INFO, "unset_env");
#endif
  for (p = envp; *p; p++) {
    char *s;
    s = strchr (*p, '=');
    if (s)
      *s = '\0';
#ifdef DEBUG_ENV
    eh_log (EH_LOG_INFO, "unsetting %s", *p);
#endif
    unsetenv (*p);
  }
}

static void
eh_ruby_gc (void)
{
  static int count = 0;
  /* TODO: make this configurable */
  if (++count > 100) {
    rb_gc ();
    count = 0;
  }
}

static int
eh_cgiruby_eval_script (eh_cgiruby_extdata_t *extdata, eh_request_t *er,
			eh_connection_t *ec, char **error_message)
{
  int r = 0;
  char **envp;
  char *str;
  if (eh_cgicommon_chdir (extdata->filename))
    return 1;
  envp = eh_cgicommon_make_env (er, ec);
  eh_set_env_variables (envp);
  str = eh_str_load_file (extdata->filename);
  if (str == NULL) {
    r = 1;
  } else {
    int state;
    eh_debug ("rb_eval_string: %s", extdata->filename);
    /* there are 3 candidates. which is suitable? */
    rb_eval_string_protect (str, &state);
    // rb_eval_string_wrap (str, &state);
    // rb_load_protect (rb_str_new2 (extdata->filename), 1, &state);
    x_free (str);
    eh_ruby_gc ();
    if (state) {
      if (state == TAG_RAISE &&
	  rb_obj_is_kind_of(ruby_errinfo, rb_eSystemExit)) {
#if 0
	VALUE v;
	v = rb_iv_get (ruby_errinfo, "status");
	if (!NIL_P(v)) {
	  eh_log (EH_LOG_INFO, "SystemExit returned non-nil");
	}
#endif
      } else {
	x_append_printf (error_message, "Script execution failed.\n");
	eh_print_ruby_error (error_message);
      }
    } else {
#if 0
      VALUE v;
      v = rb_iv_get (ruby_errinfo, "status");
      if (!NIL_P(v)) {
	eh_log (EH_LOG_INFO, "Returned non-nil");
      }
#endif
    }
    // FIXME: rb_protect(exec_end_proc);
    rb_exec_end_proc ();
    /* ruby script sometimes override the current sigprocmask, and then
       realtime signals can be 'leaked'. when such a leak is occured,
       on_leaked_rtsignal is called and the sigprocmask is resumed. the
       following line forces to resume the sigprocmask in order to avoid
       signal leaks as far as possible. in theory, it should work fine
       without the following line. */
    eh_fd_block_rtsignal ();
  }
  eh_unset_env_variables (envp);
  eh_strarr_free (envp);
  chdir ("/");
  return r;
}

static void
eh_cgiruby_tmpfile (void)
{
  FILE *fp;
  fflush (stdout);
  fp = tmpfile ();
  if (fp == NULL) {
    eh_log_perror (EH_LOG_FATAL, "tmpfile()");
    exit (-1);
  }
  dup2 (fileno (fp), 0);
  fclose (fp);
  fp = tmpfile ();
  if (fp == NULL) {
    eh_log_perror (EH_LOG_FATAL, "tmpfile()");
    exit (-1);
  }
  dup2 (fileno (fp), 1);
  fclose (fp);
}

static void
eh_cgiruby_reply (eh_cgiruby_extdata_t *extdata)
{
  int r;
  eh_connection_t *ec = extdata->connection_backref;
  char *error_message = NULL;
  
  if (access (ec->current_request.filename, X_OK) < 0) {
    eh_connection_append_wvec_response (ec, extdata->method, "403",
					NULL, NULL, 0);
    goto finish;
  }
  
  /* prepare stdin and stdout */
  eh_cgiruby_tmpfile ();
  eh_debug ("request body(%d): %s", extdata->strbuf_from_client.buffer_len,
	    extdata->strbuf_from_client.buffer);
  do {
    r = write (0, extdata->strbuf_from_client.buffer,
	       extdata->strbuf_from_client.buffer_len);
  } while (r < 0 && errno == EINTR);
  if (r < 0) {
    eh_log_perror (EH_LOG_WARNING, "stdin");
    eh_connection_append_wvec_response (ec, extdata->method, "500",
					NULL, NULL, 0);
    goto finish;
  }
  lseek (0, 0, SEEK_SET);
  if (eh_cgiruby_eval_script (extdata, &ec->current_request, ec,
			      &error_message) == 0) {
    off_t len;
    char *buf;
    fflush (stdout); // FIXME: necessary?
    len = lseek (1, 0, SEEK_CUR);
    lseek (1, 0, SEEK_SET);
    buf = (char *)x_malloc (len + 1);
    buf[len] = '\0';
    do {
      r = read (1, buf, len);
      // FIXME: it's possible that r < len
    } while (r < 0 && errno == EINTR);
    eh_debug ("read stdout %d bytes", r);
    eh_debug ("%s", buf);
    if (r != len) {
      if (r < 0) {
	eh_log_perror (EH_LOG_WARNING, "stdout");
      } else {
	eh_log (EH_LOG_WARNING, "offset is %d, read %d", len, r);
      }
      eh_connection_append_wvec_response (ec, extdata->method, "500",
					  NULL, NULL, 0);
    } else {
      /* succeed */
      if (error_message) {
	x_append_printf (&error_message, "--------\n%s", buf);
	eh_cgicommon_response_500 (ec, extdata->method, error_message);
      } else {
	eh_cgicommon_reply (ec, extdata->method, buf, len);
      }
    }
    x_free (buf);
  } else {
    eh_log (EH_LOG_WARNING, "%s: eval failed", extdata->filename);
    eh_connection_append_wvec_response (ec, extdata->method, "500",
					NULL, NULL, 0);
  }
  
 finish:
  if (error_message)
    x_free (error_message);
  eh_connection_request_finish (ec);
  eh_connection_set_request_events (ec, 0xa47);
  eh_app_increment_script_eval_count (ec->app_backref);
}

static void
eh_cgiruby_on_read_request_body (eh_rhandler_t *eh, const char *buf,
				 size_t buflen)
{
  eh_cgiruby_extdata_t *extdata = (eh_cgiruby_extdata_t *)eh->extdata;
  eh_debug ("len = %d", buflen);
  eh_strbuf_append (&extdata->strbuf_from_client, buf, buflen);
  eh->body_length_left = extdata->strbuf_from_client.read_limit;
  if (eh->body_length_left == 0) {
    eh_cgiruby_reply (extdata);
  }
  return;
}

static int
eh_cgiruby_do_timeout (eh_rhandler_t *eh)
{
  return 1;
}

static void
eh_cgiruby_on_delete (eh_rhandler_t *eh)
{
  eh_cgiruby_extdata_t *extdata = (eh_cgiruby_extdata_t *)eh->extdata;
  eh_debug ("%p", eh);
  eh_strbuf_discard (&extdata->strbuf_from_client);
  if (extdata->headers_str)
    x_free (extdata->headers_str);
  if (extdata->filename)
    x_free (extdata->filename);
  x_free (extdata);
  x_free (eh);
}

eh_rhandler_t eh_cgiruby_tmpl = {
  0, NULL,
  eh_cgiruby_on_read_request_body,
  eh_cgiruby_do_timeout,
  eh_cgiruby_on_delete,
};

eh_rhandler_t *
eh_rhandler_cgiruby_new (eh_connection_t *ec, const eh_request_t *er,
			 void *rhfunc_data)
{
  eh_rhandler_t *eh = NULL;
  eh_cgiruby_extdata_t *extdata;
  int reqbody_len = 0;
  const char *s;

  s = er->headers.predef.content_length;
  if (s) {
    reqbody_len = atoi (s);
  }

  extdata = (eh_cgiruby_extdata_t *)x_malloc (sizeof (*extdata));
  memset (extdata, 0, sizeof (*extdata));
  eh_strbuf_init (&extdata->strbuf_from_client, reqbody_len);
  extdata->connection_backref = ec;
  extdata->method = er->method;
  extdata->headers_str = eh_headers_strdup_getall (&er->headers);
  extdata->filename = x_strdup (er->filename);
  extdata->http_version_minor = er->http_version_minor;

  eh = (eh_rhandler_t *)x_malloc (sizeof (*eh));
  memcpy (eh, &eh_cgiruby_tmpl, sizeof (*eh));
  eh->extdata = extdata;
  eh->body_length_left = reqbody_len;
  if (reqbody_len == 0) {
    eh_cgiruby_reply (extdata);
    eh_cgiruby_on_delete (eh);
    return NULL;
  }
  return eh;
}

static void eh_cgiruby_init_ruby (void) __attribute__ ((constructor));
static void
eh_cgiruby_init_ruby (void)
{
  ruby_init ();
  ruby_init_loadpath ();
}


REGISTER_HANDLER ("cgi-ruby", eh_rhandler_cgiruby_new, NULL, 0);

#endif

