/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

#define EH_HASH_SIZE 389

#define x_free_if(x) if (x) x_free (x)

static void
eh_config_on_delete_passwd_hentry (eh_strhash_entry_t *ent)
{
  x_free (ent->key);
  x_free ((char *)ent->val);
}

void
eh_config_limit_delete (eh_config_limit_t *ecl)
{
  eh_strhash_delete (ecl->passwd_ht);
  if (ecl->require)
    eh_strarr_free (ecl->require);
  x_free (ecl);
}

eh_config_limit_t *
eh_config_limit_new (eh_method_t mask)
{
  eh_config_limit_t *ecl;
  ecl = (eh_config_limit_t *)x_malloc (sizeof (*ecl));
  memset (ecl, 0, sizeof (*ecl));
  ecl->auth_required = 0;
  ecl->passwd_ht = eh_strhash_new (EH_HASH_SIZE,
				   eh_config_on_delete_passwd_hentry);
  ecl->method_mask = mask;
  return ecl;
}

void
eh_config_filesmatch_delete (eh_config_filesmatch_t *ecf)
{
  int i;
  regfree (&ecf->re);
  x_free (ecf->pattern);
  for (i = 0; i < ecf->num_limit; i++)
    eh_config_limit_delete (ecf->limit_arr[i]);
  if (ecf->limit_arr)
    x_free (ecf->limit_arr);
  eh_config_limit_delete (ecf->default_limit);
  x_free (ecf);
}

eh_config_filesmatch_t *
eh_config_filesmatch_new (const char *pattern, eh_config_global_t *ecg)
{
  int r;
  eh_config_filesmatch_t *ecf;
  ecf = (eh_config_filesmatch_t *)x_malloc (sizeof (*ecf));
  memset (ecf, 0, sizeof (*ecf));
  r = regcomp (&ecf->re, pattern, REG_EXTENDED | REG_NOSUB);
  if (r) {
    char buf[1025];
    buf[1024] = '\0';
    regerror (r, &ecf->re, buf, 1024);
    x_append_printf (&ecg->error_string,
		     "invalid regular expression '%s': %s\n",
		     pattern, buf);
    regfree (&ecf->re);
    x_free (ecf);
    return NULL;
  }
  ecf->pattern = x_strdup (pattern);
  ecf->num_limit = 0;
  ecf->limit_arr = NULL;
  ecf->default_limit = eh_config_limit_new (eh_method_allmask);
  return ecf;
}

void
eh_config_dir_delete (eh_config_dir_t *ecd)
{
  int i;
  if (ecd->directoryindexes)
    eh_strarr_free (ecd->directoryindexes);
  x_free_if (ecd->defaulttype);
  x_free_if (ecd->authuserfile);
  x_free_if (ecd->authname);
  eh_strhash_delete (ecd->addhandler_ht);
  for (i = 0; i < ecd->num_filesmatch; i++)
    eh_config_filesmatch_delete (ecd->filesmatch_arr[i]);
  if (ecd->filesmatch_arr)
    x_free (ecd->filesmatch_arr);
  eh_config_filesmatch_delete (ecd->default_filesmatch);
  x_free (ecd);
}

static void
eh_config_on_delete_dir (eh_strhash_entry_t *ent)
{
  eh_config_dir_t *ecd = (eh_config_dir_t *)ent->val;
  x_free (ent->key);
  eh_config_dir_delete (ecd);
}

eh_addhandler_t *
eh_addhandler_new (int is_handler, const char *name)
{
  eh_addhandler_t *ah;
  ah = (eh_addhandler_t *)x_malloc (sizeof (*ah));
  ah->is_handler = is_handler;
  ah->name = x_strdup (name);
  ah->rhfunc = NULL;
  ah->rhfunc_data = NULL;
  ah->nofilesystem = 0;
  return ah;
}

void
eh_addhandler_delete (eh_addhandler_t *ah)
{
  x_free (ah->name);
  x_free (ah);
}

static void
eh_config_on_delete_addhandler_entry (eh_strhash_entry_t *ent)
{
  x_free (ent->key);
  eh_addhandler_delete ((eh_addhandler_t *)ent->val);
}

eh_config_dir_t *
eh_config_dir_new (eh_config_global_t *ecg)
{
  eh_config_dir_t *ecd;
  ecd = (eh_config_dir_t *)x_malloc (sizeof (*ecd));
  memset (ecd, 0, sizeof (*ecd));
  ecd->econf_global_ref = ecg;
  ecd->filesmatch_arr = NULL;
  ecd->sslverifydepth = 1;
  ecd->num_filesmatch = 0;
  ecd->default_filesmatch = eh_config_filesmatch_new (".*", ecg);
  assert (ecd->default_filesmatch);
  ecd->addhandler_ht = eh_strhash_new (EH_HASH_SIZE,
				       eh_config_on_delete_addhandler_entry);
  return ecd;
}

void
eh_config_vhost_delete (eh_config_vhost_t *ecv)
{
  if (ecv->passenv)
    eh_strarr_free (ecv->passenv);
  x_free_if (ecv->documentroot);
  x_free_if (ecv->servername);
  x_free_if (ecv->serveradmin);
  x_free_if (ecv->customlog);
  x_free_if (ecv->typesconfig);
  x_free_if (ecv->errordocumentdir);
  x_free_if (ecv->sslcertificatefile);
  x_free_if (ecv->sslcertificatekeyfile);
  x_free_if (ecv->sslcacertificatefile);
  x_free_if (ecv->ssldhparamfile);
  x_free_if (ecv->sslciphersuite);
  eh_config_dir_delete (ecv->default_dir);
  eh_strhash_delete (ecv->dir_ht);
  if (ecv->accesslog_filename_per_worker) {
    if (eh_filespool_close (ecv->econf_global_ref->accesslog_spool,
			    ecv->accesslog_filename_per_worker)) {
      eh_log (EH_LOG_FATAL, "failed to close %s",
	      ecv->accesslog_filename_per_worker);
    }
    x_free (ecv->accesslog_filename_per_worker);
  }
  if (ecv->documentroot_fullpath)
    x_free (ecv->documentroot_fullpath);
  if (ecv->dir_arr) {
    int i;
    for (i = 0; i < ecv->num_dir; i++) {
      x_free (ecv->dir_arr[i].dirname);
    }
    x_free (ecv->dir_arr);
  }
  if (ecv->alias_arr) {
    int i;
    for (i = 0; i < ecv->num_alias; i++) {
      x_free (ecv->alias_arr[i].src);
      x_free (ecv->alias_arr[i].dest);
      x_free_if (ecv->alias_arr[i].rhfunc_name);
    }
    x_free (ecv->alias_arr);
  }
  eh_strhash_delete (ecv->nolog_ht);
  x_free (ecv);
}

static void
eh_config_on_delete_vhost (eh_strhash_entry_t *ent)
{
  eh_config_vhost_t *ecv = (eh_config_vhost_t *)ent->val;
  x_free (ent->key);
  eh_config_vhost_delete (ecv);
}

static void
eh_config_on_delete_nolog_hentry (eh_strhash_entry_t *ent)
{
  x_free (ent->key);
}

eh_config_vhost_t *
eh_config_vhost_new (eh_config_global_t *ecg)
{
  eh_config_vhost_t *ecv;
  ecv = (eh_config_vhost_t *)x_malloc (sizeof (*ecv));
  memset (ecv, 0, sizeof (*ecv));
  ecv->default_dir = eh_config_dir_new (ecg);
  ecv->dir_ht = eh_strhash_new (EH_HASH_SIZE, eh_config_on_delete_dir);
  ecv->timeout = 300;
  ecv->nolog_ht = eh_strhash_new (EH_HASH_SIZE,
				  eh_config_on_delete_nolog_hentry);
  return ecv;
}

void
eh_config_global_delete (eh_config_global_t *econf)
{
  x_free_if (econf->serverroot);
  x_free_if (econf->user);
  x_free_if (econf->group);
  x_free_if (econf->bindaddress);
  x_free_if (econf->changeroot);
  x_free (econf->error_string);
  eh_config_vhost_delete (econf->default_vhost);
  eh_strhash_delete (econf->vhost_ht);
  eh_filespool_delete (econf->accesslog_spool);
  x_free_if (econf->change_root_fullpath);
  x_free (econf);
}

static void
eh_config_dirarr_add (eh_strhash_entry_t *hent, void *data)
{
  eh_config_dirarr_ent_t **entpp = (eh_config_dirarr_ent_t **)data;
  eh_config_dirarr_ent_t *entp = (*entpp)++;
  /* set chrooted dirname later */
  entp->dirname = x_strdup (hent->key);
  entp->nearest_prefix = -1;
  entp->dir = (eh_config_dir_t *)hent->val;
}

static int
eh_config_dirarr_ent_compare (const void *a, const void *b)
{
  const eh_config_dirarr_ent_t *ap, *bp;
  ap = (const eh_config_dirarr_ent_t *)a;
  bp = (const eh_config_dirarr_ent_t *)b;
  return strcmp (ap->dirname, bp->dirname);
}

static int
eh_config_aliasarr_ent_compare (const void *a, const void *b)
{
  const eh_config_aliasarr_ent_t *ap, *bp;
  ap = (const eh_config_aliasarr_ent_t *)a;
  bp = (const eh_config_aliasarr_ent_t *)b;
  return strcmp (ap->src, bp->src);
}

#if 0
static void
eh_config_vhost_dir_set_allowoverride (eh_config_dirarr_ent_t *ent)
{
  const char *const *strarr;
  eh_config_dir_t *dir = ent->dir;
  dir->allowoverride = 0;
  strarr = eh_strhash_find (dir->strarr_ht, "AllowOverride", NULL);
  while (strarr && *strarr) {
    const char *p, *q;
    for (p = *strarr, p += strspn (p, " \t"), q = p + strcspn (p, " \t");
	 *p; p = q + strspn (q, " \t"), q = p + strcspn (p, " \t")) {
      int len = q - p;
      if (len == 10 && strncmp (p, "AuthConfig", 10) == 0) {
	dir->allowoverride |= EH_ALLOWOVERRIDE_AUTHCONFIG;
      } else if (len == 7 && strncmp (p, "Options", 7) == 0) {
	dir->allowoverride |= EH_ALLOWOVERRIDE_OPTIONS;
      }
    }
    strarr++;
  }
}
#endif

static void
eh_config_on_delete_pwfilehash_entry (eh_strhash_entry_t *ent)
{
  x_free (ent->key);
  x_free ((char *)ent->val);
}

static eh_strhash_t *
eh_config_create_passwd_ht (const char *pwfilename, eh_config_global_t *ecg)
{
  FILE *fp;
  char *line;
  eh_strhash_t *ht;
  fp = fopen (pwfilename, "r");
  if (fp == NULL) {
    x_append_printf (&ecg->error_string, "failed to open password file '%s'\n",
		     pwfilename);
    ecg->fatal_error = 1;
    return NULL;
  }
  ht = eh_strhash_new (EH_HASH_SIZE, eh_config_on_delete_pwfilehash_entry);
  while ((line = eh_file_get_line (fp)) != NULL) {
    char *uname, *pw, *p, *q;
    eh_strhash_entry_t *ent;
    p = line + strspn (line, " \t");
    if (p[0] == '#')
      goto cont;
    q = strchr (p, ':');
    if (q == NULL)
      goto cont;
    *q++ = '\0';
    uname = p;
    pw = q;
    q += strcspn (q, " \t:");
    *q = '\0';
    if (pw[0] == '\0')
      goto cont;
    eh_debug ("uname = %s, pw = %s", uname, pw);
    uname = x_strdup (uname);
    ent = eh_strhash_find_create (ht, uname);
    if (ent->key == uname) {
      /* new entry */
      ent->val = x_strdup (pw);
      eh_debug ("new entry: %s, %s", uname, pw);
    } else {
      x_free (uname);
    }
    
  cont:
    x_free (line);
  }
  fclose (fp);
  return ht;
}

static void
eh_config_add_req_user (eh_strhash_entry_t *ent, void *data)
{
  eh_config_limit_t *ecl = (eh_config_limit_t *)data;
  char *key;
  eh_strhash_entry_t *reqht_ent;
  key = x_strdup (ent->key);
  reqht_ent = eh_strhash_find_create (ecl->passwd_ht, key);
  if (reqht_ent->key != key) {
    /* already exists */
    x_free (key);
    return;
  }
  reqht_ent->val = x_strdup ((char *)ent->val);
}

static void
eh_config_limit_prepare (eh_config_dir_t *dir,
			 eh_config_filesmatch_t *ecf,
			 eh_config_limit_t *ecl,
			 const char *pwfilename,
			 eh_strhash_t *pw_ht)
{
  eh_config_global_t *ecg;
  char **strarr;
  ecg = dir->econf_global_ref;
  strarr = ecl->require;
  if (strarr == NULL || strarr[0] == NULL)
    return;
  ecl->auth_required = 1;
  if (pw_ht == NULL)
    return;
  if (strcmp (strarr[0], "valid-user") == 0 && strarr[1] == NULL) {
    eh_strhash_foreach (pw_ht, eh_config_add_req_user, ecl);
  } else if (strcmp (strarr[0], "user") == 0) {
    char **strp;
    for (strp = strarr + 1; *strp; strp++) {
      char *uname;
      const char *pwstr;
      uname = x_strdup (*strp);
      pwstr = (const char *)eh_strhash_find (pw_ht, uname, NULL);
      if (pwstr) {
	eh_strhash_entry_t *ent;
	eh_debug ("req user: %s passwd: %s", uname, pwstr);
	ent = eh_strhash_find_create (ecl->passwd_ht, uname);
	if (ent->key == uname) {
	  /* new entry */
	  ent->val = x_strdup (pwstr);
	} else {
	  x_free (uname);
	}
      } else {
	x_append_printf (&ecg->error_string,
			 "user '%s' not found in '%s'\n",
			 uname, pwfilename);
	ecg->fatal_error = 1;
	x_free (uname);
      }
    }
  } else {
    x_append_printf (&ecg->error_string,
		     "'Require %s' unrecognized", strarr[0]);
    ecg->fatal_error = 1;
    return;
  }
}

static void
eh_config_filesmatch_prepare (eh_config_dir_t *dir,
			      eh_config_filesmatch_t *ecf,
			      const char *pwfilename,
			      eh_strhash_t *pw_ht)
{
  int i;
  for (i = 0; i < ecf->num_limit; i++) {
    eh_config_limit_prepare (dir, ecf, ecf->limit_arr[i], pwfilename,
			     pw_ht);
  }
  eh_config_limit_prepare (dir, ecf, ecf->default_limit, pwfilename,
			   pw_ht);
}

static void
eh_config_addhandler_hent_prepare (eh_strhash_entry_t *ent, void *data)
{
  eh_addhandler_t *ah = (eh_addhandler_t *)ent->val;
  eh_config_global_t *ecg = (eh_config_global_t *)data;
  eh_rhandler_new_func_t *rhfunc = NULL;
  void *rhfuncdata = NULL;
  int nofilesystem = 0;
  if (!ah->is_handler)
    return;
  rhfunc = eh_get_rhandler (ah->name, &rhfuncdata, &nofilesystem);
  if (rhfunc == NULL) {
    x_append_printf (&ecg->error_string,
		     "Request Handler '%s' not found", ah->name);
    ecg->fatal_error = 1;
  } else {
    ah->rhfunc = rhfunc;
    ah->rhfunc_data = rhfuncdata;
    ah->nofilesystem = nofilesystem;
  }
}

static void
eh_config_dir_prepare (eh_config_dir_t *dir, eh_config_vhost_t *vhost)
{
  int i;
  char *authuserfile = NULL;
  eh_config_global_t *ecg;
  eh_strhash_t *pw_ht = NULL;
  eh_debug ("");

  ecg = vhost->econf_global_ref;

  eh_strhash_foreach (dir->addhandler_ht, eh_config_addhandler_hent_prepare,
		      ecg);
  
  authuserfile = eh_config_get_fullpath (ecg, dir->authuserfile, NULL);
  if (authuserfile) {
    pw_ht = eh_config_create_passwd_ht (authuserfile, ecg);
  }
  for (i = 0; i < dir->num_filesmatch; i++) {
    eh_config_filesmatch_prepare (dir, dir->filesmatch_arr[i],
				  authuserfile, pw_ht);
  }
  eh_config_filesmatch_prepare (dir, dir->default_filesmatch,
				authuserfile, pw_ht);
  if (pw_ht)
    eh_strhash_delete (pw_ht);
  if (authuserfile)
    x_free (authuserfile);
}

static void
eh_config_dir_hent_prepare (eh_strhash_entry_t *ent, void *data)
{
  eh_config_dir_prepare ((eh_config_dir_t *)ent->val,
			 (eh_config_vhost_t *)data);
}

static int
eh_config_path_apply_chroot (char **strp, eh_config_global_t *ecg)
{
  char *str;
  assert (*strp);
  str = eh_config_get_fullpath (ecg, *strp, NULL);
  if (ecg->change_root_fullpath) {
    char *tstr;
    size_t len;
    if (!eh_substr (ecg->change_root_fullpath, str)) {
      x_free (*strp);
      *strp = str;
      return 1;
    }
    len = strlen (ecg->change_root_fullpath);
    assert (len > 0);
    if (str[len] == '\0') {
      tstr = x_strdup ("/");
    } else if (str[len] != '/') {
      x_free (*strp);
      *strp = str;
      return 1;
    } else {
      tstr = x_strdup (str + len);
    }
    x_free (str);
    str = tstr;
  }
  x_free (*strp);
  *strp = str;
  return 0;
}

static void
eh_config_vhost_prepare (eh_config_vhost_t *vhost, const char *hostname,
			 eh_config_global_t *ecg)
{
  /* this function is called after config file is loaded */
  int n, i;
  eh_config_dirarr_ent_t *entarr, *entp;
  struct sockaddr_in servaddr;
  struct hostent *hp;

  /* set econf_global_ref */
  vhost->econf_global_ref = ecg;

  /* resolve hostname */
  memset (&servaddr, 0, sizeof (servaddr));
  if (hostname == NULL || strcmp (hostname, "*") == 0) {
    servaddr.sin_addr.s_addr = htonl (INADDR_ANY);
  } else {
    if ((hp = gethostbyname (hostname)) == NULL) {
      x_append_printf (&ecg->error_string, "gethostbyname failed: %s: %s\n",
		       hostname, hstrerror (h_errno));
      ecg->fatal_error = 1;
    }
    memcpy (&servaddr.sin_addr, hp->h_addr, hp->h_length);
  }
  vhost->servaddr = servaddr;

  /* get full pathname of documentroot */
  vhost->documentroot_fullpath =
    eh_config_get_fullpath (ecg, vhost->documentroot, ESEHTTPD_DOCUMENTROOT);
  eh_debug ("documentroot_fullpath = %s", vhost->documentroot_fullpath);
  if (eh_config_path_apply_chroot (&vhost->documentroot_fullpath, ecg)) {
    x_append_printf (&ecg->error_string,
		     "document root '%s' is not a subdirectory of "
		     "ChangeRoot directory '%s'\n",
		     vhost->documentroot_fullpath, ecg->change_root_fullpath);
    ecg->fatal_error = 1;
  }
  
  /* prepare dirarr */
  n = vhost->dir_ht->num_ent_total;
  eh_debug ("dir_ht size: %d", n);
  if (n) {
    entarr = (eh_config_dirarr_ent_t *)x_malloc (n * sizeof (*entarr));
    entp = entarr;
    eh_strhash_foreach (vhost->dir_ht, eh_config_dirarr_add, &entp);
    assert (entarr + n == entp);
    for (i = 0; i < n; i++) {
      assert (entarr[i].dirname);
      if (eh_config_path_apply_chroot (&entarr[i].dirname, ecg)) {
	x_append_printf (&ecg->error_string,
			 "<Directory \"%s\"> is not a subdirectory of "
			 "ChangeRoot directory '%s'\n",
			 entarr[i].dirname, ecg->change_root_fullpath);
	ecg->fatal_error = 1;
      }
    }
    qsort (entarr, n, sizeof (*entarr), eh_config_dirarr_ent_compare);
    for (i = 0; i < n; i++) {
      int j;
      for (j = i - 1; j >= 0; j--) {
	if (eh_substr (entarr[j].dirname, entarr[i].dirname))
	  break;
      }
      entarr[i].nearest_prefix = j;
    }
    assert (vhost->dir_arr == NULL);
    vhost->dir_arr = entarr;
    vhost->num_dir = n;
  } else {
    vhost->dir_arr = NULL;
    vhost->num_dir = 0;
  }

  /* sort aliasarr */
  if (vhost->num_alias > 0) {
    for (i = 0; i < vhost->num_alias; i++) {
      if (eh_config_path_apply_chroot (&vhost->alias_arr[i].dest, ecg)) {
	x_append_printf (&ecg->error_string,
			 "Alias destination '%s' is not a subdirectory of "
			 "ChangeRoot directory '%s'\n",
			 vhost->alias_arr[i].dest, ecg->change_root_fullpath);
	ecg->fatal_error = 1;
      }
    }
    qsort (vhost->alias_arr, vhost->num_alias,
	   sizeof (eh_config_aliasarr_ent_t), eh_config_aliasarr_ent_compare);
  }

  /* resolve rhfuncs */
  for (i = 0; i < vhost->num_alias; i++) {
    eh_config_aliasarr_ent_t *ent;
    ent = vhost->alias_arr + i;
    if (ent->rhfunc_name) {
      eh_rhandler_new_func_t *rhfunc = NULL;
      void *rhfuncdata = NULL;
      int nofilesystem = 0;
      rhfunc = eh_get_rhandler (ent->rhfunc_name, &rhfuncdata,
				&nofilesystem);
      if (rhfunc == NULL) {
	x_append_printf (&ecg->error_string,
			 "Request Handler '%s' not found\n", ent->rhfunc_name);
	ecg->fatal_error = 1;
      } else {
	ent->rhfunc = rhfunc;
	ent->rhfunc_data = rhfuncdata;
	ent->nofilesystem = nofilesystem;
      }
    }
  }
  
  eh_strhash_foreach (vhost->dir_ht, eh_config_dir_hent_prepare, vhost);
  eh_config_dir_prepare (vhost->default_dir, vhost);
}

static void
eh_config_vhost_hent_prepare (eh_strhash_entry_t *ent, void *data)
{
  eh_config_vhost_prepare ((eh_config_vhost_t *)ent->val,
			   (const char *)ent->key,
			   (eh_config_global_t *)data);
}

eh_config_global_t *
eh_config_global_new (const char *path)
{
  eh_config_global_t *econf;
  econf = (eh_config_global_t *)x_malloc (sizeof (*econf));
  memset (econf, 0, sizeof (*econf));
  econf->error_string = x_strdup ("");
  econf->default_vhost = eh_config_vhost_new (econf);
  econf->vhost_ht = eh_strhash_new (EH_HASH_SIZE, eh_config_on_delete_vhost);
  econf->accesslog_spool = eh_filespool_new ();
  econf->num_vhosts = 0;
  econf->startservers = 1;
  econf->filecachesize = 300;
  econf->sendbuffersize = 16384;
  econf->recvbuffersize = 16384;
  econf->listenbacklog = 128;
  econf->requestheaderlimit = 8192;
  econf->requestbodylimit = 16777216;
  econf->lingeringcloselimit = 65536;
  econf->multipleaccept = 0;
  
  eh_config_global_read (econf, path);
  return econf;
}

void
eh_config_global_log_errors (eh_config_global_t *econf)
{
  const char *p, *q;
  eh_for_p_in_str (econf->error_string, "\n", p, q) {
    char *s;
    s = x_strndup (p, q - p);
    eh_log_funcname (EH_LOG_FATAL, NULL, "%s", s);
    x_free (s);
  }
}

int
eh_config_global_prepare_tables (eh_config_global_t *econf)
{
  if (econf->fatal_error)
    return 1;
  if (econf->changeroot) {
    econf->change_root_fullpath = eh_config_get_fullpath (econf,
							  econf->changeroot,
							  NULL);
  }
  eh_strhash_foreach (econf->vhost_ht, eh_config_vhost_hent_prepare, econf);
  eh_config_vhost_prepare (econf->default_vhost, econf->bindaddress, econf);
  return econf->fatal_error;
}


static char *
eh_config_get_accesslog_filename_per_worker (int worker_id,
					     const char *basename)
{
    char *filename;
    filename = x_strdup (basename);
    if (worker_id >= 0)
      x_append_printf (&filename, "_worker%d", worker_id);
    return filename;
}

static void
eh_config_vhost_prepare_accesslog (eh_config_vhost_t *vhost, long worker_id)
{
  eh_config_global_t *ecg;
  ecg = vhost->econf_global_ref;
  if (vhost->customlog) {
    char *fullpath;
    char *filename;
    FILE *fp;
    fullpath = eh_config_get_fullpath (ecg, vhost->customlog, NULL);
    filename = eh_config_get_accesslog_filename_per_worker (worker_id,
							    fullpath);
    x_free (fullpath);
    eh_debug ("customlog per worker: %s", filename);
    fp = eh_filespool_find (ecg->accesslog_spool, filename, 1);
    if (fp == NULL) {
      eh_log_perror (EH_LOG_WARNING, filename);
      x_free (filename);
      return;
    } else {
      /* do nothing. the file is cached in ecg->accesslog_spool. */
      x_free (filename);
      return;
    }
  }
}

static void
eh_config_vhost_hent_prepare_accesslog (eh_strhash_entry_t *ent, void *data)
{
  long *worker_id_p = (long *)data;
  eh_config_vhost_prepare_accesslog ((eh_config_vhost_t *)ent->val,
				     *worker_id_p);
}

void
eh_config_global_prepare_accesslog (eh_config_global_t *ecg,
				    long worker_id)
{
  eh_strhash_foreach (ecg->vhost_ht, eh_config_vhost_hent_prepare_accesslog,
		      (void *)&worker_id);
  eh_config_vhost_prepare_accesslog (ecg->default_vhost, worker_id);
}

static void
eh_config_vhost_set_worker_id (eh_config_vhost_t *vhost, long worker_id,
			       long num_workers)
{
  eh_config_global_t *ecg;
  int i;
  ecg = vhost->econf_global_ref;
  if (!vhost->customlog)
    return;
  for (i = -1; i < num_workers; i++) {
    char *fullpath;
    char *filename;
    FILE *fp;
    fullpath = eh_config_get_fullpath (ecg, vhost->customlog, NULL);
    filename = eh_config_get_accesslog_filename_per_worker (i,
							    fullpath);
    x_free (fullpath);
    fp = eh_filespool_find (ecg->accesslog_spool, filename, 0);
    if (fp == NULL) {
      x_free (filename);
      continue;
    }
    if (i == worker_id) {
      /* we've got the cached file */
      fseek (fp, 0, SEEK_END);
      vhost->access_log = fp;
      vhost->accesslog_filename_per_worker = filename;
      ecg->use_access_log = 1;
    } else {
      x_free (filename);
      eh_filespool_close (ecg->accesslog_spool, filename);
    }
  }
}

static void
eh_config_vhost_hent_set_worker_id (eh_strhash_entry_t *ent, void *data)
{
  long *arg = (long *)data;
  eh_config_vhost_set_worker_id ((eh_config_vhost_t *)ent->val,
				 arg[0], arg[1]);
}

void
eh_config_global_set_worker_id (eh_config_global_t *ecg,
				long worker_id, long num_workers)
{
  long arg[2];
  arg[0] = worker_id;
  arg[1] = num_workers;
  eh_strhash_foreach (ecg->vhost_ht, eh_config_vhost_hent_set_worker_id,
		      (void *)arg);
  eh_config_vhost_set_worker_id (ecg->default_vhost, worker_id, num_workers);
}

void
eh_config_global_flush_accesslog (eh_config_global_t *ecg)
{
  if (!ecg->use_access_log)
    return;
  eh_filespool_flush (ecg->accesslog_spool);
}

void
eh_config_setval (eh_strhash_t *strarr_ht, const char *key, const char *val)
{
  char *key_cp;
  int n;
  char **p;
  eh_strhash_entry_t *ent;
  /* eh_debug ("%s: %s", key, val); */
  key_cp = x_strdup (key);
  ent = eh_strhash_find_create (strarr_ht, key_cp);
  if (ent->key != key_cp) {
    x_free (key_cp);
  }
  for (n = 0, p = (char **)ent->val; p && p[n]; n++);
  ent->val = x_realloc (ent->val, (n + 2) * sizeof (char *));
  p = (char **)ent->val;
  p[n] = x_strdup (val);
  p[n + 1] = NULL;
}

eh_config_vhost_t *
eh_config_global_find_create_vhost (eh_config_global_t *ecg,
				    const char *hostname)
{
  eh_config_vhost_t *ecv;
  eh_strhash_entry_t *ent;
  char *name;
  name = x_strdup (hostname);
  ent = eh_strhash_find_create (ecg->vhost_ht, name);
  if (ent->key != name) {
    x_free (name);
    ecv = (eh_config_vhost_t *)ent->val;
  } else {
    ecv = eh_config_vhost_new (ecg);
    ent->val = ecv;
    ecg->num_vhosts++;
  }
  return ecv;
}

eh_config_vhost_t *
eh_config_global_find_vhost (eh_config_global_t *ecg, const char *hostname)
{
  if (ecg->num_vhosts == 0)
    return NULL;
  return (eh_config_vhost_t *)eh_strhash_find (ecg->vhost_ht, hostname, NULL);
}

eh_config_dir_t *
eh_config_vhost_find_create_dir (eh_config_vhost_t *ecv, const char *dirname)
{
  eh_config_dir_t *ecd;
  eh_strhash_entry_t *ent;
  char *name;
  name = x_strdup (dirname);
  ent = eh_strhash_find_create (ecv->dir_ht, name);
  if (ent->key != name) {
    x_free (name);
    ecd = (eh_config_dir_t *)ent->val;
  } else {
    ecd = eh_config_dir_new (ecv->econf_global_ref);
    ent->val = ecd;
  }
  return ecd;
}

static int
eh_config_vhost_find_dirarr_ent (eh_config_dirarr_ent_t *arr,
				 int n, const char *dirname)
{
  int low, high;
  
  if (n < 1 || strcmp (arr[0].dirname, dirname) > 0)
    return -1;

  /* The following loop does a binary search. In the loop, low becomes
     the greatest number s.t. arr[low] <= dirname, and high the
     smallest number s.t. dirname < arr[high]. The array arr[] must be
     sorted by '<=' already. */
  low = 0;
  high = n;
  while (high - low > 1) {
    int i = low + (high - low) / 2;
    if (strcmp (arr[i].dirname, dirname) > 0)
      high = i;
    else
      low = i;
  }

  /* Now low is the greatest number s.t. arr[low] <= dirname. We want
     to find the greatest number low' s.t. arr[low'] is a substring of
     dirname. We'll show that arr[low'] is always a substring of
     arr[low]. Let len be strlen (arr[low']). Because arr[low'] is a
     prefix of dirname, strncmp (arr[low'], dirname, len) == 0 holds.
     Moreover, arr[low'] <= arr[low] <= dirname implies that
     strncmp (arr[low'], arr[low], len) == 0 also holds, i.e.,
     arr[low'] is a substring of arr[low]. */
  while (low >= 0) {
    if (eh_substr (arr[low].dirname, dirname))
      return low;
    low = arr[low].nearest_prefix;
  }
  
  return -1;
}

eh_config_dir_t *
eh_config_vhost_get_dir (eh_config_vhost_t *ecv, const char *dirname)
{
  int num = ecv->num_dir;
  int ind;
  eh_config_dirarr_ent_t *arr = ecv->dir_arr;
  ind = eh_config_vhost_find_dirarr_ent (arr, num, dirname);
  if (ind >= 0) {
    eh_debug ("get_dir %s -> %s", dirname, arr[ind].dirname);
    return arr[ind].dir;
  } 
  return ecv->default_dir;
}

static int
eh_config_vhost_find_aliasarr_ent (eh_config_aliasarr_ent_t *arr,
				   int n, const char *url)
{
  /* almost the same as eh_config_vhost_find_dirarr_ent () */
  /* arr[] must be sorted already */
  int low, high;
#ifdef DEBUG
  int i;
  for (i = 0; i < n; i++)
    eh_debug ("aliasarr[%d].src = %s", i, arr[i].src);
#endif
  eh_debug ("url = %s", url);
  if (n < 1 || strcmp (arr[0].src, url) > 0) {
    eh_debug ("return -1 (1st)");
    return -1;
  }
  low = 0; /* last entry which is smaller than (or equal to) url */
  high = n; /* first entry which is greater than url */
  while (high - low > 1) {
    int i = low + (high - low) / 2;
    if (strcmp (arr[i].src, url) > 0)
      high = i;
    else
      low = i;
  }
  do {
    if (eh_substr (arr[low].src, url)) {
      eh_debug ("return %d", low);
      return low;
    }
    low--;
  } while (low >= 0);
  eh_debug ("return -1 (2nd)");
  return -1;
}

int
eh_config_vhost_check_alias (eh_config_vhost_t *ecv,
			     const char *url, const char **alias_dest_r,
			     eh_rhandler_new_func_t **rhfunc_r,
			     void **rhfunc_data_r,
			     int *nofilesystem_r)
{
  int num = ecv->num_alias;
  int ind;
  eh_config_aliasarr_ent_t *arr = ecv->alias_arr;
  ind = eh_config_vhost_find_aliasarr_ent (arr, num, url);
  if (ind >= 0 && eh_substr (arr[ind].src, url)) {
    *alias_dest_r = arr[ind].dest;
    *rhfunc_r = arr[ind].rhfunc;
    *rhfunc_data_r = arr[ind].rhfunc_data;
    *nofilesystem_r = arr[ind].nofilesystem;
    return strlen (arr[ind].src);
  }
  return 0; 
}

static eh_method_t
eh_config_get_method_mask (const char *str)
{
  const char *p, *q;
  eh_method_t mask, m;
  (int)mask = 0;
  eh_for_p_in_str (str, " \t", p, q) {
    char *s;
    s = x_strndup (p, q - p);
    m = eh_config_get_method (s);
    x_free (s);
    if (m == eh_method_unknown) {
      return m;
    }
    (int)mask |= m;
  }
  return mask;
}

eh_config_limit_t *
eh_config_filesmatch_create_limit (eh_config_filesmatch_t *ecf,
				   const char *str)
{
  int n;
  eh_method_t method_mask;
  eh_config_limit_t *ecl;
  method_mask = eh_config_get_method_mask (str);
  if (method_mask == eh_method_unknown)
    return NULL;
  /* HEAD is masked if and only if GET is masked */
  if ((method_mask & eh_method_get) != 0) {
    (int)method_mask |= eh_method_head;
  } else {
    (int)method_mask &= ~eh_method_head;
  }
  eh_debug ("creating new <limit>: method_mask = %x", method_mask);
  ecl = eh_config_limit_new (method_mask);
  n = ecf->num_limit++;
  ecf->limit_arr = (eh_config_limit_t **)
    x_realloc (ecf->limit_arr, ecf->num_limit * sizeof (eh_config_limit_t *));
  ecf->limit_arr[n] = ecl;
  return ecl;
}

eh_config_filesmatch_t *
eh_config_dir_create_filesmatch (eh_config_dir_t *ecd, const char *pattern)
{
  int n;
  eh_config_filesmatch_t *ecf;
  ecf = eh_config_filesmatch_new (pattern, ecd->econf_global_ref);
  if (ecf == NULL) {
    return NULL;
  }
  n = ecd->num_filesmatch++;
  ecd->filesmatch_arr = (eh_config_filesmatch_t **)
    x_realloc (ecd->filesmatch_arr,
	       ecd->num_filesmatch * sizeof (eh_config_filesmatch_t *));
  ecd->filesmatch_arr[n] = ecf;
  return ecf;
}

static const struct {
  eh_method_t method;
  const char *str;
} eh_method_table[] = {
  { eh_method_get,      "GET" },
  { eh_method_head,     "HEAD" },
  { eh_method_post,     "POST" },
  { eh_method_put,      "PUT" },
  { eh_method_delete,   "DELETE" },
  { eh_method_options,  "OPTIONS" },
  { eh_method_trace,    "TRACE" },
  { eh_method_connect,  "CONNECT" },
};

eh_method_t
eh_config_get_method (const char *str)
{
  int i, n;
  n = sizeof (eh_method_table) / sizeof (eh_method_table[0]);
  for (i = 0; i < n; i++) {
    if (strcmp (str, eh_method_table[i].str) == 0)
      return eh_method_table[i].method;
  }
  return eh_method_unknown;
}

char *
eh_config_get_method_mask_str (eh_method_t mask)
{
  char *s;
  int i, n;
  s = x_strdup ("");
  n = sizeof (eh_method_table) / sizeof (eh_method_table[0]);
  for (i = 0; i < n; i++) {
    if ((eh_method_table[i].method & mask) != 0) {
      if (eh_method_table[i].method != eh_method_head)
	x_append_printf (&s, " %s", eh_method_table[i].str);
    }
  }
  if (s[0] == '\0') {
    x_free (s);
    return NULL;
  }
  return s;
}

eh_config_limit_t *
eh_config_dir_get_limit (eh_config_dir_t *ecd, const char *filename,
			 eh_method_t method)
{
  eh_config_filesmatch_t *ecf_matched;
  eh_config_limit_t *ecl_matched;
  ecf_matched = ecd->default_filesmatch;
  if (ecd->num_filesmatch) {
    int i;
    const char *bname;
    bname = strrchr (filename, '/');
    if (bname)
      bname++;
    else
      bname = filename;
    eh_debug ("bname: %s", bname);
    for (i = 0; i < ecd->num_filesmatch; i++) {
      eh_config_filesmatch_t *ecf;
      ecf = ecd->filesmatch_arr[i];
      eh_debug ("comparing pattern: %s", ecf->pattern);
      if (regexec (&ecf->re, bname, 0, NULL, 0) == 0) {
	ecf_matched = ecf;
	break;
      }
    }
  }
  eh_debug ("matched <filesmatch>: %p, num_limit=%d", ecf_matched,
	    ecf_matched->num_limit);
  ecl_matched = ecf_matched->default_limit;
  if (ecf_matched->num_limit) {
    int i;
    for (i = 0; i < ecf_matched->num_limit; i++) {
      eh_config_limit_t *ecl;
      ecl = ecf_matched->limit_arr[i];
      eh_debug ("<limit>: %x %x", ecl->method_mask, method);
      if ((ecl->method_mask & method) != 0) {
	eh_debug ("matched <limit>: %p, auth=%d", ecl, ecl->auth_required);
	ecl_matched = ecl;
      }
    }
  }
  return ecl_matched;
}

static void
eh_remove_duplicated_slashes (char *str)
{
  const char *rp;
  char *wp;
  for (rp = wp = str; *rp; rp++) {
    if (rp[0] == '/' && rp[1] == '/') {
      /* skip */
    } else {
      *wp++ = *rp;
    }
  }
  *wp = '\0';
  if (wp > str + 1 && wp[-1] == '/')
    wp[-1] = '\0';
}

char *
eh_config_get_fullpath (eh_config_global_t *ecg, const char *path,
			const char *default_path)
{
  const char *p;
  char *str;
  if (path == NULL)
    path = default_path;
  if (path == NULL)
    return NULL;
  if (path[0] == '/') {
    str = x_strdup (path);
  } else {
    p = ecg->serverroot;
    if (p == NULL)
      p = ESEHTTPD_SERVERROOT;
    str = x_strdup_printf ("/%s/%s", p, path);
  }
  eh_remove_duplicated_slashes (str);
  return str;
}
