/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"


typedef struct {
  eh_config_vhost_t *vhost;
  eh_config_dir_t *dir;
  eh_config_filesmatch_t *filesmatch;
  eh_config_limit_t *limit;
  int unknown_context_count;
  int linenum;
  int errline;
} eh_config_context_t;

typedef enum {
  eh_s_global,
  eh_s_vhost,
  eh_s_dir,
  eh_s_fmatch,
  eh_s_limit,
} eh_config_section_t;

struct eh_config_directive_s;
typedef struct eh_config_directive_s {
  const char *name;
  int num_args;
  int (*read_cb)(void *confdata,
		 const char *args[],
		 int num_args,
		 const struct eh_config_directive_s *dinfo);
  char *(*write_cb)(void *confdata,
		    const struct eh_config_directive_s *dinfo);
  eh_config_section_t section;
  unsigned int offset;
  const void *data;
} eh_config_directive_t;

static int
eh_rcb_onoff (void *confdata, const char *args[], int num_args,
	      const eh_config_directive_t *dinfo)
{
  int *ptr, val = 0;
  ptr = (int *)(((char *)confdata) + dinfo->offset);
  assert (args[0]);
  if (strcasecmp (args[0], "on") == 0)
    val = 1;
  else if (strcasecmp (args[0], "off") == 0)
    val = 0;
  else
    return 1;
  *ptr = val;
  return 0;
}

static char *
eh_wcb_onoff (void *confdata, const struct eh_config_directive_s *dinfo)
{
  int *ptr;
  ptr = (int *)(((char *)confdata) + dinfo->offset);
  if (*ptr)
    return x_strdup_printf ("%s On\n", dinfo->name);
  else
    return x_strdup_printf ("%s Off\n", dinfo->name);
}

static int
eh_rcb_range (void *confdata, const char *args[], int num_args,
	      const eh_config_directive_t *dinfo)
{
  int *ptr;
  int val = 0, minval = 0, maxval = 0;
  ptr = (int *)(((char *)confdata) + dinfo->offset);
  sscanf ((const char *)dinfo->data, "%d %d", &minval, &maxval);
  assert (args[0]);
  val = atoi (args[0]);
  if (val < minval || val > maxval)
    return 1;
  *ptr = val;
  return 0;
}

static char *
eh_wcb_range (void *confdata, const struct eh_config_directive_s *dinfo)
{
  int *ptr;
  ptr = (int *)(((char *)confdata) + dinfo->offset);
  return x_strdup_printf ("%s %d\n", dinfo->name, *ptr);
}

static int
eh_rcb_idstr (void *confdata, const char *args[], int num_args,
	      const eh_config_directive_t *dinfo)
{
  char **ptr;
  ptr = (char **)(((char *)confdata) + dinfo->offset);
  if (*ptr)
    x_free (*ptr);
  assert (args[0]);
  if (!eh_is_valid_id_string (args[0]))
    return 1;
  *ptr = x_strdup (args[0]);
  return 0;
}

static char *
eh_wcb_idstr (void *confdata, const struct eh_config_directive_s *dinfo)
{
  char **ptr;
  ptr = (char **)(((char *)confdata) + dinfo->offset);
  if (*ptr) {
    return x_strdup_printf ("%s %s\n", dinfo->name, *ptr);
  } else {
    return x_strdup_printf ("# %s is not set\n", dinfo->name);
  }
}

static int
eh_rcb_qstr (void *confdata, const char *args[], int num_args,
	     const eh_config_directive_t *dinfo)
{
  char *str, **ptr;
  ptr = (char **)(((char *)confdata) + dinfo->offset);
  if (*ptr)
    x_free (*ptr);
  assert (args[0]);
  str = x_strdup (args[0]);
  eh_unescape_doublequotes (str);
  if (dinfo->data && strcmp ((const char *)dinfo->data, "/") == 0) {
    /* append '/' if necessary */
    int len;
    len = strlen (str);
    if (len > 0 && str[len - 1] != '/') {
      x_str_append (&str, "/");
    }
  }
  *ptr = str;
  return 0;
}

static char *
eh_wcb_qstr (void *confdata, const struct eh_config_directive_s *dinfo)
{
  char **ptr;
  ptr = (char **)(((char *)confdata) + dinfo->offset);
  if (*ptr) {
    char *s, *t;
    s = eh_strdup_escape_doublequotes (*ptr);
    t = x_strdup_printf ("%s \"%s\"\n", dinfo->name, s);
    x_free (s);
    return t;
  } else {
    return x_strdup_printf ("# %s is not set\n", dinfo->name);
  }
}

static int
eh_rcb_strarr (void *confdata, const char *args[], int num_args,
	       const eh_config_directive_t *dinfo)
{
  char ***ptr;
  ptr = (char ***)(((char *)confdata) + dinfo->offset);
  if (*ptr)
    eh_strarr_free (*ptr);
  *ptr = eh_strarr_copy (args);
  return 0;
}

static char *
eh_wcb_strarr (void *confdata, const struct eh_config_directive_s *dinfo)
{
  char ***ptr;
  ptr = (char ***)(((char *)confdata) + dinfo->offset);
  if (*ptr) {
    char **p, *r;
    r = x_strdup (dinfo->name);
    for (p = *ptr; *p; p++) {
      x_append_printf (&r, " %s", *p);
    }
    x_str_append (&r, "\n");
    return r;
  } else {
    return x_strdup_printf ("# %s is not set\n", dinfo->name);
  }
}

static int
eh_rcb_customlog (void *confdata, const char *args[], int num_args,
		  const eh_config_directive_t *dinfo)
{
  int combined = 0;
  eh_config_vhost_t *vhost = (eh_config_vhost_t *)confdata;
  if (num_args > 2 || num_args < 1)
    return 1;
  if (num_args == 2) {
    if (strcmp (args[1], "combined") == 0)
      combined = 1;
    else if (strcmp (args[1], "common") == 0)
      combined = 0;
    else
      return 1;
  }
  if (vhost->customlog)
    x_free (vhost->customlog);
  vhost->customlog = x_strdup (args[0]);
  eh_unescape_doublequotes (vhost->customlog);
  vhost->log_combined = combined;
  return 0;
}

static char *
eh_wcb_customlog (void *confdata, const struct eh_config_directive_s *dinfo)
{
  eh_config_vhost_t *vhost = (eh_config_vhost_t *)confdata;
  if (vhost->customlog) {
    char *s, *t;
    s = eh_strdup_escape_doublequotes (vhost->customlog);
    t = x_strdup_printf ("%s \"%s\" %s\n", dinfo->name, s,
			 vhost->log_combined ? "combined" : "common");
    x_free (s);
    return t;
  } else {
    return x_strdup_printf ("# %s is not set\n", dinfo->name);
  }
}

static void
eh_configread_append_alias (eh_config_vhost_t *vhost, const char *src,
			    const char *dest, const char *rhandlername)
{
  eh_config_aliasarr_ent_t *ent;
  int n;
  n = vhost->num_alias++;
  vhost->alias_arr = (eh_config_aliasarr_ent_t *)
    x_realloc (vhost->alias_arr,
	       (vhost->num_alias) * sizeof (eh_config_aliasarr_ent_t));
  ent = vhost->alias_arr + n;
  if (strlen (src) > 0 && src[strlen (src) - 1] != '/')
    ent->src = x_strdup_printf ("%s/", src);
  else
    ent->src = x_strdup (src);
  ent->dest = dest ? x_strdup (dest) : NULL;
  ent->rhfunc_name = rhandlername ? x_strdup (rhandlername) : NULL;
  ent->rhfunc = NULL;
  ent->rhfunc_data = NULL;
  ent->nofilesystem = 0;
  return;
}

static int
eh_rcb_alias (void *confdata, const char *args[], int num_args,
	      const eh_config_directive_t *dinfo)
{
  char *src, *dest;
  eh_config_vhost_t *vhost = (eh_config_vhost_t *)confdata;
  src = x_strdup (args[0]);
  dest = x_strdup (args[1]);
  eh_unescape_doublequotes (src);
  eh_unescape_doublequotes (dest);
  eh_configread_append_alias (vhost, src, dest, NULL);
  x_free (src);
  x_free (dest);
  return 0;
}

static char *
eh_wcb_alias (void *confdata, const struct eh_config_directive_s *dinfo)
{
  eh_config_vhost_t *vhost = (eh_config_vhost_t *)confdata;
  int i;
  char *s = x_strdup ("");
  for (i = 0; i < vhost->num_alias; i++) {
    eh_config_aliasarr_ent_t *ent = vhost->alias_arr + i;
    if (ent->rhfunc_name)
      continue;
    x_append_printf (&s, "Alias \"%s\" \"%s\"\n", ent->src, ent->dest);
  }
  if (s[0] == '\0') {
    x_free (s);
    return x_strdup_printf ("# %s is not set\n", dinfo->name);
  }
  return s;
}

static int
eh_rcb_scriptalias (void *confdata, const char *args[], int num_args,
		    const eh_config_directive_t *dinfo)
{
  char *src, *dest;
  const char *rhfuncname;
  eh_config_vhost_t *vhost = (eh_config_vhost_t *)confdata;
  if (num_args != 2 && num_args != 3)
    return 1;
  rhfuncname = num_args == 3 ? args[2] : "cgi-script";
  src = x_strdup (args[0]);
  dest = x_strdup (args[1]);
  eh_unescape_doublequotes (src);
  eh_unescape_doublequotes (dest);
  eh_configread_append_alias (vhost, src, dest, rhfuncname);
  x_free (src);
  x_free (dest);
  return 0;
}

static char *
eh_wcb_scriptalias (void *confdata, const struct eh_config_directive_s *dinfo)
{
  eh_config_vhost_t *vhost = (eh_config_vhost_t *)confdata;
  int i;
  char *s = x_strdup ("");
  for (i = 0; i < vhost->num_alias; i++) {
    eh_config_aliasarr_ent_t *ent = vhost->alias_arr + i;
    if (ent->rhfunc_name == NULL)
      continue;
    x_append_printf (&s, "ScriptAlias \"%s\" \"%s\" %s\n", ent->src, ent->dest,
		     ent->rhfunc_name);
  }
  if (s[0] == '\0') {
    x_free (s);
    return x_strdup_printf ("# %s is not set\n", dinfo->name);
  }
  return s;
}

static int
eh_rcb_addhandler (void *confdata, const char *args[], int num_args,
		   const eh_config_directive_t *dinfo)
{
  eh_config_dir_t *dir = (eh_config_dir_t *)confdata;
  int i;
  int is_handler;
  if (num_args < 2)
    return 1;
  is_handler = (strcmp ((const char *)dinfo->data, "handler") == 0);
  for (i = 1; i < num_args; i++) {
    const char *p;
    char *suffix;
    eh_strhash_entry_t *ent;
    p = args[i];
    if (p[0] == '.')
      p++;
    suffix = x_strdup (p);
    ent = eh_strhash_find_create (dir->addhandler_ht, suffix);
    if (ent->key != suffix) {
      x_free (suffix);
      eh_addhandler_delete ((eh_addhandler_t *)ent->val);
    }
    ent->val = eh_addhandler_new (is_handler, args[0]);
    eh_debug ("addhandler/type %s %s", (char *)ent->val, args[0]);
  }
  return 0;
}


typedef struct {
  char **strp;
  const struct eh_config_directive_s *dinfo;
} eh_write_addhandler_arg_t;

static void
eh_config_write_addhandler (eh_strhash_entry_t *ent, void *data)
{
  eh_write_addhandler_arg_t *argp = (eh_write_addhandler_arg_t *)data;
  const char *key = (const char *)ent->key;
  eh_addhandler_t *h = (eh_addhandler_t *)ent->val;
  char **strp = argp->strp;
  if (h->is_handler && strcmp (argp->dinfo->name, "AddHandler") == 0)
    x_append_printf (strp, "AddHandler %s %s\n", h->name, key);
  if (!h->is_handler && strcmp (argp->dinfo->name, "AddType") == 0)
    x_append_printf (strp, "AddType %s %s\n", h->name, key);
}

static char *
eh_wcb_addhandler (void *confdata, const struct eh_config_directive_s *dinfo)
{
  eh_config_dir_t *dir = (eh_config_dir_t *)confdata;
  eh_write_addhandler_arg_t arg;
  char *s;
  s = x_strdup ("");
  arg.strp = &s;
  arg.dinfo = dinfo;
  eh_strhash_foreach (dir->addhandler_ht, eh_config_write_addhandler, &arg);
  if (s[0] == '\0') {
    x_free (s);
    return x_strdup_printf ("# %s is not set\n", dinfo->name);
  }
  return s;
}

static int
eh_rcb_verifyclient (void *confdata, const char *args[], int num_args,
		     const eh_config_directive_t *dinfo)
{
  eh_config_dir_t *dir = (eh_config_dir_t *)confdata;
  assert (args[0]);
  if (strcmp (args[0], "require") == 0) {
    dir->ssl_verifyclient = 1;
  } else if (strcmp (args[0], "none") == 0) {
    dir->ssl_verifyclient = 0;
  } else {
    return 1;
  }
  return 0;
}

static char *
eh_wcb_verifyclient (void *confdata, const struct eh_config_directive_s *dinfo)
{
  eh_config_dir_t *dir = (eh_config_dir_t *)confdata;
  if (dir->ssl_verifyclient) {
    return x_strdup_printf ("%s require\n", dinfo->name);
  } else {
    return x_strdup_printf ("# %s is not set\n", dinfo->name);
  }
}

static int
eh_rcb_options (void *confdata, const char *args[], int num_args,
		const eh_config_directive_t *dinfo)
{
  eh_config_dir_t *dir = (eh_config_dir_t *)confdata;
  int i;
  dir->option_nofilecache = 0;
  dir->option_indexes = 0;
  for (i = 0; i < num_args; i++) {
    if (strcmp (args[i], "NoFileCache") == 0)
      dir->option_nofilecache = 1;
    else if (strcmp (args[i], "Indexes") == 0)
      dir->option_indexes = 1;
    else {
      return 1;
    }
  }
  return 0;
}

static char *
eh_wcb_options (void *confdata, const struct eh_config_directive_s *dinfo)
{
  eh_config_dir_t *dir = (eh_config_dir_t *)confdata;
  char *s;
  s = x_strdup (dinfo->name);
  if (dir->option_nofilecache)
    x_str_append (&s, " NoFileCache");
  if (dir->option_indexes)
    x_str_append (&s, " Indexes");
  if (strcmp (s, dinfo->name) == 0) {
    x_free (s);
    return x_strdup_printf ("# %s is not set\n", dinfo->name);
  }
  x_str_append (&s, "\n");
  return s;
}

static int
eh_rcb_nolog (void *confdata, const char *args[], int num_args,
	      const eh_config_directive_t *dinfo)
{
  eh_config_vhost_t *vhost = (eh_config_vhost_t *)confdata;
  int i;
  if (num_args == 0) {
    return 1;
  }
  for (i = 0; i < num_args; i++) {
    char *key;
    eh_strhash_entry_t *ent;
    eh_strhash_remove (vhost->nolog_ht, args[i]);
    key = x_strdup (args[i]);
    ent = eh_strhash_find_create (vhost->nolog_ht, key);
    assert (ent->key == key);
    ent->val = (void *)"yes";
  }
  return 0;
}

static void
eh_config_write_nolog (eh_strhash_entry_t *ent, void *data)
{
  char **strp = (char **)data;
  x_append_printf (strp, " %s", ent->key);
}  

static char *
eh_wcb_nolog (void *confdata, const struct eh_config_directive_s *dinfo)
{
  eh_config_vhost_t *vhost = (eh_config_vhost_t *)confdata;
  char *s;
  s = x_strdup (dinfo->name);
  eh_strhash_foreach (vhost->nolog_ht, eh_config_write_nolog, &s);
  if (strcmp (s, dinfo->name) == 0) {
    x_free (s);
    return x_strdup_printf ("# %s is not set\n", dinfo->name);
  }
  x_str_append (&s, "\n");
  return s;
}

#define EH_CB_qstr eh_rcb_qstr, eh_wcb_qstr
#define EH_CB_idstr eh_rcb_idstr, eh_wcb_idstr
#define EH_CB_range eh_rcb_range, eh_wcb_range
#define EH_CB_onoff eh_rcb_onoff, eh_wcb_onoff
#define EH_CB_strarr eh_rcb_strarr, eh_wcb_strarr
#define EH_CB_customlog eh_rcb_customlog, eh_wcb_customlog
#define EH_CB_alias eh_rcb_alias, eh_wcb_alias
#define EH_CB_scriptalias eh_rcb_scriptalias, eh_wcb_scriptalias
#define EH_CB_verifyclient eh_rcb_verifyclient, eh_wcb_verifyclient
#define EH_CB_nolog eh_rcb_nolog, eh_wcb_nolog
#define EH_CB_addhandler eh_rcb_addhandler, eh_wcb_addhandler
#define EH_CB_options eh_rcb_options, eh_wcb_options

#define EH_S_G(o) eh_s_global, offsetof(eh_config_global_t, o)
#define EH_S_V(o) eh_s_vhost, offsetof(eh_config_vhost_t, o)
#define EH_S_D(o) eh_s_dir, offsetof(eh_config_dir_t, o)
#define EH_S_F(o) eh_s_fmatch, offsetof(eh_config_filesmatch_t, o)
#define EH_S_L(o) eh_s_limit, offsetof(eh_config_limit_t, o)

static const eh_config_directive_t eh_config_core_directive[] = {
  { "ServerRoot", 1, EH_CB_qstr, EH_S_G(serverroot), "/" },
  { "BindAddress", 1, EH_CB_qstr, EH_S_G(bindaddress), NULL },
  { "User", 1, EH_CB_idstr, EH_S_G(user), NULL },
  { "Group", 1, EH_CB_idstr, EH_S_G(group), NULL },
  { "MaxFiles", 1, EH_CB_range, EH_S_G(maxfiles), "256 65535" },
  { "StartServers", 1, EH_CB_range, EH_S_G(startservers), "1 32" },
  { "ChangeRoot", 1, EH_CB_qstr, EH_S_G(changeroot), "/" },
  { "ListenBacklog", 1, EH_CB_range, EH_S_G(listenbacklog), "10 1024" },
  { "SendBufferSize", 1, EH_CB_range, EH_S_G(sendbuffersize), "1024 65536" },
  { "RecvBufferSize", 1, EH_CB_range, EH_S_G(recvbuffersize), "1024 65536" },
  { "FileCacheSize", 1, EH_CB_range, EH_S_G(filecachesize), "100 65535" },
  { "FileCacheThreshold", 1, EH_CB_range, EH_S_G(filecachethreshold),
    "0 1073741824" },
  { "SendCGIErrors", 1, EH_CB_onoff, EH_S_G(sendcgierrors), NULL },
  { "MultipleAccept", 1, EH_CB_onoff, EH_S_G(multipleaccept), NULL },
  { "ForceLingeringClose", 1, EH_CB_onoff, EH_S_G(forcelingeringclose), NULL },
  { "LingeringCloseLimit", 1, EH_CB_range, EH_S_G(lingeringcloselimit),
    "1024 1048576" },
  { "RequestHeaderLimit", 1, EH_CB_range, EH_S_G(requestheaderlimit),
    "1024 65536" },
  { "RequestBodyLimit", 1, EH_CB_range, EH_S_G(requestbodylimit),
    "1024 1073741824" },
  { "ScriptEvalMax", 1, EH_CB_range, EH_S_G(scriptevalmax), "0 65536" },
  { "Port", 1, EH_CB_range, EH_S_V(port), "0 65535" },
  { "ServerName", 1, EH_CB_qstr, EH_S_V(servername), NULL },
  { "ServerAdmin", 1, EH_CB_qstr, EH_S_V(serveradmin), NULL },
  { "PassEnv", 1, EH_CB_strarr, EH_S_V(passenv), NULL },
  { "CustomLog", -1, EH_CB_customlog, eh_s_vhost, 0, NULL },
  { "DocumentRoot", 1, EH_CB_qstr, EH_S_V(documentroot), "/" },
  { "TypesConfig", 1, EH_CB_qstr, EH_S_V(typesconfig), NULL },
  { "ErrorDocumentDir", 1, EH_CB_qstr, EH_S_V(errordocumentdir), "/" },
  { "TimeOut", 1, EH_CB_range, EH_S_V(timeout), "10 600" },
  { "SSLPort", 1, EH_CB_range, EH_S_V(sslport), "0 65535" },
  { "SSLCertificateFile", 1, EH_CB_qstr, EH_S_V(sslcertificatefile), NULL },
  { "SSLCertificateKeyFile", 1, EH_CB_qstr, EH_S_V(sslcertificatekeyfile),
    NULL },
  { "SSLCACertificateFile", 1, EH_CB_qstr, EH_S_V(sslcacertificatefile),
    NULL },
  { "SSLDHParamFile", 1, EH_CB_qstr, EH_S_V(ssldhparamfile), NULL },
  { "SSLCipherSuite", 1, EH_CB_qstr, EH_S_V(sslciphersuite), NULL },
  { "Alias", 2, EH_CB_alias, eh_s_vhost, 0, NULL },
  { "ScriptAlias", -1, EH_CB_scriptalias, eh_s_vhost, 0, NULL },
  { "NoLog", -1, EH_CB_nolog, eh_s_vhost, 0, NULL },
  { "AuthUserFile", 1, EH_CB_qstr, EH_S_D(authuserfile), NULL },
  { "AuthName", 1, EH_CB_qstr, EH_S_D(authname), NULL },
  { "AuthRequireSSL", 1, EH_CB_onoff, EH_S_D(authrequiressl), NULL },
  { "DirectoryIndex", -1, EH_CB_strarr, EH_S_D(directoryindexes), NULL },
  { "DefaultType", 1, EH_CB_qstr, EH_S_D(defaulttype), NULL },
  { "SSLVerifyClient", 1, EH_CB_verifyclient, eh_s_dir, 0, NULL },
  { "SSLVerifyDepth", 1, EH_CB_range, EH_S_D(sslverifydepth), "1 100" },
  { "SSLRequireSSL", 1, EH_CB_onoff, EH_S_D(sslrequiressl), NULL },
  { "AddType", -1, EH_CB_addhandler, eh_s_dir, 0, "type" },
  { "AddHandler", -1, EH_CB_addhandler, eh_s_dir, 0, "handler" },
  { "Options", -1, EH_CB_options, eh_s_dir, 0, NULL },
  { "Require", -1, EH_CB_strarr, EH_S_L(require), NULL },
  { NULL, -1, NULL, NULL, eh_s_global, 0, NULL },
};
  
static void
eh_config_change_context (eh_config_global_t *econf, eh_config_context_t *con,
			  char *line, int linenum)
{
  char *p, *q, *s;
  eh_debug ("%s", line);
  if (strncmp (line, "</", 2) == 0) {
    if (con->unknown_context_count) {
      con->unknown_context_count--;
      return;
    }
    p = line + 2;
    q = p + strcspn (p, " \t<>");
    if (q[0] != '>') {
      con->errline = linenum;
      return;
    }
    q[0] = '\0';
    if (strcmp (p, "Limit") == 0) {
      if (con->limit == NULL)
	con->errline = linenum;
      con->limit = NULL;
    } else if (strcmp (p, "FilesMatch") == 0) {
      if (con->filesmatch == NULL)
	con->errline = linenum;
      con->filesmatch = NULL;
    } else if (strcmp (p, "Directory") == 0) {
      if (con->dir == NULL)
	con->errline = linenum;
      con->dir = NULL;
    } else if (strcmp (p, "VirtualHost") == 0) {
      if (con->vhost == NULL)
	con->errline = linenum;
      con->vhost = NULL;
    }
    return;
  }
  p = line + 1;
  p += strspn (p, " \t");
  q = p + strcspn (p, " \t");
  if (q[0] == '\0') {
    con->errline = linenum;
    return;
  }
  *q++ = '\0';
  q += strspn (q, " \t");
  s = q + eh_unescape_doublequotes (q);
  s += strspn (s, " \t");
  if (s[0] == '>')
    s++;
  s += strspn (s, " \t");
  if (s[0] != '\0') {
    con->errline = linenum;
    return;
  }
  if (strcmp (p, "Limit") == 0) {
    eh_config_filesmatch_t *ecf;
    if (con->limit) {
      con->errline = linenum;
      return;
    }
    if (con->filesmatch) {
      ecf = con->filesmatch;
    } else if (con->dir) {
      ecf = con->dir->default_filesmatch;
    } else if (con->vhost) {
      ecf = con->vhost->default_dir->default_filesmatch;
    } else {
      ecf = econf->default_vhost->default_dir->default_filesmatch;
    }
    eh_debug ("new limit context: '%s'", q);
    con->limit = eh_config_filesmatch_create_limit (ecf, q);
    eh_debug ("new ecl %p", con->limit);
    if (con->limit == NULL) {
      /* invalid method */
      con->errline = linenum;
      return;
    }
    return;
  } else if (strcmp (p, "FilesMatch") == 0) {
    eh_config_dir_t *ecd;
    if (con->filesmatch || con->limit) {
      con->errline = linenum;
      return;
    }
    if (con->dir) {
      ecd = con->dir;
    } else if (con->vhost) {
      ecd = con->vhost->default_dir;
    } else {
      ecd = econf->default_vhost->default_dir;
    }
    eh_debug ("new filesmatch context: '%s'", q);
    con->filesmatch = eh_config_dir_create_filesmatch (ecd, q);
    eh_debug ("new ecf %p", con->filesmatch);
    if (con->filesmatch == NULL) {
      /* invalid pattern */
      con->errline = linenum;
      return;
    }
    return;
  } else if (strcmp (p, "Directory") == 0) {
    eh_config_vhost_t *ecv;
    if (con->dir || con->filesmatch || con->limit) {
      con->errline = linenum;
      return;
    }
    ecv = con->vhost ? con->vhost : econf->default_vhost;
    eh_debug ("new directory context: '%s'", q);
    con->dir = eh_config_vhost_find_create_dir (ecv, q);
    eh_debug ("new ecd %p", con->dir);
    return;
  } else if (strcmp (p, "VirtualHost") == 0) {
    if (con->vhost || con->dir || con->filesmatch || con->limit) {
      con->errline = linenum;
      return;
    }
    eh_debug ("new virtualhost context: '%s'", q);
    con->vhost = eh_config_global_find_create_vhost (econf, q);
    eh_debug ("new ecv %p", con->vhost);
  } else {
    con->unknown_context_count++;
    return;
  }
}

static void
eh_config_parse_directive (eh_config_global_t *econf, eh_config_context_t *con,
			   const char *key, const char *val)
{
  eh_config_vhost_t *vhost;
  eh_config_dir_t *dir;
  eh_config_filesmatch_t *filesmatch;
  eh_config_limit_t *limit;
  void *confdata = econf;
  char **args;
  int num_args;
  const eh_config_directive_t *p;
  for (p = eh_config_core_directive; p->name; p++) {
    if (strcasecmp (p->name, key) == 0)
      break;
  }
  if (p->name == NULL) {
    x_append_printf (&econf->error_string,
		     "%d: unknown keyword '%s'\n", con->linenum, key);
    con->errline = con->linenum;
    return;
  }
  vhost = con->vhost;
  if (vhost == NULL)
    vhost = econf->default_vhost;
  dir = con->dir;
  if (dir == NULL)
    dir = vhost->default_dir;
  filesmatch = con->filesmatch;
  if (filesmatch == NULL)
    filesmatch = dir->default_filesmatch;
  limit = con->limit;
  if (limit == NULL)
    limit = filesmatch->default_limit;
  switch (p->section) {
  case eh_s_global:
    if (con->vhost || con->dir || con->filesmatch || con->limit) {
      x_append_printf (&econf->error_string,
		       "%d: keyword '%s' must be in the global section\n",
		       con->linenum, key);
      con->errline = con->linenum;
    }
    confdata = econf;
    break;
  case eh_s_vhost:
    if (con->dir || con->filesmatch || con->limit) {
      x_append_printf (&econf->error_string,
		       "%d: keyword '%s' must be in a <VirtualHost> section\n",
		       con->linenum, key);
      con->errline = con->linenum;
    }
    confdata = vhost;
    break;
  case eh_s_dir:
    if (con->filesmatch || con->limit) {
      x_append_printf (&econf->error_string,
		       "%d: keyword '%s' must be in a <Directory> section\n",
		       con->linenum, key);
      con->errline = con->linenum;
    }
    confdata = dir;
    break;
  case eh_s_fmatch:
    if (con->limit) {
      x_append_printf (&econf->error_string,
		       "%d: keyword '%s' must be in a <FilesMatch> section\n",
		       con->linenum, key);
      con->errline = con->linenum;
    }
    confdata = filesmatch;
    break;
  case eh_s_limit:
    confdata = limit;
    break;
  }
  if (con->errline)
    return;
  args = eh_get_token_arr (val);
  for (num_args = 0; args[num_args] != NULL; num_args++);
  if (p->num_args >= 0 && num_args != p->num_args) {
    x_append_printf (&econf->error_string,
		     "%d: wrong number of arguments for '%s'\n",
		     con->linenum, key);
    con->errline = con->linenum;
  } else if ((*p->read_cb)(confdata, (const char **)args, num_args, p)) {
    x_append_printf (&econf->error_string,
		     "%d: wrong argument for '%s'\n",
		     con->linenum, key);
    con->errline = con->linenum;
  }
  eh_strarr_free (args);
}

static void
eh_config_parse_line (eh_config_global_t *econf, eh_config_context_t *con,
		      char *line, int linenum)
{
  char *p, *q;
  p = line + strspn (line, " \t");
  if (p[0] == '#')
    return;
  if (p[0] == '<') {
    eh_config_change_context (econf, con, p, linenum);
    return;
  }
  if (con->unknown_context_count) {
    eh_debug("unknown context");
    return;
  }
  q = p + strcspn (p, " \t");
  if (*q != '\0') {
    *q++ = '\0';
  }
  q = q + strspn (p, " \t");
  if (p[0] == '\0')
    return;
  eh_config_parse_directive (econf, con, p, q);
}

#if 0
static void
eh_config_entry_dump (eh_strhash_entry_t *ent, void *data)
{
  int i, n = *(int *)data;
  char **p;
  for (i = 0; i < n; i++) putchar ('\t');
  printf ("%s ", ent->key);
  for (p = (char **)ent->val; *p; p++)
    printf ("[%s]", *p);
  putchar ('\n');
}

static void
eh_config_dir_dump (eh_strhash_entry_t *ent, void *data)
{
  int v = 2;
  eh_config_dir_t *ecd = (eh_config_dir_t *)ent->val;
  printf ("<Directory %s>\n", ent->key);
  eh_strhash_foreach (ecd->strarr_ht, eh_config_entry_dump, &v);
  printf ("</Directory>\n");
}

static void
eh_config_global_dump (eh_config_global_t *econf)
{
  int v = 0;
  eh_strhash_foreach (econf->strarr_ht, eh_config_entry_dump, &v);
  eh_strhash_foreach (econf->default_vhost->dir_ht, eh_config_dir_dump, NULL);
}
#endif

int
eh_config_global_read (eh_config_global_t *econf, const char *path)
{
  FILE *fp;
  char *line;
  int linenum;
  eh_config_context_t context;
  memset (&context, 0, sizeof (context));
  fp = fopen (path, "r");
  if (fp == NULL) {
    return 1;
  }
  for (linenum = 1; (line = eh_file_get_line (fp)) != NULL; linenum++) {
    context.linenum = linenum;
    eh_config_parse_line (econf, &context, line, linenum);
    x_free (line);
    if (context.errline)
      break;
  }
  fclose (fp);
  if (context.errline) {
    x_append_printf (&econf->error_string,
		     "An error is found in %s, line %d\n",
		     path, context.errline);
    econf->fatal_error = 1;
  }
  return econf->fatal_error;
}

static void
eh_config_write_directive (void *confdata, eh_config_section_t section,
			   FILE *fp, int indent)
{
  const eh_config_directive_t *p;
  char *s;
  for (p = eh_config_core_directive; p->name; p++) {
    if (p->section == section) {
      const char *t1, *t2;
      s = (*p->write_cb)(confdata, p);
      if (s == NULL) {
	fprintf (stderr, "%s: null\n", p->name);
	continue;
      }
      if (strchr (s, '\n') == NULL) {
	fprintf (stderr, "%s: no newline\n", p->name);
      }
      eh_for_p_in_str (s, "\n", t1, t2) {
	int i;
	for (i = 0; i < indent; i++)
	  fputc ('\t', fp);
	fwrite (t1, t2 - t1, 1, fp);
	fputc ('\n', fp);
      }
      x_free (s);
    }
  }
}

static void
eh_config_limit_write (eh_config_limit_t *lim, FILE *fp, int is_default)
{
  char *s;
  s = eh_config_get_method_mask_str (lim->method_mask);
  if (!is_default && s == NULL)
    return;
  if (is_default)
    fprintf (fp, "\t\t\t# Default Limit\n");
  else
    fprintf (fp, "\t\t\t<Limit%s>\n", s);
  x_free (s);
  eh_config_write_directive (lim, eh_s_limit, fp, 4);
  if (is_default)
    fprintf (fp, "\t\t\t# End of Default Limit\n");
  else
    fprintf (fp, "\t\t\t</Limit>\n");
}

static void
eh_config_filesmatch_write (eh_config_filesmatch_t *fm, FILE *fp,
			    int is_default)
{
  char *s;
  int i;
  s = eh_strdup_escape_doublequotes (fm->pattern);
  if (is_default)
    fprintf (fp, "\t\t# Default FilesMatch\n");
  else
    fprintf (fp, "\t\t<FilesMatch \"%s\">\n", s);
  x_free (s);
  for (i = 0; i < fm->num_limit; i++) {
    eh_config_limit_write (fm->limit_arr[i], fp, 0);
  }
  eh_config_limit_write (fm->default_limit, fp, 1);
  if (is_default)
    fprintf (fp, "\t\t# End of Default FilesMatch\n");
  else
    fprintf (fp, "\t\t</FilesMatch>\n");
}

static void
eh_config_dir_write (eh_config_dir_t *dir, const char *name, FILE *fp,
		     int is_default)
{
  char *s;
  int i;
  s = eh_strdup_escape_doublequotes (name);
  if (is_default)
    fprintf (fp, "\t# Default Directory\n");
  else
    fprintf (fp, "\t<Directory \"%s\">\n", s);
  x_free (s);
  eh_config_write_directive (dir, eh_s_dir, fp, 2);
  for (i = 0; i < dir->num_filesmatch; i++) {
    eh_config_filesmatch_write (dir->filesmatch_arr[i], fp, 0);
  }
  eh_config_filesmatch_write (dir->default_filesmatch, fp, 1);
  if (is_default)
    fprintf (fp, "\t# End of Default Directory\n");
  else
    fprintf (fp, "\t</Directory>\n");
}

static void
eh_config_dir_write_cb (eh_strhash_entry_t *ent, void *data)
{
  eh_config_dir_write ((eh_config_dir_t *)ent->val,
		       (const char *)ent->key, (FILE *)data, 0);
}

static void
eh_config_vhost_write (eh_config_vhost_t *vhost, const char *name, FILE *fp,
		       int is_default)
{
  char *s;
  s = eh_strdup_escape_doublequotes (name);
  if (is_default)
    fprintf (fp, "# Default VirtualHost\n");
  else
    fprintf (fp, "<VirtualHost \"%s\">\n", s);
  x_free (s);
  eh_config_write_directive (vhost, eh_s_vhost, fp, 1);
  eh_strhash_foreach (vhost->dir_ht, eh_config_dir_write_cb, fp);
  eh_config_dir_write (vhost->default_dir, "/", fp, 1);
  if (is_default)
    fprintf (fp, "# End of Default VirtualHost\n");
  else
    fprintf (fp, "</VirtualHost>\n");
}

static void
eh_config_vhost_write_cb (eh_strhash_entry_t *ent, void *data)
{
  eh_config_vhost_write ((eh_config_vhost_t *)ent->val,
			 (const char *)ent->key, (FILE *)data, 0);
}

void
eh_config_global_write (eh_config_global_t *econf, FILE *fp)
{
  eh_config_write_directive (econf, eh_s_global, fp, 0);
  eh_strhash_foreach (econf->vhost_ht, eh_config_vhost_write_cb, fp);
  eh_config_vhost_write (econf->default_vhost, "*", fp, 1);
}
