/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

static void become_daemon (int log_fd)
{
  int i;
  if (fork () > 0) {
    /* parent */
    exit (0);
  }
  /* first child */
  setpgrp ();
  if (fork () > 0) {
    /* first child */
    exit (0);
  }
  /* second child */
  for (i = 0; i < 3; i++) {
    if (log_fd != i)
      close (i);
  }
  if (log_fd >= 0) {
    dup2 (log_fd, 0);
    dup2 (log_fd, 1);
    dup2 (log_fd, 2);
  }
  setsid ();
  chdir ("/");
}

static void
show_usage (const char *name)
{
  fprintf (stderr,
	   "Usage: %s [OPTIONS]\n\n"
	   "  -n               don't fork. run in foreground\n"
	   "  -s               output log messages to standard error\n"
	   "  -d               enable debugging\n"
	   "  -t               test configuration file\n"
	   "  -l FILENAME      set logfile to FILENAME\n"
	   "  -f FILENAME      read configuration from FILENAME\n"
	   "  -p               force to use the standard poll() method\n"
	   , name);
  exit (-1);
}

void
set_stdio_null (void)
{
  FILE *fp;
  fflush (stdout);
  fp = fopen ("/dev/null", "r");
  if (fp == NULL) {
    eh_log_perror (EH_LOG_FATAL, "/dev/null");
    exit (-1);
  }
  dup2 (fileno (fp), 0);
  fclose (fp);
  fp = fopen ("/dev/null", "w");
  if (fp == NULL) {
    eh_log_perror (EH_LOG_FATAL, "/dev/null");
    exit (-1);
  }
  dup2 (fileno (fp), 1);
  fclose (fp);
}

static void
unset_env_variables (void)
{
  char **env_cp, **p;
  env_cp = eh_strarr_copy ((const char **)environ);
  for (p = env_cp; *p; p++) {
    char *s;
    char *t;
    if (strncmp (*p, "ESEHTTPD", 8) == 0) {
      continue;
    } else if (strncmp (*p, "PATH=", 5) == 0) {
      continue;
    }
    s = strchr (*p, '=');
    if (s == NULL) {
      continue;
    }
    t = x_strndup (*p, s - *p);
    unsetenv (t);
    x_free (t);
  }
  eh_strarr_free (env_cp);
}

static void
eh_configtest (const char *conf_file)
{
  eh_config_global_t *econf;

  eh_log (EH_LOG_INFO, "testing configuration file:");
  eh_log (EH_LOG_INFO, "'%s'", conf_file);
  econf = eh_config_global_new (conf_file);
  if (eh_config_global_prepare_tables (econf) == 0) {
    eh_log (EH_LOG_INFO, "OK");
    eh_config_global_delete (econf);
    exit (0);
  } else {
    eh_config_global_log_errors (econf);
    eh_config_global_delete (econf);
    exit (-1);
  }
}

int main (int argc, char **argv)
{
  eh_app_t app;
  int log_fd;
  int nodaemon, log_stderr, use_poll, configtest;
  int i;
  int r;
  const char *conf_file = ESEHTTPD_CONF;
  const char *log_file = ESEHTTPD_LOGFILE;
  const char *log_pattern = NULL;
#ifdef DEBUG
  extern void mtrace ();
  extern void muntrace ();
  mtrace ();
  atexit (muntrace);
#endif
  
  unset_env_variables ();
  
  log_fd = -1;
  nodaemon = 0;
  log_stderr = 0;
  use_poll = 0;
  configtest = 0;
  
  if (getenv ("ESEHTTPD_NODAEMON"))
    nodaemon = 1;
  if ((log_pattern = getenv ("ESEHTTPD_DEBUG")) != NULL) {
    debug = 1;
    nodaemon = 1;
    log_stderr = 1;
  }
  for (i = 1; i < argc; i++) {
    if (argv[i][0] != '-' || strlen (argv[i]) != 2) {
      show_usage (argv[0]);
    }
    switch (argv[i][1]) {
    case 'n':
      nodaemon = 1;
      break;
    case 'p':
      use_poll = 1;
      break;
    case 's':
      log_stderr = 1;
      nodaemon = 1;
      break;
    case 'd':
      debug = 1;
      nodaemon = 1;
      log_stderr = 1;
      break;
    case 'l':
      if (i + 1 >= argc) {
	show_usage (argv[0]);
      }
      log_file = argv[++i];
      break;
    case 't':
      configtest = 1;
      nodaemon = 1;
      log_stderr = 1;
      break;
    case 'f':
      if (i + 1 >= argc) {
	show_usage (argv[0]);
      }
      conf_file = argv[++i];
      break;
    default:
      show_usage (argv[0]);
    }
  }

  if (!log_stderr) {
    log_fd = eh_log_open (log_file, 1, log_pattern);
    if (log_fd < 0) {
      perror (log_file);
      exit (-1);
    }
  } else {
    eh_log_open (NULL, 0, log_pattern);
  }

  eh_log (EH_LOG_INFO, ESEHTTPD_NAME " " VERSION);

  SSL_library_init ();
  SSL_load_error_strings ();
#if 0
  if (RAND_load_file ("/dev/random", 32) < 32) {
    eh_log (EH_LOG_INFO, "failed to read from /dev/random");
  }
#endif
#if SSL_LIBRARY_VERSION >= 0x00905100
  if (RAND_status () == 0) {
    eh_log (EH_LOG_INFO,
	    "warning: PRNG still contains not sufficient entropy");
  }
#endif

  if (configtest) {
    eh_configtest (conf_file);
  }
  
  eh_fd_init (use_poll);

  eh_app_init (&app, conf_file);
  
  if (!nodaemon) {
    become_daemon (log_fd);
    eh_fd_getpid ();
    eh_log_getpid ();
  }
  umask (0077);
  
  set_stdio_null ();

  /* create sockets */
  eh_app_start_servers (&app);
  
  /* drop root privilege */
  eh_app_chroot_setuid (&app);

  /* fork worker processes */
  eh_app_start_workers (&app);

  /* the event loop */
  r = eh_app_select_loop (&app);

  /* we need to free fds before app so that app->filecache is freed properly */
  eh_fd_free_all ();
  
  eh_app_discard (&app);
  
  if (getenv ("ESEHTTPD_MEMPROF")) {
    while (1)
      sleep(1000);
  }
  x_mem_check ();
  
  eh_log (EH_LOG_INFO, "stop");
  eh_log_close ();

  return r;
}
