/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

typedef struct {
  unsigned int offset;
  const char name[32];
} eh_predef_entry_t;

#define EHENT(x, y) {offsetof(eh_predef_fields_t, y), x}

static const eh_predef_entry_t eh_predef_entries[] = {
  /* this array must be sorted */
  EHENT ("authorization", authorization),
  EHENT ("cache-control", cache_control),
  EHENT ("connection", connection),
  EHENT ("content-length", content_length),
  EHENT ("content-type", content_type),
  EHENT ("expect", expect),
  EHENT ("host", host),
  EHENT ("if-modified-since", if_modified_since),
  EHENT ("if-range", if_range),
  EHENT ("if-unmodified-since", if_unmodified_since),
  EHENT ("location", location),
  EHENT ("pragma", pragma),
  EHENT ("range", range),
  EHENT ("referer", referer),
  EHENT ("status", status),
  EHENT ("transfer-encoding", transfer_encoding),
  EHENT ("user-agent", user_agent),
};

static void
eh_headers_make_space (eh_headers_t *eh)
{
  if (eh->hentry_len >= eh->hentry_alloclen) {
    eh->hentry_alloclen *= 2;
    eh->hentry = (eh_header_entry_t *)
      x_realloc (eh->hentry, eh->hentry_alloclen * sizeof (eh->hentry[0]));
  }
}

static unsigned char eh_ctable[256];
static int eh_ctable_initialized = 0;

static void
eh_ctable_init (void)
{
  size_t i;
  /* initialize ctable */
  static const char sep[] = "()<>@,;:\\\"/[]?={} \t";
  for (i = 32; i < 127; i++) {
    if (i >= 'A' && i <= 'Z')
      eh_ctable[i] = i - ('A' - 'a');
    else
      eh_ctable[i] = i;
  }
  for (i = 0; i < sizeof(sep); i++)
    eh_ctable[(int)sep[i]] = 0;
}

static int
eh_get_token (char *str)
{
  unsigned char *p;
  char c;
  for (p = (unsigned char *)str; (c = eh_ctable[*p]) != 0; p++)
    *p = c;
  return (char *)p - str;
}

static const eh_predef_entry_t *
eh_find_predef_entry (const char *name)
{
  size_t l, u, i;
  size_t len = strlen (name) + 1;
  l = 0;
  u = sizeof (eh_predef_entries) / sizeof (eh_predef_entries[0]);
  while (l < u) {
    int cmp;
    i = (l + u) / 2;
    cmp = memcmp (name, eh_predef_entries[i].name, len);
    /*    cmp = strcmp (name, eh_predef_entries[i].name); */
    if (cmp < 0) {
      u = i;
    } else if (cmp > 0) {
      l = i + 1;
    } else {
      return eh_predef_entries + i;
    }
  }
  return NULL;
}

static const char **
eh_headers_find_predef_entry (eh_headers_t *eh, const char *name)
{
  const eh_predef_entry_t *pos;
  eh_debug ("[%s]", name);
  pos = eh_find_predef_entry (name);
  if (pos) {
    eh_debug ("found");
    return (const char **)((char *)(&eh->predef) + pos->offset);
  }
  return NULL;
}

void
eh_headers_remove (eh_headers_t *eh, const char *name)
{
  const char **pp;
  size_t i;
  eh_debug ("[%s: %s]", name);
  pp = eh_headers_find_predef_entry (eh, name);
  if (pp)
    *pp = NULL;
  for (i = 0; i < eh->hentry_len; i++) {
    if (strcmp (eh->hentry[i].name, name) == 0)
      eh->hentry[i].value = NULL;
  }
}

void
eh_headers_append (eh_headers_t *eh, const char *name, const char *value)
{
  const char **pp;
  eh_debug ("[%s: %s]", name, value);
  /* check if name is in the 'pre-defined' list */
  pp = eh_headers_find_predef_entry (eh, name);
  if (pp && *pp == NULL) {
    *pp = value;
    return;
  }
  /* not a predef field, or more than one values are set for a predef field */
  eh_headers_make_space (eh);
  eh->hentry[eh->hentry_len].name = name;
  eh->hentry[eh->hentry_len].value = value;
  eh->hentry_len++;
}

static int
eh_get_field_value (char *str, char *str_end)
{
  int r;
  char *p;
#if 0
  for (p = strchr (str, '\n'); p; p = strchr (p + 1, '\n')) {
    if (p[1] != ' ' && p[1] != '\t')
      break;
  }
#endif
  for (p = (char *)memchr (str, '\n', str_end - str); p;
       p = (char *)memchr (p + 1, '\n', str_end - p - 1)) {
    if (p[1] != ' ' && p[1] != '\t')
      break;
  }
  if (p == NULL)
    return strlen (str);
  r = p - str;
  while (p != str && (p[0] == '\n' || p[0] == '\r'))
    *p-- = '\0';
  return r + 1;
}

static void
eh_headers_parse (eh_headers_t *eh, char *headers_str, char *headers_str_end)
{
  char *field_name, *field_value;
  field_name = headers_str;
  assert (headers_str_end[-1] == '\0');

  eh_debug ("");
  if (!eh_ctable_initialized) {
    eh_ctable_init ();
    eh_ctable_initialized = 1;
  }
  
  while (field_name[0] && field_name[0] != '\r' && field_name[0] != '\n') {
    int namelen, e;

    /* get field_name */
    namelen = eh_get_token (field_name);
    if (field_name[namelen] != ':') {
      eh_debug ("got char %d instead of ':'", field_name[namelen]);
      eh->ill_formed = 1;
      return;
    }
    field_name[namelen] = '\0';
    eh_debug ("field_name: %s", field_name);
    field_value = field_name + namelen + 1;
    while (field_value[0] == ' ')
      field_value++;

    /* get field_value */
    e = eh_get_field_value (field_value, headers_str_end);
    eh_debug ("field_value: %s", field_value);
    eh_headers_append (eh, field_name, field_value);
    field_name = field_value + e;
  }
  return;
}
			    
void
eh_headers_initnc (eh_headers_t *eh, char *headers_str, char *headers_str_end)
{
  assert (headers_str_end[-1] == '\0');
  eh->hentry_len = 0;
  eh->hentry_alloclen = 10;
  eh->ill_formed = 0;
  eh->hentry = (eh_header_entry_t *)
    x_malloc (eh->hentry_alloclen * sizeof (eh->hentry[0]));
  eh_headers_parse (eh, headers_str, headers_str_end);
}

void
eh_headers_discard (eh_headers_t *eh)
{
  x_free (eh->hentry);
  eh->hentry = NULL;
  eh->hentry_len = 0;
  eh->hentry_alloclen = 0;
  eh->ill_formed = 0;
}

char *
eh_headers_strdup_find (const eh_headers_t *eh, const char *name)
{
  eh_header_entry_t *he;
  char *val = NULL;
  size_t val_len = 0;
  for (he = eh->hentry; he < eh->hentry + eh->hentry_len; he++) {
    if (he->value == NULL)
      continue;
    if (strcmp (he->name, name) == 0) {
      size_t app_len, new_len;
      if (val) {
	/*
	  [RFC2616, 4.2 Message Headers]
	  Multiple message-header fields with the same field-name MAY be
	  present in a message if and only if the entire field-value for that
	  header field is defined as a comma-separated list [i.e., #(values)].
	  It MUST be possible to combine the multiple header fields into one
	  "field-name: field-value" pair, without changing the semantics of the
	  message, by appending each subsequent field-value to the first, each
	  separated by a comma.
	*/
	app_len = strlen (he->value);
	new_len = val_len + 1 + app_len;
	val = (char *)x_realloc (val, new_len + 1);
	val[val_len] = ',';
	memcpy (val + val_len + 1, he->value, app_len);
	val_len = new_len;
	val[new_len] = '\0';
      } else {
	val_len = strlen (he->value);
	val = x_strndup (he->value, val_len);
      }
    }
  }
  return val;
}

void
eh_headers_forall (const eh_headers_t *eh,
		   void (*func)(const char *key, const char *val, void *data),
		   void *data)
{
  size_t i;
  for (i = 0; i < sizeof (eh_predef_entries) / sizeof (eh_predef_entries[0]);
       i++) {
    const eh_predef_entry_t *ent = eh_predef_entries + i;
    const char *key, *val, **valp;
    key = ent->name;
    valp = (const char **)((char *)(&eh->predef) + ent->offset);
    val = *valp;
    if (!val)
      continue;
    (*func)(key, val, data);
  }
  for (i = 0; i < eh->hentry_len; i++) {
    if (eh->hentry[i].value == NULL)
      continue;
    (*func)(eh->hentry[i].name, eh->hentry[i].value, data);
  }
}

static void
eh_headers_append_headerstr_one (const char *key, const char *val, void *data)
{
  eh_strbuf_t *strbuf = (eh_strbuf_t *)data;
  char *str;
  str = x_strdup_printf ("%s: %s\r\n", key, val);
  eh_strbuf_append (strbuf, str, strlen (str));
  x_free (str);
}

char *
eh_headers_strdup_getall (const eh_headers_t *eh)
{
  char *headers_str;
  eh_strbuf_t strbuf;
  eh_strbuf_init (&strbuf, (size_t)-1);
  eh_headers_forall (eh, eh_headers_append_headerstr_one, (void *)&strbuf);
  headers_str = x_strdup (strbuf.buffer);
  eh_strbuf_discard (&strbuf);
  return headers_str;
}

