/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"
#include <regex.h>

static pid_t pid = 0;
static FILE *log_file = NULL;
static int need_locking = 0;
static int use_log_regex = 0;
static regex_t log_regex;

static void
eh_log_setlock (int lck)
{
  if (log_file == NULL)
    return;
  while (1) {
    int r;
    struct flock lk;
    lk.l_type = lck ? F_WRLCK : F_UNLCK;
    lk.l_whence = 0;
    lk.l_start = 0;
    lk.l_len = 0;
    lk.l_pid = 0;
    r = fcntl (fileno (log_file), F_SETLKW, &lk);
    if (r < 0 && errno != EINTR) {
      syslog (LOG_NOTICE, "locking logfile %s", strerror (errno));
    } else {
      break;
    }
  }
}

int eh_log_open (const char *filename, int lock_logfile,
		 const char *log_pattern)
{
  if (log_file != NULL || use_log_regex != 0) {
    errno = EINVAL;
    return -1;
  }
  pid = getpid ();
  if (log_pattern) {
    int r;
    use_log_regex = 1;
    if ((r = regcomp (&log_regex, log_pattern, REG_EXTENDED | REG_NOSUB))
	!= 0) {
      char buf[1025];
      buf[1024] = '\0';
      regerror (r, &log_regex, buf, 1024);
      fprintf (stderr, "invalid regular expression: '%s': %s\n",
	       log_pattern, buf);
      regfree (&log_regex);
      use_log_regex = 0;
    }
  }
  if (filename == NULL) {
    return -1;
  }
  log_file = fopen (filename, "a");
  if (log_file == NULL) {
    syslog (LOG_NOTICE, "opening %s: %s", filename, strerror (errno));
    fprintf (stderr, "opening %s: %s\n", filename, strerror (errno));
    return -1;
  }
  need_locking = lock_logfile;
  setvbuf (log_file, NULL, _IOLBF, 0);
  fflush (stderr);
  return fileno (log_file);
}

void eh_log_getpid (void)
{
  pid = getpid ();
}

int eh_log_close (void)
{
  int r;
  if (log_file == NULL)
    return 0;
  r = fclose (log_file);
  if (r < 0) {
    syslog (LOG_NOTICE, "closing logfile: %s", strerror (errno));
  }
  return r;
}

void eh_log_funcname (int level, const char *funcname, const char *format, ...)
{
  FILE *fp;
  va_list ap;
  if (!debug)
    level &= ~EH_LOG_DEBUG;
  if (level == 0)
    return;
  fp = log_file ? log_file : stderr;

  if (use_log_regex) {
    if (format == NULL || regexec (&log_regex, format, 0, NULL, 0) != 0)
      return;
    if (funcname == NULL || regexec (&log_regex, funcname, 0, NULL, 0) != 0)
      return;
  }
  
  if (pid == 0)
    pid = getpid ();

  eh_log_setlock (1);
  fprintf (fp, "%s [%u] ", eh_fd_get_time_string_logformat (), pid);
  va_start (ap, format);
  vfprintf (fp, format, ap);
  va_end (ap);
  if (funcname && level != EH_LOG_INFO)
    fprintf (fp, " (%s)", funcname);
  fprintf (fp, "\n");
  eh_log_setlock (0);
}

void eh_log_ssl_error (int level, const char *str)
{
  unsigned long err;
  err = ERR_get_error ();
  eh_log_funcname (level, NULL, "%s: %s", str, ERR_error_string (err, NULL));
}
