/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

void
eh_mimetype_discard (eh_mimetype_t *mt)
{
  eh_strhash_delete (mt->ht);
  x_free (mt->default_mimetype);
  mt->ht = NULL;
}

static void
eh_mimetype_append (eh_mimetype_t *mt, const char *ext, int ext_len,
		    const char *mtstr, int mtstr_len)
{
  eh_strhash_entry_t *ent;
  char *key, *val;
  key = x_strndup (ext, ext_len);
  val = x_strndup (mtstr, mtstr_len);
  ent = eh_strhash_find_create (mt->ht, key);
  if (key != ent->key) x_free (key);
  if (ent->val) x_free (ent->val);
  ent->val = val;
}

static void
eh_mimetype_parse_line (eh_mimetype_t *mt, const char *line)
{
  const char *p;
  const char *mtstr;
  int mtstr_len;
  line += strspn (line, " \t");
  if (line[0] == '#')
    return;
  mtstr = line;
  mtstr_len = strcspn (mtstr, " \t");
  if (mtstr_len == 0)
    return;
  p = mtstr + mtstr_len;
  p += strspn (p, " \t");
  while (1) {
    int ext_len;
    ext_len = strcspn (p, " \t");
    if (ext_len == 0)
      break;
    eh_mimetype_append (mt, p, ext_len, mtstr, mtstr_len);
    p += ext_len;
    p += strspn (p, " \t");
  }
}

static void
on_delete_entry (eh_strhash_entry_t *ent)
{
  x_free (ent->key);
  x_free (ent->val);
}

void
eh_mimetype_init (eh_mimetype_t *mt, const char *path,
		  const char *default_mimetype)
{
  FILE *fp;
  char *line;

  memset (mt, 0, sizeof (*mt));
  mt->ht = eh_strhash_new (389, on_delete_entry);
  mt->default_mimetype = x_strdup (default_mimetype);

  fp = fopen (path, "r");
  if (fp == NULL) {
    eh_log_perror (EH_LOG_FATAL, "fopen");
    return;
  }
  while ((line = eh_file_get_line (fp)) != NULL) {
    eh_mimetype_parse_line (mt, line);
    x_free (line);
  }
  fclose (fp);
#if 0
  if (debug) eh_strhash_dump (mt->ht);
#endif
}

const char *
eh_mimetype_find_ext (eh_mimetype_t *mt, const char *ext)
{
  return (const char *)eh_strhash_find (mt->ht, ext, NULL);
}

