/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

static void
eh_check_keepalive (eh_request_t *req)
{
  const char *val;
  val = req->headers.predef.connection;
  if (val == NULL) {
    /* not specified */
    req->rconn = req->http_version_minor == 0
      ? eh_rconn_close : eh_rconn_keepalive;
    req->need_connection_field = 0;
  } else if (strcasecmp (val, "keep-alive") == 0) {
    req->rconn = eh_rconn_keepalive;
    req->need_connection_field = req->http_version_minor == 0 ? 1 : 0;
  } else if (strcasecmp (val, "close") == 0) {
    req->rconn = eh_rconn_close;
    req->need_connection_field = 0;
  } else {
    /* unknown value */
    req->rconn = eh_rconn_close;
    req->need_connection_field = 1;
  }
}

static void
eh_check_request_body_length (eh_request_t *req)
{
  int r;
  size_t len;
  if (req->headers.predef.content_length == NULL)
    return;
  r = eh_parse_sizestr (req->headers.predef.content_length, &len);
  if (r < 0) {
    req->bad_request = 1;
  }
  req->request_body_length = len;
}

void
eh_request_init (eh_request_t *req_r, char *request_str,
		 size_t request_len, size_t extra_len,
		 eh_config_vhost_t *econf_default_vhost_ref,
		 int use_access_log, eh_accesslog_t *ea)
{
  char *p;
  int headers_offset = 0;

  if (req_r->not_finished) {
    eh_log (EH_LOG_WARNING, "previous request not finished");
    eh_request_discard (req_r);
  }
  memset (req_r, 0, sizeof (*req_r));
  req_r->method = eh_method_unknown;
  req_r->econf_vhost_ref = econf_default_vhost_ref;
  req_r->not_finished = 1;
  req_r->bad_request = 1;
  req_r->request_str_ref = request_str;
  req_r->request_len = request_len;
  req_r->extra_len = extra_len;

  while (request_str[0] == '\r' || request_str[0] == '\n')
    request_str++;
  req_r->method_str = request_str;
  
  if ((p = strchr (request_str, '\n')) == NULL)
    goto do_headers;
  p[0] = '\0';
  if (p > request_str && p[-1] == '\r')
    p[-1] = '\0';
  eh_accesslog_new_request (ea, use_access_log ? request_str : NULL);
  
  headers_offset = p + 1 - request_str;
  p = request_str + strcspn (request_str, " \t");
  if (*p == '\0')
    goto do_headers;
  *p++ = '\0';

  req_r->method = eh_config_get_method (request_str);

  p += strspn (p, " \t");
  req_r->url = p;
  p += strcspn (p, "? ");
  if (*p == '?') {
    *p = '\0';
    p++;
    req_r->query_string = p;
    p += strcspn (p, " ");
  }
  if (*p != ' ')
    goto do_headers;
  *p++ = '\0';
  p += strspn (p, " \t");
  if (strncmp (p, "HTTP/1.", 7) != 0)
    goto do_headers;
  p += 7;
  
  if (p[0] > '9' || p[0] < '0')
    goto do_headers;
  req_r->http_version_minor = (*p++) - '0';
  while (p[0] >= '0' && p[0] <= '9') {
    req_r->http_version_minor *= 10;
    req_r->http_version_minor += (*p++) - '0';
  }

  req_r->bad_request = 0;
  
 do_headers:
  
  if (req_r->bad_request == 0) {
    eh_headers_initnc (&req_r->headers, request_str + headers_offset,
		       request_str + request_len);
  } else {
    static char *tmpstr = "";
    eh_headers_initnc (&req_r->headers, tmpstr, tmpstr);
  }
  if (req_r->headers.ill_formed)
    req_r->bad_request = 1;
  eh_check_keepalive (req_r);
  eh_check_request_body_length (req_r);

  eh_debug ("url %s", req_r->url);
  return;
}

void
eh_request_determine_mimetype (eh_request_t *er, eh_connection_t *ec)
{
  const char *dot, *mimetype = NULL;
  dot = strrchr (er->filename, '.');
  if (dot == NULL || dot[1] == '\0') {
    goto defaulttype;
  }
  if (er->econf_dir_ref->addhandler_ht->num_ent_total > 0) {
    eh_addhandler_t *h;
    h = (eh_addhandler_t *)eh_strhash_find (er->econf_dir_ref->addhandler_ht,
					    dot + 1, NULL);
    if (h) {
      if (h->is_handler) {
	er->mime_type = NULL;
	er->rhfunc = h->rhfunc;
	er->rhfunc_data = h->rhfunc_data;
      } else {
	er->mime_type = h->name;
      }
      return;
    }
  }
  mimetype = eh_mimetype_find_ext (&ec->vserver_backref->mt, dot + 1);
  if (mimetype) {
    er->mime_type = mimetype;
    return;
  }
  
 defaulttype:
  if (er->econf_dir_ref->defaulttype) {
    er->mime_type = er->econf_dir_ref->defaulttype;
  } else {
    er->mime_type = ec->vserver_backref->mt.default_mimetype;
  }
}

int
eh_request_url_to_filename (eh_request_t *req, eh_connection_t *ec)
{
  int alias_src_len;
  const char *url_path;
  const char *document_root, *alias_dest;
  eh_rhandler_new_func_t *rhfunc = NULL;
  void *rhfunc_data = NULL;
  int nofilesystem = 0;

  assert (req->host == NULL);
  assert (req->filename == NULL);
  
  req->rhfunc = NULL;
  if (req->url[0] != '/') {
    int prefix_len;
    int host_len;
    const char *hostp, *p;
    if (strncmp (req->url, "http://", 7) == 0) {
      prefix_len = 7;
    } else if (strncmp (req->url, "https://", 8) == 0) {
      prefix_len = 8;
    } else {
      eh_debug ("wrong url: %s", req->url);
      return -1; /* Bad Request */
    }
    hostp = req->url + prefix_len;
    host_len = strcspn (hostp, ":/ ");
    req->host = x_strndup (hostp, host_len);
    p = hostp + host_len;
    p += strcspn (p, "/ ");
    url_path = p[0] == '/' ? p : "/";
  } else {
    url_path = req->url;
  }
  if (req->http_version_minor > 0) {
    if (req->headers.predef.host == NULL) {
      /* HTTP 1.1 request must have Host header */
      return -1;
    }
    if (ec->vserver_backref->namevirtualhost) {
      const char *host;
      eh_config_vhost_t *ecv;
      host = req->host; /* non-null if URI contains hostname */
      if (host == NULL) {
	host = req->headers.predef.host;
      }
      ecv = eh_config_global_find_vhost (ec->app_backref->econf, host);
      if (ecv &&
	  ecv->servaddr.sin_addr.s_addr
	  == req->econf_vhost_ref->servaddr.sin_addr.s_addr) {
	req->econf_vhost_ref = ecv;
	eh_debug ("vhost '%s' found: %p, root=%s", host, ecv,
		  ecv->documentroot_fullpath);
      } else {
	eh_debug ("vhost '%s' not found. using default %p, root=%s",
		  host, req->econf_vhost_ref,
		  req->econf_vhost_ref->documentroot_fullpath);
	return 1;
      }
    }
  } else if (ec->vserver_backref->namevirtualhost) {
    eh_debug ("http 1.0 request for a namevirtualhost");
    return 1; /* not found */
  }
  alias_src_len = eh_config_vhost_check_alias (req->econf_vhost_ref, url_path,
					       &alias_dest, &rhfunc,
					       &rhfunc_data,
					       &nofilesystem);
  if (alias_src_len > 0) {
    eh_debug ("alias_dest = %s, url_path = %s, alias_src_len = %d",
	      alias_dest, url_path, alias_src_len);
    req->filename = x_strdup_printf ("%s/%s", alias_dest,
				     url_path + alias_src_len);
  } else {
    document_root = req->econf_vhost_ref->documentroot_fullpath;
    assert (document_root);
    req->filename = x_strdup_printf ("%s%s", document_root, url_path);
  }
  if (eh_decode_url (req->filename)) {
    return 1;
  }
  if (eh_normalize_path (req->filename)) {
    return 1;
  }
  req->rhfunc = rhfunc;
  req->rhfunc_data = rhfunc_data;
  req->econf_dir_ref = eh_config_vhost_get_dir (req->econf_vhost_ref,
						req->filename);
  if (nofilesystem) {
    /* if rhfunc does not access to the filesystem, we need not to stat() */
    return 0;
  }
  if (req->headers.predef.pragma == NULL &&
      req->headers.predef.cache_control == NULL) {
    req->fcent_ref = eh_filecache_find (ec->app_backref->fcache,
					req->filename);
    eh_debug ("%s: cached: %p", req->filename, req->fcent_ref);
  } else {
    eh_debug ("no-cache is specified");
  }
  if (req->fcent_ref != NULL &&
      req->fcent_ref->last_stat_call_time == eh_fd_get_time ()) {
    eh_debug ("using cached statbuf");
    req->statbuf = req->fcent_ref->statbuf;
    req->mime_type = req->fcent_ref->mime_type;
    return 0;
  }
  if (stat (req->filename, &req->statbuf) < 0) {
    return 1;
  }
  if (S_ISDIR (req->statbuf.st_mode)) {
    struct stat statbuf_alt;
    const char *const *indexes;
    int len = strlen (req->filename);
    assert (len > 0);
    if (req->filename[len - 1] == '/') {
      len--;
    } else {
      return 1;
    }
    indexes = (const char *const *)req->econf_dir_ref->directoryindexes;
    if (indexes == NULL) {
      static const char *default_indexes[] = {
	"index.html", NULL,
      };
      indexes = default_indexes;
    }
    while (*indexes) {
      int indlen;
      indlen = strlen (*indexes);
      req->filename = (char *)x_realloc (req->filename, len + indlen + 2);
      req->filename[len] = '/';
      strcpy (req->filename + len + 1, *indexes);
      if (stat (req->filename, &statbuf_alt) == 0)
	break;
      indexes++;
    }
    if (*indexes) {
      /* index file is found */
      req->statbuf = statbuf_alt;
      if (req->headers.predef.pragma == NULL &&
	  req->headers.predef.cache_control == NULL) {
	req->fcent_ref = eh_filecache_find (ec->app_backref->fcache,
					    req->filename);
      }
    } else {
      req->filename = (char *)x_realloc (req->filename, len + 2);
      req->filename[len] = '/';
      req->filename[len + 1] = '\0';
    }
  }
  
  /* determine mimetype and handler */
  eh_request_determine_mimetype (req, ec);

  /* set limit section to use */
  req->econf_limit_ref = eh_config_dir_get_limit (req->econf_dir_ref,
						  req->filename, req->method);
  eh_debug ("filename = %s", req->filename);
  return 0;
}

void
eh_request_set_remote_user (eh_request_t *req, const char *remote_user)
{
  if (req->remote_user)
    x_free (req->remote_user);
  req->remote_user = x_strdup (remote_user);
}

void
eh_request_discard (eh_request_t *req)
{
  assert (req->not_finished);
  eh_debug ("");
  eh_headers_discard (&req->headers);
  if (req->request_line)
    x_free (req->request_line);
  if (req->host)
    x_free (req->host);
  if (req->filename)
    x_free (req->filename);
  if (req->remote_user)
    x_free (req->remote_user);
  memset (req, 0, sizeof (*req));
}
