/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

typedef struct {
  char *code;
  int force_close;
  char *phrase;
} eh_status_code_entry_t;

static const eh_status_code_entry_t eh_sentry[] = {
  /* The following list are status codes which may contain response body
     with it. Note that 1xx, 204, and 304 responses must not contain
     response body. Seems apache does not append response body for 412
     responses, but RFC does not have mention about it.
  */
  { "200", 0, "OK" },
  { "302", 0, "Found" },
#if 0
  { "304", 0, "Not Modified" },
#endif
  { "400", 1, "Bad Request" },
  { "401", 0, "Authorization Required" },
  { "403", 0, "Forbidden" },
  { "404", 0, "Not Found" },
  { "405", 0, "Method Not Allowed" },
  { "411", 1, "Length Required" },
#if 0
  { "412", 0, "Precondition Failed" },
#endif
  { "413", 1, "Request Entity Too Large" },
  { "417", 1, "Expectation Failed" },
  { "500", 1, "Internal Server Error" },
  { "501", 1, "Not Implemented"},
  { "503", 1, "Service Unavailable" },
};

void
eh_statuscode_table_discard (eh_statuscode_table_t *st)
{
  eh_strhash_delete (st->ht);
}

typedef struct {
  const char *phrase;
  int force_close;
  char *body;
} eh_statuscode_hash_ent_val_t;

static eh_statuscode_hash_ent_val_t *
eh_statuscode_hash_ent_val_new (const char *phrase, int force_close)
{
  eh_statuscode_hash_ent_val_t *ev;
  ev = (eh_statuscode_hash_ent_val_t *)x_malloc (sizeof (*ev));
  ev->phrase = phrase;
  ev->force_close = force_close;
  ev->body = NULL;
  return ev;
}

static void
eh_statuscode_hash_ent_val_free (eh_statuscode_hash_ent_val_t *ev)
{
  if (ev->body)
    x_free (ev->body);
  x_free (ev);
}

static void
eh_statuscode_hash_entry_free (eh_strhash_entry_t *ent)
{
  eh_statuscode_hash_ent_val_free ((eh_statuscode_hash_ent_val_t *)ent->val);
}

void
eh_statuscode_table_init (eh_statuscode_table_t *st, const char *path)
{
  size_t i;
  st->ht = eh_strhash_new (389, eh_statuscode_hash_entry_free);
  for (i = 0; i < sizeof(eh_sentry) / sizeof(eh_sentry[0]); i++) {
    eh_strhash_entry_t *ent;
    eh_statuscode_hash_ent_val_t *ev;
    ent = eh_strhash_find_create (st->ht, eh_sentry[i].code);
    ev = eh_statuscode_hash_ent_val_new (eh_sentry[i].phrase,
					 eh_sentry[i].force_close);
    if (path != NULL) {
      char *str;
      str = x_strdup_printf ("%s/%s.html", path, eh_sentry[i].code);
      ev->body = eh_str_load_file (str);
      if (ent->val == NULL) {
	eh_log (EH_LOG_WARNING, "failed to load %s", str);
      }
      x_free (str);
    }
    ent->val = ev;
  }
}

static void
str_free (void *str)
{
  x_free (str);
}

void
eh_connection_append_wvec_response_with_content_length (eh_connection_t *ec,
							eh_method_t method,
							const char *
							status_code,
							const char *headers,
							const char *body,
							size_t body_len,
							size_t content_length)
{
  eh_statuscode_hash_ent_val_t *ev;
  char *mess;
  size_t mess_len;
  int status;
  eh_strhash_t *ht;
  char *tmpbody = NULL;

  ht = ec->vserver_backref->statuscode_table.ht;
  status = atoi (status_code);
  
  ev = (eh_statuscode_hash_ent_val_t *)eh_strhash_find (ht, status_code, NULL);
  if (ev == NULL) {
    eh_log (EH_LOG_WARNING, "status code '%s' not found", status_code);
    ev = (eh_statuscode_hash_ent_val_t *)eh_strhash_find (ht, "500", NULL);
    content_length = 0;
    assert (ev);
  }
  
  if (ev->force_close ||
      ec->current_request.headers.predef.content_length) {
    ec->rconn = eh_rconn_close;
  }
  
  if (body == NULL) {
    if (ev->body) {
      body = ev->body;
      body_len = strlen (body);
    } else {
      char *str;
      str = eh_html_strdup_server_address (&ec->current_request);
      tmpbody = x_strdup_printf ("%s"
				 "<html><head>"
				 "<title>%s %s</title>"
				 "</head><body>"
				 "%s %s"
				 "<hr>"
				 "%s"
				 "</body></html>",
				 eh_html_get_doctype_decl (),
				 status_code, ev->phrase,
				 status_code, ev->phrase,
				 str);
      x_free (str);
      body = tmpbody;
      body_len = strlen (body);
    }
  }
  if (headers == NULL) {
    headers = "Content-Type: text/html\r\n";
  }
  /* only head method can override content_length */
  if (method != eh_method_head || content_length == 0) {
    content_length = body_len;
  }
  mess = x_strdup_printf ("HTTP/1.1 %s %s\r\n"
			  "Connection: %s\r\n"
			  "Content-Length: %lu\r\n"
			  "%s\r\n",
			  status_code, ev->phrase,
			  ec->rconn == eh_rconn_keepalive ?
			  "Keep-Alive" : "Close",
			  content_length, headers ? headers : "");
  mess_len = strlen (mess);
  if (method != eh_method_head) {
    mess = (char *)x_realloc (mess, mess_len + body_len);
    memcpy (mess + mess_len, body, body_len);
    eh_connection_append_wvec (ec, (void *)mess, mess_len + body_len,
			       str_free, mess);
    eh_accesslog_set_response_status (&ec->accesslog, status, body_len);
  } else {
    eh_connection_append_wvec (ec, (void *)mess, mess_len, str_free, mess);
    eh_accesslog_set_response_status (&ec->accesslog, status, 0);
  }
  if (tmpbody)
    x_free (tmpbody);
}

void
eh_connection_append_wvec_response (eh_connection_t *ec,
				    eh_method_t method,
				    const char *status_code,
				    const char *headers,
				    const char *body, size_t body_len)
{
  eh_connection_append_wvec_response_with_content_length (ec, method,
							  status_code,
							  headers, body,
							  body_len, body_len);
}
