/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

int
eh_normalize_path (char *path)
{
  char *rp, *wp;
  int bad;
  rp = path;
  if (rp[0] != '/')
    return 1;
  for (bad = 0, wp = ++rp; *rp; ) {
    int n;
    n = strcspn (rp, "/");
    if (n == 0) {
      rp++;
    } else if (rp[0] == '.') {
      if (n == 1) {
	/* we found xxx/./yyy */
	rp += 2;
      } else {
	bad = 1;
	break;
      }
    } else {
      memcpy (wp, rp, n);
      rp += n;
      wp += n;
      if (*rp == '/')
	*wp++ = *rp++;
    }
  }
  *wp = '\0';
  return bad;
}

static int
eh_xdigit2char (int c)
{
  if (c >= '0' && c <= '9')
    return c - '0';
  if (c >= 'a' && c <= 'f')
    return c - 'a' + 10;
  if (c >= 'A' && c <= 'F')
    return c - 'A' + 10;
  return -1;
}

int
eh_substr (const char *sub, const char *str)
{
  return strncmp (sub, str, strlen (sub)) == 0;
}

int
eh_str_is_member (const char *memb, const char *str, const char *delim)
{
  const char *p, *q;
  int memb_len;
  memb_len = strlen (memb);
  eh_for_p_in_str (str, delim, p, q) {
    if (memb_len == q - p && memcmp (memb, p, memb_len) == 0)
      return 1;
  }
  return 0;
}

static char *
eh_get_token (const char **str)
{
  int len;
  char *r;
  if (**str != '\"') {
    len = strcspn (*str, " \t");
    r = x_strndup (*str, len);
    *str += len;
  } else {
    const char *rp;
    char *wp;
    (*str)++;
    r = (char *)x_malloc (strlen (*str) + 1);
    for (rp = *str, wp = r; *rp; rp++, wp++) {
      if (*rp == '\\' && (rp[1] == '\\' || rp[1] == '\"')) {
	rp++;
      } else if (*rp == '\"') {
	rp++;
	break;
      }
      *wp = *rp;
    }
    *wp = '\0';
    r = (char *)x_realloc (r, wp - r + 1);
    *str = rp;
  }
  *str += strspn (*str, " \t");
  return r;
}

char **
eh_get_token_arr (const char *str)
{
  int num_str = 0;
  char **strarr;
  strarr = (char **)x_malloc (sizeof (char *));
  strarr[0] = NULL;
  str += strspn (str, " \t");
  while (*str) {
    char *p;
    p = eh_get_token (&str);
    num_str++;
    strarr = (char **)x_realloc (strarr, (num_str + 1) * sizeof (char *));
    strarr[num_str - 1] = p;
    strarr[num_str] = NULL;
  }
  return strarr;
}

#if 0
char *
eh_find_end_of_header (char *buf, size_t len)
{
  char *p, *endp;
  p = buf;
  endp = buf + len;
  while (endp - p > 1) {
    char *q;
    q = memchr (p, '\n', endp - p - 1);
    if (q == NULL)
      return NULL;
    if (q[1] == '\r' &&
	q + 2 < endp && q[2] == '\n' &&
	q > buf && q[-1] == '\r') {
      q[2] = '\0';
      return q + 3;
    }
    if (q[1] == '\n') {
      q[1] = '\0';
      return q + 2;
    }
    p = q + 1;
  }
  return NULL;
}
#endif
     

#if 1
char *
eh_find_end_of_header (char *buf, size_t len)
{
  for (; len > 1; buf++, len--) {
    switch (buf[0]) {
    case '\r':
      if (len >= 4 && buf[1] == '\n' && buf[2] == '\r' && buf[3] == '\n') {
	buf[3] = '\0';
	return buf + 4;
      }
      /* we don't allow '\r' as CRLF */
#if 0
      if (buf[1] == '\r') {
	buf[1] = '\0';
	return buf + 2;
      }
#endif
      break;
    case '\n':
      if (buf[1] == '\n') {
	buf[1] = '\0';
	return buf + 2;
      }
    default:
      ;
    }
  }
  return NULL;
}
#endif

char **
eh_strarr_copy (const char **strarr)
{
  char **s;
  int i, n;
  for (n = 0; strarr[n]; n++);
  s = (char **)x_malloc ((n + 1) * sizeof (char *));
  for (i = 0; i < n; i++) {
    s[i] = x_strdup (strarr[i]);
  }
  s[n] = NULL;
  return s;
}

void
eh_strarr_free (char **strarr)
{
  char **p;
  for (p = strarr; *p; p++)
    x_free (*p);
  x_free (strarr);
}

char *
eh_str_load_file (const char *fname)
{
  struct stat sbuf;
  void *ptr;
  char *str;
  int fd;
  if (stat (fname, &sbuf) < 0)
    return NULL;
  if ((fd = open (fname, O_RDONLY)) < 0)
    return NULL;
  if ((ptr = mmap (NULL, sbuf.st_size, PROT_READ, MAP_SHARED, fd, 0))
      == (void *)-1) {
    close (fd);
    return NULL;
  }
  close (fd);
  str = x_strndup ((char *)ptr, sbuf.st_size);
  munmap (ptr, sbuf.st_size);
  return str;
}

int
eh_decode_url (char *str)
{
  char *rp, *wp;
  int bad;
  rp = strchr (str, '%');
  if (rp == NULL)
    return 0;
  for (bad = 0, wp = rp; *rp; rp++, wp++) {
    if (*rp != '%') {
      *wp = *rp;
    } else {
      int x1 = 0, x2 = 0, x3 = 0;
      if ((x1 = eh_xdigit2char (rp[1])) >= 0 &&
	  (x2 = eh_xdigit2char (rp[2])) >= 0 &&
	  (x3 = (x1 << 4) | x2) != '/') {
	*wp = x3;
	rp += 2;
      } else {
	bad = 1;
	*wp = *rp;
      }
    }
  }
  *wp = '\0';
  return bad;
}

char *
eh_strdup_encode_url (const char *str)
{
  char *r, *wp;
  int len, wlen;
  const char *rp;
  len = wlen = strlen (str);
  for (rp = str; *rp; rp++) {
    if (*rp == '&' || *rp == '<' || *rp == '>' || *rp == '\"')
      wlen += 2;
  }
  r = (char *)x_malloc (wlen + 1);
  for (rp = str, wp = r; *rp; rp++) {
    if ((*rp & 0x80) != 0 ||
	*rp == '&' || *rp == '<' || *rp == '>' || *rp == '\"') {
      snprintf (wp, 4, "%%%02x", *rp);
      wp += 3;
    } else {
      *wp++ = *rp;
    }
  }
  assert (wlen == (wp - r));
  *wp = '\0';
  return r;
}

int
eh_parse_sizestr (const char *str, size_t *val)
{
  size_t x, y;
  const char *p;
  for (x = 0, p = str; *p >= '0' && *p <= '9'; p++) {
    y = x;
    x *= 10;
    if (x < y)
      return -1; /* overflow */
    y = x;
    x += (*p - '0');
    if (x < y)
      return -1; /* overflow */
  }
  *val = x;
  return p - str;
}

int
eh_is_valid_id_string (const char *str)
{
  const char *p;
  for (p = str; *p; p++) {
    char c = *p;
    if (c >= '0' && c <= '9')
      continue;
    if (c >= 'a' && c <= 'z')
      continue;
    if (c >= 'A' && c <= 'Z')
      continue;
    if (c == '-' || c == '_')
      continue;
    return 0;
  }
  return 1;
}

char *
eh_strdup_escape_shell_chars (const char *str)
{
  char *ptr, *wp;
  const char *rp;
  ptr = (char *)x_malloc (strlen (str) * 2 + 1);
  for (rp = str, wp = ptr; *rp; *wp++ = *rp++) {
    switch (*rp) {
    case 10:
    case 34:
    case 36:
    case 38:
    case 39:
    case 40:
    case 41:
    case 42:
    case 59:
    case 60:
    case 62:
    case 63:
    case 91:
    case 92:
    case 93:
    case 94:
    case 96:
    case 123:
    case 124:
    case 125:
    case 126:
      *wp++ = '\\';
    default:
      ;
    }
  }
  *wp = '\0';
  return ptr;
}

int
eh_unescape_doublequotes (char *str)
{
  char *rp, *wp;
  if (str[0] != '\"') {
    wp = str + strcspn (str, " \t<>");
    if (*wp != '\0')
      *wp++ = '\0';
    return wp - str;
  }
  for (rp = str + 1, wp = str; *rp != '\0' && *rp != '\"'; rp++, wp++) {
    if (*rp == '\\' && (rp[1] == '\"' || rp[1] == '\\'))
      rp++;
    *wp = *rp;
  }
  if (*wp != '\0') {
    *wp++ = '\0';
  }
  if (*rp == '\"')
    rp++;
  return rp - str;
}

int
eh_strsplit_unescape_doublequotes (const char *str, ...)
{
  int num;
  char *s, *p;
  va_list ap;
  va_start (ap, str);
  s = x_strdup (str);
  p = s;
  num = 0;
  p += strspn (p, " \t");
  while (1) {
    int len;
    char **r;
    r = va_arg (ap, char **);
    if (r == NULL)
      break;
    len = eh_unescape_doublequotes (p);
    if (len == 0)
      break;
    *r = x_strdup (p);
    num++;
    p += len;
    p += strspn (p, " \t");
  }
  x_free (s);
  va_end (ap);
  return num;
}

char *
eh_strdup_escape_doublequotes (const char *str)
{
  char *ptr, *wp;
  const char *rp;
  ptr = (char *)x_malloc (strlen (str) * 2 + 1);
  for (rp = str, wp = ptr; *rp; *wp++ = *rp++) {
    switch (*rp) {
    case '\"':
    case '\\':
      *wp++ = '\\';
    default:
      ;
    }
  }
  *wp = '\0';
  return ptr;
}
