
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <netinet/tcp.h>
#include <netinet/in.h>
#include <openssl/ssl.h>
#include <openssl/err.h>

static SSL_CTX *ctx = NULL;
static SSL *ssl = NULL;
static int sd = -1;
static int fd = -1;

static void
ssl_error (const char *str, int fatal)
{
  unsigned long err;
  err = ERR_get_error ();
  fprintf (stderr, "%s: %s\n", str, ERR_error_string (err, NULL));
  if (fatal)
    exit (-1);
}

static void
syscall_error (const char *str, int fatal)
{
  perror (str);
  if (fatal)
    exit (-1);
}

static RSA *
tmp_rsa_callback (SSL *ssl, int is_export, int keylength)
{
  static RSA *rsa = NULL;
  if (keylength != 512)
    keylength = 1024;
  if (rsa)
    RSA_free (rsa);
  rsa = RSA_generate_key (keylength, RSA_F4, NULL, NULL);
  return rsa;
}

static void
init_ssl_context (void)
{
  SSL_library_init ();
  SSL_load_error_strings ();
  ctx = SSL_CTX_new (SSLv23_server_method ());
  if (ctx == NULL)
    ssl_error ("new", 1);
  if (!SSL_CTX_load_verify_locations (ctx, "/etc/esehttpd/ca.crt", NULL))
    ssl_error ("ca.crt", 0);
  else
    SSL_CTX_set_client_CA_list
      (ctx, SSL_load_client_CA_file ("/etc/esehttpd/ca.crt"));
  if (!SSL_CTX_use_certificate_file (ctx, "/etc/esehttpd/server.crt",
				     SSL_FILETYPE_PEM))
    ssl_error ("server.crt", 1);
  if (!SSL_CTX_use_PrivateKey_file (ctx, "/etc/esehttpd/server.key",
				    SSL_FILETYPE_PEM))
    ssl_error ("server.key", 1);
  SSL_CTX_set_tmp_rsa_callback (ctx, tmp_rsa_callback);
}

static void
create_socket (void)
{
  int v;
  struct sockaddr_in servaddr;
  if ((sd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
    syscall_error ("socket", 1);
  v = 1;
  setsockopt (sd, SOL_SOCKET, SO_REUSEADDR, &v, sizeof (v));
  memset (&servaddr, 0, sizeof (servaddr));
  servaddr.sin_addr.s_addr = htonl (INADDR_ANY);
  servaddr.sin_family = AF_INET;
  servaddr.sin_port = htons (443);
  if (bind (sd, (const struct sockaddr *)&servaddr, sizeof (servaddr)) < 0)
    syscall_error ("bind", 1);
  if (listen (sd, 10) < 0)
    syscall_error ("listen", 1);
}

static int
check_error (int r, int *want_read, int *want_write)
{
  int e;
  e = SSL_get_error (ssl, r);
  switch (e) {
  case SSL_ERROR_NONE:
    return 0;
  case SSL_ERROR_WANT_READ:
    *want_read = 1;
    fprintf (stderr, "WANT_READ\n");
    return 0;
  case SSL_ERROR_WANT_WRITE:
    fprintf (stderr, "WANT_WRITE\n");
    *want_write = 1;
    return 0;
  case SSL_ERROR_SYSCALL:
    syscall_error ("read/write", 0);
    return 1;
  default:
    ;
  }
  ssl_error ("read/write", 0);
  return 1;
}

static void
ssl_get_peer (void)
{
  X509 *peer;
  long errnum;
  if ((peer = SSL_get_peer_certificate (ssl)) != NULL) {
    fprintf (stderr, "SSL_get_peer_certificate returns %p\n", peer);
    if ((errnum = SSL_get_verify_result (ssl)) == X509_V_OK) {
      fprintf (stderr, "SSL_get_verify_result returns X509_V_OK\n");
    } else {
      fprintf (stderr, "SSL_get_verify_result failed: %s\n",
	       X509_verify_cert_error_string (errnum));
    }
  } else {
    fprintf (stderr, "SSL_get_peer_certificate returns NULL\n");
  }
}

static void
do_io (void)
{
  char rbuf[1024];
  char wbuf[1024];
  int wbuflen;
  int want_read, want_write;

  wbuflen = 0;
  fcntl (fd, F_SETFL, O_RDWR | O_NONBLOCK);
  SSL_set_accept_state (ssl);
  
  want_read = want_write = 0;
  while (1) {
    fd_set rfds, wfds;
    int r;
    FD_ZERO (&rfds);
    FD_ZERO (&wfds);
    if (want_read) {
      FD_SET (fd, &rfds);
    } else if (want_write) {
      FD_SET (fd, &wfds);
    } else if (wbuflen > 0) {
      FD_SET (fd, &wfds);
    } else {
      FD_SET (fd, &rfds);
    }
    if (wbuflen < 1024)
      FD_SET (0, &rfds);
    if (select (fd + 1, &rfds, &wfds, NULL, NULL) <= 0)
      continue;
    want_read = want_write = 0;
    if (FD_ISSET (0, &rfds)) {
      assert (wbuflen < 1024);
      r = read (0, wbuf + wbuflen, 1024 - wbuflen);
      if (r <= 0) {
	fprintf (stderr, "0: %d\n", r);
	return;
      }
      wbuflen += r;
      if (wbuflen == 2 && strncmp (wbuf, "R\n", 2) == 0) {
	SSL_set_verify (ssl, SSL_VERIFY_PEER | SSL_VERIFY_CLIENT_ONCE, NULL);
	SSL_renegotiate (ssl);
	r = SSL_do_handshake (ssl);
	fprintf (stderr, "SSL_do_handshake: %d\n", r);
	wbuflen = 0;
      }
    } else if (!SSL_is_init_finished (ssl)) {
      r = SSL_accept (ssl);
      fprintf (stderr, "SSL_accept: %d\n", r);
      if (r > 0) {
	ssl_get_peer ();
      } else if (r <= 0 && check_error (r, &want_read, &want_write)) {
	return;
      }
    } else if (wbuflen > 0) {
      r = SSL_write (ssl, wbuf, wbuflen);
      fprintf (stderr, "SSL_write: %d\n", r);
      if (r > 0) {
	memmove (wbuf, wbuf + wbuflen, wbuflen - r);
	wbuflen -= r;
      } else {
	if (check_error (r, &want_read, &want_write))
	  return;
      }
    } else {
      r = SSL_read (ssl, rbuf, 1024);
      fprintf (stderr, "SSL_read: %d\n", r);
      if (r > 0) {
	write (1, rbuf, r);
      } else if (r == 0) {
	fprintf (stderr, "EOF from client\n");
	return;
      } else {
	if (check_error (r, &want_read, &want_write))
	  return;
      }
    }
  }
}

static void
accept_loop (void)
{
  while (1) {
    struct sockaddr_in cliaddr;
    socklen_t clilen;
    fd = accept (sd, (struct sockaddr *)&cliaddr, &clilen);
    if (fd < 0) {
      syscall_error ("accept", 0);
      continue;
    }
    ssl = SSL_new (ctx);
    SSL_set_fd (ssl, fd);
    do_io ();
    fprintf (stderr, "close\n");
    SSL_free (ssl);
    close (fd);
  }
}

int main (void)
{
  init_ssl_context ();
  create_socket ();
  accept_loop ();
  return 0;
}
