/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

#define EH_FCACHE_MAX_DEFAULT 300

#define DEBUG1(x)
#define DEBUG2(x)
#define DEBUG3(x)

#ifdef DEBUG_FILECACHE_DELAYED_REMOVE
#define EH_FCACHE_MAX_DEFAULT 10
#define DEBUG3(x) x
#endif

static const unsigned long prime_list[] =
{
  53ul,         97ul,         193ul,       389ul,       769ul,
  1543ul,       3079ul,       6151ul,      12289ul,     24593ul,
  49157ul,      98317ul,      196613ul,    393241ul,    786433ul,
  1572869ul,    3145739ul,    6291469ul,   12582917ul,  25165843ul,
  50331653ul,   100663319ul,  201326611ul, 402653189ul, 805306457ul,
  1610612741ul, 3221225473ul, 4294967291ul
};

#ifdef DEBUG
static void
eh_filecache_assert_valid (eh_filecache_t *ef)
{
  eh_filecache_entry_t *ent;
  if (ef->first)
    assert (ef->first->prev == NULL);
  if (ef->last)
    assert (ef->last->next == NULL);
  assert ((ef->first == NULL) == (ef->last == NULL));
  if (ef->first) {
    for (ent = ef->first; ; ent = ent->next) {
      assert (ent);
      assert (ent->header_len != (size_t)-1);
      assert (ent->ef_backref == ef);
      if (ent->next)
	assert (ent->next->prev = ent);
      if (ent->prev)
	assert (ent->prev->next = ent);
      if (ent == ef->last)
	break;
    }
  }
}
#endif

static eh_filecache_entry_t *
eh_filecache_entry_new (eh_filecache_t *ef, const char *key_backref,
			char *malloced_header, size_t header_len,
			void *mapped_body, const struct stat *statbuf,
			time_t last_stat_call_time, const char *mime_type)
{
  eh_filecache_entry_t *ent;
  ent = (eh_filecache_entry_t *)x_malloc (sizeof (*ent));
  ent->header = malloced_header;
  ent->ef_backref = ef;
  ent->hash_key_backref = key_backref;
  ent->header_len = header_len;
  ent->body = mapped_body;
  ent->prev = NULL;
  ent->next = ef->first;
  ent->statbuf = *statbuf;
  ent->last_stat_call_time = last_stat_call_time;
  ent->mime_type = mime_type;
  ent->wvec_refcount = 0;
  ent->remove_delayed_flag = 0;
  if (ef->first) {
    ef->first->prev = ent;
  }
  ef->first = ent;
  if (ef->last == NULL) {
    ef->last = ent;
  }
  ef->num_ent++;
#ifdef DEBUG
  eh_filecache_assert_valid (ef);
#endif
  return ent;
}

static void
eh_filecache_remove_entry (eh_filecache_t *ef, eh_filecache_entry_t *ent)
{
  assert (ent->wvec_refcount == 0);
  assert (ent->body);
  assert (ent->header);
  x_free (ent->header);
  munmap (ent->body, ent->statbuf.st_size);
  ent->header_len = (size_t)-1;
  DEBUG1 (fprintf (stderr, "delete %p %s\n", ent, ent->hash_key_backref));
  x_free (ent);
}

void
eh_filecache_entry_wvec_ref (eh_filecache_entry_t *ent)
{
  ent->wvec_refcount++;
}

void
eh_filecache_entry_wvec_unref (void *ptr)
{
  eh_filecache_entry_t *ent = (eh_filecache_entry_t *)ptr;
  assert (ent->wvec_refcount);
  ent->wvec_refcount--;
  if (ent->remove_delayed_flag && ent->wvec_refcount == 0) {
    eh_filecache_t *ef;
    eh_filecache_entry_t *next, *prev;
    ef = ent->ef_backref;
    next = ent->next;
    prev = ent->prev;
    if (next)
      next->prev = prev;
    if (prev)
      prev->next = next;
    if (ef->remove_delayed_entries == ent)
      ef->remove_delayed_entries = next;
    eh_filecache_remove_entry (ef, ent);
    DEBUG3 (fprintf (stderr, "wvec_unref: removed %p\n", ent));
  }
}

static void
eh_filecache_maybe_remove (eh_filecache_t *ef,
			   eh_filecache_entry_t *ent)
{
  eh_filecache_entry_t *next, *prev;
  assert (ef->num_ent > 0);
  assert (ent->remove_delayed_flag == 0);
  next = ent->next;
  prev = ent->prev;
  if (next)
    next->prev = prev;
  if (prev)
    prev->next = next;
  if (ef->first == ent)
    ef->first = next;
  if (ef->last == ent)
    ef->last = prev;
  ef->num_ent--;
  if (ent->wvec_refcount == 0) {
    eh_filecache_remove_entry (ef, ent);
  } else {
    /* this entry is reffered by wvec and therefore we can't free it.
       move it to the 'remove_delayed_entries' list */
    DEBUG3 (fprintf (stderr, "delayed: inserted %p %s\n", ent,
		     ent->hash_key_backref));
    ent->hash_key_backref = NULL;
    ent->remove_delayed_flag = 1;
    next = ef->remove_delayed_entries;
    if (next)
      next->prev = ent;
    ef->remove_delayed_entries = ent;
    ent->prev = NULL;
    ent->next = next;
  }
#ifdef DEBUG
  eh_filecache_assert_valid (ef);
#endif
}

static void
eh_filecache_ht_on_destroy (eh_strhash_entry_t *hent)
{
  eh_filecache_entry_t *ent;
  ent = (eh_filecache_entry_t *)hent->val;
  eh_filecache_maybe_remove (ent->ef_backref, ent);
  x_free (hent->key);
}

static unsigned long
eh_filecache_htsize (int n)
{
  size_t i;
  for (i = 0; i < sizeof (prime_list) / sizeof (prime_list[0]); i++) {
    if (prime_list[i] > (unsigned long)n)
      return prime_list[i];
  }
  return prime_list[--i];
}

eh_filecache_t *
eh_filecache_new (int ent_max)
{
  eh_filecache_t *ef;
  unsigned long htsize;
  ent_max = ent_max > 0 ? ent_max : EH_FCACHE_MAX_DEFAULT;
  htsize = eh_filecache_htsize (ent_max);
  ef = (eh_filecache_t *)x_malloc (sizeof (*ef));
  ef->num_ent = 0;
  ef->ent_max = ent_max;
  ef->first = NULL;
  ef->last = NULL;
  ef->cent_ht = eh_strhash_new (htsize, eh_filecache_ht_on_destroy);
  eh_log (EH_LOG_INFO, "filecache size: %d, hash table size: %ul",
	  ent_max, htsize);
  ef->remove_delayed_entries = NULL;
  return ef;
}

void
eh_filecache_delete (eh_filecache_t *ef)
{
  eh_strhash_delete (ef->cent_ht);
  assert (ef->num_ent == 0);
  /* ef->remove_delayed_entries != NULL means (possibly) wvec is not freed
     yet */
  assert (ef->remove_delayed_entries == NULL);
  x_free (ef);
}

void
eh_filecache_ht_entry_dump (eh_strhash_entry_t *hent, void *data)
{
  eh_filecache_entry_t *ent;
  ent = (eh_filecache_entry_t *)hent->val;
  DEBUG1 (fprintf (stderr, "%s: %p\n", hent->key, ent->ef_backref));
}

eh_filecache_entry_t *
eh_filecache_find (eh_filecache_t *ef, const char *filename)
{
  eh_filecache_entry_t *ent, *prev, *next;
  ent = (eh_filecache_entry_t *)eh_strhash_find (ef->cent_ht, filename, NULL);
  if (ent == NULL) {
    eh_debug ("find: failed %s", filename);
    return NULL;
  }
#ifdef DEBUG
  if (ent->ef_backref != ef) {
    /* broken */
    eh_strhash_foreach (ef->cent_ht, eh_filecache_ht_entry_dump, NULL);
  }
#endif
  assert (ent->ef_backref == ef);
  eh_debug ("find: succeed %s", filename);
  assert (ent->ef_backref == ef);
  if (ent != ef->first) {
    prev = ent->prev;
    next = ent->next;
    if (prev)
      prev->next = next;
    if (next)
      next->prev = prev;
    if (ent == ef->last)
      ef->last = prev;
    ent->next = ef->first;
    ent->prev = NULL;
    ef->first = ent;
    if (ent->next)
      ent->next->prev = ent;
  }
#ifdef DEBUG
  eh_filecache_assert_valid (ef);
#endif
  return ent;
}

static void
eh_filecache_truncate (eh_filecache_t *ef, eh_filecache_entry_t *keepent)
{
  eh_filecache_entry_t *ent;
  for (ent = ef->last; ent != NULL && ef->num_ent > ef->ent_max; ) {
    eh_filecache_entry_t *ent_prev;
    ent_prev = ent->prev;
    if (ent != keepent && ent->wvec_refcount == 0)
      eh_strhash_remove (ef->cent_ht, ent->hash_key_backref);
    ent = ent_prev;
  }
#ifdef DEBUG
  eh_filecache_assert_valid (ef);
#endif
}

eh_filecache_entry_t *
eh_filecache_insert (eh_filecache_t *ef, const char *filename,
		     char *malloced_header, size_t header_len,
		     void *mapped_body, const struct stat *statbuf,
		     time_t last_stat_call_time, const char *mime_type)
{
  eh_strhash_entry_t *hent;
  eh_filecache_entry_t *ent;
  char *key;
  key = x_strdup (filename);
  hent = eh_strhash_find_create (ef->cent_ht, key);
  DEBUG2 (fprintf (stderr, "insert %s: %p\n", filename, hent));
  if (hent->key != key) {
    /* already exists. keep the old_ent in the 'remove_delayed_entries'
       list, and replace this hash entry with the new filecache_entry. */
    eh_filecache_entry_t *old_ent;
    DEBUG2 (fprintf (stderr, "  %s: already exists\n", key));
    x_free (key);
    key = hent->key;
    old_ent = (eh_filecache_entry_t *)hent->val;
    eh_filecache_maybe_remove (ef, old_ent);
  }
  ent = eh_filecache_entry_new (ef, key, malloced_header, header_len,
				mapped_body, statbuf, last_stat_call_time,
				mime_type);
  hent->val = ent;
  eh_filecache_truncate (ef, ent);
  assert (hent->key);
  assert (hent->val);
  return ent;
}

void
eh_filecache_entry_set_last_stat_call_time (eh_filecache_entry_t *ent,
					    time_t last_stat_call_time)
{
  ent->last_stat_call_time = last_stat_call_time;
}

int
eh_filecache_get_num_entries (eh_filecache_t *ef)
{
  int i;
  eh_filecache_entry_t *ent;
  for (i = 0, ent = ef->first; ent; ent = ent->next, i++);
  return i;
}

