/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define EH_DONT_PROHIBIT_MALLOC
#include "esehttpd.h"
#include "modbase.h"

typedef struct {
  eh_modbase *mb;
  eh_connection_t *ec; // reference
  eh_strbuf_t strbuf;
} eh_modbase_extdata_t;

static void
eh_modbase_set_header (const char *key, const char *val, void *data)
{
  map<string, string> *hp = (map<string, string> *)data;
  if (strcmp (key, "authorization") == 0 ||
      strcmp (key, "proxy-authorization") == 0)
    return;
  (*hp)[key] = val;
}

void eh_modbase_run (const char *buf, const char *bufend, void *data)
{
  char *s;
  eh_modbase_extdata_t *const em = (eh_modbase_extdata_t *)data;
  eh_modbase *const mb = em->mb;
  eh_request_t *const req = &em->ec->current_request;
  mb->body_begin = buf;
  mb->body_end = bufend;
  eh_headers_forall (&req->headers, eh_modbase_set_header, &mb->headers);
  mb->query_string = req->query_string ? req->query_string : "";
  mb->request_method = req->method_str;
  mb->server_name = req->econf_vhost_ref->servername ?
    req->econf_vhost_ref->servername : "localhost";
  s = x_strdup_printf ("HTTP/1.%d", req->http_version_minor);
  mb->server_protocol = s;
  x_free (s);
  unsigned long raddr = ntohl (em->ec->client_addr.sin_addr.s_addr);
  s = x_strdup_printf ("%lu.%lu.%lu.%lu",
		       (raddr >> 24) & 0xff,
		       (raddr >> 16) & 0xff,
		       (raddr >>  8) & 0xff,
		       (raddr >>  0) & 0xff);
  mb->remote_addr = s;
  x_free (s);
  mb->remote_user = req->remote_user ? req->remote_user : "";
  mb->request_uri = req->url;
  mb->script_filename = req->filename;
  mb->run ();
  s = x_strdup_printf ("%d", mb->response.status);
  if (mb->response.body.length () > 0)
    eh_connection_append_wvec_response (em->ec, req->method, s,
					mb->response.headers.c_str (),
					mb->response.body.data (),
					mb->response.body.length ());
  else
    eh_connection_append_wvec_response (em->ec, req->method, s,
					mb->response.headers.c_str (),
					NULL, 0);
  x_free (s);
}

static void
eh_modbase_do_reply (eh_modbase_extdata_t *em)
{
  eh_modbase_run (em->strbuf.buffer,
		  em->strbuf.buffer + em->strbuf.buffer_len,
		  em);
  eh_debug ("buffer_len = %d", em->strbuf.buffer_len);
  eh_connection_request_finish (em->ec);
}

static void
eh_modbase_on_read_reqbody (eh_rhandler_t *eh, const char *buf, size_t buflen)
{
  eh_modbase_extdata_t *em = (eh_modbase_extdata_t *)eh->extdata;
  assert (buflen <= eh->body_length_left);
  eh_debug ("");
  eh_strbuf_append (&em->strbuf, buf, buflen);
  eh->body_length_left = em->strbuf.read_limit;
  if (eh->body_length_left == 0) {
    eh_modbase_do_reply (em);
  }
}

static void
eh_modbase_on_delete (eh_rhandler_t *eh)
{
  eh_modbase_extdata_t *em = (eh_modbase_extdata_t *)eh->extdata;
  eh_debug ("");
  eh_strbuf_discard (&em->strbuf);
  delete em->mb;
  x_free (em);
  x_free (eh);
}

eh_rhandler_t eh_modbase_tmpl = {
  0, NULL, eh_modbase_on_read_reqbody, NULL, eh_modbase_on_delete,
};

eh_rhandler_t *
eh_rhandler_modbase_new (struct eh_connection_s *ec,
			 const eh_request_t *er,
			 void *rhfunc_data)
{
  eh_modbase *(*mod_new_func)(void) = (eh_modbase *(*)(void))rhfunc_data;
  int reqbody_len = 0;
  eh_rhandler_t *eh;
  eh_modbase_extdata_t *em;
  const char *s;
  eh_debug ("");
  s = er->headers.predef.content_length;
  if (s) {
    reqbody_len = atoi (s);
  }
  eh = (eh_rhandler_t *)x_malloc (sizeof (*eh));
  memcpy (eh, &eh_modbase_tmpl, sizeof (*eh));
  eh->body_length_left = reqbody_len;
  em = (eh_modbase_extdata_t *)x_malloc (sizeof (*em));
  em->ec = ec;
  em->mb = (*mod_new_func) ();
  eh_strbuf_init (&em->strbuf, reqbody_len);
  eh->extdata = (void *)em;
  
  if (reqbody_len == 0) {
    eh_modbase_do_reply (em);
    eh_modbase_on_delete (eh);
    return NULL;
  }
  return eh;
}

void eh_register_hmod (const char *name, eh_modbase *(*mod_new_func)(void),
		       int nofs)
{
  eh_register_rhandler (name, eh_rhandler_modbase_new, (void *)mod_new_func,
			nofs);
}


eh_modbase::eh_modbase ()
  : body_begin (0), body_end (0)
{
  response.status = 200;
}

eh_modbase::~eh_modbase ()
{
}

void eh_modbase::run ()
{
  response.status = 200;
  response.headers = "Content-Type: text/html\r\n";
  response.body = "<HTML><BODY>hello, world</BODY></HTML>";
}

static unsigned int eh_gethex (const char *ptr)
{
  int x1 = ptr[0], x2 = ptr[1];
  unsigned int val = 0;
  if (x1 >= '0' && x1 <= '9')
    val = x1 - '0';
  else if (x1 >= 'a' && x1 <= 'f')
    val = 10 + x1 - 'a';
  else if (x1 >= 'A' && x1 <= 'F')
    val = 10 + x1 - 'A';
  else
    return 256;
  val *= 16;
  if (x2 >= '0' && x2 <= '9')
    val += x2 - '0';
  else if (x2 >= 'a' && x2 <= 'f')
    val += 10 + x2 - 'a';
  else if (x2 >= 'A' && x2 <= 'F')
    val += 10 + x2 - 'A';
  else
    return 256;
  return val;
}

static string eh_modbase_unescape_form_data (const char *begin,
					     const char *end)
{
  string str;
  for (const char *p = begin; p != end; p++) {
    unsigned int val;
    if (p + 2 <= end && p[0] == '%' && (val = eh_gethex (p + 1)) <= 255) {
      str += val;
      p += 2;
    } else if (p[0] == '+') {
      str += ' ';
    } else {
      str += p[0];
    }
  }
  return str;
}

map <string, string>
eh_modbase::decode_form_data (const char *begin, const char *end)
{
  map <string, string> fdata;
  const char *p, *key_begin, *key_end;
  if (begin == NULL) {
    begin = body_begin;
    end = body_end;
  }
  if (begin == NULL || end == NULL)
    return fdata;
  for (p = begin, key_begin = p, key_end = NULL; ; p++) {
    if (p == end || *p == '&') {
      if (key_end) {
	string key, val;
	key = eh_modbase_unescape_form_data (key_begin, key_end);
	val = eh_modbase_unescape_form_data (key_end + 1, p);
	fdata[key] = val;
      }
      if (p == end)
	break;
      key_begin = p + 1;
      key_end = NULL;
    } else if (*p == '=') {
      key_end = p;
    }
  }
  return fdata;
}

string
eh_modbase::escape_html (const string& str)
{
  char *p = eh_html_strdup_escape (str.c_str ());
  string s = p;
  x_free (p);
  return s;
}

REGISTER_HMOD("modbase", eh_modbase, 1);
