/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Softare
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

#ifdef USE_ONESIG
/* copied from linux/fcntl.h */
#define F_LINUX_SPECIFIC_BASE   1024
#define F_SETAUXFL      (F_LINUX_SPECIFIC_BASE+3)
#define F_BULK          (F_LINUX_SPECIFIC_BASE+5)
#define O_ONESIGFD      (2<<17)
#endif

static eh_fd_multiplexing_method_t method = eh_fd_multiplexing_method_poll;
static int rtsignum = -1;
static pid_t my_pid = -1;

/* file descriptors table */
static int ef_fdmax = 0;
static eh_fd_t **ef_table = NULL;

/* 'active_fds' is the queue of fds that have delayed events.
   eh_fd_request_events() itself never call event handlers, so that we
   avoid reentrance into event handlers. */
static int *active_fds = NULL;
static int active_fds_len = 0;
static int active_fds_alloclen = 0;

/* set by signal handlers */
static int got_signal_mask;
enum {
  EH_SIGNAL_MASK_INT = 1,
  EH_SIGNAL_MASK_HUP = 2,
  EH_SIGNAL_MASK_ALRM = 4,
  EH_SIGNAL_MASK_CHLD = 8,
  EH_SIGNAL_MASK_IO = 16,
};

static int shutdown_is_in_progress = 0;

static time_t current_time = 0;
static time_t next_timer_event = 0;
static char current_time_string[81] = "";
static char current_time_string_logformat[81] = "";

#ifdef USE_ONESIG
static int disable_fcntl_setauxfl = 0;
static int disable_fcntl_bulk = 0;
#endif

static int open_count = 0;

static void (*eh_fd_on_childexit)(pid_t pid, int status, void *data) = NULL;
static void *eh_fd_on_childexit_data = NULL;
static void (*eh_fd_global_timer_callback)(void *data) = NULL;
static void *eh_fd_global_timer_callback_data = NULL;

static int eh_fd_status_code = -1;

#ifdef DEBUG
#define DEBUG_DO(x) x
#else
#define DEBUG_DO(x)
#endif

#define DEBUG_LEAKED_RTSIGNAL(x)

/* the following eh_debug calls are only for debugging. eh_debug is not
   signal-handler safe. */

static void on_sigchld (int x)
{
  eh_debug ("SIGCHLD");
  got_signal_mask |= EH_SIGNAL_MASK_CHLD;
}

static void on_sigint (int x)
{
  eh_debug ("SIGINT");
  got_signal_mask |= EH_SIGNAL_MASK_INT;
}

static void on_sigalrm (int x)
{
  eh_debug ("SIGALRM");
  got_signal_mask |= EH_SIGNAL_MASK_ALRM;
}

static void on_sighup (int x)
{
  eh_debug ("SIGHUP");
  got_signal_mask |= EH_SIGNAL_MASK_HUP;
}

static void on_rtsignal (int x)
{
  /* if this handler is called, the signal block is broken somewhere
     (e.g., cgiruby). we need to do poll() to resume. */
  eh_debug ("SIGIO/rtsignal");
  DEBUG_LEAKED_RTSIGNAL (eh_log (EH_LOG_INFO, "LEAKED RTSIGNAL"));
  got_signal_mask |= EH_SIGNAL_MASK_IO;
}

static void all_signals_mask (sigset_t *sset)
{
  assert(rtsignum > 0);
  sigemptyset (sset);
  if (method == eh_fd_multiplexing_method_rtsignal)
    sigaddset (sset, rtsignum);
  sigaddset (sset, SIGIO);
  sigaddset (sset, SIGALRM);
  sigaddset (sset, SIGCHLD);
  sigaddset (sset, SIGINT);
  sigaddset (sset, SIGTERM);
  sigaddset (sset, SIGQUIT);
  sigaddset (sset, SIGHUP);
}

static void got_signal_mask_lock ()
{
  /* since got_signal_mask is modified by signal handlers, we need to
     block these signals when we modify it */
  sigset_t sset;
  all_signals_mask(&sset);
  sigprocmask (SIG_BLOCK, &sset, NULL);
}

static void got_signal_mask_unlock ()
{
  sigset_t sset;
  /* when we use rtsignals, these signals must always be blocked. hence
     we do nothing then. */
  if (method == eh_fd_multiplexing_method_rtsignal)
    return;
  all_signals_mask(&sset);
  sigprocmask (SIG_UNBLOCK, &sset, NULL);
}

static void eh_fd_update_time (void)
{
  current_time = time (NULL);
  eh_rfc1123_date (current_time_string, 80, &current_time);
  strftime (current_time_string_logformat, 80, "%d/%b/%Y:%H:%M:%S %z",
	    localtime (&current_time));
}

static void eh_fd_init_timer (void)
{
  eh_fd_update_time ();
  next_timer_event = current_time + 10; /* 10 sec. */
}

static void
eh_fd_try_waitpid (void)
{
  int status;
  pid_t pid;
  while ((pid = waitpid (-1, &status, WNOHANG)) > 0) {
    eh_debug ("waitpid: pid = %d", pid);
    if (WIFEXITED (status)) {
      eh_log (EH_LOG_DEBUG, "process %d exited with %d",
	      pid, WEXITSTATUS(status));
    } else if (WIFSIGNALED (status)) {
      eh_log (EH_LOG_WARNING, "process %d died with signal %d",
	      pid, WTERMSIG(status));
    }
    if (eh_fd_on_childexit) {
      (*eh_fd_on_childexit) (pid, status, eh_fd_on_childexit_data);
    }
  }
}

static void eh_fd_graceful_shutdown (int status)
{
  int i;
  eh_log (EH_LOG_INFO, "graceful shutdown");
  for (i = 0; i < ef_fdmax; i++) {
    eh_fd_t *ef = ef_table[i];
    if (ef && ef->on_timer)
      (*ef->on_timer)(ef, shutdown_is_in_progress);
  }
  if (eh_fd_status_code < 0)
    eh_fd_status_code = status;
}

void eh_fd_schedule_graceful_shutdown (int status)
{
  if (eh_fd_status_code < 0)
    eh_fd_status_code = status;
  /* behave as if SIGHUP is received */
  got_signal_mask_lock();
  got_signal_mask |= EH_SIGNAL_MASK_HUP;
  got_signal_mask_unlock();
}

static void eh_fd_on_timer (void)
{
  eh_debug ("");
  eh_fd_update_time ();
  if (current_time >= next_timer_event) {
    int nfds = 0;
    int i;
    next_timer_event = current_time + 10;
    for (i = 0; i < ef_fdmax; i++) {
      eh_fd_t *ef = ef_table[i];
      if (ef) {
	nfds++;
	if (ef->on_timer)
	  (*ef->on_timer)(ef, shutdown_is_in_progress);
      }
    }
    /* eh_debug ("current_time = %ld, nfds = %d", current_time, nfds); */
  }
  if (eh_fd_global_timer_callback)
    (*eh_fd_global_timer_callback)(eh_fd_global_timer_callback_data);
  /* it's possible that we miss a SIGCHLD when signals are rushed. */
  eh_fd_try_waitpid ();
}

time_t
eh_fd_get_time (void)
{
  return current_time;
}

const char *
eh_fd_get_time_string (void)
{
  if (current_time_string[0] == '\0') {
    eh_fd_update_time ();
  }
  return current_time_string;
}

int
eh_fd_get_open_count (void)
{
  return open_count;
}

const char *
eh_fd_get_time_string_logformat (void)
{
  if (current_time_string_logformat[0] == '\0') {
    eh_fd_update_time ();
  }
  return current_time_string_logformat;
}

void
eh_fd_block_rtsignal (void)
{
  if (method == eh_fd_multiplexing_method_rtsignal) {
    sigset_t sset;
    all_signals_mask(&sset);
    sigprocmask (SIG_BLOCK, &sset, NULL);
  }
}

void
eh_fd_init (int force_to_use_poll)
{
  struct sigaction sact;
  memset (&sact, 0, sizeof (sact));
  
#ifdef HAVE_LINUX_RTSIG
  if (!force_to_use_poll)
    method = eh_fd_multiplexing_method_rtsignal;
#endif

  rtsignum = SIGRTMIN + 1;
  eh_fd_block_rtsignal ();
  
  if (method == eh_fd_multiplexing_method_rtsignal) {
    sact.sa_handler = on_rtsignal;
    sact.sa_flags = SA_RESTART;
    all_signals_mask(&sact.sa_mask);
    sigaction (rtsignum, &sact, NULL);
    sigaction (SIGIO, &sact, NULL);
    eh_log (EH_LOG_INFO, "using linux realtime signals");
  } else {
    eh_log (EH_LOG_INFO, "using the standard poll method");
  }
  
  sact.sa_handler = on_sigchld;
  sact.sa_flags = SA_RESTART;
  all_signals_mask(&sact.sa_mask);
  sigaction (SIGCHLD, &sact, NULL);
  
  sact.sa_handler = on_sigint;
  sact.sa_flags = SA_RESTART;
  all_signals_mask(&sact.sa_mask);
  sigaction (SIGINT, &sact, NULL);
  sigaction (SIGQUIT, &sact, NULL);
  sigaction (SIGTERM, &sact, NULL);
  
  sact.sa_handler = on_sigalrm;
  sact.sa_flags = SA_RESTART;
  all_signals_mask(&sact.sa_mask);
  sigaction (SIGALRM, &sact, NULL);

  sact.sa_handler = on_sighup;
  sact.sa_flags = SA_RESTART;
  all_signals_mask(&sact.sa_mask);
  sigaction (SIGHUP, &sact, NULL);

  signal (SIGPIPE, SIG_IGN);
  /* signal (SIGSEGV, SIG_DFL); */

  eh_fd_init_timer ();
  my_pid = getpid ();
}

eh_fd_multiplexing_method_t
eh_fd_get_multiplexing_method (void)
{
  return method;
}

void
eh_fd_getpid (void)
{
  my_pid = getpid ();
}

int
eh_fd_delegate (pid_t pid, sigval_t sval)
{
  int r;
  r = sigqueue (pid, rtsignum, sval);
#ifdef DEBUG_RTSIG
  if (r < 0) {
    eh_log_perror (EH_LOG_WARNING, "sigqueue");
    eh_log (EH_LOG_WARNING, "sigqueue failed (dest pid=%d)", pid);
  }
#endif
  return r;
}

static void
eh_fd_move_to_active (eh_fd_t *ef)
{
  if (active_fds_len >= active_fds_alloclen - 1) {
    active_fds_alloclen += 32;
    active_fds = (int *)x_realloc (active_fds,
				   active_fds_alloclen * sizeof (int));
  }
  ef->active_flag = 1;
  active_fds[active_fds_len++] = ef->pfd.fd;
}

static void
eh_fd_resize (int newfd)
{
  if (newfd >= ef_fdmax) {
    int oldlen, i;
    oldlen = ef_fdmax;
    ef_fdmax = newfd + 1;
    ef_table = (eh_fd_t **)
      x_realloc (ef_table, ef_fdmax * sizeof (*ef_table));
    for (i = oldlen; i < ef_fdmax; i++)
      ef_table[i] = NULL;
  }
}

static void
eh_fd_table_set (eh_fd_t *ef)
{
  int fd = ef->pfd.fd;
  eh_debug ("%d: set", fd);
  eh_fd_resize (fd);
  ef_table[fd] = ef;
}

static void
eh_fd_table_unset (int fd)
{
  assert (ef_table[fd]);
  eh_debug ("%d: clear", fd);
  ef_table[fd] = NULL;
}

eh_fd_t *
eh_fd_new (int fd, void (*on_event)(eh_fd_t *ef),
	   void (*on_timer)(eh_fd_t *ef, int graceful_shutdown),
	   void (*on_delete)(eh_fd_t *ef),
	   void *user_data)
{
  eh_fd_t *ef;
  eh_debug ("");
  ef = (eh_fd_t *)x_malloc (sizeof (*ef));
  open_count++;
  ef->pfd.fd = fd;
  ef->pfd.events = 0;
  ef->pfd.revents = POLLIN | POLLOUT;
  ef->on_event = on_event;
  ef->on_timer = on_timer;
  ef->on_delete = on_delete;
  ef->user_data = user_data;
  ef->active_flag = 0;
  ef->show_status = NULL;
  eh_fd_table_set (ef);
  if (method == eh_fd_multiplexing_method_rtsignal) {
#ifdef USE_ONESIG
    if (!disable_fcntl_bulk) {
      unsigned long fcs[10];
      int i = 0;
      fcs[i++] = F_SETFL;
      fcs[i++] = O_RDWR | O_NONBLOCK | O_ASYNC;
      fcs[i++] = F_SETSIG;
      fcs[i++] = rtsignum;
      fcs[i++] = F_SETOWN;
      fcs[i++] = my_pid;
      fcs[i++] = F_SETAUXFL;
      fcs[i++] = O_ONESIGFD;
      fcs[i++] = F_BULK;             /* end mark */
      assert (i <= 10);
      errno = 0;
      i = fcntl (fd, F_BULK, fcs);
      if (i < 0) {
	if (errno == EINVAL) {
	  eh_log (EH_LOG_INFO, "F_BULK disabled");
	  disable_fcntl_bulk = 1;
	} else {
	  eh_log_perror (EH_LOG_INFO, "fcntl F_BULK");
	}
      }
    }
    if (disable_fcntl_bulk) {
      fcntl (fd, F_SETFL, O_RDWR | O_NONBLOCK | O_ASYNC);
      fcntl (fd, F_SETSIG, rtsignum);
      fcntl (fd, F_SETOWN, my_pid);
      if (!disable_fcntl_setauxfl) {
	if (fcntl (fd, F_SETAUXFL, O_ONESIGFD) < 0) {
	  if (errno == EINVAL) {
	    eh_log (EH_LOG_INFO, "F_SETAUXFL disabled");
	    disable_fcntl_setauxfl = 1;
	  }
	}
      }
    }
#else
    fcntl (fd, F_SETFL, O_RDWR | O_NONBLOCK | O_ASYNC);
    fcntl (fd, F_SETSIG, rtsignum);
    fcntl (fd, F_SETOWN, my_pid);
#endif
  } else {
    fcntl (fd, F_SETFL, O_RDWR | O_NONBLOCK | O_ASYNC);
  }
  eh_debug ("%d: events = %x, revents = %x", fd, 0, POLLIN | POLLOUT);
  return ef;
}

void
eh_fd_set_data (eh_fd_t *ef, void (*on_event)(eh_fd_t *ef),
		void (*on_timer)(eh_fd_t *ef, int graceful_shutdown),
		void (*on_delete)(eh_fd_t *ef),
		void *user_data)
{
  ef->on_event = on_event;
  ef->on_timer = on_timer;
  ef->on_delete = on_delete;
  ef->user_data = user_data;
}

void
eh_fd_reset_pid (eh_fd_t *ef)
{
  if (method == eh_fd_multiplexing_method_rtsignal) {
    fcntl (ef->pfd.fd, F_SETOWN, my_pid);
    eh_log (EH_LOG_INFO, "%d: pid=%d", ef->pfd.fd, my_pid);
  }
}

static int
eh_fd_has_masked_event (eh_fd_t *ef)
{
  return ((ef->pfd.events & ef->pfd.revents));
}

static void
eh_fd_free (eh_fd_t *ef)
{
  eh_debug ("%d: close", ef->pfd.fd);
  if (*ef->on_delete)
    (*ef->on_delete)(ef);
  close (ef->pfd.fd);
  eh_fd_table_unset (ef->pfd.fd);
  open_count--;
  x_free (ef);
}

static void
eh_fd_dispatch_events (eh_fd_t *ef)
{
  if (ef->on_event) {
    eh_debug ("%d: dispatch %x - %x", ef->pfd.fd, ef->pfd.events,
	      ef->pfd.revents);
    (*(ef->on_event))(ef);
  } else {
    eh_fd_free (ef);
  }
}

void
eh_fd_delete_request (eh_fd_t *ef)
{
  ef->on_event = NULL;
  if (!ef->active_flag)
    eh_fd_move_to_active (ef);
}

void
eh_fd_request_events (eh_fd_t *ef, short events)
{
  eh_fd_debug_trace (ef, 0x90000 + events);
  ef->pfd.events = events;
  eh_debug ("%d: events = %x (revents = %x)", ef->pfd.fd,
	    events, ef->pfd.revents);
  if (eh_fd_has_masked_event (ef) && !ef->active_flag)
    eh_fd_move_to_active (ef);
}

void
eh_fd_clear_revents (eh_fd_t *ef, short revents)
{
  revents &= ( POLLIN | POLLOUT);
  ef->pfd.revents &= ~revents;
  eh_debug ("%d: revents = %x (cleared %x)", ef->pfd.fd, ef->pfd.revents,
	    revents);
}

static int
eh_fd_poll (int timeout_msec)
{
  size_t nfds;
  int i, v;
  eh_fd_t *ef;
  struct pollfd *ufds;
  ufds = (struct pollfd *)x_alloca (ef_fdmax * sizeof (*ufds));
  for (i = 0, nfds = 0; i < ef_fdmax; i++) {
    if (ef_table[i])
      ufds[nfds++] = ef_table[i]->pfd;
  }
  v = poll (ufds, nfds, timeout_msec);
  if (v > 0) {
    int j;
    for (i = 0, j = 0; i < ef_fdmax; i++) {
      if (ef_table[i]) {
	eh_fd_debug_trace (ef_table[i], 0x0 + ufds[j].revents);
        /* we must do or'ing because ufds[].revents is masked by
         * ufds[].events. remember that this function may be called
         * from eh_fd_resume_rtsignal_method(). */
	ef_table[i]->pfd.revents |= ufds[j++].revents;
      }
    }
    assert (j == nfds);
    for (i = 0; i < ef_fdmax; i++) {
      ef = ef_table[i];
      if (ef == NULL)
	continue;
      if (ef->pfd.revents & (POLLHUP | POLLERR)) {
	eh_debug ("POLLHUP | POLLERR");
	ef->pfd.revents |= POLLIN | POLLOUT;
      }
      if ((ef->pfd.revents & ef->pfd.events)) {
	eh_fd_dispatch_events (ef);
      }
    }
  }
  x_alloca_free (ufds);
  return v;
}

static void
eh_fd_check_without_polling_one (void)
{

  eh_fd_t *ef;
  int i;
  int len;
  int *active_fds_prev;
  len = active_fds_len;
  active_fds_prev = (int *)x_alloca (len * sizeof (int));
  memcpy (active_fds_prev, active_fds, len * sizeof (int));
  active_fds_len = 0;
  for (i = 0; i < len; i++) {
    int fd;
    fd = active_fds_prev[i];
    assert (fd >= 0 && fd < ef_fdmax);
    ef = ef_table[fd];
    if (ef) {
      ef->active_flag = 0;
      eh_fd_debug_trace (ef, 0x10000 + ef->pfd.revents);
      eh_fd_dispatch_events (ef);
    }
  }
  x_alloca_free (active_fds_prev);
}

static void
eh_fd_check_without_polling (void)
{
  while (active_fds_len)
    eh_fd_check_without_polling_one ();
}

#if 0
static void
eh_fd_dump (void)
{
  int i;
  eh_fd_t *ef;
  for (i = 0; i < ef_fdmax; i++) {
    ef = ef_table[i];
    if (ef == NULL)
      continue;
    eh_log (EH_LOG_INFO, "%d: events %x revents %x",
	    ef->pfd.fd, ef->pfd.events, ef->pfd.revents);
  }
  for (i = 0; i < ef_active_len; i++) {
    ef = ef_active[i];
    if (ef == NULL)
      continue;
    eh_log (EH_LOG_INFO, "%d: active", ef->pfd.fd);
  }
}
#endif

void
eh_fd_set_childexit_callback (void (*on_childexit)(pid_t pid,
						   int status,
						   void *data),
			      void *data)
{
  eh_fd_on_childexit = on_childexit;
  eh_fd_on_childexit_data = data;
}

void
eh_fd_set_global_timer_callback (void (*global_timer_callback)(void *data),
				 void *data)
{
  eh_fd_global_timer_callback = global_timer_callback;
  eh_fd_global_timer_callback_data = data;
}

static int
eh_fd_examine_misc_signals (void)
{
  int retval = 0;
  int mask = got_signal_mask; /* always atomic in practice */
  int interesting_mask =
    EH_SIGNAL_MASK_INT |
    EH_SIGNAL_MASK_CHLD |
    EH_SIGNAL_MASK_HUP |
    EH_SIGNAL_MASK_ALRM;
  if ((mask & interesting_mask) == 0)
    return 0;
  got_signal_mask_lock();
  got_signal_mask &= ~interesting_mask;
  got_signal_mask_unlock();
  if ((mask & EH_SIGNAL_MASK_INT)) {
    eh_fd_status_code = 0;
    retval = 1;
  }
  if ((mask & EH_SIGNAL_MASK_CHLD)) {
    eh_fd_try_waitpid ();
  }
  if ((mask & EH_SIGNAL_MASK_HUP)) {
    shutdown_is_in_progress = 1;
    eh_fd_graceful_shutdown (0);
  }
  if ((mask & EH_SIGNAL_MASK_ALRM)) {
    eh_fd_on_timer ();
  }
  return retval;
}

static void
eh_fd_resume_rtsignal_method (void)
{
  /* resume from SIGIO or leaked rtsignal */
  struct timespec ts;
  siginfo_t si;
  sigset_t sset_rt;
  int sig;
  DEBUG_LEAKED_RTSIGNAL (eh_log (EH_LOG_INFO, "RESUME RTSIGNAL"));

  /* purge rtsignals */
  sigemptyset(&sset_rt);
  sigaddset(&sset_rt, rtsignum);
  ts.tv_sec = 0;
  ts.tv_nsec = 0;
  while ((sig = sigtimedwait (&sset_rt, &si, &ts)) > 0);

  /* do polling without blocking */
  eh_fd_poll (0);
}

void
eh_fd_loop_rtsignal (int timeout_msec)
{
  struct timespec ts;
  siginfo_t si;
  sigset_t sset;
  int sig;
  memset (&si, 0, sizeof (si));

  /* for safety, we block signals again */
  eh_fd_block_rtsignal ();

  all_signals_mask(&sset);

  /* do we need to poll here? */
  eh_fd_poll (0);

  while (open_count > 0) {
    if (got_signal_mask) {
      /* sigprocmask is overridden somewhere! we need to resume. */
      eh_fd_block_rtsignal ();
      eh_fd_resume_rtsignal_method ();
      if (eh_fd_examine_misc_signals ())
	goto loop_break;
      got_signal_mask_lock();
      got_signal_mask = 0;
      got_signal_mask_unlock();
    }

    /* examine all the pending events, before calling sigtimedwait. */
    eh_fd_check_without_polling ();
    
    ts.tv_sec = timeout_msec / 1000;
    ts.tv_nsec = (timeout_msec % 1000) * 1000;
    sig = sigtimedwait (&sset, &si, &ts);
    eh_debug ("sig = %d", sig);
    
    if (sig == rtsignum) {
      eh_fd_t *ef;
      short band;
      int fd;
      if (si.si_code == SI_QUEUE) {
	/* delegated */
	fd = si.si_int;
	band = POLLIN | POLLOUT;
      } else {
	fd = si.si_fd;
	band = si.si_band;
      }
      assert (fd > 0 && fd < ef_fdmax);
      ef = ef_table[fd];
      if (ef) {
	ef->pfd.revents |= band;
	eh_debug ("%d: band = 0x%x", ef->pfd.fd, band);
	eh_fd_debug_trace (ef, 0x20000 + band);
	if (ef->pfd.revents & (POLLHUP | POLLERR)) {
	  eh_fd_debug_trace (ef, 0x22200);
	  eh_debug ("POLLHUP | POLLERR %x", band);
	  ef->pfd.revents |= POLLIN | POLLOUT;
	}
	ef->pfd.revents &= ~(POLLHUP | POLLERR);
	if ((ef->pfd.revents & ef->pfd.events)) {
	  eh_fd_dispatch_events (ef);
	}
      } else {
	eh_debug ("ef == NULL");
      }
    } else if (sig == SIGIO) {
#ifdef DEBUG_RTSIG
      eh_log (EH_LOG_WARNING, "got SIGIO pid=%d", getpid ());
#endif
      eh_fd_resume_rtsignal_method ();
    } else if (sig == -1) {
      /* EAGAIN */
    } else {
      got_signal_mask_lock ();
      switch (sig) {
      case SIGINT:
      case SIGTERM:
      case SIGQUIT:
	got_signal_mask |= EH_SIGNAL_MASK_INT;
	break;
      case SIGHUP:
	got_signal_mask |= EH_SIGNAL_MASK_HUP;
	break;
      case SIGCHLD:
	got_signal_mask |= EH_SIGNAL_MASK_CHLD;
	break;
      case SIGALRM:
	got_signal_mask |= EH_SIGNAL_MASK_ALRM;
	break;
      }
      got_signal_mask_unlock ();
      if (eh_fd_examine_misc_signals ())
	goto loop_break;
    }
  }
 loop_break:
  ;
}

void
eh_fd_loop_poll (int timeout_msec)
{
  while (open_count > 0) {
    eh_fd_check_without_polling ();
    eh_fd_poll (timeout_msec);
    if (eh_fd_examine_misc_signals ()) {
      break;
    }
  }
}

void
eh_fd_start_timer (int timeout_msec)
{
  struct itimerval itval;

  itval.it_interval.tv_sec = timeout_msec / 1000;
  itval.it_interval.tv_usec = (timeout_msec % 1000) * 1000;
  itval.it_value.tv_sec = timeout_msec / 1000;
  itval.it_value.tv_usec = (timeout_msec % 1000) * 1000;
  setitimer (ITIMER_REAL, &itval, NULL);
}

int
eh_fd_check_and_dispatch_events (int timeout_msec)
{
  struct itimerval itval;

  shutdown_is_in_progress = 0;
  
  if (method == eh_fd_multiplexing_method_rtsignal)
    eh_fd_loop_rtsignal (timeout_msec);
  else
    eh_fd_loop_poll (timeout_msec);

  itval.it_interval.tv_sec = 0;
  itval.it_interval.tv_usec = 0;
  itval.it_value.tv_sec = 0;
  itval.it_value.tv_usec = 0;
  setitimer (ITIMER_REAL, &itval, NULL);
  return eh_fd_status_code;
}

#ifdef DEBUG_TRACE_EVENTS

typedef struct {
  size_t len;
  int *tr;
} dtent_t;
static dtent_t dtarr[32768];

void
eh_fd_debug_trace (eh_fd_t *ef, int val)
{
  int fd = ef->pfd.fd;
  dtent_t *dp;
  assert (fd >= 0 && fd < 32768);
  dp = &dtarr[fd];
  dp->len++;
  dp->tr = (int *)x_realloc (dp->tr, dp->len * sizeof (int));
  dp->tr[dp->len - 1] = val;
}

static void
eh_fd_debug_trace_dump (void)
{
  int i, j;
  dtent_t *dp;
  for (i = 0; i < 32768; i++) {
    const char *st;
    short events = 0;
    dp = &dtarr[i];
    if (i >= ef_fdmax)
      break;
    if (dp == NULL || dp->len < 1)
      continue;
    if (ef_table[i]) {
      st = "open";
      events = ef_table[i]->pfd.events;
    } else {
      st = "closed";
    }
    printf ("fd %d (%s %x):", i, st, events);
    j = dp->len > 100 ? dp->len - 100 : 0;
    for ( ; j < dp->len; j++) {
      printf (" %x", dp->tr[j]);
    }
    printf ("\n");
  }
  printf ("-\n");
}

#endif

void
eh_fd_free_all (void)
{
  int i;

#ifdef DEBUG_TRACE_EVENTS  
  eh_fd_debug_trace_dump ();
#endif
  for (i = 0; i < ef_fdmax; i++)
    if (ef_table[i])
      eh_fd_free (ef_table[i]);
  if (ef_table)
    x_free (ef_table);
  if (active_fds)
    x_free (active_fds);
  ef_table = NULL;
  ef_fdmax = 0;
#ifdef DEBUG_TRACE_EVENTS  
  for (i = 0; i < 32768; i++)
    dtarr[i].tr = x_realloc (dtarr[i].tr, 0);
#endif
}

void
eh_fd_set_show_status_callback (eh_fd_t *ef,
				void (*show_status)(eh_fd_t *ef, char **strp))
{
  ef->show_status = show_status;
}

void
eh_fd_show_status (char **strp, int my_fd)
{
  int i;
  for (i = 0; i < ef_fdmax; i++) {
    eh_fd_t *ef;
    if (i == my_fd)
      continue;
    ef = ef_table[i];
    if (ef == NULL)
      continue;
    x_append_printf (strp, "%5d: watchev: %c%c retev: %c%c(%x) act: %c ",
		     i,
		     (ef->pfd.events & POLLIN) ? 'I' : ' ',
		     (ef->pfd.events & POLLOUT) ? 'O' : ' ',
		     (ef->pfd.revents & POLLIN) ? 'I' : ' ',
		     (ef->pfd.revents & POLLOUT) ? 'O' : ' ',
		     ef->pfd.revents,
		     (ef->active_flag) ? 'Y' : 'N');
    /* try polling */
    {
      struct pollfd ufd;
      int v;
      ufd = ef->pfd;
      v = poll (&ufd, 1, 0);
      x_append_printf (strp, "pretev: %c%c ",
		       (ufd.revents & POLLIN) ? 'I' : ' ',
		       (ufd.revents & POLLOUT) ? 'O' : ' ');
    }
    if (ef->show_status) {
      (*ef->show_status)(ef, strp);
    }
    x_append_printf (strp, "\n");
  }
}
