/*************************************************************************
 *
 * Copyright 2009 by bBreak Systems.
 *
 * ExCella Core - Excelt@CJava痘p邽߂̋ʊ
 *
 * $Id: BookController.java 96 2009-05-29 07:48:02Z yuta-takahashi $
 * $Revision: 96 $
 *
 * This file is part of ExCella Core.
 *
 * ExCella Core is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * ExCella Core is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the COPYING.LESSER file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with ExCella Core.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0-standalone.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/
package org.bbreak.excella.core;

import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.bbreak.excella.core.exception.ExportException;
import org.bbreak.excella.core.exception.ParseException;
import org.bbreak.excella.core.exporter.book.BookExporter;
import org.bbreak.excella.core.exporter.sheet.SheetExporter;
import org.bbreak.excella.core.handler.ParseErrorHandler;
import org.bbreak.excella.core.listener.SheetParseListener;
import org.bbreak.excella.core.tag.TagParser;

/**
 * [NubN̉͂sNX
 * 
 * @since 1.0
 */
public class BookController {

    /** O */
    private static Log log = LogFactory.getLog( BookController.class);

    /** Excel2007̃t@C */
    public static final String XSSF_SUFFIX = ".xlsx";

    /** Excel2003ȑÕt@C */
    public static final String HSSF_SUFFIX = ".xls";

    /** Rg̃vtBbNX */
    public static final String COMMENT_PREFIX = "-";

    /** Ώۂ̃ubN */
    private Workbook workbook;

    /** V[g̈ꗗ */
    private List<String> sheetNames = new ArrayList<String>();

    /** p[T[̃Xg */
    private List<SheetTagParserInfo> tagParsers = new ArrayList<SheetTagParserInfo>();

    /** V[g̓CxgXi */
    private List<SheetListenerInfo> sheetListeners = new ArrayList<SheetListenerInfo>();

    /** ubN͌ʏo̓NX */
    private List<BookExporter> bookExporters = new ArrayList<BookExporter>();

    /** V[g͌ʏo̓NX */
    private List<SheetExporterInfo> sheetExporters = new ArrayList<SheetExporterInfo>();

    /** G[nhONX */
    private ParseErrorHandler errorHandler = null;

    /** ubNf[^ */
    private BookData bookData = new BookData();

    /**
     * RXgN^<BR>
     * t@Č`𔻒肵Workbook𐶐
     * 
     * @param filepath t@CpX
     * @throws IOException t@C̓ǂݍ݂Ɏsꍇ
     */
    public BookController( String filepath) throws IOException {
        if ( log.isInfoEnabled()) {
            log.info( filepath + "̓ǂݍ݂Jn܂");
        }
        if ( filepath.endsWith( XSSF_SUFFIX)) {
            // XSSF`
            workbook = new XSSFWorkbook( filepath);
        } else {
            // HSSF`
            FileInputStream stream = new FileInputStream( filepath);
            POIFSFileSystem fs = new POIFSFileSystem( stream);
            workbook = new HSSFWorkbook( fs);
            stream.close();
        }

        // V[g
        int numOfSheets = workbook.getNumberOfSheets();
        for ( int sheetCnt = 0; sheetCnt < numOfSheets; sheetCnt++) {
            String sheetName = workbook.getSheetName( sheetCnt);
            sheetNames.add( sheetName);
        }
    }

    /**
     * RXgN^
     * 
     * @param workbook Ώۂ̃ubN
     */
    public BookController( Workbook workbook) {
        this.workbook = workbook;
        // V[g
        int numOfSheets = workbook.getNumberOfSheets();
        for ( int sheetCnt = 0; sheetCnt < numOfSheets; sheetCnt++) {
            String sheetName = workbook.getSheetName( sheetCnt);
            sheetNames.add( sheetName);
        }
    }

    /**
     * ubNɊ܂܂SV[g(RgV[g)̉͂̎s
     * 
     * @throws ParseException p[XɎsꍇ
     * @throws IOException G[t@C݂̏Ɏsꍇ
     */
    public void parseBook() throws ParseException, ExportException {
        parseBook( null);
    }

    /**
     * ubNɊ܂܂SV[g(RgV[g)̉͂̎s
     * 
     * @param object BookControllerparseBook(), parseSheet()\bhA 
     *                SheetParserparseSheet\bhňnꍇ
     *                TagParser܂ňp鏈f[^
     * @throws ParseException p[XɎsꍇ
     * @throws ExportException o͏Ɏsꍇ
     */
    public void parseBook( Object data) throws ParseException, ExportException {
        bookData.clear();
        for ( String sheetName : sheetNames) {
            if ( sheetName.startsWith( COMMENT_PREFIX)) {
                continue;
            }
            // ͂̎s
            SheetData sheetData = parseSheet( sheetName, data);
            // ʂ̒ǉ
            bookData.putSheetData( sheetName, sheetData);
        }

        // o͏̎s
        for ( BookExporter exporter : bookExporters) {
            if ( exporter != null) {
                exporter.setup();
                try {
                    exporter.export( workbook, bookData);
                } finally {
                    exporter.tearDown();
                }
            }
        }
    }

    /**
     * _ł̉̓f[^̎擾
     * 
     * @return _ł̉̓f[^
     */
    public BookData getBookData() {
        return bookData;
    }

    /**
     * _łWorkbook̎擾
     * 
     * @return _łWorkbook
     */
    public Workbook getBook() {
        return workbook;
    }

    /**
     * V[gf[^̉
     * 
     * @param sheetName ͑Ώۂ̃V[g
     * @return V[g̉͌
     * @throws ParseException p[XɎsꍇ
     * @throws ExportException GNX|[gG[
     */
    public SheetData parseSheet( String sheetName) throws ParseException, ExportException {
        return parseSheet( sheetName, null);
    }

    /**
     * V[gf[^̉
     * 
     * @param sheetName ͑Ώۂ̃V[g
     * @param object BookControllerparseBook(), parseSheet()\bhA 
     *                SheetParserparseSheet\bhňnꍇ
     *                TagParser܂ňp鏈f[^
     * @return V[g̉͌
     * @throws ParseException p[XG[
     * @throws ExportException GNX|[gG[
     */
    public SheetData parseSheet( String sheetName, Object data) throws ParseException, ExportException {
        Sheet sheet = workbook.getSheet( sheetName);

        SheetData sheetData = null;
        try {
            SheetParser sheetParser = new SheetParser();

            if ( log.isInfoEnabled()) {
                log.info( "V[g[" + sheetName + "]̏Jn܂");
            }

            // ^Op[T̒ǉ
            for ( SheetTagParserInfo parserInfo : tagParsers) {
                String targetSheetName = parserInfo.getSheetName();
                if ( targetSheetName == null || targetSheetName.equals( sheetName)) {
                    sheetParser.addTagParser( parserInfo.getParser());
                }
            }

            // V[gOCxg̒ʒm
            for ( SheetListenerInfo listenerInfo : sheetListeners) {
                String targetSheetName = listenerInfo.getSheetName();
                if ( targetSheetName == null || targetSheetName.equals( sheetName)) {
                    listenerInfo.getListener().preParse( sheet, sheetParser);
                }
            }

            // V[g͏
            sheetData = sheetParser.parseSheet( sheet, data);

            // V[gCxg̒ʒm
            for ( SheetListenerInfo listnerInfo : sheetListeners) {
                String targetSheetName = listnerInfo.getSheetName();
                if ( targetSheetName == null || targetSheetName.equals( sheetName)) {
                    listnerInfo.getListener().postParse( sheet, sheetParser, sheetData);
                }
            }
        } catch ( ParseException ex) {
            if ( log.isWarnEnabled()) {
                log.warn( sheetName + "ɗO܂", ex);
            }

            // G[nh[ւ̒ʒm
            if ( errorHandler != null) {
                errorHandler.notifyException( workbook, sheet, ex);
            }
            throw ex;
        }

        // o͏̎s
        for ( SheetExporterInfo exporterInfo : sheetExporters) {
            String targetSheetName = exporterInfo.getSheetName();
            if ( targetSheetName == null || targetSheetName.equals( sheetName)) {
                SheetExporter exporter = exporterInfo.getExporter();
                exporter.setup();
                try {
                    exporter.export( sheet, sheetData);
                } finally {
                    exporter.tearDown();
                }
            }
        }
        return sheetData;
    }

    /**
     * ubNɊ܂܂V[g̈ꗗ擾(RgV[g܂)
     * 
     * @return
     */
    public List<String> getSheetNames() {
        return sheetNames;
    }

    /**
     * ^Op[T̒ǉ
     * 
     * @param parser ǉ^Op[T
     */
    public void addTagParser( TagParser<?> parser) {
        tagParsers.add( new SheetTagParserInfo( parser));
    }

    /**
     * ΏۃV[gwł̃^Op[T̒ǉ
     * 
     * @param sheetName ΏۃV[g
     * @param parser ǉ^Op[T
     */
    public void addTagParser( String sheetName, TagParser<?> parser) {
        tagParsers.add( new SheetTagParserInfo( sheetName, parser));
    }

    /**
     * w^Õ^Op[T폜
     * 
     * @param tag ^O
     */
    public void removeTagParser( String tag) {
        List<SheetTagParserInfo> removeList = new ArrayList<SheetTagParserInfo>();
        for (SheetTagParserInfo sheetTagParserInfo : tagParsers) {
            if (sheetTagParserInfo.getParser().getTag().equals( tag)) {
                removeList.add( sheetTagParserInfo);
            }
        }
        tagParsers.removeAll( removeList);
    }
    
    /**
     * ׂẴ^Op[T폜
     */
    public void clearTagParsers() {
        tagParsers.clear();
    }
    
    /**
     * V[gXi̒ǉ
     * 
     * @param listener ǉ郊Xi
     */
    public void addSheetParseListener( SheetParseListener listener) {
        sheetListeners.add( new SheetListenerInfo( listener));
    }

    /**
     * V[gXi̒ǉ
     * 
     * @param sheetName ΏۃV[g
     * @param listener ǉ郊Xi
     */
    public void addSheetParseListener( String sheetName, SheetParseListener listener) {
        sheetListeners.add( new SheetListenerInfo( sheetName, listener));
    }

    /**
     * SẴV[gXi폜
     */
    public void clearSheetParseListeners() {
        sheetListeners.clear();
    }
    
    /**
     * o͏NX̎擾
     * 
     * @return o͏NX
     */
    public List<BookExporter> getExporter() {
        return bookExporters;
    }

    /**
     * V[g͌ʏo̓NX̒ǉ
     * 
     * @param exporter ǉo̓NX
     */
    public void addSheetExporter( SheetExporter exporter) {
        sheetExporters.add( new SheetExporterInfo( exporter));
    }

    /**
     * V[g͌ʏo̓NX̒ǉ
     * 
     * @param sheetName ΏۃV[g
     * @param exporter ǉo̓NX
     */
    public void addSheetExporter( String sheetName, SheetExporter exporter) {
        sheetExporters.add( new SheetExporterInfo( sheetName, exporter));
    }

    /**
     * ׂẲ͌ʏo̓NX폜
     */
    public void clearSheetExporters() {
        sheetExporters.clear();
    }
    
    /**
     * G[nh̎擾
     * 
     * @return G[nh
     */
    public ParseErrorHandler getErrorHandler() {
        return errorHandler;
    }

    /**
     * G[nh̐ݒ
     * 
     * @return G[nh
     */
    public void setErrorHandler( ParseErrorHandler errorHandler) {
        this.errorHandler = errorHandler;
    }

    /**
     * ubNo͏NX̒ǉ
     * 
     * @param exporter ubNo͏NX
     */
    public void addBookExporter( BookExporter exporter) {
        bookExporters.add( exporter);
    }

    /**
     * SẴubNo͏NX폜
     */
    public void clearBookExporters() {
        bookExporters.clear();
    }
    
    /**
     * V[gƂ̃p[TێNX
     * 
     * @since 1.0
     */
    private class SheetTagParserInfo {

        /** V[g */
        private String sheetName = null;

        /** p[T */
        private TagParser<?> parser = null;

        /**
         * RXgN^
         * 
         * @param sheetName V[g
         * @param parser p[T
         */
        public SheetTagParserInfo( String sheetName, TagParser<?> parser) {
            this.sheetName = sheetName;
            this.parser = parser;
        }

        /**
         * RXgN^
         * 
         * @param parser p[T
         */
        public SheetTagParserInfo( TagParser<?> parser) {
            this.parser = parser;
        }

        public String getSheetName() {
            return sheetName;
        }

        public TagParser<?> getParser() {
            return parser;
        }
    }

    /**
     * V[gƂ̃XiێNX
     * 
     * @since 1.0
     */
    private class SheetListenerInfo {

        /** V[g */
        private String sheetName = null;

        /** Xi */
        private SheetParseListener listener = null;

        /**
         * RXgN^
         * 
         * @param sheetName V[g
         * @param listener Xi
         */
        public SheetListenerInfo( String sheetName, SheetParseListener listener) {
            this.sheetName = sheetName;
            this.listener = listener;
        }

        /**
         * RXgN^
         * 
         * @param listener Xi
         */
        public SheetListenerInfo( SheetParseListener listener) {
            this.listener = listener;
        }

        public String getSheetName() {
            return sheetName;
        }

        public SheetParseListener getListener() {
            return listener;
        }
    }

    /**
     * V[gƂExporterێNX
     * 
     * @since 1.0
     */
    private class SheetExporterInfo {

        /** V[g */
        private String sheetName = null;

        /** GNX|[^ */
        private SheetExporter exporter = null;

        /**
         * RXgN^
         * 
         * @param sheetName V[g
         * @param exporter GNX|[^
         */
        public SheetExporterInfo( String sheetName, SheetExporter exporter) {
            this.sheetName = sheetName;
            this.exporter = exporter;
        }

        /**
         * RXgN^
         * 
         * @param exporter GNX|[^
         */
        public SheetExporterInfo( SheetExporter exporter) {
            this.exporter = exporter;
        }

        public String getSheetName() {
            return sheetName;
        }

        public SheetExporter getExporter() {
            return exporter;
        }
    }
}
