<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.XML
 */
/**
 * @file XMLUtils.php
 * @brief XMLUtils.
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: XMLUtils.php 2 2007-07-11 10:37:48Z ishitoya $
 *
 */
require_once('ficus/lang/Unicode.php');
/**
 * @class Ficus_XMLUtils
 */
class Ficus_XMLUtils
{

    /**
     * @var $nameStartChars array.
     */
    static $nameStartChars;

    /**
     * @var $nameChars array.
     */
    static $nameChars;

    /**
     * @var $nCNameStartChars array.
     */
    static $nCNameStartChars;

    /**
     * @var $nCNameChars array.
     */
    static $nCNameChars;

    /**
     * Char range compare.
     *
     * @param $a array of char range.
     * @param $b array of char range.
     * @return int see strcmp.
     */
    static function rangeComp($a, $b) {
        if ($a[0] == $b[0]) {
            return 0;
        }
        return ($a[0] < $b[0]) ? -1 : 1;
    }

    /**
     * Get name start char.
     *
     * @return array of name start char ranges.
     */
    static function getNameStartChar() {
        if (!isset(self::$nameStartChars)) {
            self::$nameStartChars = array_merge(self::getNCNameStartChar(),
                                               array(array(0x3A, 0x3A)));   // :
            usort(self::$nameStartChars, array('Ficus_XMLUtils', 'rangeComp'));
        }
        return self::$nameStartChars;
    }

    /**
     * Get no colon name start char.
     *
     * @return array of no colon name start char ranges.
     */
    static function getNCNameStartChar() {
        if (!isset(self::$nCNameStartChars)) {
            self::$nCNameStartChars
                = array(
                    array(0x41, 0x5A),  // A-Z
                    array(0x5F, 0x5F),  // _
                    array(0x61, 0x7A),  // a-z
                    array(0xC0, 0xD6),
                    array(0xD8, 0xF6),
                    array(0xF8, 0x2FF),
                    array(0x370, 0x37D),
                    array(0x37F, 0x1FFF),
                    array(0x200C, 0x200D),
                    array(0x2070, 0x218F),
                    array(0x2C00, 0x2FEF),
                    array(0x3001, 0xD7FF),
                    array(0xF900, 0xFDCF),
                    array(0xFDF0, 0xFFFD),
                    array(0x10000, 0xEFFFF));
        }
        return self::$nCNameStartChars;
    }

    /**
     * Get name char.
     *
     * @return array of name char ranges.
     */
    static function getNameChar() {
        if (!isset(self::$nameChars)) {
            self::$nameChars = array_merge(self::getNCNameChar(),
                                          array(array(0x3A, 0x3A)));    // :
            usort(self::$nameChars, array('Ficus_XMLUtils', 'rangeComp'));
        }
        return self::$nameChars;
    }

    /**
     * Get no colon name char.
     *
     * @return array of no colon name char ranges.
     */
    static function getNCNameChar() {
        if (!isset(self::$nameStartChars)) {
            $nameStartChar = array (
                array(0x2D, 0x2D),  // -
                array(0x2E, 0x2E),  // .
                array(0x30, 0x39),  // 0-9
                array(0xB7, 0xB7),
                array(0x0300, 0x036F),
                array(0x203F, 0x2040)
            );
        self::$nCNameChars = array_merge(self::getNCNameStartChar(), $nameStartChar);
        usort(self::$nCNameChars, array('Ficus_XMLUtils', 'rangeComp'));
        }
        return self::$nCNameChars;
    }

    /**
     * Validate NCName.
     *
     * @param $utf string of utf.
     * @param $encoding string of unicode encoding name.
     * @return boolean true if validate.
     */
    static public function validateNCName($utf, $encoding = Ficus_Unicode::UTF8) {
        $nCNameCharRanges = self::getNCNameStartChar();
        $ret = '';
        for ($i = 0; $i < mb_strlen($utf, $encoding); $i++) {
            $c = mb_substr($utf, $i, 1, $encoding);
            // Check NCName char ranges.
            $unicode = Ficus_Unicode::getUnicode($c, $encoding);
            $outOfRange = true;
            foreach ($nCNameCharRanges as $crange) {
                if ($crange[0] <= $unicode && $unicode <= $crange[1]) {
                    $outOfRange = false;
                    break;
                } else if ($unicode < $crange[0]) {
                    break;
                }
            }
            if ($outOfRange) {
                return false;
            }
            $nCNameCharRanges = self::getNCNameChar();
        }
        return true;
    }

    /**
     * Encode URI.
     *
     * not implemented.
     *
     * @param $utf string of utf.
     * @param $encoding string of unicode encoding name.
     * @return string of encoded NCName.
     */
    static public function encodeURI($utf, $encoding = Ficus_Unicode::UTF8) {
        //$URICharRanges = self::getURIChar();
        $nCNameCharRanges = self::getNCNameStartChar();
        $ret = '';
        for ($i = 0; $i < mb_strlen($utf, $encoding); $i++) {
            $c = mb_substr($utf, $i, 1, $encoding);

            // Check NCName char ranges.
            $unicode = Ficus_Unicode::getUnicode($c, $encoding);
            $outOfRange = true;
            foreach ($nCNameCharRanges as $crange) {
                if ($crange[0] <= $unicode && $unicode <= $crange[1]) {
                    $outOfRange = false;
                    break;
                } else if ($unicode < $crange[0]) {
                    break;
                }
            }

            if ($outOfRange) {
                // encode.
                $hex = strtoupper(bin2hex($c));
                $hex = (strlen($hex) % 2 == 0) ? $hex : '0' . $hex;
                $c = '%' . join('%', str_split($hex, 2));
            }
            $ret .= $c;

            //$URICharRanges = self::getURIChar();
            $nCNameCharRanges = self::getNCNameChar();
        }
        return $ret;
    }

    /**
     * Decode NCName.
     *
     * Decoding indiscriminately.
     *
     * @param $nCName string of NCName.
     * @param $encoding string of unicode encoding name.
     * @return string of decoded NCName.
     */
    static public function decodeURI($nCName) {
        return rawurldecode($nCName);
    }
}
?>
