<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.cli.ffmpeg
 */
/**
 * @file FFMpegCreateThumbnail.php
 * @brief svn add
 * @author <a href="kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: FFMpegCreateThumbnail.php 2 2007-07-11 10:37:48Z ishitoya $
 *
 * executes ffmpeg abstract class
 */

require_once("ficus/exception/IllegalArgumentException.php");
require_once("ficus/cli/ffmpeg/FFMpeg.php");
require_once("ficus/media/video/VideoBean.php");
require_once("ficus/media/audio/AudioBean.php");


/**
 * @class Ficus_FFMpegCreateThumbnail
 */
class Ficus_FFMpegCreateThumbnail extends Ficus_FFMpeg
{
    const DEFAULT_HEIGHT = 160;
    const DEFAULT_WIDTH = 120;
    const DEFAULT_START = 0;
    const DEFAULT_DURATION = 1;
    const DEFAULT_FORMAT = Ficus_ImageConstants::FORMAT_JPEG;
    const DEFAULT_FILENAME_FORMAT = "{\$num}";
    
    /**
     * @var $overwrite boolean
     */
    protected $overwrite = false;

    /**
     * @var $width integer width
     */
    protected $width = self::DEFAULT_WIDTH;

    /**
     * @var $height integer height
     */
    protected $height = self::DEFAULT_HEIGHT;

    /**
     * @var $start
     */
    protected $start = self::DEFAULT_START;

    /**
     * @var $end
     */
    protected $end = null;

    /**
     * @var $offset
     */
    protected $offset = self::DEFAULT_START;

    /**
     * @var $num
     */
    protected $num = 0;

    /**
     * @var $duration
     */
    protected $duration = self::DEFAULT_DURATION;

    /**
     * @var $format
     */
    protected $format = self::DEFAULT_FORMAT;

    /**
     * @var $outputDirectory
     */
    protected $outputDirectory = null;

    /**
     * @var $outputFilenameFormat
     */
    protected $filenameFormat = self::DEFAULT_FILENAME_FORMAT;

    /**
     * execute method
     * @return array of result
     */
    protected function onExecute(){
        if($this->isEmptyEnd()){
            $command = new Ficus_FFMpegVideoInfo();
            $command->setFilename($this->filename());
            $command->execute();
            $this->setEnd($command->video()->length());
        }
        $offset = $this->start;
        $this->num = 0;
        while($offset < $this->end){
            $this->offset = $this->getTimeString($offset);
            $this->result = Ficus_Runtime::exec($this->createCommand());
            if($this->processResult() == false){
                return false;
            }
            $this->num++;
            $offset += $this->duration * 1000;
        }
        return true;
    }
    
    /**
     * create command
     * @return string ./ffmpeg -f build.xml
     */
    protected function createCommand(){
        $command = parent::createCommand();
        if($this->overwrite){
            $command = $this->addOption($command, self::OVERWRITE);
        }
        $size = $this->getSizeString($this->width, $this->height);
        $command = $this->addOption($command, self::FORMAT,
                                    self::FORMAT_IMAGE);
        $command = $this->addOption($command, self::OFFSET, $this->offset);
        $command = $this->addOption($command, self::VIDEO_FRAMES, 1);
        $command = $this->addOption($command, self::VIDEO_SIZE, $size);

        if(is_dir($this->outputDirectory) == false){
            mkdir($this->outputDirectory, 0755, true);
        }
        $filename = str_replace('{$num}', $this->num, $this->filenameFormat);
        $filename .= "." . $this->format;
        $filename = $this->outputDirectory . "\\$filename";

        $command = $this->addOption($command, $filename);
        return $command;
    }

    /**
     * process result
     * @return boolean true if no error
     */
    protected function processResult(){
        if($this->result === true){
            return true;
        }else if($this->result === false){
            return false;
        }
        $result = $this->getErrorStream();
        if(preg_match('/muxing overhead/', $result)){
            return true;
        }
        return false;
    }
}
?>
