<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file Assert.php
 * @brief Assert for php
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: Assert.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * Util Assert
 */
require_once("ficus/exception/AssertException.php");
require_once("ficus/exception/TypeMismatchException.php");
require_once("ficus/exception/IllegalArgumentException.php");
/**
 * @class Ficus_Assert
 */
class Ficus_Assert
{
    /**
     * type check.
     *
     * throw TypeMismatchException if $var not instance of type.
     * subclass instance not equal superclass.
     *
     * @param $type of class
     * @param $var of var
     * @throw Ficus_TypeMismatchException type mismatch.
     */
    public static function isType($var, $type) {
		if (get_class($var) != $type) {
            throw new Ficus_TypeMismatchException("$type: " . get_class($var));
        }
        return true;
    }

    /**
     * type check.
     *
     * throw TypeMismatchException if $var not a specified primitive type.
     *
     * @param $type of primitive
     * @param $var of var
     * @throw Ficus_TypeMismatchException type mismatch.
     */
    public static function isPrimitiveType($var, $type) {
		$assertFunction = "is_$type";
        if (function_exists($assertFunction)  === false){
			throw new Ficus_TypeMismatchException("php type $type was not found");
		} else if ($assertFunction($var) === false){
			throw new Ficus_TypeMismatchException("$type: " . gettype($var));
		}
        return true;
    }

    /**
     * type check.
     *
     * throw TypeMismatchException if $var not instance of type.
     *
     * @param $type of class
     * @param $var of var
     * @throw Ficus_TypeMismatchException type mismatch.
     */
    public static function isInstanceOf($var, $type) {
        if (!($var instanceof $type)) {
            throw new Ficus_TypeMismatchException("$type: " . get_class($var));
        }
        return true;
    }

    /**
     * type hinting check.
     *
     * throw TypeMismatchException if $var not instance of type.
     *
     * @param $type of class
     * @param $var of var
     * @throw Ficus_TypeMismatchException type mismatch.
     */
    public static function typeHinting($type, $var) {
        if (!($var instanceof $type)) {
            self::throwTypeMismatch($type, $var);
        }
        return true;
    }

    /**
     * type check.
     *
     * throw TypeMismatchException if $var not a specified primitive type.
     * But allow null.
     *
     * @param $type of primitive
     * @param $var of var
     * @throw Ficus_TypeMismatchException type mismatch.
     */
    public static function isPrimitiveTypeAllowNull($var, $type) {
        if(is_null($var)){
            return true;
        }
        
		$assertFunction = "is_$type";
        if (function_exists($assertFunction)  === false){
			throw new Ficus_TypeMismatchException("php type $type was not found");
		} else if ($assertFunction($var) === false){
			throw new Ficus_TypeMismatchException("$type: " . gettype($var));
		}
        return true;
    }

    /**
     * type check allow null value.
     *
     * throw TypeMismatchException if $var not instance of type.
     * subclass instance not equal superclass.
     *
     * @param $type of class
     * @param $var of var
     * @throw Ficus_TypeMismatchException type mismatch.
     */
    public static function isTypeAllowNull($var, $type) {
        if(is_null($var)){
            return true;
        }
        
		if (get_class($var) != $type) {
            throw new Ficus_TypeMismatchException("$type: " . get_class($var));
        }
        return true;
    }

    /**
     * array key exists
     *
     * throws TypeMismatchException when $array is not Array, and
     * throws IllegalArgumentException when $array key $key is not exists.
     *
     * @param $array Array of array
     * @param $key String key
     * @throw Ficus_TypeMismatchException
     * @throw Ficus_IllegalArgumentException
     */
    public static function isKeyExists($array, $key){
        Ficus_Assert::isPrimitiveType($array, "array");
        if(array_key_exists($key, $array) == false){
            throw new Ficus_IllegalArgumentException("$key is not exists in array");
        }
        if(is_null($array[$key])){
            throw new Ficus_IllegalArgumentException("$key exists in array, but is null");
        }
        return true;
    }
    
    /**
     * array key exists allow null
     *
     * throws TypeMismatchException when $array is not Array, and
     * throws IllegalArgumentException when $array key $key is not exists.
     *
     * @param $array Array of array
     * @param $key String key
     * @throw Ficus_TypeMismatchException
     * @throw Ficus_IllegalArgumentException
     */
    public static function isKeyExistsAllowNull($array, $key){
        Ficus_Assert::isPrimitiveType($array, "array");
        if(array_key_exists($key, $array) == false){
            throw new Ficus_IllegalArgumentException("$key is not exists in array");
        }
        return true;
    }
    
    /**
     * type hinting check allow null.
     *
     * throw TypeMismatchException if $var not instance of type.
     *
     * @param $type of class
     * @param $var of var
     * @throw Ficus_TypeMismatchException type mismatch.
     */
    public static function typeHintingAllowNull($type, $var) {
        if ($var != null && !($var instanceof $type)) {
            self::throwTypeMismatch($type, $var);
        }
        return true;
    }

    /**
     * throw TypeMismatchException
     *
     * @param $type of class
     * @param $var of var
     * @throw Ficus_TypeMismatchException type mismatch.
     */
    protected static function throwTypeMismatch($type, $var) {
        $backtrace = debug_backtrace();
        throw new Ficus_TypeMismatchException("Argument " . self::getType($var)
                                            . " mismatch $type " . " called by "
                                            . "{$backtrace[1]['file']}"
                                            . "({$backtrace[1]['line']})");
    }

    /**
     * get type or object type.
     *
     * @param $type mixed
     * @return string of type name or class name.
     */
    public static function getType($type) {
        if (gettype($type) == "object") {
            return get_class($type);
        } else {
            return gettype($type);
        }
    }

    /**
     * get backtrace.
     *
     * @param $level int backtrace level.
     * @return array of backtrace.
     */
    protected static function getBacktrace($level) {
        $backtrace = debug_backtrace();
        return $backtrace[$level+1];
    }

    /**
     * assert callback.
     *
     * @param $file string file name
     * @param $line int line number
     * @param $code string source code
     * @exception Ficus_AssertException assert.
     */
    public static function callback($file, $line, $code) {
        throw new Ficus_AssertException("$code in $file($line)");
    }

}
assert_options(ASSERT_ACTIVE, 1);
assert_options(ASSERT_WARNING, 0);
assert_options(ASSERT_CALLBACK, array("Ficus_Assert", "callback"));
?>
