<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file Class.php
 * @brief class utility class
 * @author <a href="mailto:kent@guarana.cc">ISHTIOYA Kentaro</a>
 * @version $Id: Class.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * class utility class
 */

/**
 * @class Ficus_Class
 */
class Ficus_Class
{
    const PREFIX_SEPARATOR = "_";
    const PACKAGE_SEPARATOR = ".";
    
    /**
     * @var $target object target object
     */
    private $target = null;

    /**
     * @var $classname string target object class name cache purpose
     */
    private $classname = null;
    
    /**
     * constructor
     * @param $target object target object
     */
    public function __construct($target){
        $this->setTarget($target);
    }

    /**
     * set target object
     * @param $target object target object
     */
    public function setTarget($target){
        if(is_object($target)){
            $this->target = $target;
            $this->classname = get_class($target);
        }else if(is_string($target)){
            $this->classname = $target;
        }else{
            throw
                new Ficus_IllegalArgumentException("target must be an object");
        }
    }
    
    /**
     * get prefix from object class name
     * @return string prefix
     */
    public function prefix(){
        $classname = $this->classname();
        $index = strpos($classname, self::PREFIX_SEPARATOR);
        return substr($classname, 0, $index + 1);
    }

    /**
     * get package name
     * @return string package name
     */
    public function package(){
        $index = strrpos($this->classname, self::PACKAGE_SEPARATOR);
        if($index !== false){
            return substr($this->classname, 0, $index);
        }else{
            return "";
        }
    }

    /**
     * get class name
     * @return string class name
     */
    public function shortname(){
        $classname = $this->classname();
        $index = strpos($classname, self::PREFIX_SEPARATOR);
        if($index !== false){
            return substr($classname, $index + 1);
        }else{
            return $classname;
        }
    }

    /**
     * get Target
     * @return object target object
     */
    public function target(){
        return $this->target;
    }
    
    /**
     * get class name
     * @return string class name
     */
    public function classname(){
        $index = strrpos($this->classname, self::PACKAGE_SEPARATOR);
        if($index !== false){
            return substr($this->classname, $index + 1);
        }else{
            return $this->classname;
        }
    }

    /**
     * class qname
     * @return string class qname
     */
    public function qname($prefix = null){
        if(is_null($prefix)){
            return $this->classname;
        }else{
            $classname = $this->package();
            if(empty($classname) == false){
                $classname .= self::PACKAGE_SEPARATOR;
            }
            return $classname . $prefix . ucfirst($this->classname());
        }
    }

    /**
     * file name
     * @param $ext string extension for file
     * @return string file name
     */
    public function filename($ext = ""){
        $classname = $this->package();
        if(empty($classname) == false){
            $classname .= self::PACKAGE_SEPARATOR;
        }
        $classname .= $this->shortname();
        return str_replace(self::PACKAGE_SEPARATOR, "/", $classname) . $ext;
    }

    /**
     * cast class
     * @param $src Object target object
     * @param $dest mixed string of classname or object
     * @return casted class
     */
    public function cast($src, $dest){
        if(($src instanceof Ficus_Serializable) &&
           is_object($dest) &&
           ($dest instanceof Ficus_Serializable)){
            $serialized = $src->serialize();
            $dest->unserialize($serialized);
            return $dest;
        }else if(is_string($dest) && class_exists($dest)){
            $serialized = serialize($src);
            $object = str_replace(get_class($src), $dest, $serialized);
            $retval = unserialize($object);
            if($retval == false){
                throw new Ficus_ClassCastException("Can not cast class " . get_class($src) . " to $dest.");
            }
            return $retval;
        }else{
            throw new Ficus_ClassCastException("Can not cast class. illegal argument.");
        }
    }
}
?>
