<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file ClassPathElement.php
 * @brief class path element
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: ClassPathElement.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * class path element
 */

/**
 * @class Ficus_ClassPathElement
 */
class Ficus_ClassPathElement
{
    const CACHE_RECURSIVE = "recursive";
    const CACHE_NORMAL = "normal";
    
    /**
     * @var $pattern Ficus_PathPattern pattern
     */
    private $pattern;
    
    /**
     * @var $dir string target dir
     */
    private $dir;
    
    /**
     * @var $paths array of path
     */
    private $paths = array();

    /**
     * @var $cache array of cache
     */
    private $cache = array();
    
    /**
     * construct class path element
     * @param $dir string root dir
     * @param $pattern Ficus_PathPattern pattern to apply
     */
    public function __construct($dir, $pattern = null){
        if(empty($dir)){
            throw new Ficus_IllegalArgumentException("dir is empty");
        }
        if(is_string($dir)){
            $dir = new Ficus_Dir($dir);
        }

        Ficus_Assert::isType($dir, "Ficus_Dir");
        Ficus_Assert::isTypeAllowNull($pattern, "Ficus_PathPattern");
        if(is_null($pattern)){
            $this->paths[] = $dir;
        }else{
            $dir->setPattern($pattern);
            $this->paths[] = $dir;
        }
        $this->pattern = $pattern;
    }

    /**
     * get paths
     * @return array of path
     */
    public function paths(){
        return $this->paths;
    }

    /**
     * search
     * @param $needle string path
     * @param $recursive boolean true to search recursive
     * @return array found paths
     */
    public function search($needle, $recursive = false){
        if($recursive){
            return $this->recursiveSearch($needle);
        }
        
        if(isset($this->cache[self::CACHE_NORMAL][$needle])){
            return $this->cache[self::CACHE_NORMAL][$needle];
        }
        $result = array();
        foreach($this->paths as $path){
            $path = $path . "/" . $needle;
            $path = Ficus_Dir::normalize($path);
            if(is_readable($path)){
                $result[] = $path;
            }
        }

        $this->cache[self::CACHE_NORMAL][$needle] = $result;
        return $result;
    }

    /**
     * recursive search
     * @param $needle string path
     * @return array found paths
     */
    public function recursiveSearch($needle){
        if(isset($this->cache[self::CACHE_RECURSIVE]) == false){
            $this->cache[self::CACHE_RECURSIVE] = array();
            foreach($this->paths as $path){
                $paths = $path->getList();
                $this->cache[self::CACHE_RECURSIVE]
                    = array_merge($this->cache[self::CACHE_RECURSIVE], $paths);
            }
        }
        if(isset($this->cache[self::CACHE_RECURSIVE][$needle])){
            return array($this->cache[self::CACHE_RECURSIVE][$needle]);
        }
        return array();
    }

    /**
     * to string
     * @return string
     */
    public function __toString(){
        return implode(",", $this->paths);
    }
}
?>
