<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file Runtime.php
 * @brief execute command line commands
 * @author <a href="kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: Runtime.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * utility to execute command line command
 */
require_once("ficus/lang/Assert.php");
require_once("ficus/exception/NotReadyException.php");
require_once("ficus/exception/NotSupportedException.php");
require_once("ficus/lang/PlatformInformation.php");
/**
 * @class Ficus_Runtime
 */
class Ficus_Runtime
{
    const ERROR = "error";

    /**
     * command search path
     */
    protected static $paths = array();

    /**
     * add path
     * @param $path string path
     */
    public static function addPath($path){
        self::$paths[] = $path;
    }

    /**
     * create env
     * @return array of env
     */
    private static function createEnv(){
        if(empty(self::$paths) == false){
            $path = implode(PATH_SEPARATOR, self::$paths);
            return array('PATH' => $path);
        }
        return null;
    }
    
    /**
     * execute command.
     * @param $command string command line command
     * @param $parameters array array of parameters
     * @return array of command output.
     * @throw Ficus_TypeMismatchException
     */
    public static function exec($command, $parameters = null){
        Ficus_Assert::isPrimitiveTypeAllowNull($parameters, "array");
        if(is_null($parameters) == false){
            $param = implode(' ', $parameters);
            $command .= ' ' . $param;
        }
        $specs = array(
            0 => array('pipe', 'r'),
            1 => array('pipe', 'w'),
            2 => array('pipe', 'w'));
        $env = self::createEnv();

        $command = str_replace("\\", "/", $command);
        $command = escapeshellcmd($command);
        $process = proc_open($command, $specs, $pipes, null, $env);

        if(is_resource($process) == false){
            throw new Ficus_NotReadyException("Process is not started. command is $command.");
        }        

        $output = stream_get_contents($pipes[1]);
        fclose($pipes[1]);
        $error = stream_get_contents($pipes[2]);
        fclose($pipes[2]);
        $output = explode("\n", $output);
        $output[self::ERROR] = $error;
        proc_close($process);

        return $output;
    }

    /**
     * execute command with input.
     * @param $command string command line command
     * @param $inputs array array of command line input
     * @param $parameters array array of parameters
     * @return array of command output.
     * @throw Ficus_TypeMismatchException
     * @throw Ficus_NotReadyException
     */
    public static function execWithInput($command, $inputs, $parameter = null){
        Ficus_Assert::isPrimitiveTypeAllowNull($inputs, "array");
        Ficus_Assert::isPrimitiveTypeAllowNull($command, "array");
        if(is_null($parameters) == false){
            $param = implode(' ', $parameters);
            $command .= ' ' . $param;
        }

        $specs = array(
            0 => array('pipe', 'r'),
            1 => array('pipe', 'w'),
            2 => array('pipe', 'w'));
        $env = self::createEnv();
        $process = proc_open($command, $specs, $pipes, null, $env);

        if(is_resource($process) == false){
            throw new Ficus_NotReadyException("Process is not started. command is $command.");
        }
        
        if(is_null($inputs) == false){
            foreach($inputs as $input){
                fwrite($pipes[0], $input);
                sleep(1);
            }
            fclose($pipes[0]);
        }
        
        $output = stream_get_contents($pipes[1]);
        $error = stream_get_contents($pipes[2]);
        $output = explode("\n", $output);
        $output[self::ERROR] = $error;
        fclose($pipes[1]);
        fclose($pipes[2]);
        proc_close($process);
        return $output;
    }

    /**
     * is command exists
     * @param $command string target command
     * @return boolean return true if command exists
     */
    public static function isCommandExists($command){
        if(Ficus_PlatformInformation::isWindows()){
            throw new Ficus_NotSupportedException("windows platform is not supported for this function.");
        }
        $output = self::exec("which " . $command);
        $output = implode(' ', $output);
        if(count($output) == 0 ||
           preg_match('/command not found/', $output) ||
           preg_match("/no $command/", $output)){
            return false;
        }else{
            return true;
        }
    }
}
?>
