<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file String.php
 * @brief String for php
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: String.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * String wrapper class.
 */

require_once('ficus/exception/MethodNotFoundException.php');

/**
 * @class Ficus_String
 */
class Ficus_String
{

    /**
     * @var $string string.
     */
    private $string;

    /**
     * Constructor.
     *
     * @param $string string.
     */
    public function __construct($string) {
        $this->string = $string;
    }

    /**
     * Contains str.
     *
     * @param $str string needle.
     * @return boolean if $str contains this string.
     */
    public function contains($str) {
        return $this->strstr($str) != FALSE;
    }

    /**
     * To string.
     *
     * @return string this string.
     */
    public function toString() {
        return $this->string;
    }

    /**
     * To string.
     *
     * @return string this string.
     */
    public function __toString() {
        return $this->string;
    }

    /**
     * equals
     * @param $string Ficus_String right
     * @return boolean true if equals
     */
    public function equals($right){
        if(is_object($right) == false){
            $string = (string)$right;
        }else if($right instanceof Ficus_String){
            $string = $right->string;
        }else if(is_object($right) && method_exists($right, "__toString")){
            $string = $right->__toString();
        }else{
            throw new Ficus_IllegalArgumentException("right object is not able to convert to string.");
        }
        return ($this->string == $string);
    }         
    
    /**
     * starts with
     * @param $pattern string
     */
    public function startsWith($pattern){
        return (strpos($this->string, $pattern) === 0);
    }

    /**
     * ends with
     * @param $pattern string
     */
    public function endsWith($pattern){
        $plength = strlen($pattern);
        $index = strrpos($this->string, $pattern);
        if($index === false){
            return false;
        }
        if(strlen($this->string) - $plength == $index){
            return true;
        }
        return false;
    }

    /**
     * make lowercase first caractor
     * @return String processed string
     */
    public function lowerCaseFirst(){
        $first = substr($this->string, 0, 1);
        return new Ficus_String(strtolower($first) . substr($this->string, 1));
    }
           
    /**
     * Copy.
     *
     * @return Ficus_String clone myself.
     */
    public function copy() {
        return new Ficus_String($this->string);
    }

    /**
     * Call PHP strings function.
     *
     * @param $name string function name.
     * @param $arguments function arguments.
     * @return mixed function return value.
     * @throw Ficus_MethodNotFoundException method not found.
     */
    public function __call($name, $arguments) {
        if (substr($name, -1) == '_') {
            $strip = true;
            $name = substr($name, 0, -1);
        } else {
            $strip = false;
        }
        $ret = $this->call($name, $arguments);
        if (!$strip && is_string($ret)) {
            return new Ficus_String($ret);
        } else {
            return $ret;
        }
    }

    /**
     * Call PHP strings function.
     *
     * @param $name string function name.
     * @param $arguments function arguments.
     * @return mixed function return value.
     * @throw Ficus_MethodNotFoundException method not found.
     */
    protected function call($name, $arguments) {
        switch ($name) {

            case 'addcslashes':
            case 'addslashes':
            case 'bin2hex':
            case 'chop':
            case 'chunk_split':
            case 'convert_cyr_string':
            case 'convert_uudecode':
            case 'convert_uuencode':
            case 'count_chars':
            case 'crc32':
            case 'crypt':
            case 'hebrev':
            case 'hebrevc':
            case 'html_entity_decode':
            case 'htmlentities':
            case 'htmlspecialchars_decode':
            case 'htmlspecialchars':
            case 'levenshtein':
            case 'ltrim':
            case 'md5_file':
            case 'md5':
            case 'metaphone':
            case 'money_format':
            case 'nl2br':
            case 'number_format':
            case 'ord':
            case 'parse_str':
            case 'quoted_printable_decode':
            case 'QuoteMeta':
            case 'rtrim':
            case 'sha1':
            case 'similar_text':
            case 'soundex':
            case 'sscanf':
            case 'str_pad':
            case 'str_repeat':
            case 'str_rot13':
            case 'str_shuffle':
            case 'str_split':
            case 'str_word_count':
            case 'strcasecmp':
            case 'strchr':
            case 'strcmp':
            case 'strcoll':
            case 'strcspn':
            case 'strip_tags':
            case 'stripcslashes':
            case 'stripos':
            case 'stripslashes':
            case 'stristr':
            case 'strlen':
            case 'strnatcasecmp':
            case 'strnatcmp':
            case 'strncasecmp':
            case 'strncmp':
            case 'strpbrk':
            case 'strpos':
            case 'strrchr':
            case 'strrev':
            case 'strripos':
            case 'strrpos':
            case 'strspn':
            case 'strstr':
            case 'strtok':
            case 'strtolower':
            case 'strtoupper':
            case 'strtr':
            case 'substr_compare':
            case 'substr_count':
            case 'substr_replace':
            case 'substr':
            case 'trim':
            case 'ucfirst':
            case 'ucwords':
            case 'wordwrap':
            if (is_array($arguments)) {
                $argumentsWithString = array_merge(array($this->string), $arguments);
            } else {
                $argumentsWithString = array($this->string);
            }
            return call_user_func_array($name, $argumentsWithString);
            break;

            // this string is second argument.
            case 'explode':
            $arrangedArguments = array();
            $arrangedArguments []= $arguments[0];
            $arrangedArguments []= $this->string;
            $arrangedArguments += array_slice($arguments, 1);
            return call_user_func_array($name, $arrangedArguments);
            break;

            // this string is third argument.
            case 'str_ireplace':
            case 'str_replace':
            $arrangedArguments = array();
            $arrangedArguments += array_slice($arguments, 0, 2);
            $arrangedArguments []= $this->string;
            $arrangedArguments += array_slice($arguments, 2);
            return call_user_func_array($name, $arrangedArguments);
            break;

            //case 'chr': // string chr(int ascii)
            //case 'fprintf':
            //case 'get_html_translation_table':
            //case 'implode':
            //case 'join':
            //case 'localeconv':
            //case 'nl_langinfo':
            //case 'printf':
            //case 'setlocale':
            //case 'sha1_file':
            //case 'sprintf':
            //case 'vfprintf':
            //case 'vprintf':
            //case 'vsprintf':
            //case 'echo':
            //case 'print':

            default:
            break;

        }
        throw new Ficus_MethodNotFoundException($name);
    }

    /**
     * convert uri to html link
     * @param $string string to convert
     * @return string converted string
     */
    public static function convertLink($string){
        return ereg_replace(
                 "(https?|ftp|file)(://[[:alnum:]\+\$\;\?\.%,!#~*/:@&=_-]+)",
                 "<a href=\"\\1\\2\" target=\"_blank\">\\1\\2</a>", $string);
    }

    /**
     * convert mail address to mailto link
     * @param $string string address
     * @return string converted string
     */
    public static function convertMailAddress($string){
        if(preg_match('/(.+) <(.*@.*)>/', $string, $regs)){
            return '<a href="mailto:' . $regs[2] . '">' . $regs[1] . "</a> " .  '&lt;<a href="mailto:' . $regs[2] . '">' . $regs[2] . "</a>&gt;";;
        }else if(preg_match('/<(.*@.*)>/', $string, $regs)){
            return '&lt;<a href="mailto:' . $regs[1] . '">' . $regs[1] . "</a>&gt;";
        }
    }
}
?>
