<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages
 */
/**
 * @file PageLoader.php
 * @brief controll page
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: PageLoader.php 2 2007-07-11 10:37:48Z ishitoya $
 */

require_once("ficus/pages/ConcretePage.php");
require_once("ficus/pages/PageModeExtractor.php");
require_once("ficus/lang/ClassLoader.php");
require_once("ficus/exception/ClassNotFoundException.php");
require_once("ficus/exception/PageNotFoundException.php");
require_once("ficus/lang/Class.php");
require_once("ficus/lang/String.php");

/**
 * @class Ficus_PageLoader
 */
class Ficus_PageLoader{
    const PAGES_DIR = "pages";
    const PAGE_SUFFIX = "Page";

    /**
     * page dir
     */
    protected $pageDir = null;
    /**
     * default page
     */
    protected $defaultPage = null;
    /**
     * prefixes
     */
    protected $prefixes = array();
    /**
     * suffix
     */
    protected $suffix = self::PAGE_SUFFIX;

    /**
     * set base dir
     */
    public function setBaseDir($base){
        $this->setPageDir($base . "/" . self::PAGES_DIR);        
    }

    /**
     * set default page
     * @param $pagename string default page
     */
    public function setDefaultPage($pagename){
        $this->defaultPage = $pagename;
    }

    /**
     * set prefixes
     * @param $prefixes array page class prefixes
     */
    public function setPrefixes($prefixes){
        if(is_string($prefixes)){
            $prefixes = array($prefixes);
        }
        $this->prefixes = $prefixes;
    }

    /**
     * set suffix
     * @param $suffix String to set
     */
    public function setSuffix($suffix){
        $this->suffix = $suffix;
    }
    
    /**
     * load page
     * @param $request array of request
     * @return string name of found page
     */
    public function dispatch($request){
        $request = $request->requests();
        foreach($request as $key => $value){
            if(empty($value)){
                try{
                    $page = $this->load($key);
                    return $key;
                }catch(Ficus_PageNotFoundException $e){
                }
            }
        }
        return $this->defaultPage;
    }
    
    /**
     * get class name
     * @param $prefix String prefix
     * @param $name String name of page
     * @return string classname
     */
    protected function getClassName($prefix, $name){
        $class = new Ficus_Class($name);
        $qname = $class->qname($prefix);
        return $qname . $this->getSuffix();
    }

    /**
     * load page
     * @param $name string classname of page
     * @return Ficus_Page instance of page
     */
    public function load($name){
        if(empty($name)){
            throw new Ficus_IllegalArgumentException("template name is empty");
        }
        
        $name = str_replace("_", ".", $name);
        foreach($this->prefixes as $prefix){
            $classname = $this->getClassName($prefix, $name);
            try{
                $class = new Ficus_Class($classname);
                $package = $class->package();
                $page = Ficus_ClassLoader::load($classname,
                                               array($package, $name));
            }catch(Ficus_ClassNotFoundException $e){
                continue;
            }
            if($page instanceof Ficus_Page){
                return $page;
            }
        }
        $loader = Ficus_PageComponentFactory::getTemplateLoader();
        $template = $loader->load($name);
        $fullpath = $loader->getFullPath($template);
        if(is_file($fullpath)){
            return new Ficus_ConcretePage($template, $name);
        }
            
        throw new Ficus_PageNotFoundException("page, $name is not found in class path");
    }
        
    /**
     * set Pages dir
     * @param $pageDir string page dir
     */
    protected function setPageDir($pageDir){
        $this->pageDir = $pageDir;
        Ficus_ClassPath::add(new Ficus_ClassPathElement($pageDir), "pages");
        Ficus_FicusPathInitializer::addIncludePath($pageDir);
    }

    /**
     * get suffix
     * @return string suffix of the page
     */
    public function getSuffix(){
        return $this->suffix;
    }

    /**
     * get Page name
     * @param $page Ficus_Page
     * @return string page name
     */
    public function getPageName($page){
        $classname = get_class($page);
        $pattern = '/(.+?)_([^_]+)' . $this->getSuffix() . '/';
        if(preg_match($pattern, $classname, $regs)){
           return $regs[2];
        }
        return "";
    }
}
?>