/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Watanabe Lab, School of Media Science,
 *	Tokyo University of Technology, All rights reserved.
 *
 *	Redistribution and use in source and binary forms,
 *	with or without modification, are permitted provided that the
 *	following conditions are met:
 *
 *		- Redistributions of source code must retain the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer.
 *
 *		- Redistributions in binary form must reproduce the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer in the documentation and/or
 *			other materials provided with the distribution.
 *
 *		- Neither the name of the copyright holders nor the names
 *			of its contributors may be used to endorse or promote
 *			products derived from this software without specific
 *			prior written permission.
 *
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 *	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *	POSSIBILITY OF SUCH DAMAGE. 
 *
 ****************************************************************************/
/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Watanabe Lab, School of Media Science,
 *	Tokyo University of Technology, All rights reserved.
 *
 *	本ソフトウェアおよびソースコードのライセンスは、基本的に
 *	「修正 BSD ライセンス」に従います。以下にその詳細を記します。
 *
 *	ソースコード形式かバイナリ形式か、変更するかしないかを問わず、
 *	以下の条件を満たす場合に限り、再頒布および使用が許可されます。
 *
 *	- ソースコードを再頒布する場合、上記の著作権表示、本条件一覧、
 *		および下記免責条項を含めること。
 *
 *	- バイナリ形式で再頒布する場合、頒布物に付属のドキュメント等の
 *		資料に、上記の著作権表示、本条件一覧、および下記免責条項を
 *		含めること。
 *
 *	- 書面による特別の許可なしに、本ソフトウェアから派生した製品の
 *		宣伝または販売促進に、本ソフトウェアの著作権者の名前または
 *		コントリビューターの名前を使用してはならない。
 *
 *	本ソフトウェアは、著作権者およびコントリビューターによって「現
 *	状のまま」提供されており、明示黙示を問わず、商業的な使用可能性、
 *	および特定の目的に対する適合性に関す暗黙の保証も含め、またそれ
 *	に限定されない、いかなる保証もないものとします。著作権者もコン
 *	トリビューターも、事由のいかんを問わず、損害発生の原因いかんを
 *	問わず、かつ責任の根拠が契約であるか厳格責任であるか(過失その
 *	他の)不法行為であるかを問わず、仮にそのような損害が発生する可
 *	能性を知らされていたとしても、本ソフトウェアの使用によって発生
 *	した(代替品または代用サービスの調達、使用の喪失、データの喪失、
 *	利益の喪失、業務の中断も含め、またそれに限定されない)直接損害、
 *	間接損害、偶発的な損害、特別損害、懲罰的損害、または結果損害に
 *	ついて、一切責任を負わないものとします。
 *
 ****************************************************************************/
#include <FK/FK.h>

const double	BUILDWIDTH	= 25.0;		// 建物幅の基本単位
const double	SPEED		= 0.5;		// 車のスピード
const double	CIRCUITX	= 150.0;	// コースの X 方向幅
const double	CIRCUITY	= 250.0;	// コースの Y 方向幅
const double	EPS			= 0.001;	// 誤差判定用数値

class Car {
private:
	fk_Model	carModel;		// 車全体モデル
	fk_Model	bodyModel;		// 車体モデル
	fk_Model	tireModel[4];	// 各タイヤモデル
	fk_Model	driverModel[2];	// 運転者モデル
	fk_Model	birdModel;		// 鳥瞰視点モデル

	fk_Block	body;			// 車体形状
	fk_Circle	tire;			// タイヤ形状
	fk_Sphere	driver;			// 運転者形状

public:
	Car(void) {		// コンストラクタ
		init();
	}

	void		init(void);
	void		entryScene(fk_Scene *, bool);	
	fk_Vector	getCarPosition(void);
	fk_Model *	getBirdModel(void);
	void		forward(void);
	void		rotate(fk_Model *, fk_Vector, fk_Vector);
};

class World {
private:
	fk_Model	buildModel[6], groundModel, lightModel[2];
	fk_Block	buildShape, groundShape;
	fk_Light	lightShape;

	void		defLight(int, double, double, double);
	void		makeBuild(int, double, double, double, fk_Material *);

public:
	World(void) {
		init();
	}

	void		init(void);
	void		entryScene(fk_Scene *, bool);
};

int main(int, char *[])
{
	Car				carObj;
	World			worldObj;
	Fl_Window		mainWindow(940, 320, "FK TEST");
	fk_Scene		carViewScene, buildViewScene, birdViewScene;
	fk_Window		carViewWindow(10, 10, 300, 300);
	fk_Window		buildViewWindow(320, 10, 300, 300);
	fk_Window		birdViewWindow(630, 10, 300, 300);
	fk_Color		bgColor(0.2, 0.7, 1.0);

	fk_Model		buildViewModel, birdViewModel;

	mainWindow.end();
	fk_InitMaterial();

	carObj.init();
	worldObj.init();


	// 各ウィンドウにバックグラウンドカラー設定
	carViewScene.setBGColor(bgColor);
	buildViewScene.setBGColor(bgColor);
	birdViewScene.setBGColor(bgColor);


	// 各モデルをディスプレイリストに登録
	worldObj.entryScene(&carViewScene, false);
	worldObj.entryScene(&buildViewScene, true);
	worldObj.entryScene(&birdViewScene, false);

	carObj.entryScene(&carViewScene, false);
	carObj.entryScene(&buildViewScene, true);
	carObj.entryScene(&birdViewScene, true);

	// 建物ウィンドウの視点設定
	buildViewModel.glMoveTo(-250.0, 100.0, 100.0);
	buildViewModel.glFocus(carObj.getCarPosition());
	buildViewModel.glUpvec(0.0, 0.0, 1.0);
	buildViewScene.entryCamera(&buildViewModel);

	// 鳥瞰ウィンドウの視点設定
	birdViewScene.entryCamera(carObj.getBirdModel());

	// ウィンドウへディスプレイリストを登録
	carViewWindow.setScene(&carViewScene);
	buildViewWindow.setScene(&buildViewScene);
	birdViewWindow.setScene(&birdViewScene);

	mainWindow.show();
	carViewWindow.show();
	buildViewWindow.show();
	birdViewWindow.show();

	while(true) {

		if(mainWindow.visible() == 0) {
			if(Fl::wait() == 0) {
				break;
			} else {
				continue;
			}
		}

		if(carViewWindow.drawWindow() == 0 ||
		   buildViewWindow.drawWindow() == 0 ||
		   birdViewWindow.drawWindow() == 0) break;

		if(Fl::check() == 0) break;

		if(carViewWindow.winOpenStatus() == false ||
		   buildViewWindow.winOpenStatus() == false ||
		   birdViewWindow.winOpenStatus() == false) {
			continue;
		}	  

		carObj.forward();
		buildViewModel.glFocus(carObj.getCarPosition());
		buildViewModel.glUpvec(0.0, 0.0, 1.0);
	}
	
	return 0;
}

void Car::init(void)
{
	int		i;

	body.setSize(7.0, 6.0, 20.0);
	tire.setRadius(2.0);
	tire.setDivide(2);
	driver.setRadius(2.0);
	driver.setDivide(2);

	bodyModel.setShape(&body);
	bodyModel.glMoveTo(0.0, 5.0, 0.0);
	bodyModel.setMaterial(Yellow);
	bodyModel.setParent(&carModel);


	tireModel[0].glMoveTo(-4.0, 1.0, -8.0);
	tireModel[0].glVec(1.0, 0.0, 0.0);
	tireModel[1].glMoveTo(4.0, 1.0, -8.0);
	tireModel[1].glVec(-1.0, 0.0, 0.0);
	tireModel[2].glMoveTo(-4.0, 1.0, 8.0);
	tireModel[2].glVec(1.0, 0.0, 0.0);
	tireModel[3].glMoveTo(4.0, 1.0, 8.0);
	tireModel[3].glVec(-1.0, 0.0, 0.0);

	for(i = 0; i < 4; i++) {
		tireModel[i].setShape(&tire);
		tireModel[i].setMaterial(Gray2);
		tireModel[i].setParent(&carModel);
	}

	driverModel[0].setShape(&driver);
	driverModel[1].setShape(&driver);
	driverModel[0].glMoveTo(-2.0, 10.0, 0.0);
	driverModel[1].glMoveTo(2.0, 10.0, 0.0);
	driverModel[0].setMaterial(Cream);
	driverModel[1].setMaterial(Cream);
	driverModel[0].setSmoothMode(true);
	driverModel[1].setSmoothMode(true);
	driverModel[0].setParent(&carModel);
	driverModel[1].setParent(&carModel);

	birdModel.glMoveTo(0.0, 100.0, 200.0);
	birdModel.glFocus(0.0, 5.0, 0.0);
	birdModel.glUpvec(0.0, 1.0, 0.0);
	birdModel.setParent(&carModel);

	carModel.glMoveTo(CIRCUITX, CIRCUITY, 0.0);
	carModel.glVec(0.0, -1.0, 0.0);
	carModel.glUpvec(0.0, 0.0, 1.0);

	return;
}

void Car::entryScene(fk_Scene *scene, bool viewFlag)
{
	int		i;

	scene->entryModel(&bodyModel);

	for(i = 0; i < 4; i++) {
		scene->entryModel(&tireModel[i]);
	}

	scene->entryModel(&driverModel[0]);
	if(viewFlag == true) {
		scene->entryModel(&driverModel[1]);
	} else {
		scene->entryCamera(&driverModel[1]);
	}

	return;
}

fk_Vector Car::getCarPosition(void)
{
	return carModel.getPosition();
}

fk_Model * Car::getBirdModel(void)
{
	return &birdModel;
}

void Car::forward(void)
{
	fk_Vector	carPosition, carVelocity;
	fk_Vector	Xplus(1.0, 0.0, 0.0), Xminus(-1.0, 0.0, 0.0);
	fk_Vector	Yplus(0.0, 1.0, 0.0), Yminus(0.0, -1.0, 0.0);
	double		X = CIRCUITX;
	double		Y = CIRCUITY;

	carPosition = carModel.getPosition();
	carVelocity = carModel.getVec();
	carModel.loTranslate(0.0, 0.0, -SPEED); // 前進

	// サーキットの外にでた場合、回転する。
	if(carPosition.x > X) {
		rotate(&carModel, carVelocity, Xplus);
	}
	if(carPosition.x < -X) {
		rotate(&carModel, carVelocity, Xminus);
	}
	if(carPosition.y > Y) {
		rotate(&carModel, carVelocity, Yplus);
	}
	if(carPosition.y < -Y) {
		rotate(&carModel, carVelocity, Yminus);
	}

	return;
}

void Car::rotate(fk_Model *model, fk_Vector velocity, fk_Vector orgVec)
{
	// velocity と orgVec の内積値が正、つまり角度が 90 度以内の場合回転
	if(velocity * orgVec > EPS) {
		model->loAngle(FK_PI/400.0, 0.0, 0.0);
	}
	return;
}

void World::defLight(int lightID, double x, double y, double z)
{
	lightModel[lightID].setShape(&lightShape);
	lightModel[lightID].setMaterial(White);
	lightModel[lightID].glTranslate(0.0, 0.0, 0.0);
	lightModel[lightID].glFocus(x, y, z);

	return;
}

void World::makeBuild(int buildID, double x, double y,
					  double heightScale, fk_Material *buildMat)
{
	buildModel[buildID].setShape(&buildShape);
	buildModel[buildID].setScale(heightScale, fk_Z);
	buildModel[buildID].glMoveTo(x, y, (BUILDWIDTH * heightScale)/2.0);
	buildModel[buildID].setMaterial(*buildMat);

	return;
}

void World::init(void)
{
	// 照明の設定
	defLight(0, 1.0, 1.0, -1.0);
	defLight(1, -1.0, -1.0, -1.0);

	// 建物の設定
	buildShape.setSize(BUILDWIDTH, BUILDWIDTH, BUILDWIDTH);
	makeBuild(0, -250.0, 100.0, 5.0, &Red);
	makeBuild(1, -150.0, 400.0, 2.0, &DimYellow);
	makeBuild(2, 50.0, 250.0, 4.0, &Blue); 
	makeBuild(3, 300.0, 200.0, 3.0, &Gray1); 
	makeBuild(4, 250.0, -250.0, 0.5, &Green);
	makeBuild(5, -50.0, -350.0, 6.0, &Orange);

	// 地面の設定
	groundShape.setSize(1000, 1000, 2.0);
	groundModel.setShape(&groundShape);
	groundModel.glTranslate(0.0, 0.0, -1.0);
	groundModel.setMaterial(Brown);

	return;
}

void World::entryScene(fk_Scene *scene, bool buildFlag)
{
	int		i;

	scene->entryModel(&groundModel);

	for(i = 0; i < 2; i++) {
		scene->entryModel(&lightModel[i]);
	}

	for(i = 0; i < 6; i++) {
		scene->entryModel(&buildModel[i]);
	}

	if(buildFlag == true) scene->removeModel(&buildModel[0]);
	return;
}
