/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Watanabe Lab, School of Media Science,
 *	Tokyo University of Technology, All rights reserved.
 *
 *	Redistribution and use in source and binary forms,
 *	with or without modification, are permitted provided that the
 *	following conditions are met:
 *
 *		- Redistributions of source code must retain the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer.
 *
 *		- Redistributions in binary form must reproduce the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer in the documentation and/or
 *			other materials provided with the distribution.
 *
 *		- Neither the name of the copyright holders nor the names
 *			of its contributors may be used to endorse or promote
 *			products derived from this software without specific
 *			prior written permission.
 *
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 *	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *	POSSIBILITY OF SUCH DAMAGE. 
 *
 ****************************************************************************/
/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Watanabe Lab, School of Media Science,
 *	Tokyo University of Technology, All rights reserved.
 *
 *	本ソフトウェアおよびソースコードのライセンスは、基本的に
 *	「修正 BSD ライセンス」に従います。以下にその詳細を記します。
 *
 *	ソースコード形式かバイナリ形式か、変更するかしないかを問わず、
 *	以下の条件を満たす場合に限り、再頒布および使用が許可されます。
 *
 *	- ソースコードを再頒布する場合、上記の著作権表示、本条件一覧、
 *		および下記免責条項を含めること。
 *
 *	- バイナリ形式で再頒布する場合、頒布物に付属のドキュメント等の
 *		資料に、上記の著作権表示、本条件一覧、および下記免責条項を
 *		含めること。
 *
 *	- 書面による特別の許可なしに、本ソフトウェアから派生した製品の
 *		宣伝または販売促進に、本ソフトウェアの著作権者の名前または
 *		コントリビューターの名前を使用してはならない。
 *
 *	本ソフトウェアは、著作権者およびコントリビューターによって「現
 *	状のまま」提供されており、明示黙示を問わず、商業的な使用可能性、
 *	および特定の目的に対する適合性に関す暗黙の保証も含め、またそれ
 *	に限定されない、いかなる保証もないものとします。著作権者もコン
 *	トリビューターも、事由のいかんを問わず、損害発生の原因いかんを
 *	問わず、かつ責任の根拠が契約であるか厳格責任であるか(過失その
 *	他の)不法行為であるかを問わず、仮にそのような損害が発生する可
 *	能性を知らされていたとしても、本ソフトウェアの使用によって発生
 *	した(代替品または代用サービスの調達、使用の喪失、データの喪失、
 *	利益の喪失、業務の中断も含め、またそれに限定されない)直接損害、
 *	間接損害、偶発的な損害、特別損害、懲罰的損害、または結果損害に
 *	ついて、一切責任を負わないものとします。
 *
 ****************************************************************************/
#include <FK/D3DXShapeParser.h>
#include <FK/D3DXData.h>
#include <FK/IFSTexture.h>
#include <FK/Error.h>

using namespace std;

fk_D3DXShapeParser::fk_D3DXShapeParser(void)
{
	Clear();
	return;
}

fk_D3DXShapeParser::~fk_D3DXShapeParser()
{
	return;
}

void fk_D3DXShapeParser::Clear(void)
{
	vData.clear(); optVData.clear();
	fData.clear(); optFData.clear();
	tData.clear(); optTData.clear();
	mData.clear();
	vMapData.clear();

	return;
}

bool fk_D3DXShapeParser::ReadMeshData(fk_TreeData *argData, FILE *argFP)
{
	if(SetFrameMatrix(argData) == false) return false;
	if(ReadVectorData(argFP, FK_D3DX_V_MODE) == false) return false;
	if(ReadFaceData(argFP) == false) return false;
	return true;
}

bool fk_D3DXShapeParser::SetFrameMatrix(fk_TreeData *argData)
{
	fk_TreeData				*parent = argData->getParent();
	fk_D3DXPropertyList		*prop;

	if(parent == NULL) return false;
	prop = (fk_D3DXPropertyList *)parent->getObject();
	if(prop == NULL) return false;

	frameMatrix = *(prop->GetFrameMatrix());
	//if(frameMatrix.inverse() == false) return false;

	return true;
}

bool fk_D3DXShapeParser::ReadVectorData(FILE *argFP, fk_D3DX_VecMode argMode)
{
	string				word, line;
	int					i, vNum;
	fk_Vector			tmpVec;
	fk_TexCoord			tmpTex;

	word.clear();
	while(word.size() == 0) {
		if(ReadLine(argFP, &line) == false) return false;
		word = PopWord(&line);
	}

	word.erase(word.size()-1);
	if(IsInteger(word) == false) return false;
	vNum = Str2Int(word);

	if(argMode == FK_D3DX_V_MODE) {
		for(i = 0; i < vNum; i++) {
			if(GetVector(argFP, &line, &tmpVec) == false) return false;
			//vData.push_back(frameMatrix * tmpVec);
			vData.push_back(tmpVec);
		}
	} else {
		for(i = 0; i < vNum; i++) {
			if(GetTexCoord(argFP, &line, &tmpTex) == false) return false;
			tData.push_back(tmpTex);
		}
	}

	return true;
}

bool fk_D3DXShapeParser::GetVector(FILE *argFP,
								   string *argLine, fk_Vector *argVec)
{
	string		word;
	fk_Vector	vec;

	if(GetWord(argFP, ";,", argLine, &word) == false) return false;
	if(IsNumeric(word) == false) return false;
	argVec->x = Str2Double(word);

	if(GetWord(argFP, ";,", argLine, &word) == false) return false;
	if(IsNumeric(word) == false) return false;
	argVec->y = Str2Double(word);

	if(GetWord(argFP, ";,", argLine, &word) == false) return false;
	if(IsNumeric(word) == false) return false;
	argVec->z = Str2Double(word);

	return true;
}

bool fk_D3DXShapeParser::GetTexCoord(FILE *argFP, string *argLine,
								fk_TexCoord *argTexCoord)
{
	string		word;
	fk_TexCoord	coord;

	if(GetWord(argFP, ";,", argLine, &word) == false) return false;
	if(IsNumeric(word) == false) return false;
	argTexCoord->x = float(Str2Double(word));

	if(GetWord(argFP, ";,", argLine, &word) == false) return false;
	if(IsNumeric(word) == false) return false;
	argTexCoord->y = float(1.0 - Str2Double(word));

	return true;
}

bool fk_D3DXShapeParser::ReadFaceData(FILE *argFP)
{
	int					i, fNum, fCount, vNum;
	string				word, line, lineList;
	vector<int>			vIDArray;

	word.clear();
	while(word.size() == 0) {
		if(ReadLine(argFP, &line) == false) return false;
		word = PopWord(&line);
	}

	word.erase(word.size()-1);
	if(IsInteger(word) == false) return false;
	fNum = Str2Int(word);

	fCount = 0;
	lineList.clear();
	while(fCount < fNum) {
		word.clear();
		while(word.size() == 0) {
			if(AddLineList(argFP, &lineList) == false) return false;
			word = PopWord(&lineList, ";,");
		}

		if(IsInteger(word) == false) return false;
		vNum = Str2Int(word);
		if(vNum != 3 && vNum != 4) return false;

		lineList.erase(0, 1);

		vIDArray.clear();
		for(i = 0; i < vNum; i++) {
			word = PopWord(&lineList, ";,");
			while(word.size() == 0) {
				if(AddLineList(argFP, &lineList) == false) return false;
				word = PopWord(&lineList, ";,");
			}
			if(IsInteger(word) == false) return false;
			vIDArray.push_back(Str2Int(word)+1);
		}

		fData.push_back(vIDArray);
		lineList.erase(0, 1);
		fCount++;
	}

	return true;
}

bool fk_D3DXShapeParser::ReadMaterialData(FILE *argFP)
{
	string		word, lineList;
	int			i, matMaxID, fNum, matID;

	lineList.clear();
	word.clear();

	if(GetWord(argFP, ";,", &lineList, &word) == false) return false;
	if(IsNumeric(word) == false) return false;
	matMaxID = Str2Int(word) - 1;

	if(GetWord(argFP, ";,", &lineList, &word) == false) return false;
	if(IsNumeric(word) == false) return false;
	fNum = Str2Int(word);

	mData.clear();
	for(i = 0; i < fNum; i++) {
		if(GetWord(argFP, ";,", &lineList, &word) == false) return false;
		if(IsNumeric(word) == false) return false;
		matID = Str2Int(word);
		if(matID < 0 || matID > matMaxID) return false;
		mData.push_back(matID);
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////

void fk_D3DXShapeParser::SetQuadFace(int argIndex)
{
	vector<int>		tmpIDArray1, tmpIDArray2;
	
	tmpIDArray1.push_back(fData[argIndex][0]);
	tmpIDArray1.push_back(fData[argIndex][1]);
	tmpIDArray1.push_back(fData[argIndex][2]);

	tmpIDArray2.push_back(fData[argIndex][0]);
	tmpIDArray2.push_back(fData[argIndex][2]);
	tmpIDArray2.push_back(fData[argIndex][3]);

	optFData.push_back(tmpIDArray1);
	optFData.push_back(tmpIDArray2);

	return;
}

void fk_D3DXShapeParser::SetIFSTexCoord(fk_IFSTexture *argIFST)
{
	int		i, j, index;

	for(i = 0; i < int(optFData.size()); i++) {
		for(j = 0; j < 3; j++) {
			index = optFData[i][j] - 1;
			argIFST->setTextureCoord(i, j, optTData[index]);
		}
	}
	return;
}

void fk_D3DXShapeParser::OptimizeData(int argMateID)
{
	unsigned int		i, j;
	bool				texDataFlg;
	vector<bool>		tmpArray;
	int					index, count;
	string				outStr;

	optVData.clear();
	optFData.clear();
	optTData.clear();

	count = 0;
	vMapData.clear();

	if(mData.size() == fData.size()) texDataFlg = true;
	else texDataFlg = false;

	for(i = 0; i < fData.size(); i++) {
		if(texDataFlg == true && argMateID >= 0) {
			if(mData[i] != argMateID) continue;
		}

		if(fData[i].size() == 3) {
			optFData.push_back(fData[i]);
		} else {
			SetQuadFace(i);
		}
	}

	tmpArray.resize(vData.size());
	for(i = 0; i < tmpArray.size(); i++) tmpArray[i] = false;

	for(i = 0; i < optFData.size(); i++) {
		for(j = 0; j < optFData[i].size(); j++) {
			index = optFData[i][j]-1;
			if(index < 0 || index >= int(tmpArray.size())) return;
			tmpArray[index] = true;
		}
	}

	vMapData.resize(tmpArray.size());
	for(i = 0; i < tmpArray.size(); i++) {
		if(tmpArray[i] == true) {
			vMapData[i] = count;
			optVData.push_back(vData[i]);
			optTData.push_back(tData[i]);
			count++;
		} else {
			vMapData[i] = -1;
		}
	}

	for(i = 0; i < optFData.size(); i++) {
		for(j = 0; j < optFData[i].size(); j++) {
			index = optFData[i][j]-1;
			optFData[i][j] = vMapData[index]+1;
		}
	}

	return;
}

bool fk_D3DXShapeParser::MakeMesh(fk_ParserData *argMesh, bool argFlg)
{
	argMesh->Init();
	return argMesh->MakeMesh(&optVData, &optFData, NULL, argFlg);
}

int fk_D3DXShapeParser::GetVMap(int argIndex)
{
	if(argIndex < 0 || argIndex >= int(vMapData.size())) return -1;
	return vMapData[argIndex];
}

int fk_D3DXShapeParser::GetOptVSize(void)
{
	return int(optVData.size());
}

int fk_D3DXShapeParser::GetOrgVSize(void)
{
	return int(vData.size());
}

void fk_D3DXShapeParser::Print(void)
{
	unsigned int		i, j;
	string				outStr;

	fk_Printf("vsize = %d", optVData.size());
	for(i = 0; i < optVData.size(); i++) {
		fk_Printf("v[%d]\t= (%f, %f, %f)",
				  i, optVData[i].x, optVData[i].y, optVData[i].z);
	}

	fk_Printf("fSize = %d", optFData.size());
	for(i = 0; i < optFData.size(); i++) {
		outStr = fk_StrPrintf("f[%d]\t= (", i);
		for(j = 0; j < optFData[i].size(); j++) {
			outStr += fk_StrPrintf("%d", optFData[i][j]);
			if(j == optFData[i].size()-1) {
				outStr += ")";
			} else {
				outStr += ", ";
			}
		}
		fk_PutError(outStr);
	}

	fk_Printf("tSize = %d", optTData.size());
	for(i = 0; i < optTData.size(); i++) {
		fk_Printf("t[%d]\t= (%f, %f)", i, optTData[i].x, optTData[i].y);
	}

	fk_Printf("mSize = %d", mData.size());
	for(i = 0; i < mData.size(); i++) {
		fk_Printf("m[%d]\t = %d", i, mData[i]);
	}

	return;
}
